﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>

#include <nn/fssystem/fs_SubdirectoryFileSystem.h>
#include <nn/fssystem/fs_PathTool.h>


namespace nn { namespace fssystem {

    Result SubdirectoryFileSystem::Initialize(const char* pRootPath) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(
            strnlen(pRootPath, EntryNameLengthMax + 1) <= EntryNameLengthMax,
            nn::fs::ResultTooLongPath());

        char pathWorkBuffer[EntryNameLengthMax + 1 + 1]; // '/' + '\0'

        size_t normalizedPathLength;
        nn::Result result = PathTool::Normalize(pathWorkBuffer, &normalizedPathLength, pRootPath, sizeof(pathWorkBuffer), m_IsUncPreserved);
        if(result.IsFailure())
        {
            NN_SDK_ASSERT(false);
            return result;
        }

        if( pathWorkBuffer[normalizedPathLength - 1] != '/' )
        {
            NN_SDK_ASSERT_LESS_EQUAL(normalizedPathLength + 2, sizeof(pathWorkBuffer));
            strncat(pathWorkBuffer, "/", 2); // m_RootPath は必ずスラッシュ終わりとする
            pathWorkBuffer[sizeof(pathWorkBuffer) - 1] = '\0';
            normalizedPathLength++;
        }

        m_RootPathBufferLength = normalizedPathLength + 1;
        m_RootPath = reinterpret_cast<char*>(nn::fs::detail::Allocate(m_RootPathBufferLength));
        if( m_RootPath == nullptr ) // TODO: Initialize() に分離してエラーを返す等
        {
            return nn::fs::ResultAllocationMemoryFailedInSubdirectoryFileSystemA();
        }

        strncpy(m_RootPath, pathWorkBuffer, m_RootPathBufferLength);
        m_RootPath[m_RootPathBufferLength - 1] = '\0';
        NN_RESULT_SUCCESS;
    }

    SubdirectoryFileSystem::SubdirectoryFileSystem(nn::fs::fsa::IFileSystem* pBaseFileSystem, const char* pRootPath) NN_NOEXCEPT
        : m_pBaseFileSystem(pBaseFileSystem),
          m_IsUncPreserved(false)
    {
        Result result = Initialize(pRootPath);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
    }

    SubdirectoryFileSystem::SubdirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* pRootPath) NN_NOEXCEPT
      : m_pBaseFileSystem(pBaseFileSystem.get()),
        m_pSharedBaseFileSystem(std::move(pBaseFileSystem)),
        m_IsUncPreserved(false)
    {
        Result result = Initialize(pRootPath);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
    }

    SubdirectoryFileSystem::SubdirectoryFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* pRootPath, bool isUncPreserved) NN_NOEXCEPT
      : m_pBaseFileSystem(pBaseFileSystem.get()),
        m_pSharedBaseFileSystem(std::move(pBaseFileSystem)),
        m_IsUncPreserved(isUncPreserved)
    {
        Result result = Initialize(pRootPath);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
    }

    SubdirectoryFileSystem::~SubdirectoryFileSystem() NN_NOEXCEPT
    {
        if( m_RootPath != nullptr )
        {
            nn::fs::detail::Deallocate(m_RootPath, m_RootPathBufferLength);
        }
    }

    Result SubdirectoryFileSystem::ResolveFullPath(char* outPath, size_t outPathSize, const char* relativePath) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_RootPathBufferLength <= outPathSize);
        NN_RESULT_THROW_UNLESS(
            m_RootPathBufferLength + strnlen(relativePath, EntryNameLengthMax + 1) <= outPathSize,
            nn::fs::ResultTooLongPath());

        strncpy(outPath, m_RootPath, outPathSize);
        outPath[outPathSize - 1] = '\0';

        size_t normalizedPathLength;
        NN_SDK_ASSERT(m_RootPathBufferLength >= 2); // '/' + '\0'
        NN_RESULT_DO(PathTool::Normalize(outPath + m_RootPathBufferLength - 2, &normalizedPathLength, relativePath, outPathSize - (m_RootPathBufferLength - 2), m_IsUncPreserved));

        NN_RESULT_SUCCESS;
    }

    Result SubdirectoryFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->GetEntryType(outValue, fullPath);
    }

    Result SubdirectoryFileSystem::DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->GetFreeSpaceSize(outValue, fullPath);
    }

    Result SubdirectoryFileSystem::DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->GetTotalSpaceSize(outValue, fullPath);
    }

    Result SubdirectoryFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->GetFileTimeStampRaw(outTimeStamp, fullPath);
    }

    Result SubdirectoryFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->OpenFile(outValue, fullPath, mode);
    }

    Result SubdirectoryFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->OpenDirectory(outValue, fullPath, mode);
    }

    Result SubdirectoryFileSystem::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->CreateFile(fullPath, size, option);
    }

    Result SubdirectoryFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->DeleteFile(fullPath);
    }

    Result SubdirectoryFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->CreateDirectory(fullPath);
    }

    Result SubdirectoryFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->DeleteDirectory(fullPath);
    }

    Result SubdirectoryFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->DeleteDirectoryRecursively(fullPath);
    }

    Result SubdirectoryFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->CleanDirectoryRecursively(fullPath);
    }

    Result SubdirectoryFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        char currentFullPath[EntryNameLengthMax + 1];
        char newFullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(currentFullPath, sizeof(currentFullPath), currentPath));
        NN_RESULT_DO(ResolveFullPath(newFullPath,     sizeof(newFullPath),     newPath));

        return m_pBaseFileSystem->RenameFile(currentFullPath, newFullPath);
    }

    Result SubdirectoryFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        char currentFullPath[EntryNameLengthMax + 1];
        char newFullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(currentFullPath, sizeof(currentFullPath), currentPath));
        NN_RESULT_DO(ResolveFullPath(newFullPath,     sizeof(newFullPath),     newPath));

        return m_pBaseFileSystem->RenameDirectory(currentFullPath, newFullPath);
    }

    Result SubdirectoryFileSystem::DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT
    {
        char fullPath[EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPath, sizeof(fullPath), path));

        return m_pBaseFileSystem->QueryEntry(outBuffer, outBufferSize, inBuffer, inBufferSize, queryId, fullPath);
    }

    Result SubdirectoryFileSystem::DoCommit() NN_NOEXCEPT
    {
        return m_pBaseFileSystem->Commit();
    }

    Result SubdirectoryFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
    {
        return m_pBaseFileSystem->CommitProvisionally(counter);
    }

    Result SubdirectoryFileSystem::DoRollback() NN_NOEXCEPT
    {
        return m_pBaseFileSystem->Rollback();
    }

}}

