﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_SparseStorage.h>
#include <nn/fssystem/fs_IndirectStorageImpl.h>

namespace nn { namespace fssystem {

/**
 * @brief   ストレージを読み込みます。
 *
 * @param[in]   offset  オフセット
 * @param[in]   buffer  バッファの先頭アドレス
 * @param[in]   size    サイズ
 *
 * @return  関数の実行結果を返します。
 */
Result SparseStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
    NN_SDK_REQUIRES(IsInitialized());

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    // エントリが無い場合は 0 埋めする（ZeroStorage 側を読み込む処理と同等）
    if( GetEntryTable().IsEmpty() )
    {
        NN_RESULT_THROW_UNLESS(GetEntryTable().IsInclude(offset, size), fs::ResultOutOfRange());

        std::memset(buffer, 0, size);
    }
    else
    {
        NN_RESULT_DO(OperatePerEntry<false>(
            offset,
            size,
            [=](fs::IStorage* pStorage, int64_t dataOffset, int64_t processOffset, int64_t processSize) NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(pStorage->Read(
                    dataOffset,
                    reinterpret_cast<char*>(buffer) + (processOffset - offset),
                    size_t(processSize)
                ));
                NN_RESULT_SUCCESS;
            }
        ));
    }

    NN_RESULT_SUCCESS;
}

}}
