﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstring>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/os/os_Mutex.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_HtcFileSystem.h>
#include <nn/fssystem/fs_Utility.h>
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
#include <nn/fs/fs_QueryRange.h>
#endif

#include <nn/htcfs.h>

using namespace nn::fs;
using namespace nn::fs::detail;

namespace nn { namespace fssystem {

namespace {
    bool ExistsEntry(const char* path) NN_NOEXCEPT
    {
        {
            auto exists = false;
            const auto result = nn::htcfs::FileExists(&exists, path);
            if( result.IsSuccess() && exists )
            {
                return true;
            }
        }
        {
            auto exists = false;
            const auto result = nn::htcfs::DirectoryExists(&exists, path);
            if( result.IsSuccess() && exists )
            {
                return true;
            }
        }
        return false;
    }

}

class HtcFileSystemFile: public fsa::IFile, public fs::detail::Newable
{
public:
    NN_IMPLICIT HtcFileSystemFile(nn::htcfs::FileHandle handle, nn::fs::OpenMode mode) NN_NOEXCEPT
        : m_Handle(handle)
        , m_Mode(mode)
    {
    }

    virtual ~HtcFileSystemFile() NN_NOEXCEPT
    {
        nn::htcfs::CloseFile(m_Handle);
    }

    virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        size_t readSize = 0;
        NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));
        return nn::htcfs::ReadFile(outValue, buffer, m_Handle, offset, readSize, option);
    }

    virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        if( offset < 0 )
        {
            return nn::fs::ResultOutOfRange();
        }

        NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Write) != 0, ResultInvalidOperationForOpenMode() );

        if ((m_Mode & nn::fs::OpenMode_AllowAppend) == 0)
        {
            int64_t currentSize;
            NN_RESULT_DO(GetSize(&currentSize));
            NN_FSP_REQUIRES((offset + static_cast<int64_t>(size)) <= currentSize && (offset + static_cast<int64_t>(size)) >= offset,
                ResultFileExtensionWithoutOpenModeAllowAppend(), "OpenMode_AllowAppend is required for implicit extension of file size by WriteFile().");
        }
        return nn::htcfs::WriteFile(buffer, m_Handle, offset, size, option);
    }

    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
    {
        if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(nn::htcfs::FlushFile(m_Handle));
        NN_RESULT_SUCCESS;
    }

    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(DrySetSize(size, m_Mode));
        return nn::htcfs::SetFileSize(size, m_Handle);
    }
    virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::htcfs::GetFileSize(outValue, m_Handle);
    }

    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t,
        int64_t,
        const void*,
        size_t) NN_NOEXCEPT NN_OVERRIDE
    {
        switch( operationId )
        {
        case OperationId::Invalidate:
            NN_RESULT_SUCCESS;
        case OperationId::QueryRange:
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

            reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
            NN_RESULT_SUCCESS;
        default:
            return nn::fs::ResultUnsupportedOperation();
        }
    }

private:
    nn::htcfs::FileHandle m_Handle;
    nn::fs::OpenMode m_Mode;
};

//===========================================================================================================

class HtcFileSystemDirectory : public fsa::IDirectory, public fs::detail::Newable
{
public:

    NN_IMPLICIT HtcFileSystemDirectory(nn::htcfs::DirectoryHandle handle) NN_NOEXCEPT
        : m_Handle(handle)
    {
    }

    virtual ~HtcFileSystemDirectory() NN_NOEXCEPT
    {
        nn::htcfs::CloseDirectory(m_Handle);
    }

    virtual Result DoRead(int64_t *outValue, DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        if( 0 < entryBufferCount )
        {
            return nn::htcfs::ReadDirectory(outValue, entryBuffer, entryBufferCount, m_Handle);
        }
        else
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }
    }

    virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
    {
        return nn::htcfs::GetEntryCount(outValue, m_Handle);
    }

private:
    nn::htcfs::DirectoryHandle m_Handle;
};

//===========================================================================================================

namespace {
    bool g_Initialized = false;
    nn::os::Mutex g_InitializedMutex(false);
}

HtcFileSystem::HtcFileSystem() NN_NOEXCEPT
{
}

HtcFileSystem::~HtcFileSystem() NN_NOEXCEPT
{
}

Result GenerateHtcPath(const char** pPath, const char* path, size_t pathLengthMax) NN_NOEXCEPT
{
    *pPath = path;
    if( path[0] == '/' )
    {
        ++*pPath;
    }

    static const auto NameLengthMax = 255;
    const auto driveLetterLength = IsWindowsDrive(*pPath) ? 2 : 0;
    return VerifyPath(
        *pPath + driveLetterLength,
        static_cast<int>(pathLengthMax - driveLetterLength),
        NameLengthMax);
}

Result GenerateHtcFilePath(const char** pPath, const char* path) NN_NOEXCEPT
{
    static const auto LengthMax = 260 - 1; // MAX_PATH
    return GenerateHtcPath(pPath, path, LengthMax);
}

Result GenerateHtcDirectoryPath(const char** pPath, const char* path) NN_NOEXCEPT
{
    static const auto LengthMax = 248 - 1; // ::CreateDirectory() の最大パス長
    return GenerateHtcPath(pPath, path, LengthMax);
}

Result HtcFileSystem::Initialize() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(g_InitializedMutex);
    if (!g_Initialized)
    {
        nn::htcfs::Initialize();
        g_Initialized = true;
    }
    NN_RESULT_SUCCESS;
}

Result HtcFileSystem::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
{
    NN_UNUSED(option);
    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pPath, path ));
    return nn::htcfs::CreateFile( pPath, size );
}

Result HtcFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
{
    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pPath, path ));
    return nn::htcfs::DeleteFile( pPath );
}

Result HtcFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
{
    const char* pPath;
    NN_RESULT_DO(GenerateHtcDirectoryPath( &pPath, path ));
    return nn::htcfs::CreateDirectory( pPath );
}

Result HtcFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
{
    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pPath, path )); // ファイルパスの最大長まで許容される
    return nn::htcfs::DeleteDirectory( pPath, false );
}

Result HtcFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
{
    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pPath, path )); // ファイルパスの最大長まで許容される
    return nn::htcfs::DeleteDirectory( pPath, true );
}

Result HtcFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    const auto parentLength = strnlen(path, fs::EntryNameLengthMax + 1);
    const auto separatorLength = path[parentLength - 1] == '/' ? 0 : 1;
    const auto childLengthMax = fs::EntryNameLengthMax - parentLength - separatorLength;
    NN_RESULT_THROW_UNLESS(
        parentLength + separatorLength <= fs::EntryNameLengthMax,
        nn::fs::ResultTooLongPath());

    const auto childEntry = fs::detail::MakeUnique<DirectoryEntry>();
    NN_RESULT_THROW_UNLESS(
        childEntry != nullptr,
        nn::fs::ResultAllocationMemoryFailedMakeUnique());

    const auto childPath = fs::detail::MakeUnique<char[]>(fs::EntryNameLengthMax + 1);
    NN_RESULT_THROW_UNLESS(
        childPath != nullptr,
        nn::fs::ResultAllocationMemoryFailedMakeUnique());

    std::strncpy(childPath.get(), path, fs::EntryNameLengthMax);
    if( 0 < separatorLength )
    {
        childPath[parentLength] = '/';
    }
    childPath[parentLength + separatorLength] = '\0';

    for( ; ; )
    {
        std::unique_ptr<fsa::IDirectory> directory;
        NN_RESULT_DO(OpenDirectory(&directory, path, OpenDirectoryMode_All));

        int64_t readCount = 0;
        NN_RESULT_DO(directory->Read(&readCount, childEntry.get(), 1));

        if( readCount == 0 )
        {
            break;
        }

        NN_RESULT_THROW_UNLESS(
            strnlen(childEntry->name, childLengthMax + 1) <= childLengthMax,
            nn::fs::ResultTooLongPath());
        std::strncpy(
            childPath.get() + parentLength + separatorLength,
            childEntry->name,
            childLengthMax);
        childPath[fs::EntryNameLengthMax] = '\0';

        if( childEntry->directoryEntryType == fs::DirectoryEntryType_File )
        {
            NN_RESULT_DO(DeleteFile(childPath.get()));
        }
        else
        {
            NN_SDK_ASSERT_EQUAL(
                childEntry->directoryEntryType,
                fs::DirectoryEntryType_Directory);
            NN_RESULT_DO(DeleteDirectoryRecursively(childPath.get()));
        }
    }

    NN_RESULT_SUCCESS;
}

Result HtcFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
{
    const char* pCurrentPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pCurrentPath, currentPath ));
    const char* pNewPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pNewPath, newPath ));
    NN_RESULT_TRY(nn::htcfs::RenameFile( pCurrentPath, pNewPath ))
        NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
        {
            if( strncmp(pCurrentPath, pNewPath, fs::EntryNameLengthMax) == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            else if( ExistsEntry(pNewPath) )
            {
                return nn::fs::ResultPathAlreadyExists();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result HtcFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
{
    const char* pCurrentPath;
    NN_RESULT_DO(GenerateHtcDirectoryPath( &pCurrentPath, currentPath ));
    const char* pNewPath;
    NN_RESULT_DO(GenerateHtcDirectoryPath( &pNewPath, newPath ));
    NN_RESULT_TRY(nn::htcfs::RenameDirectory( pCurrentPath, pNewPath ))
        NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
        {
            if( strncmp(pCurrentPath, pNewPath, fs::EntryNameLengthMax) == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            else if( ExistsEntry(pNewPath) )
            {
                return nn::fs::ResultPathAlreadyExists();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result HtcFileSystem::DoGetEntryType( DirectoryEntryType* outValue, const char *path ) NN_NOEXCEPT
{
    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath( &pPath, path ));
    return nn::htcfs::GetEntryType( outValue, pPath );
}

Result HtcFileSystem::DoOpenFile(std::unique_ptr<fsa::IFile>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT
{
    nn::htcfs::FileHandle handle;
    NN_FSP_REQUIRES( (mode & (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)) != 0, ResultInvalidArgument(), "OpenMode_Read or OpenMode_Write is required.");

    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath(&pPath, path));
    Result result = nn::htcfs::OpenFile(&handle, pPath, mode);
    if (result.IsSuccess())
    {
        std::unique_ptr<HtcFileSystemFile> file(new HtcFileSystemFile(handle, mode));
        if (!file)
        {
            nn::htcfs::CloseFile(handle);
            return ResultAllocationMemoryFailedInHtcFileSystemA();
        }
        outValue->reset(file.release());
    }

    return result;
}

Result HtcFileSystem::DoOpenDirectory(std::unique_ptr<fsa::IDirectory>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT
{
    const char* pPath;
    NN_RESULT_DO(GenerateHtcDirectoryPath( &pPath, path ));

    nn::htcfs::DirectoryHandle handle;
    nn::Result result = nn::htcfs::OpenDirectory( &handle, pPath, mode );
    if (result.IsSuccess())
    {
        std::unique_ptr<HtcFileSystemDirectory> directory(new HtcFileSystemDirectory(handle));
        if (!directory)
        {
            nn::htcfs::CloseDirectory(handle);
            return ResultAllocationMemoryFailedInHtcFileSystemB();
        }
        outValue->reset(directory.release());
    }

    return result;
}

Result HtcFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
{
    uint64_t createTime = 0;
    uint64_t accessTime = 0;
    uint64_t modifyTime = 0;

    const char* pPath;
    NN_RESULT_DO(GenerateHtcFilePath(&pPath, path));
    NN_RESULT_DO(nn::htcfs::GetFileTimeStamp(&createTime, &accessTime, &modifyTime, pPath));

    outTimeStamp->create = createTime;
    outTimeStamp->access = accessTime;
    outTimeStamp->modify = modifyTime;
    outTimeStamp->isLocalTime = false; // UTC なので false
    NN_RESULT_SUCCESS;
}

Result HtcFileSystem::DoCommit() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

}}
