﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_Utility.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_FROM_WINDOWS_SDK_HEADERS
#include <nn/nn_Windows.h>
#include <winioctl.h>
NN_PRAGMA_POP_WARNINGS

using namespace nn;
using namespace nn::fs;

namespace nn { namespace fssystem {

    namespace
    {
        // TMA から返ってくるシステム時刻(先発グレゴリオ暦) から UNIX 時間への変換オフセット(秒)
        // 先発グレゴリオ暦 : 1601/1/1 0:00 開始
        // UNIX時間         : 1970/1/1 0:00 開始
        static const uint64_t OffsetToConvertToPosixTime = 11644473600;

        Result ConvertLastErrorToResult() NN_NOEXCEPT;

        const int FileSystemNameLengthMax = 7;
        const int PathLengthMax = FileSystemNameLengthMax + 1 + nn::fs::EntryNameLengthMax;

        class HostFile: public nn::fs::fsa::IFile, public nn::fs::detail::Newable
        {
        public:
            NN_IMPLICIT HostFile(HANDLE handle, nn::fs::OpenMode mode) NN_NOEXCEPT
                : m_Handle(handle), m_Mode(mode)
            {
            }

            virtual ~HostFile() NN_NOEXCEPT NN_OVERRIDE
            {
                ::CloseHandle(m_Handle);
            }

            virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

        private:
            const HANDLE m_Handle;
            const nn::fs::OpenMode m_Mode;
        };

        Result HostFile::DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_UNUSED(option);

            NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Read) != 0, nn::fs::ResultInvalidOperationForOpenMode() );

            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            if( buffer == nullptr )
            {
                return nn::fs::ResultNullptrArgument();
            }
            if( offset < 0 )
            {
                return nn::fs::ResultOutOfRange();
            }

            OVERLAPPED overlapped = {0};
            overlapped.OffsetHigh = (offset >> 32) & 0xFFFFFFFF;
            overlapped.Offset = offset & 0xFFFFFFFF;
            overlapped.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
            if( overlapped.hEvent == NULL )
            {
                return nn::fs::ResultUnexpectedInHostFileSystemA();
            }

            NN_UTIL_SCOPE_EXIT
            {
                CloseHandle(overlapped.hEvent);
            };

            DWORD value;
            BOOL success = ::ReadFile(m_Handle, buffer, static_cast<DWORD>(size), &value, &overlapped);

            if (!success)
            {
                int64_t currentSize;
                NN_RESULT_DO(GetSize(&currentSize));
                if( offset > currentSize )
                {
                    return nn::fs::ResultOutOfRange();
                }

                DWORD error = GetLastError();
                switch (error)
                {
                case ERROR_IO_PENDING:
                    {
                        BOOL waitResult = GetOverlappedResult(m_Handle, &overlapped, &value, true);
                        if( !waitResult )
                        {
                            LARGE_INTEGER fileSize;
                            if (GetLastError() == ERROR_HANDLE_EOF &&
                                GetFileSizeEx(m_Handle, &fileSize) &&
                                fileSize.QuadPart == offset)
                            {
                                *outValue = value;
                                return ResultSuccess();
                            }
                            else
                            {
                                return ConvertLastErrorToResult();
                            }
                        }
                    }
                    break;
                default:
                    return ConvertLastErrorToResult();
                }
            }

            *outValue = value;
            return ResultSuccess();
        }

        Result HostFile::DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Write) != 0, ResultInvalidOperationForOpenMode() );

            // IFile::Write() でハンドリングしている想定
            NN_SDK_REQUIRES(size > 0);

            if( buffer == nullptr )
            {
                return nn::fs::ResultNullptrArgument();
            }
            if( offset < 0 )
            {
                return nn::fs::ResultOutOfRange();
            }

            if ((m_Mode & nn::fs::OpenMode_AllowAppend) == 0)
            {
                int64_t currentSize;
                NN_RESULT_DO(GetSize(&currentSize));
                NN_FSP_REQUIRES(static_cast<int64_t>(offset + size) <= currentSize && static_cast<int64_t>(offset + size) >= offset,
                    ResultFileExtensionWithoutOpenModeAllowAppend(), "OpenMode_AllowAppend is required for implicit extension of file size by WriteFile().");
            }

            if (size != 0)
            {
                OVERLAPPED overlapped = {0};
                overlapped.OffsetHigh = (offset >> 32) & 0xFFFFFFFF;
                overlapped.Offset = offset & 0xFFFFFFFF;
                overlapped.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
                if( overlapped.hEvent == NULL )
                {
                    return nn::fs::ResultUnexpectedInHostFileSystemB();
                }

                NN_UTIL_SCOPE_EXIT
                {
                    CloseHandle(overlapped.hEvent);
                };

                DWORD value;
                BOOL success = ::WriteFile(m_Handle, buffer, static_cast<DWORD>(size), &value, &overlapped);
                if (!success)
                {
                    DWORD error = GetLastError();
                    switch (error)
                    {
                    case ERROR_IO_PENDING:
                    {
                        BOOL waitResult = GetOverlappedResult(m_Handle, &overlapped, &value, true);
                        if( !waitResult )
                        {
                            return ConvertLastErrorToResult();
                        }
                    }
                    break;
                    default:
                        return ConvertLastErrorToResult();
                    }
                }
                if (value < size)
                {
                    return nn::fs::ResultUsableSpaceNotEnough();
                }
                NN_SDK_ASSERT(value == size);
            }

            if ((option.flags & nn::fs::WriteOptionFlag_Flush) != 0)
            {
                NN_RESULT_DO(Flush());
            }

            return ResultSuccess();
        }

        Result HostFile::DoFlush() NN_NOEXCEPT
        {
            if ((m_Mode & nn::fs::OpenMode_Write) == 0)
            {
                NN_RESULT_SUCCESS;
            }

            // TODO 適切な事前条件チェック
            if (!::FlushFileBuffers(m_Handle))
            {
                NN_RESULT_THROW(nn::fs::ResultUnexpectedInHostFileSystemC());
            }

            NN_RESULT_SUCCESS;
        }

        Result SetFileSize(HANDLE handle, int64_t size) NN_NOEXCEPT
        {
            LONG high = size >> 32;
            ::SetFilePointer(handle, size & 0xFFFFFFFF, &high, FILE_BEGIN);

            if (!::SetEndOfFile(handle))
            {
                return ConvertLastErrorToResult();
            }

            return ResultSuccess();
        }

        Result HostFile::DoSetSize(int64_t size) NN_NOEXCEPT
        {
            NN_RESULT_DO(DrySetSize(size, m_Mode));
            return SetFileSize(m_Handle, size);
        }

        Result HostFile::DoGetSize(int64_t *outValue) NN_NOEXCEPT
        {
            LARGE_INTEGER size;
            if (!::GetFileSizeEx(m_Handle, &size))
            {
                return nn::fs::ResultUnexpectedInHostFileSystemD();
            }

            *outValue = size.QuadPart;
            return ResultSuccess();
        }

        Result HostFile::DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t,
            int64_t,
            const void*,
            size_t) NN_NOEXCEPT
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
                NN_RESULT_SUCCESS;
            case OperationId::QueryRange:
                NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
                NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

                reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
                NN_RESULT_SUCCESS;
            default:
                return nn::fs::ResultUnsupportedOperation();
            }
        }

        inline bool IsDirectory(const WIN32_FIND_DATAW& fd)
        {
            return (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) ? true : false;
        }

        class HostDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
        {
        public:
            HostDirectory(HANDLE dirHandle, HANDLE findHandle, nn::fs::OpenDirectoryMode mode, const wchar_t* path) NN_NOEXCEPT
                : m_DirectoryHandle(dirHandle), m_FindHandle(findHandle), m_Mode(mode)
            {
                wcscpy_s(m_Path, path);
            }

            virtual ~HostDirectory() NN_NOEXCEPT
            {
                ::FindClose(m_FindHandle);
                ::CloseHandle(m_DirectoryHandle);
            }

            virtual Result DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT;
            virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT;

        private:
            const HANDLE m_DirectoryHandle;
            const HANDLE m_FindHandle;
            const nn::fs::OpenDirectoryMode m_Mode;
            wchar_t m_Path[PathLengthMax + 1];

            bool IsReadTarget(const WIN32_FIND_DATAW& fd)
            {
                if (wcsncmp( fd.cFileName, L"..", 3 ) == 0)
                {
                    return false;
                }

                bool isDirectory = IsDirectory(fd);
                if ((isDirectory && m_Mode == nn::fs::OpenDirectoryMode_File) ||
                    (!isDirectory && m_Mode == nn::fs::OpenDirectoryMode_Directory))
                {
                    return false;
                }

                return true;
            }
        };

        Result HostDirectory::DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
        {
            int readNum = 0;
            while (readNum < entryBufferCount)
            {
                WIN32_FIND_DATAW fd;
                std::memset(fd.cFileName, 0, sizeof(fd.cFileName));
                if (! ::FindNextFile(m_FindHandle, &fd))
                {
                    break;
                }

                if (!IsReadTarget(fd))
                {
                    continue;
                }

                std::memset(entryBuffer[readNum].name, 0, sizeof(entryBuffer[readNum].name));
                int wideResult = ::WideCharToMultiByte(CP_UTF8, NULL, fd.cFileName, -1, entryBuffer[readNum].name, sizeof(entryBuffer[readNum].name), NULL, NULL);
                if( wideResult == 0 )
                {
                    NN_SDK_LOG("Error(0x%08x) has occurred in WideCharToMultiByte.\n", GetLastError());
                    return nn::fs::ResultInvalidPath();
                }
                entryBuffer[readNum].directoryEntryType = static_cast<int8_t>(IsDirectory(fd) ? nn::fs::DirectoryEntryType_Directory : nn::fs::DirectoryEntryType_File);
                entryBuffer[readNum].fileSize = fd.nFileSizeLow + (static_cast<int64_t>(fd.nFileSizeHigh) << 32);
                ++readNum;
            }
            *outValue = readNum;
            return nn::ResultSuccess();
        }

        Result HostDirectory::DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
        {
            WIN32_FIND_DATAW fd;
            int count = 0;

            auto handle = ::FindFirstFile(m_Path, &fd);
            NN_SDK_ASSERT(handle != INVALID_HANDLE_VALUE);

            while (::FindNextFile(handle, &fd))
            {
                if (IsReadTarget(fd))
                {
                    ++count;
                }
            }

            ::FindClose(handle);
            *outValue = count;
            return ResultSuccess();
        }

        Result ConvertLastErrorToResult() NN_NOEXCEPT
        {
            auto error = GetLastError();
            switch (error)
            {
            case ERROR_FILE_EXISTS:
            case ERROR_ALREADY_EXISTS:
                return nn::fs::ResultPathAlreadyExists();
            case ERROR_SPACES_NOT_ENOUGH_DRIVES:
            case ERROR_DISK_FULL:
                return nn::fs::ResultUsableSpaceNotEnough();
            case ERROR_FILE_NOT_FOUND:
            case ERROR_PATH_NOT_FOUND:
            case ERROR_INVALID_NAME:
                return nn::fs::ResultPathNotFound();
            case ERROR_ACCESS_DENIED:
            case ERROR_SHARING_VIOLATION:
                return nn::fs::ResultTargetLocked();
            case ERROR_DIR_NOT_EMPTY:
                return nn::fs::ResultDirectoryNotEmpty();
            case ERROR_DIRECTORY:
                return nn::fs::ResultPathNotFound();
            case ERROR_HANDLE_EOF:
                return nn::fs::ResultOutOfRange();
            default:
                NN_SDK_LOG("HostFileSystem: Mapped unknown error %d => ResultUnexpectedInHostFileSystemE.\n", error);
                return nn::fs::ResultUnexpectedInHostFileSystemE();
            }
        }

        void CheckPath(const char* path) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(path != NULL, "path is null.\n");
            // TODO 文字数チェックとか
            NN_UNUSED(path);
        }


        Result WaitDeletionCompletion(const wchar_t* pathW)
        {
            WIN32_FIND_DATAW fd;
            HANDLE handle;
            while(NN_STATIC_CONDITION(true))
            {
                while( (handle = ::FindFirstFile(pathW, &fd)) != INVALID_HANDLE_VALUE )
                {
                    ::FindClose(handle);
                    Sleep(500);
                }

                NN_RESULT_TRY(ConvertLastErrorToResult())
                    NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
                    {
                        // 稀に ResultTargetLocked で失敗する場合があるのでその際はリトライ
                        continue;
                    }
                    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
                    {
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY
            }
        }

        Result GetEntryTypeW(nn::fs::DirectoryEntryType* outValue, const wchar_t *path) NN_NOEXCEPT
        {
            auto result = ::GetFileAttributes(path);
            if (result == -1)
            {
                auto error = GetLastError();
                switch (error)
                {
                case ERROR_PATH_NOT_FOUND:
                case ERROR_FILE_NOT_FOUND:
                case ERROR_ACCESS_DENIED:
                case ERROR_INVALID_NAME:
                    return nn::fs::ResultPathNotFound();
                default:
                    return nn::fs::ResultUnexpectedInHostFileSystemF();
                }
            }

            *outValue = (result & FILE_ATTRIBUTE_DIRECTORY) ? nn::fs::DirectoryEntryType_Directory : nn::fs::DirectoryEntryType_File;
            return ResultSuccess();
        }

    }

    HostFileSystem::~HostFileSystem() NN_NOEXCEPT
    {
    }

    inline Result HostFileSystem::ResolveFullPath(wchar_t* fullPathBuffer, int bufferSize, const char* path) NN_NOEXCEPT
    {
        static const auto LengthMax = MAX_PATH - 1;
        return ResolveFullPath(fullPathBuffer, bufferSize, path, LengthMax);
    }

    Result HostFileSystem::ResolveFullPath(wchar_t* fullPathBuffer, int bufferSize, const char* path, size_t pathLengthMax) NN_NOEXCEPT
    {
        int bufferLength = bufferSize / sizeof(fullPathBuffer[0]);
        wcscpy_s(fullPathBuffer, bufferLength, m_RootPath);

        auto rootPathLength = static_cast<int>(wcsnlen_s(fullPathBuffer, bufferLength));
        wchar_t* pathBuffer = fullPathBuffer + rootPathLength;
        int pathBufferLength = bufferLength - rootPathLength;

        // 入力Pathのバッファ長判定
        if (nn::fs::EntryNameLengthMax < strnlen(path, nn::fs::EntryNameLengthMax + 1))
        {
            return nn::fs::ResultTooLongPath();
        }

        char* pSubPath = const_cast<char*>(path);
        if (path[0] == '/')
        {
            pSubPath += 1;
        }

        static const auto NameLengthMax = 255;
        const auto driveLetterLength = IsWindowsDrive(pSubPath) ? 2 : 0;
        NN_RESULT_DO(VerifyPath(
            pSubPath + driveLetterLength,
            static_cast<int>(pathLengthMax) - driveLetterLength - rootPathLength,
            NameLengthMax));

        int wideResult = ::MultiByteToWideChar(CP_UTF8, NULL, pSubPath, -1, pathBuffer, pathBufferLength);

        if (pathBufferLength > 0)
        {
            // 変換文字列がある条件で結果をチェック（変換対象の文字列長が０の場合、戻りも０となるため）
            NN_RESULT_THROW_UNLESS(wideResult != 0 && (rootPathLength + wideResult) <= WindowsPathLengthMax, nn::fs::ResultTooLongPath());
        }

        NN_RESULT_SUCCESS;
    }

    Result HostFileSystem::Initialize(const char* rootPath) NN_NOEXCEPT
    {
        std::memset(m_RootPath, 0, sizeof(m_RootPath));
        auto rootPathLength = static_cast<int>(strnlen_s(rootPath, WindowsPathLengthMax * 3));
        int wideResult = ::MultiByteToWideChar(CP_UTF8, NULL, rootPath, -1, m_RootPath, WindowsPathLengthMax + 1);

        // 変換文字列がある条件で結果をチェック（変換対象の文字列長が０の場合、戻りも０となるため）
        if( rootPathLength > 0 && wideResult == 0 )
        {
            NN_SDK_LOG("Error(0x%08x) has occurred in MultiByteToWideChar.\n", GetLastError());
            return nn::fs::ResultInvalidPath();
        }

        if (rootPath[0] != '\0')
        {
            if (wideResult >= WindowsPathLengthMax)
            {
                return nn::fs::ResultTooLongPath();
            }
            wcscat_s(m_RootPath, L"/");
        }

        if (rootPath[0] != '\0')
        {
            nn::fs::DirectoryEntryType entryType;
            NN_RESULT_DO(GetEntryType(&entryType, ""));
            if (entryType != nn::fs::DirectoryEntryType_Directory)
            {
                return nn::fs::ResultPathNotFound();
            }
        }
        return ResultSuccess();
    }

    Result HostFileSystem::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        if( size < 0 )
        {
            return nn::fs::ResultOutOfRange();
        }

        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        auto handle = ::CreateFileW(fullPathBuffer, GENERIC_WRITE, 0, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);

        if (handle == INVALID_HANDLE_VALUE)
        {
            NN_RESULT_TRY(ConvertLastErrorToResult())
                NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
                {
                    const auto attribute = ::GetFileAttributesW(fullPathBuffer);
                    if( attribute != INVALID_FILE_ATTRIBUTES && (attribute & FILE_ATTRIBUTE_DIRECTORY) != 0 )
                    {
                        // 既存のディレクトリのパスが指定された
                        return nn::fs::ResultPathAlreadyExists();
                    }
                    else
                    {
                        NN_RESULT_RETHROW;
                    }
                }
            NN_RESULT_END_TRY
        }

        NN_UTIL_SCOPE_EXIT
        {
            ::CloseHandle(handle);
        };

        {
            // スパースファイル化する。失敗した場合も通常ファイルのまま続行する
            DWORD bytesReturned;
            auto result = DeviceIoControl(handle, FSCTL_SET_SPARSE, nullptr, 0, nullptr, 0, &bytesReturned, nullptr);
            NN_UNUSED(result);
            NN_UNUSED(bytesReturned);
        }

        if (size > 0)
        {
            NN_RESULT_DO(SetFileSize(handle, size));
        }

        return ResultSuccess();
    }

    Result HostFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        if (!::DeleteFileW(fullPathBuffer))
        {
            if( GetLastError() == ERROR_ACCESS_DENIED )
            {
                nn::fs::DirectoryEntryType type;
                NN_RESULT_DO(GetEntryTypeW(&type, fullPathBuffer));
                if( type == nn::fs::DirectoryEntryType_Directory )
                {
                    return nn::fs::ResultPathNotFound();
                }
                else
                {
                    return ConvertLastErrorToResult();
                }
            }
            else
            {
                return ConvertLastErrorToResult();
            }
        }

        NN_RESULT_DO(WaitDeletionCompletion(fullPathBuffer));

        return ResultSuccess();
    }

    Result HostFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        CheckPath(path);
        static const auto LengthMax = 248 - 1; // ::CreateDirectory() の最大パス長
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path, LengthMax));

        if (!::CreateDirectoryW(fullPathBuffer, NULL))
        {
            return ConvertLastErrorToResult();
        }

        return ResultSuccess();
    }

    Result HostFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        if (!::RemoveDirectory(fullPathBuffer))
        {
            return ConvertLastErrorToResult();
        }

        NN_RESULT_DO(WaitDeletionCompletion(fullPathBuffer));

        return ResultSuccess();
    }

    Result HostFileSystem::DeleteDirectoryRecursivelyInternal(const wchar_t *path, bool deleteItself) NN_NOEXCEPT
    {
        WIN32_FIND_DATAW fd;
        wchar_t pathW[WindowsPathLengthMax + 1] = L"";
        wcscat_s(pathW, path);
        wcscat_s(pathW, L"/*");

        auto handle = ::FindFirstFile(pathW, &fd);
        if (handle == INVALID_HANDLE_VALUE)
        {
            return ConvertLastErrorToResult();
        }
        wchar_t* subPathPtr = pathW + wcslen(pathW) - 1;
        auto subPathLen = sizeof(pathW) / sizeof(pathW[0]) - (wcslen(pathW) - 1) - 1;

        while (::FindNextFile(handle, &fd))
        {
            if (wcsncmp( fd.cFileName, L"..", 3 ) == 0 ||
                wcsncmp( fd.cFileName, L".", 2 ) == 0)
            {
                continue;
            }
            if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
            {
                wcscpy_s(subPathPtr, subPathLen, fd.cFileName);
                NN_RESULT_TRY(DeleteDirectoryRecursivelyInternal(pathW, true))
                    NN_RESULT_CATCH_ALL
                    {
                        ::FindClose(handle);
                        NN_RESULT_RETHROW;
                    }
                NN_RESULT_END_TRY
            }
            else
            {
                wcscpy_s(subPathPtr, subPathLen, fd.cFileName);
                if (! ::DeleteFileW(pathW))
                {
                    ::FindClose(handle);
                    return ConvertLastErrorToResult();
                }

                NN_RESULT_DO(WaitDeletionCompletion(pathW));
            }
        }

        ::FindClose(handle);

        if( deleteItself )
        {
            if (!::RemoveDirectoryW(path))
            {
                return ConvertLastErrorToResult();
            }
            NN_RESULT_DO(WaitDeletionCompletion(path));
        }

        return ResultSuccess();
    }

    Result HostFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        return DeleteDirectoryRecursivelyInternal(fullPathBuffer, true);
    }

    Result HostFileSystem::DoCleanDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        return DeleteDirectoryRecursivelyInternal(fullPathBuffer, false);
    }

    Result HostFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        CheckPath(currentPath);
        wchar_t currentFullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(currentFullPathBuffer, sizeof(currentFullPathBuffer), currentPath));

        CheckPath(newPath);
        wchar_t newFullPathBuffer[nn::fs::EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(newFullPathBuffer, sizeof(newFullPathBuffer), newPath));

        nn::fs::DirectoryEntryType entryType;
        NN_RESULT_DO(GetEntryTypeW(&entryType, currentFullPathBuffer));
        if (entryType != nn::fs::DirectoryEntryType_File)
        {
            return nn::fs::ResultPathNotFound();
        }

        if (!::MoveFileW(currentFullPathBuffer, newFullPathBuffer))
        {
            NN_RESULT_TRY(ConvertLastErrorToResult())
                NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
                {
                    if( wcscmp(currentFullPathBuffer, newFullPathBuffer) == 0 )
                    {
                        // 同一パスへのリネームを試みた
                        NN_RESULT_SUCCESS;
                    }

                    const auto attribute = ::GetFileAttributesW(newFullPathBuffer);
                    if( attribute != INVALID_FILE_ATTRIBUTES )
                    {
                        // 既存のエントリのパスが指定された
                        return nn::fs::ResultPathAlreadyExists();
                    }

                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }

        return ResultSuccess();
    }

    Result HostFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        CheckPath(currentPath);
        wchar_t currentFullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(currentFullPathBuffer, sizeof(currentFullPathBuffer), currentPath));

        CheckPath(newPath);
        wchar_t newFullPathBuffer[nn::fs::EntryNameLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(newFullPathBuffer, sizeof(newFullPathBuffer), newPath));

        nn::fs::DirectoryEntryType entryType;
        NN_RESULT_DO(GetEntryTypeW(&entryType, currentFullPathBuffer));
        if (entryType != nn::fs::DirectoryEntryType_Directory)
        {
            return nn::fs::ResultPathNotFound();
        }

        if (!::MoveFile(currentFullPathBuffer, newFullPathBuffer))
        {
            NN_RESULT_TRY(ConvertLastErrorToResult())
                NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
                {
                    if( wcscmp(currentFullPathBuffer, newFullPathBuffer) == 0 )
                    {
                        // 同一パスへのリネームを試みた
                        NN_RESULT_SUCCESS;
                    }

                    const auto attribute = ::GetFileAttributesW(newFullPathBuffer);
                    if( attribute != INVALID_FILE_ATTRIBUTES )
                    {
                        // 既存のエントリのパスが指定された
                        return nn::fs::ResultPathAlreadyExists();
                    }

                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }

        return ResultSuccess();
    }

    Result HostFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char *path) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        return GetEntryTypeW(outValue, fullPathBuffer);
    }

    Result HostFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        // OpenMode_Read or OpenMode_Write is required
        NN_FSP_REQUIRES( (mode & (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)) != 0, ResultInvalidArgument(), "OpenMode_Read or OpenMode_Write is required.");

        auto handle = ::CreateFileW(fullPathBuffer,
            ((mode & nn::fs::OpenMode_Read) ? GENERIC_READ : 0) | ((mode & nn::fs::OpenMode_Write) ? GENERIC_WRITE : 0),
            FILE_SHARE_READ,
            NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL);

        if (handle == INVALID_HANDLE_VALUE)
        {
            auto lastError = GetLastError();
            if (lastError == ERROR_ACCESS_DENIED)
            {
                nn::fs::DirectoryEntryType type;
                NN_RESULT_DO(GetEntryTypeW(&type, fullPathBuffer));
                if (type == nn::fs::DirectoryEntryType_Directory)
                {
                    return nn::fs::ResultPathNotFound();
                }
                else
                {
                    return ConvertLastErrorToResult();
                }
            }
            else
            {
                return ConvertLastErrorToResult();
            }
        }

        std::unique_ptr<HostFile> file(new HostFile(handle, mode));
        if (file == nullptr)
        {
            ::CloseHandle(handle);
            return nn::fs::ResultAllocationMemoryFailedInHostFileSystemA();
        }

        *outValue = std::move(file);

        return ResultSuccess();
    }

    Result HostFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        WIN32_FIND_DATAW fd;
        CheckPath(path);
        static const auto LengthMax = MAX_PATH - 2 - 1;
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path, LengthMax));

        auto dirHandle = ::CreateFileW(fullPathBuffer, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, NULL);
        if (dirHandle == INVALID_HANDLE_VALUE)
        {
            return ConvertLastErrorToResult();
        }

        wcscat_s(fullPathBuffer, WindowsPathLengthMax + 1, L"/*");
        auto handle = ::FindFirstFile(fullPathBuffer, &fd);
        if (handle == INVALID_HANDLE_VALUE)
        {
            ::CloseHandle(dirHandle);

            nn::fs::DirectoryEntryType type;
            NN_RESULT_DO(GetEntryTypeW(&type, fullPathBuffer));
            if( type != nn::fs::DirectoryEntryType_Directory )
            {
                return nn::fs::ResultPathNotFound();
            }
            else
            {
                return ConvertLastErrorToResult();
            }
        }

        // TODO: recycle.bin が無いドライブのルートディレクトリ(リムーバブルメディア等)では
        //       ここで 1 つ目のエントリが取得されてしまうので、保持して次の ReadDirectory() で返す必要がある

        std::unique_ptr<HostDirectory> directory(new HostDirectory(dirHandle, handle, mode, fullPathBuffer));

        if (directory == nullptr)
        {
            ::FindClose(handle);
            ::CloseHandle(dirHandle);
            return nn::fs::ResultAllocationMemoryFailedInHostFileSystemB();
        }

        *outValue = std::move(directory);

        return ResultSuccess();
    }

    Result HostFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
        CheckPath(path);
        wchar_t fullPathBuffer[WindowsPathLengthMax + 1];
        NN_RESULT_DO(ResolveFullPath(fullPathBuffer, sizeof(fullPathBuffer), path));

        nn::fs::DirectoryEntryType type;
        NN_RESULT_DO(GetEntryTypeW(&type, fullPathBuffer));
        if (type == nn::fs::DirectoryEntryType_Directory)
        {
            return nn::fs::ResultPathNotFound();
        }

        auto handle = :: CreateFileW(fullPathBuffer, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);

        if (handle == INVALID_HANDLE_VALUE)
        {
            return ConvertLastErrorToResult();
        }

        NN_UTIL_SCOPE_EXIT
        {
            CloseHandle(handle);
        };

        FILETIME ftC;
        FILETIME ftA;
        FILETIME ftR;

        if (!(::GetFileTime(handle, &ftC, &ftA, &ftR)))
        {
            return nn::fs::ResultInvalidPath();
        }

        // 得られた値は 100ns 単位なので s に変換した後、 UNIX 時間に変換
        outTimeStamp->create = ((static_cast<uint64_t>(ftC.dwHighDateTime) << 32) + ftC.dwLowDateTime) / 10000000 - OffsetToConvertToPosixTime;
        outTimeStamp->access = ((static_cast<uint64_t>(ftA.dwHighDateTime) << 32) + ftA.dwLowDateTime) / 10000000 - OffsetToConvertToPosixTime;
        outTimeStamp->modify = ((static_cast<uint64_t>(ftR.dwHighDateTime) << 32) + ftR.dwLowDateTime) / 10000000 - OffsetToConvertToPosixTime;
        outTimeStamp->isLocalTime = false; // UTC なので false

        return ResultSuccess();
    }

    Result HostFileSystem::DoCommit() NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    Result HostFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
    {
        NN_UNUSED(counter);
        NN_RESULT_SUCCESS;
    }

    Result HostFileSystem::DoRollback() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
}}
