﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <mutex>
#include <memory>
#include <algorithm>
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>
#include <nn/fssystem/buffers/fs_FileSystemBuddyHeap.h>

//#define     NN_DETAIL_FSSYSTEM_ENABLE_CHECKPEAK

#if defined(NN_DETAIL_FSSYSTEM_ENABLE_CHECKPEAK)
#if !defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_SdkLog.h>
#define     NN_FSPEAK_LOG           NN_SDK_LOG
#else
#define     NN_FSPEAK_LOG(...)      (void)0
#endif
#endif

namespace nn { namespace fssystem {

namespace {
    const auto RetryWait = nn::TimeSpan::FromMilliSeconds(10);

    const int64_t RunnerCountMax = 3;

    const auto HeapBlockSize = BufferPoolAlignment;       // 4K
    const auto HeapOrderTrim = 3;                         // 32K  (4K << 3)
    const auto HeapOrderMax = 7;                          // 512K (4K << 7)
    const auto HeapOrderMaxForLarge = HeapOrderMax + 3;   // 4M   (4K << 10)

    const size_t HeapAllocatableSizeTrim = HeapBlockSize * (1 << HeapOrderTrim);
    const size_t HeapAllocatableSizeMax = HeapBlockSize * (1 << HeapOrderMax);
    const size_t HeapAllocatableSizeMaxForLarge = HeapBlockSize * (1 << HeapOrderMaxForLarge);

    const size_t SplitAccessAlignmentSize = 16 * 1024;

    bool IsSplitAccessRequiredByAccessPriority(int64_t* outValue, bool isWriteAccess) NN_NOEXCEPT
    {
        const int64_t LowerPriorityReadAccessSplitSizeMax = 128 * 1024;
        const int64_t HigherPriorityWriteAccessSplitSizeMax = 512 * 1024;
        const int64_t LowerPriorityWriteAccessSplitSizeMax = 64 * 1024;

        switch( GetServiceContextReadOnly().GetPriority() )
        {
        case fs::PriorityRaw_Realtime:
        case fs::PriorityRaw_Normal:
            if( isWriteAccess )
            {
                *outValue = HigherPriorityWriteAccessSplitSizeMax;
                return true;
            }
            else
            {
                return false;
            }

        case fs::PriorityRaw_Background:
        case fs::PriorityRaw_Low:
            if( isWriteAccess )
            {
                *outValue = LowerPriorityWriteAccessSplitSizeMax;
            }
            else
            {
                *outValue = LowerPriorityReadAccessSplitSizeMax;
            }
            return true;

        default:
            // SetPriority で弾いているのでここにはこない
            NN_UNEXPECTED_DEFAULT;
        }
    }

    int64_t GetInvocationCount(int64_t offset, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(static_cast<size_t>(0), size);
        int64_t firstHeapBlockOffset = util::align_down<int64_t>(offset, HeapBlockSize);
        int64_t endOffset = offset + static_cast<int64_t>(size);
        return nn::util::DivideUp(endOffset - firstHeapBlockOffset, HeapAllocatableSizeMax);
    }

    nn::os::Mutex g_HeapMutex(false);

    FileSystemBuddyHeap g_Heap;
    size_t g_SizeHeapFreePeak = 0;
#if defined(NN_DETAIL_FSSYSTEM_ENABLE_CHECKPEAK)
    size_t g_CountAllocated = 0;
#endif
    size_t g_CountRetried = 0;
    size_t g_CountReduceAllocation = 0;

    void* g_HeapBuffer = nullptr;
    size_t g_HeapSize = 0;

    ThreadPool* g_pThreadPool = nullptr;
}

AsynchronousAccessFile::AsynchronousAccessFile(
    std::unique_ptr<fs::fsa::IFile>&& pBaseFile,
    fs::OpenMode mode,
    ThreadPool* pThreadPool) NN_NOEXCEPT
    : m_pBaseFile(std::move(pBaseFile)),
    m_Mode(mode),
    m_pThreadPool(pThreadPool)
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBaseFile);
    NN_SDK_REQUIRES_NOT_NULL(m_pThreadPool);
}

AsynchronousAccessFile::~AsynchronousAccessFile() NN_NOEXCEPT
{
}

Result AsynchronousAccessFile::DoRead(
    size_t* outValue,
    int64_t offset,
    void* buffer,
    size_t size,
    const fs::ReadOption& option) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);

    NN_FSP_REQUIRES((m_Mode & fs::OpenMode_Read) != 0, fs::ResultReadUnpermitted());

    int64_t accessSize = 0;
    if( IsSplitAccessRequiredByAccessPriority(&accessSize, false) )
    {
        // 分割 Read
        NN_SDK_REQUIRES_LESS(0, accessSize);

        size_t readSize = 0;
        NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));
        if( readSize == 0 )
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }

        const int64_t endOffset = offset + readSize;
        int64_t currentOffset = offset;
        size_t totalReadSize = 0;

        // ファイルの分割読み込み
        while( currentOffset < endOffset )
        {
            const auto nextOffset = std::min(
                endOffset,
                util::align_down<int64_t>(currentOffset + accessSize, SplitAccessAlignmentSize)
            );
            size_t currentReadSize;
            NN_RESULT_DO(DoReadImpl(
                &currentReadSize,
                currentOffset,
                reinterpret_cast<char*>(buffer) + (currentOffset - offset),
                static_cast<size_t>(nextOffset - currentOffset),
                option
            ));
            totalReadSize += currentReadSize;

            currentOffset = nextOffset;
        }

        *outValue = totalReadSize;

        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(DoReadImpl(outValue, offset, buffer, size, option));
        NN_RESULT_SUCCESS;
    }
}

Result AsynchronousAccessFile::DoReadImpl(
    size_t* outValue,
    int64_t offset,
    void* buffer,
    size_t size,
    const fs::ReadOption& option) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);

    // HeapAllocatableSizeMax 以下は分割しない
    if( size <= HeapAllocatableSizeMax )
    {
        return m_pBaseFile->Read(outValue, offset, buffer, size, option);
    }
    else
    {
        class AsynchronousFileReader : public AsynchronousRunner
        {
        public:
            AsynchronousFileReader() NN_NOEXCEPT
                : m_ReadSize(0)
            {
            }

            virtual ~AsynchronousFileReader() NN_NOEXCEPT NN_OVERRIDE
            {
                Release();
            }

            void SetParam(
                nn::fs::fsa::IFile* pFile,
                int64_t offset,
                void* buffer,
                size_t size,
                const fs::ReadOption& option) NN_NOEXCEPT
            {
                m_pFile = pFile;
                m_Offset = offset;
                m_Buffer = buffer;
                m_Size = size;
                m_Option = option;
                m_ReadSize = 0;
            }

            size_t GetReadSize() const NN_NOEXCEPT
            {
                return m_ReadSize;
            }

        protected:
            virtual nn::Result DoRun() NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pFile->Read(&m_ReadSize, m_Offset, m_Buffer, m_Size, m_Option);
            }

        private:
            nn::fs::fsa::IFile* m_pFile;
            size_t m_ReadSize;
            int64_t m_Offset;
            void* m_Buffer;
            size_t m_Size;
            fs::ReadOption m_Option;
        };

        size_t readSize = 0;
        NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));
        if( readSize == 0 )
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }

        AsynchronousFileReader readers[RunnerCountMax];

        int64_t invocationCount = GetInvocationCount(offset, readSize);
        int64_t activeRunnerCount = 0;
        // ThreadPool で AsynchronousStorageReader を実行可能状態にする処理を行う
        {
            int64_t requiredRunnerCount = std::min(RunnerCountMax, invocationCount);
            m_pThreadPool->ActivateAsynchronousRunners(&activeRunnerCount, readers, requiredRunnerCount);
            NN_SDK_ASSERT_LESS(0, activeRunnerCount);
        }
        const int64_t endOffset = offset + readSize;
        int64_t currentOffset = offset;
        size_t totalReadSize = 0;
        Result result = ResultSuccess();

        // ファイルの分割読み込み
        for( int loopIndex = 0; loopIndex < invocationCount + activeRunnerCount; ++loopIndex )
        {
            int runnerIndex = loopIndex % activeRunnerCount;
            auto& reader = readers[runnerIndex];
            if( loopIndex >= activeRunnerCount )
            {
                result = reader.GetResult();
                if( result.IsFailure() )
                {
                    break;
                }
                totalReadSize += reader.GetReadSize();
            }
            if( loopIndex >= invocationCount )
            {
                continue;
            }

            const auto nextOffset = std::min(
                endOffset,
                util::align_down<int64_t>(currentOffset + HeapAllocatableSizeMax, HeapBlockSize)
            );
            reader.SetParam(
                m_pBaseFile.get(),
                currentOffset,
                reinterpret_cast<char*>(buffer) + (currentOffset - offset),
                static_cast<size_t>(nextOffset - currentOffset),
                option
            );
            reader.Start();

            currentOffset = nextOffset;
        }

        if( result.IsSuccess() )
        {
            NN_SDK_ASSERT_EQUAL(readSize, totalReadSize);
            *outValue = totalReadSize;
        }

        return result;
    }
} // NOLINT(impl/function_size)

Result AsynchronousAccessFile::DoWrite(
    int64_t offset,
    const void* buffer,
    size_t size,
    const fs::WriteOption& option) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);

    NN_FSP_REQUIRES((m_Mode & fs::OpenMode_Write) != 0, fs::ResultWriteUnpermitted());

    // NOTE: 書き込み側は分散処理をすると悪影響が大きいのでそのまま流すか、単に分割する
    int64_t accessSize = 0;
    bool isSplitAccess = IsSplitAccessRequiredByAccessPriority(&accessSize, true);
    NN_SDK_REQUIRES(isSplitAccess);
    NN_UNUSED(isSplitAccess);
    NN_SDK_REQUIRES_LESS(0, accessSize);

    int64_t fileSize;
    NN_RESULT_DO(GetSize(&fileSize));

    // 書き込み開始位置をチェックします。
    // offset がファイルサイズより大きく、且つ自動拡張が不許可ならエラーを返します
    NN_RESULT_THROW_UNLESS(((offset <= fileSize) || ((m_Mode & nn::fs::OpenMode_AllowAppend) != 0)), fs::ResultFileExtensionWithoutOpenModeAllowAppend());

    // 拡張が必要なら自動拡張が許可されているかどうかチェックして拡張します
    if( static_cast<int64_t>(offset + size) > fileSize )
    {
        NN_RESULT_THROW_UNLESS((m_Mode & nn::fs::OpenMode_AllowAppend) != 0, fs::ResultFileExtensionWithoutOpenModeAllowAppend());
        NN_RESULT_DO(SetSize(offset + size));
    }

    const fs::WriteOption optionWithoutFlush = fs::WriteOption::MakeValue(option.flags & ~fs::WriteOptionFlag_Flush);

    const int64_t endOffset = offset + size;
    int64_t currentOffset = offset;

    // ファイルの分割書き込み
    while( currentOffset < endOffset )
    {
        const auto nextOffset = std::min(
            endOffset,
            util::align_down<int64_t>(currentOffset + accessSize, SplitAccessAlignmentSize)
        );
        NN_RESULT_DO(m_pBaseFile->Write(
            currentOffset,
            reinterpret_cast<const char*>(buffer) + (currentOffset - offset),
            static_cast<size_t>(nextOffset - currentOffset),
            optionWithoutFlush
        ));

        currentOffset = nextOffset;
    }

    if( (option.flags & fs::WriteOptionFlag_Flush) != 0 )
    {
        NN_RESULT_DO(m_pBaseFile->Flush());
    }

    NN_RESULT_SUCCESS;
}

Result AsynchronousAccessFile::DoFlush() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);
    return m_pBaseFile->Flush();
}

Result AsynchronousAccessFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);
    return m_pBaseFile->SetSize(size);
}

Result AsynchronousAccessFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);
    return m_pBaseFile->GetSize(outValue);
}

Result AsynchronousAccessFile::DoOperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseFile);
    return m_pBaseFile->OperateRange(
        outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
}

AsynchronousAccessStorage::AsynchronousAccessStorage(
    const std::shared_ptr<fs::IStorage>& pBaseStorage,
    ThreadPool* pThreadPool) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage),
    m_pThreadPool(pThreadPool)
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES_NOT_NULL(m_pThreadPool);
}

AsynchronousAccessStorage::~AsynchronousAccessStorage() NN_NOEXCEPT
{
}

Result AsynchronousAccessStorage::Read(
    int64_t offset,
    void* buffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);

    int64_t accessSize = 0;
    if( IsSplitAccessRequiredByAccessPriority(&accessSize, false) )
    {
        NN_SDK_REQUIRES_LESS(0, accessSize);

        int64_t storageSize = 0;
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [&]() NN_NOEXCEPT->Result
            {
                NN_RESULT_DO(m_pBaseStorage->GetSize(&storageSize));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));

        NN_RESULT_THROW_UNLESS(IStorage::CheckAccessRange(offset, size, storageSize), nn::fs::ResultOutOfRange());

        const int64_t endOffset = offset + size;
        int64_t currentOffset = offset;

        // ストレージの分割読み込み
        while( currentOffset < endOffset )
        {
            const auto nextOffset = std::min(
                endOffset,
                util::align_down<int64_t>(currentOffset + accessSize, SplitAccessAlignmentSize)
            );

            if( currentOffset < storageSize )
            {
                NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
                    [=]() NN_NOEXCEPT->Result
                    {
                        NN_RESULT_DO(ReadImpl(
                            currentOffset,
                            reinterpret_cast<char*>(buffer) + (currentOffset - offset),
                            static_cast<size_t>(nextOffset - currentOffset)
                        ));
                        NN_RESULT_SUCCESS;
                    },
                    NN_CURRENT_FUNCTION_NAME
                ));
            }

            currentOffset = nextOffset;
        }

        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(ReadImpl(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }
}

Result AsynchronousAccessStorage::ReadImpl(
    int64_t offset,
    void* buffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);

    // HeapAllocatableSizeMax 以下は分割しない
    if( size <= HeapAllocatableSizeMax )
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT->Result
            {
                NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, size));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }
    else
    {
        class AsynchronousStorageReader : public AsynchronousRunner
        {
        public:
            virtual ~AsynchronousStorageReader() NN_NOEXCEPT NN_OVERRIDE
            {
                Release();
            }

            void SetParam(
                nn::fs::IStorage* pStroage,
                int64_t offset,
                void* buffer,
                size_t size) NN_NOEXCEPT
            {
                m_pStorage = pStroage;
                m_Offset = offset;
                m_Buffer = buffer;
                m_Size = size;
            }

        protected:
            virtual nn::Result DoRun() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
                    [=]() NN_NOEXCEPT -> Result
                    {
                        NN_RESULT_DO(m_pStorage->Read(m_Offset, m_Buffer, m_Size));
                        NN_RESULT_SUCCESS;
                    },
                    NN_CURRENT_FUNCTION_NAME
                ));
                NN_RESULT_SUCCESS;
            }

        private:
            nn::fs::IStorage* m_pStorage;
            int64_t m_Offset;
            void* m_Buffer;
            size_t m_Size;
        };

        int64_t storageSize = 0;
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [&]() NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(m_pBaseStorage->GetSize(&storageSize));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        size_t readSize = std::min(static_cast<size_t>(storageSize - offset), size);

        AsynchronousStorageReader readers[RunnerCountMax];
        int64_t invocationCount = GetInvocationCount(offset, readSize);
        int64_t activeRunnerCount = 0;

        // ThreadPool で AsynchronousStorageReader を実行可能状態にする処理を行う
        {
            int64_t requiredRunnerCount = std::min(RunnerCountMax, invocationCount);
            m_pThreadPool->ActivateAsynchronousRunners(&activeRunnerCount, readers, requiredRunnerCount);
            NN_SDK_ASSERT_LESS(0, activeRunnerCount);
        }
        const int64_t endOffset = offset + readSize;
        int64_t currentOffset = offset;
        Result result = ResultSuccess();

        // ストレージの分割読み込み
        for( int loopIndex = 0; loopIndex < invocationCount + activeRunnerCount; ++loopIndex )
        {
            int runnerIndex = loopIndex % activeRunnerCount;
            auto& reader = readers[runnerIndex];
            if( loopIndex >= activeRunnerCount )
            {
                result = reader.GetResult();
                if( result.IsFailure() )
                {
                    break;
                }
            }
            if( loopIndex >= invocationCount )
            {
                continue;
            }

            const auto nextOffset = std::min(
                endOffset,
                util::align_down<int64_t>(currentOffset + HeapAllocatableSizeMax, HeapBlockSize)
            );
            reader.SetParam(
                m_pBaseStorage.get(),
                currentOffset,
                reinterpret_cast<char*>(buffer) + (currentOffset - offset),
                static_cast<size_t>(nextOffset - currentOffset)
            );
            reader.Start();

            currentOffset = nextOffset;
        }

        NN_RESULT_DO(result);
        NN_RESULT_SUCCESS;
    }
} // NOLINT(impl/function_size)

Result AsynchronousAccessStorage::Write(
    int64_t offset,
    const void* buffer,
    size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);

    // NOTE: 書き込み側は分散処理をすると悪影響が大きいのでそのまま流すか、単に分割する
    int64_t accessSize = 0;
    bool isSplitAccess = IsSplitAccessRequiredByAccessPriority(&accessSize, true);
    NN_SDK_REQUIRES(isSplitAccess);
    NN_UNUSED(isSplitAccess);
    NN_SDK_REQUIRES_LESS(0, accessSize);

    int64_t storageSize = 0;
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [&]() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(m_pBaseStorage->GetSize(&storageSize));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));

    NN_RESULT_THROW_UNLESS(IStorage::CheckAccessRange(offset, size, storageSize), fs::ResultOutOfRange());

    const int64_t endOffset = offset + size;
    int64_t currentOffset = offset;

    // ストレージの分割書き込み
    while( currentOffset < endOffset )
    {
        const auto nextOffset = std::min(
            endOffset,
            util::align_down<int64_t>(currentOffset + accessSize, SplitAccessAlignmentSize)
        );

        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(m_pBaseStorage->Write(
                    currentOffset,
                    reinterpret_cast<const char*>(buffer) + (currentOffset - offset),
                    static_cast<size_t>(nextOffset - currentOffset)
                ));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));

        currentOffset = nextOffset;
    }

    NN_RESULT_SUCCESS;
}

Result AsynchronousAccessStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(m_pBaseStorage->GetSize(outValue));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

Result AsynchronousAccessStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(m_pBaseStorage->SetSize(size));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

Result AsynchronousAccessStorage::Flush() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(m_pBaseStorage->Flush());
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

Result AsynchronousAccessStorage::OperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBaseStorage);
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(
                m_pBaseStorage->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize
                )
            );
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

size_t PooledBuffer::GetAllocatableSizeMaxCore(bool isEnableLargeCapacity) NN_NOEXCEPT
{
    return isEnableLargeCapacity ? HeapAllocatableSizeMaxForLarge : HeapAllocatableSizeMax;
}

PooledBuffer::PooledBuffer() NN_NOEXCEPT
    : m_Buffer(nullptr),
      m_Size(0)
{
}

PooledBuffer::PooledBuffer(PooledBuffer&& other) NN_NOEXCEPT
    : m_Buffer(other.m_Buffer),
      m_Size(other.m_Size)
{
    other.m_Buffer = nullptr;
    other.m_Size = 0;
}

PooledBuffer::PooledBuffer(size_t idealSize, size_t requiredSize) NN_NOEXCEPT
    : m_Buffer(nullptr),
      m_Size(0)
{
    Allocate(idealSize, requiredSize);
}

PooledBuffer::~PooledBuffer() NN_NOEXCEPT
{
    Deallocate();
}

void PooledBuffer::AllocateCore(
    size_t idealSize,
    size_t requiredSize,
    bool isEnableLargeCapacity) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(g_HeapBuffer);
    NN_SDK_REQUIRES(m_Buffer == nullptr);
    NN_SDK_ASSERT_EQUAL(g_Heap.GetBlockSize(), HeapBlockSize);

    NN_SDK_REQUIRES_LESS_EQUAL(
        requiredSize,
        GetAllocatableSizeMaxCore(isEnableLargeCapacity));

    const auto allocateSize
        = std::min(
            std::max(idealSize, requiredSize),
            GetAllocatableSizeMaxCore(isEnableLargeCapacity));

    for( ; ; )
    {
        {
            std::lock_guard<nn::os::Mutex> lock(g_HeapMutex);

            // TODO: TotalFreeSize に余裕がある場合は >> 1 しなくてもいいかも
            auto allocatableSize = g_Heap.GetAllocatableSizeMax();
            if( allocatableSize > HeapBlockSize )
            {
                allocatableSize >>= 1;
            }

            if( allocatableSize >= requiredSize )
            {
                auto order = g_Heap.GetOrderFromBytes(std::min(allocateSize, allocatableSize));
                m_Buffer = reinterpret_cast<char*>(g_Heap.AllocateByOrder(order));
                m_Size = g_Heap.GetBytesFromOrder(order);
            }
        }

        if( m_Buffer != nullptr )
        {
            // HeapAllocatableSizeTrim 以上に多く割り当てしている場合はトリミングする
            if (GetSize() >= allocateSize + HeapAllocatableSizeTrim)
            {
                Shrink(nn::util::align_up(allocateSize, HeapAllocatableSizeTrim));
            }
            NN_SDK_ASSERT_GREATER_EQUAL(GetSize(), requiredSize);

            if( GetSize() < std::min(allocateSize, HeapAllocatableSizeMax))
            {
                g_CountReduceAllocation++;
            }
            break;
        }
        else
        {
            nn::os::SleepThread(RetryWait);
            g_CountRetried++;
        }
    }

    // ピークデバッグ
    {
        std::lock_guard<nn::os::Mutex> lock(g_HeapMutex);

#if defined(NN_DETAIL_FSSYSTEM_ENABLE_CHECKPEAK)
        bool peakUpdated = false;
#endif
        const size_t sizeFree = g_Heap.GetTotalFreeSize();
        if (sizeFree < g_SizeHeapFreePeak)
        {
            g_SizeHeapFreePeak = sizeFree;
#if defined(NN_DETAIL_FSSYSTEM_ENABLE_CHECKPEAK)
            peakUpdated = true;
#endif
        }

#if defined(NN_DETAIL_FSSYSTEM_ENABLE_CHECKPEAK)
        // 1000 回ごと、もしくは peak 更新時
        ++g_CountAllocated;
        if ((0 == (g_CountAllocated % 1000)) || peakUpdated)
        {
            NN_FSPEAK_LOG("[fs] [PooledBuffer] peak=%4zu, req=%4zu, %sssign=%4zu, free=%4zu retry=%zu\n"
                , g_SizeHeapFreePeak / 1024, idealSize / 1024, isEnableLargeCapacity ? "A" : "a"
                , GetSize() / 1024, sizeFree / 1024, g_CountRetried );
        }
#endif
    }
}

void PooledBuffer::Deallocate() NN_NOEXCEPT
{
    Shrink(0);
    NN_SDK_ASSERT_EQUAL(m_Buffer, nullptr);
}

void PooledBuffer::Shrink(size_t idealSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(
        idealSize,
        GetAllocatableSizeMaxCore(true));

    if (m_Size > idealSize)
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
        NN_SDK_ASSERT_EQUAL(g_Heap.GetBlockSize(), HeapBlockSize);

        const auto shrunkSize = nn::util::align_up(idealSize, HeapBlockSize);

        {
            std::lock_guard<nn::os::Mutex> lock(g_HeapMutex);

            while (shrunkSize < m_Size)
            {
                // 開放するには、以下の 2 つを満たす必要がある
                // - 開放するサイズ が 2 のべき乗
                // - 開放する領域が開放するサイズでアライメントされている
                const auto tailAlignment = nn::util::isols1b(m_Size);
                const auto freeSize = std::min(
                                            nn::util::floorp2(m_Size - shrunkSize),
                                            tailAlignment);
                const auto freeOrder = g_Heap.GetOrderFromBytes(freeSize);

                NN_SDK_ASSERT(nn::util::is_aligned(freeSize, HeapBlockSize));
                NN_SDK_ASSERT_EQUAL(freeSize, g_Heap.GetBytesFromOrder(freeOrder));

                g_Heap.Free(m_Buffer + m_Size - freeSize, freeOrder);
                m_Size -= freeSize;
            }
        }

        if (m_Size == 0)
        {
            m_Buffer = nullptr;
        }
    }
}

char* PooledBuffer::GetBuffer() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    return m_Buffer;
}

size_t PooledBuffer::GetSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
    return m_Size;
}

Result InitializeBufferPool(char* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_HeapBuffer == nullptr);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, BufferPoolAlignment);
    NN_RESULT_DO(g_Heap.Initialize(
        reinterpret_cast<uintptr_t>(buffer),
        size,
        HeapBlockSize,
        HeapOrderMaxForLarge + 1));
    g_HeapBuffer = buffer;
    g_HeapSize = size;
    g_SizeHeapFreePeak = size;

    NN_RESULT_SUCCESS;
}

Result InitializeBufferPool(
    char* buffer,
    size_t size,
    char* workBuffer,
    size_t workSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_HeapBuffer == nullptr);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, BufferPoolAlignment);
    NN_SDK_REQUIRES(workSize >= BufferPoolWorkSize);
    NN_RESULT_DO(g_Heap.Initialize(
        reinterpret_cast<uintptr_t>(buffer),
        size,
        HeapBlockSize,
        HeapOrderMaxForLarge + 1,
        workBuffer,
        workSize));
    g_HeapBuffer = buffer;
    g_HeapSize = size;
    g_SizeHeapFreePeak = size;

    NN_RESULT_SUCCESS;
}

bool IsPooledBuffer(const void* buffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    return g_HeapBuffer <= buffer && buffer < reinterpret_cast<char*>(g_HeapBuffer) + g_HeapSize;
}

void RegisterThreadPool(ThreadPool* pThreadPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pThreadPool == nullptr || g_pThreadPool == nullptr);
    g_pThreadPool = pThreadPool;
}

ThreadPool* GetRegisteredThreadPool() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(g_pThreadPool);
    return g_pThreadPool;
}

size_t GetPooledBufferFreeSizePeak() NN_NOEXCEPT
{
    return g_SizeHeapFreePeak;
}

size_t GetPooledBufferRetriedCount() NN_NOEXCEPT
{
    return g_CountRetried;
}

size_t GetPooledBufferReduceAllocationCount() NN_NOEXCEPT
{
    return g_CountReduceAllocation;
}

void ClearPooledBufferPeak() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_HeapMutex);
    g_SizeHeapFreePeak = g_HeapSize;
    g_CountRetried = 0;
    g_CountReduceAllocation = 0;
}

}}
