﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/dbm/fs_DbmPathTool.h>

namespace nn { namespace fssystem { namespace dbm {

namespace {

    /**
    * @brief        ファイル名/ディレクトリ名をコピーします。
    *
    * @param[out]   outDest    コピー先バッファ
    * @param[in]    sizeDest  コピー先バッファのサイズ
    * @param[in]    pSrc      コピー元バッファ
    * @param[in]    sizeSrc   コピー元バッファのサイズ
    *
    * @pre          outDest が NULL ではない。
    * @pre          pSrc が NULL ではない。
    *
    * @details      ファイル名/ディレクトリ名をコピーします。
    *               コピー先のバッファサイズの方が大きい場合、
    *               後ろの部分は空文字列で埋めます。
    */
    inline void CopyAsImpl(
                    PathChar* outDest,
                    size_t sizeDest,
                    const PathChar* pSrc,
                    size_t sizeSrc
                ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outDest);
        NN_SDK_REQUIRES_NOT_NULL(pSrc);

        PathChar* pDestCurrent = outDest;
        PathChar* pDestEnd = pDestCurrent + sizeDest;
        const PathChar* pSrcCurrent = pSrc;
        const PathChar* pSrcEnd = pSrcCurrent + sizeSrc;

        // 文字の存在する間は普通にコピーします
        while( (pDestCurrent < pDestEnd) && (pSrcCurrent < pSrcEnd) )
        {
            *pDestCurrent = *pSrcCurrent;
            pSrcCurrent++;
            pDestCurrent++;
        }

        // 後ろの部分は空文字列で埋めます
        while( pDestCurrent < pDestEnd )
        {
            *pDestCurrent = StringTraits::NullCharactor;
            pDestCurrent++;
        }
    }

    /**
    * @brief        ディレクトリ文字列としてコピーします。
    *
    * @param[out]   outDest コピー先バッファ
    * @param[in]    pSrc      コピー元バッファ
    * @param[in]    sizeSrc   コピー元バッファのサイズ
    *
    * @pre          outDest が NULL ではない。
    * @pre          pSrc が NULL ではない。
    *
    * @details      ディレクトリ文字列としてコピーします。
    *               コピー先のバッファサイズの方が大きい場合、
    *               後ろの部分は空文字列で埋めます。
    */
    inline void CopyAsDirectoryName(
                    DirectoryName* outDest,
                    const PathChar* pSrc,
                    size_t sizeSrc
                ) NN_NOEXCEPT
    {
        CopyAsImpl(outDest->name, sizeof(DirectoryName) / sizeof(PathChar), pSrc, sizeSrc);
    }

    /**
    * @brief        ファイル名文字列としてコピーします。
    *
    * @param[out]   outDest コピー先バッファ
    * @param[in]    pSrc    コピー元バッファ
    * @param[in]    sizeSrc コピー元バッファのサイズ
    *
    * @pre          outDest が NULL ではない。
    * @pre          pSrc が NULL ではない。
    *
    * @details      ファイル名文字列としてコピーします。
    *               コピー先のバッファサイズの方が大きい場合、
    *               後ろの部分は空文字列で埋めます。
    */
    inline void CopyAsFileName(
                    FileName* outDest,
                    const PathChar* pSrc,
                    size_t sizeSrc
                ) NN_NOEXCEPT
    {
        CopyAsImpl(outDest->name, sizeof(FileName) / sizeof(PathChar), pSrc, sizeSrc);
    }

    /**
    * @brief        ファイル名/ディレクトリ名の長さを取得します。
    *
    * @param[in]    pSrc    コピー元バッファ
    * @param[in]    sizeSrc コピー元バッファのサイズ
    *
    * @return       ファイル名 / ディレクトリ名の長さ。
    *
    * @pre          pPath が NULL ではない。
    *
    * @details      ファイル名/ディレクトリ名の長さを取得します。
    */
    size_t GetLengthImpl(const PathChar* pPath, size_t sizeMax) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pPath);

        for( size_t length = 0; length <= sizeMax; ++length )
        {
            if( pPath[length] == StringTraits::NullCharactor )
            {
                return length;
            }
        }
        return sizeMax;
    }
}

/**
* @brief        指定したディレクトリ名の長さを取得します。
*
* @param[in]    pName ディレクトリ名
*
* @return       ディレクトリ名の長さ。
*
* @pre          pName が NULL ではない。
*
* @details      指定したディレクトリ名の長さを取得します。
*/
size_t PathTool::GetDirectoryNameLength(const DirectoryName* pName) NN_NOEXCEPT
{
    return GetLengthImpl(pName->name, sizeof(DirectoryName) / sizeof(PathChar));
}

/**
* @brief        バイトデータをディレクトリ名に変換します。
*
* @param[out]   outName 変換後のディレクトリ名
* @param[in]    pSrc    文字列を含んだバイトデータ
* @param[in]    size    バイトデータの長さ
*
* @return       ディレクトリ名への変換に成功した場合は true、失敗した場合は false。
*
* @pre          outName が NULL ではない。
* @pre          pSrc が NULL ではない。
*
* @details      バイトデータをディレクトリ名に変換します。
*/
bool PathTool::GetDirectoryName(
         DirectoryName* outName,
         const char* pSrc,
         size_t length
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outName);
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    // データの長さをチェックします。
    if( length > sizeof(DirectoryName) / sizeof(PathChar) )
    {
        return false;
    }

    CopyAsDirectoryName(outName, pSrc, length);

    return true;
}

/**
* @brief        指定したファイル名の長さを取得します。
*
* @param[in]    pName   ファイル名
*
* @return       ファイル名の長さ。
*
* @pre          pName が NULL ではない。
*
* @details      指定したファイル名の長さを取得します。
*/
size_t PathTool::GetFileNameLength(const FileName* pName) NN_NOEXCEPT
{
    return GetLengthImpl(pName->name, sizeof(FileName) / sizeof(PathChar));
}

/**
* @brief        バイトデータをファイル名に変換します。
*
* @param[out]   outName 変換後のファイル名
* @param[in]    pSrc    文字列を含んだバイトデータ
* @param[in]    length  バイトデータの長さ
*
* @return       ファイル名への変換に成功した場合は true、失敗した場合は false。
*
* @pre          outName が NULL ではない。
* @pre          pSrc が NULL ではない。
*
* @details      バイトデータをファイル名に変換します。
*/
bool PathTool::GetFileName(FileName* outName, const char* pSrc, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outName);
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    // データの長さをチェックします。
    if( length > sizeof(FileName) / sizeof(PathChar) )
    {
        return false;
    }

    CopyAsFileName(outName, pSrc, length);

    return true;
}

/**
* @brief        ディレクトリ名データをファイル名データに変換します。
*
* @param[out]   outFileName 変換後のファイル名
* @param[in]    pDirectoryName ディレクトリ名
*
* @return       変換に成功した場合は true、そうでなければ false。
*
* @pre          outFileName が NULL ではない。
* @pre          pDirectoryName が NULL ではない。
*
* @details      ディレクトリ名データをファイル名データに変換します。
*/
bool PathTool::ConvertDirectoryNameToFileName(
         FileName* outFileName,
         const DirectoryName* pDirectoryName
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outFileName);
    NN_SDK_REQUIRES_NOT_NULL(pDirectoryName);

    NN_STATIC_ASSERT(sizeof(FileName) >= sizeof(DirectoryName));
    CopyAsFileName(outFileName, pDirectoryName->name, sizeof(DirectoryName) / sizeof(PathChar));
    return true;
}

/**
* @brief        ファイル名データをディレクトリ名データに変換します。
*
* @param[out]   outDirectoryName 変換後のディレクトリ名
* @param[in]    pFileName ファイル名
*
* @return       変換に成功した場合は true、そうでなければ false。
*
* @pre          outDirectoryName が NULL ではない。
* @pre          pFileName が NULL ではない。
*
* @details      ファイル名データをディレクトリ名データに変換します。
*/
bool PathTool::ConvertFileNameToDirectoryName(
         DirectoryName* outDirectoryName,
         const FileName* pFileName
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryName);
    NN_SDK_REQUIRES_NOT_NULL(pFileName);

    NN_STATIC_ASSERT(sizeof(FileName) >= sizeof(DirectoryName));
    CopyAsDirectoryName(outDirectoryName, pFileName->name, sizeof(FileName) / sizeof(PathChar));

    return true;
}

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
PathTool::PathParser::PathParser() NN_NOEXCEPT
    : m_PreviousStartPath(nullptr),
      m_PreviousEndPath(nullptr),
      m_NextPath(nullptr),
      m_IsParseFinished(false)
{
}

/**
* @brief        パーサを初期化します。
*
* @param[in]    pFullPath   フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess               処理が正常に終了しました。
* @retval       ResultInvalidPathFormat     フルパスがルートディレクトリから始まっていません。
*
* @pre          pFullPath が NULL ではない。
*
* @details      パーサを初期化します。
*/
Result PathTool::PathParser::Initialize(const PathChar* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    const PathChar* pIter = pFullPath;

    // フルパスがルートディレクトリから始まっているかどうかチェックします。
    if( ! fssystem::PathTool::IsSeparator(pIter[0]) )
    {
        // 不正なパスです。
        return nn::fs::ResultInvalidPathFormat();
    }
    // セパレーターが連続していたら読み進めておきます
    while( fssystem::PathTool::IsSeparator(pIter[1]) )
    {
        pIter++;
    }

    // 初回のみ先頭のセパレーターを指し示すようにします。
    m_PreviousStartPath = pIter;

    // 初回のみ先頭のセパレーターを指し示すようにします。
    m_PreviousEndPath = m_PreviousStartPath;

    // m_NextPath は常に区切り文字の次の文字を指すようにします。
    m_NextPath = &pIter[1];
    while( fssystem::PathTool::IsSeparator(m_NextPath[0]) )
    {
        m_NextPath ++;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        パーサを破棄します。
*
* @details      パーサを破棄します。
*/
void PathTool::PathParser::Finalize() NN_NOEXCEPT
{
    m_PreviousStartPath = nullptr;
    m_PreviousEndPath = nullptr;
    m_NextPath = nullptr;
    m_IsParseFinished = false;
}

/**
* @brief        パスのパースが完了したかどうかを取得します。
*
* @return       パスのパースが完了したかどうか。
*
* @details      パスのパースが完了したかどうかを取得します。
*/
bool PathTool::PathParser::IsParseFinished() const NN_NOEXCEPT
{
    return m_IsParseFinished;
}

/**
* @brief        パスがディレクトリかどうかを取得します。
*
* @return       パスがディレクトリかどうか。
*
* @pre          初期化している。
* @pre          IsParseFinished が true の時のみ使用できます。
*
* @details      パスがディレクトリかどうかを取得します。
*/
bool PathTool::PathParser::IsDirectoryPath() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_NextPath);
    NN_SDK_REQUIRES(IsParseFinished());

    // パスの最後が区切り文字であれば true を返します。
    if( (m_NextPath[0] == StringTraits::NullCharactor)
     && (fssystem::PathTool::IsSeparator(m_NextPath[-1])) )
    {
        return true;
    }

    // カレントディレクトリを示す文字であれば true を返します。
    if( fssystem::PathTool::IsCurrentDirectory(m_NextPath) )
    {
        return true;
    }

    // 親ディレクトリを示す文字であれば true を返します。
    if( fssystem::PathTool::IsParentDirectory(m_NextPath) )
    {
        return true;
    }

    return false;
}

/**
* @brief        最上位ディレクトリ部分を取得します。
*
* @param[out]   outDirectoryName ディレクトリ名
* @param[out]   outDirectoryNameLength ディレクトリ名の長さ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess               処理が正常に終了しました。
* @retval       ResultTooLongPath           ディレクトリ名が長すぎます。
*
* @pre          初期化している。
* @pre          outDirectoryName が NULL ではない。
* @pre          outDirectoryNameLength が NULL ではない。
*
* @details      最上位ディレクトリ部分を取得します。
*/
Result PathTool::PathParser::GetNextDirectoryName(
           DirectoryName* outDirectoryName,
           size_t* outDirectoryNameLength
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_PreviousStartPath);
    NN_SDK_REQUIRES_NOT_NULL(m_PreviousEndPath);
    NN_SDK_REQUIRES_NOT_NULL(m_NextPath);
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryName);
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryNameLength);

    // 現在のディレクトリ名をコピーします。
    *outDirectoryNameLength = (m_PreviousEndPath - m_PreviousStartPath);
    CopyAsDirectoryName(outDirectoryName, m_PreviousStartPath, *outDirectoryNameLength);

    // 次のパスの始点へ進めます
    m_PreviousStartPath = m_NextPath;

    // 次の区切り文字を探索します。
    const PathChar* p = m_NextPath;
    for( size_t dirNameLength = 0; ; ++dirNameLength )
    {
        // 区切り文字が見つかれば探索終了です。
        if( fssystem::PathTool::IsSeparator(p[dirNameLength]) )
        {
            if( dirNameLength > (sizeof(DirectoryName) / sizeof(PathChar)) )
            {
                // ディレクトリ名が長すぎます。
                return nn::fs::ResultTooLongPath();
            }

            // 区切り文字の一文字手前を指すようにします。
            m_PreviousEndPath = &p[dirNameLength];
            m_NextPath = m_PreviousEndPath + 1;

            // 連続区切り文字の次の文字を指すようにします。
            while( fssystem::PathTool::IsSeparator(*m_NextPath) )
            {
                m_NextPath ++;
            }
            if( *m_NextPath == StringTraits::NullCharactor )
            {
                m_IsParseFinished = true;
            }
            break;
        }

        // 終端文字が見つかれば探索終了です。
        if( p[dirNameLength] == StringTraits::NullCharactor )
        {
            m_IsParseFinished = true;
            m_PreviousEndPath = m_NextPath = &p[dirNameLength];
            break;
        }
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        文字列の先頭から次の１節をディレクトリ名として取得します。
*
* @param[out]   outName         ディレクトリ名
* @param[out]   outNameLength   ディレクトリ名の長さ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess           処理が正常に終了しました。
* @retval       ResultTooLongPath       ディレクトリ名が長すぎます。
*
* @pre          初期化している。
* @pre          outName が NULL ではない。
* @pre          outNameLength が NULL ではない。
*
* @details      文字列の先頭から次の１節をディレクトリ名として取得します。
*/
Result PathTool::PathParser::GetAsDirectoryName(
           DirectoryName* outName,
           size_t* outNameLength
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_NextPath);
    NN_SDK_REQUIRES_NOT_NULL(m_PreviousStartPath);
    NN_SDK_REQUIRES_NOT_NULL(m_PreviousEndPath);
    NN_SDK_REQUIRES_NOT_NULL(outName);
    NN_SDK_REQUIRES_NOT_NULL(outNameLength);

    size_t numChar = m_PreviousEndPath - m_PreviousStartPath;

    if( numChar > sizeof(DirectoryName) )
    {
        // ディレクトリ名が長すぎます。
        return nn::fs::ResultTooLongPath();
    }

    // ディレクトリ名をコピーします。
    *outNameLength = numChar;
    CopyAsDirectoryName(outName, m_PreviousStartPath, numChar);

    NN_RESULT_SUCCESS;
}

/**
* @brief        文字列の先頭から次の１節をファイル名として取得します。
*
* @param[out]   outName ファイル名
* @param[out]   outNameLength ファイル名の長さ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess           処理が正常に終了しました。
* @retval       ResultTooLongPath       ファイル名が長すぎます。
* @retval       ResultInvalidPathFormat パスの最後が不正です。
*
* @pre          初期化している。
* @pre          outName が NULL ではない。
* @pre          outNameLength が NULL ではない。
*
* @details      文字列の先頭から次の１節をファイル名として取得します。
*/
Result PathTool::PathParser::GetAsFileName(
           FileName* outName,
           size_t* outNameLength
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_NextPath);
    NN_SDK_REQUIRES_NOT_NULL(m_PreviousStartPath);
    NN_SDK_REQUIRES_NOT_NULL(m_PreviousEndPath);
    NN_SDK_REQUIRES_NOT_NULL(outName);
    NN_SDK_REQUIRES_NOT_NULL(outNameLength);

    size_t numChar = m_PreviousEndPath - m_PreviousStartPath;
    if( numChar > sizeof(FileName) )
    {
        // ファイル名が長すぎます。
        return nn::fs::ResultTooLongPath();
    }

    if( fssystem::PathTool::IsParentDirectory(m_PreviousStartPath)
     || fssystem::PathTool::IsCurrentDirectory(m_PreviousStartPath) )
    {
        // パスの最後が"."、".."です。
        return nn::fs::ResultInvalidPathFormat();
    }

    // ファイル名をコピーします。
    *outNameLength = numChar;
    CopyAsFileName(outName, m_PreviousStartPath, numChar);

    NN_RESULT_SUCCESS;
}

}}}

