﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>
#include <nn/fssystem/buffers/fs_IBufferManager.h>
#include <nn/fssystem/dbm/fs_BufferedAllocationTableStorage.h>

namespace nn { namespace fssystem { namespace dbm {

namespace {
    const int64_t InvalidCacheOffset = -1;

    bool Overlaps(
        int64_t lhsOffset,
        int64_t lhsSize,
        int64_t rhsOffset,
        int64_t rhsSize) NN_NOEXCEPT
    {
        return lhsOffset < rhsOffset + rhsSize && rhsOffset < lhsOffset + lhsSize;
    }
}

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
BufferedAllocationTableStorage::BufferedAllocationTableStorage() NN_NOEXCEPT
    : m_CacheOffsets(),
      m_CacheCount(0),
      m_NextCacheIndex(0),
      m_Buffer(nullptr),
      m_BufferSize(0),
      m_EnableCount(0),
      m_pBufferManager(nullptr)
{
}

/**
* @brief        デストラクタです。
*
* @details      デストラクタです。
*/
BufferedAllocationTableStorage::~BufferedAllocationTableStorage() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_EnableCount, 0);
    NN_SDK_ASSERT(m_CacheOffsets == nullptr);
    NN_SDK_ASSERT(m_Buffer == nullptr);
    NN_SDK_ASSERT_EQUAL(m_CacheCount, 0);
    NN_SDK_ASSERT_EQUAL(m_BufferSize, static_cast<size_t>(0));
}

/**
* @brief        ストレージをマウントします。
*
* @param[in]    pAllocationTable    アロケーションテーブル
* @param[in]    countManageBlock    管理用ブロックの数
* @param[in]    sizeBlock           ブロックのサイズ
* @param[in]    storage             実データ用ストレージ
* @param[in]    pBufferManager      バッファマネージャ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       処理が正常に終了しました。
* @retval       上記以外            アロケーションテーブルからのデータ読み込みに失敗しました。
*
* @pre          pAllocationTable != nullptr
* @pre          pBufferManager != nullptr
* @pre          offsetStorage >= 0
* @pre          sizeBlock >= MinBlockSize
*
* @details      ストレージをマウントします。
*/
Result BufferedAllocationTableStorage::InitializeBuffered(
            AllocationTable* pAllocationTable,
            uint32_t countManageBlock,
            uint32_t sizeBlock,
            fs::SubStorage storage,
            IBufferManager* pBufferManager
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBufferManager);

    NN_RESULT_DO(
        AllocationTableStorage::Initialize(
            pAllocationTable,
            countManageBlock,
            sizeBlock,
            storage
        )
    );

    m_pBufferManager = pBufferManager;
    m_CacheOffsets.reset();
    m_CacheCount = 0;
    m_NextCacheIndex = 0;
    m_Buffer = nullptr;
    m_BufferSize = 0;
    m_EnableCount = 0;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージをアンマウントします。
*
* @details      ストレージをアンマウントします。
*/
void BufferedAllocationTableStorage::FinalizeBuffered() NN_NOEXCEPT
{
    DisableCacheBuffer();

    AllocationTableStorage::Finalize();
}

/**
* @brief        実ストレージの offset から size バイト読み込み buffer にコピーします。
*
* @param[in]    offset              オフセット
* @param[out]   outBuffer           読み込み先バッファ
* @param[in]    size                読み込むサイズ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   読み込みが終了しました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             offset が範囲外です。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          offset は 0 以上。
*
* @details      実ストレージの offset から size バイト読み込み buffer にコピーします。
*/
Result BufferedAllocationTableStorage::Read(
           int64_t offset,
           void* outBuffer,
           size_t size
       ) NN_NOEXCEPT
{
    if( 0 < m_EnableCount )
    {
        // 必ずロックを獲得した状態で呼び出されることとパフォーマンスを重視することからロックしない
        // そのため、このブロックの処理はスレッドセーフでない
        const auto blockSize = static_cast<size_t>(AllocationTableStorage::GetBlockSize(1));
        auto currentOffset = offset;
        auto remainSize = size;

        while( 0 < remainSize )
        {
            const auto alignedOffset = nn::util::align_down(currentOffset, blockSize);
            const auto paddingSize = static_cast<size_t>(currentOffset - alignedOffset);
            const auto readSize = 0 < paddingSize || remainSize <= blockSize
                ? std::min(blockSize - paddingSize, remainSize)
                : remainSize;
            const auto currentBuffer
                = reinterpret_cast<char*>(outBuffer) + (currentOffset - offset);

            if( readSize <= blockSize )
            {
                // キャッシュを探索する
                auto cacheIndex = 0;
                auto invalidCacheIndex = m_CacheCount;
                for( ; cacheIndex < m_CacheCount; ++cacheIndex )
                {
                    const auto cacheOffset = m_CacheOffsets[cacheIndex];
                    if( cacheOffset != InvalidCacheOffset )
                    {
                        if( Overlaps(currentOffset, readSize, cacheOffset, blockSize) )
                        {
                            break;
                        }
                    }
                    else if( invalidCacheIndex == m_CacheCount )
                    {
                        invalidCacheIndex = cacheIndex;
                    }
                }

                // 見つからなければ新たにキャッシュに読み込む
                if( cacheIndex == m_CacheCount )
                {
                    if( invalidCacheIndex == m_CacheCount )
                    {
                        cacheIndex = m_NextCacheIndex;
                    }
                    else
                    {
                        cacheIndex = invalidCacheIndex;
                    }
                    m_CacheOffsets[cacheIndex] = alignedOffset;

                    NN_RESULT_DO(
                        AllocationTableStorage::Read(
                            alignedOffset,
                            m_Buffer + (cacheIndex * blockSize),
                            blockSize
                        )
                    );
                }

                // キャッシュから読み込む
                NN_SDK_ASSERT_RANGE(cacheIndex, 0, m_CacheCount);
                std::memcpy(
                    currentBuffer,
                    m_Buffer + (cacheIndex * blockSize) + paddingSize,
                    readSize
                );
            }
            else
            {
                // キャッシュせずに読み込む
                NN_RESULT_DO(AllocationTableStorage::Read(currentOffset, currentBuffer, readSize));
            }

            currentOffset += readSize;
            remainSize -= readSize;
        }

        NN_RESULT_SUCCESS;
    }
    else
    {
        // キャッシュせずに読み込む
        return AllocationTableStorage::Read(offset, outBuffer, size);
    }
}

/**
* @brief        実ストレージの offset 以降に buffer を size バイト分コピーします。
*
* @param[in]    offset              オフセット
* @param[in]    srcBuffer           書き込み元バッファ
* @param[in]    size                書き込むサイズ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にコピーしました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             offset が範囲外です。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          offset が 0 以上。
*
* @details      実ストレージの offset 以降に buffer を size バイト分コピーします。
*/
Result BufferedAllocationTableStorage::Write(
           int64_t offset,
           const void* srcBuffer,
           size_t size
       ) NN_NOEXCEPT
{
    // 書き込む範囲のキャッシュを無効化する
    if( 0 < m_EnableCount )
    {
        // 必ずロックを獲得した状態で呼び出されることとパフォーマンスを重視することからロックしない
        // そのため、このブロックの処理はスレッドセーフでない
        InvalidateCache(offset, size);
    }

    // 書き込む
    return AllocationTableStorage::Write(offset, srcBuffer, size);
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
* @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
* @param[in]    operationId     範囲指定処理の種類
* @param[in]    offset          範囲指定処理開始位置
* @param[in]    size            範囲指定処理を行うデータサイズ
* @param[in]    inBuffer        範囲指定処理に渡すバッファ
* @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result BufferedAllocationTableStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    if( operationId == fs::OperationId::Invalidate && 0 < m_EnableCount )
    {
        InvalidateCache(offset, size);
    }
    return AllocationTableStorage::OperateRange(
        outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
}

void BufferedAllocationTableStorage::InvalidateCache(int64_t offset, int64_t size) NN_NOEXCEPT
{
    const auto blockSize = static_cast<size_t>(AllocationTableStorage::GetBlockSize(1));
    for( auto cacheIndex = m_CacheCount - 1; 0 <= cacheIndex; --cacheIndex )
    {
        auto& cacheOffset = m_CacheOffsets[cacheIndex];
        if( cacheOffset != InvalidCacheOffset
            && Overlaps(offset, size, cacheOffset, blockSize) )
        {
            cacheOffset = InvalidCacheOffset;
            m_NextCacheIndex = cacheIndex;
        }
    }
}

/**
* @brief        バッファリングを有効化します。
*
* @param[in]    cacheCount   キャッシュ数
*
* @return       関数の処理結果を返します。
*
* @pre
*               - 0 < cacheCount
*
* @details      バッファリングを有効化します。
*/
Result BufferedAllocationTableStorage::EnableCacheBuffer(int cacheCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBufferManager);
    NN_SDK_REQUIRES_GREATER(cacheCount, 0);

    if( m_EnableCount == 0 )
    {
        NN_SDK_ASSERT(m_CacheOffsets == nullptr);
        NN_SDK_ASSERT(m_Buffer == nullptr);
        NN_SDK_ASSERT_EQUAL(m_CacheCount, 0);
        NN_SDK_ASSERT_EQUAL(m_BufferSize, static_cast<size_t>(0));

        // この関数を呼び出している CacheBufferEnabler::CacheBufferEnabler は
        // ResultBufferAllocationFailed を無視するので、単に AllocateBuffer を呼ぶ
        const auto blockSize = static_cast<size_t>(AllocationTableStorage::GetBlockSize(cacheCount));
        const auto buffer = m_pBufferManager->AllocateBuffer(
                                blockSize,
                                IBufferManager::BufferAttribute()
                            );
        NN_RESULT_THROW_UNLESS(buffer.first != 0, nn::fs::ResultBufferAllocationFailed());

        m_CacheOffsets = nn::fs::detail::MakeUnique<int64_t[]>(cacheCount);
        if( m_CacheOffsets != nullptr )
        {
            for( auto cacheIndex = 0; cacheIndex < cacheCount; ++cacheIndex )
            {
                m_CacheOffsets[cacheIndex] = InvalidCacheOffset;
            }
        }
        else
        {
            m_pBufferManager->DeallocateBuffer(buffer.first, buffer.second);
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedMakeUnique());
        }

        m_CacheCount = cacheCount;
        m_NextCacheIndex = 0;
        m_Buffer = reinterpret_cast<char*>(buffer.first);
        m_BufferSize = buffer.second;
    }

    ++m_EnableCount;

    NN_SDK_ASSERT_GREATER(m_EnableCount, 0);
    NN_SDK_ASSERT_NOT_NULL(m_CacheOffsets);
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);
    NN_SDK_ASSERT_GREATER(m_CacheCount, 0);
    NN_SDK_ASSERT_GREATER(m_BufferSize, static_cast<size_t>(0));

    NN_RESULT_SUCCESS;
}

/**
* @brief        バッファリングを無効化します。
*
* @details      バッファリングを無効化します。
*/
void BufferedAllocationTableStorage::DisableCacheBuffer() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBufferManager);

    if( 0 < m_EnableCount )
    {
        --m_EnableCount;
    }
    else
    {
        return;
    }

    if( m_EnableCount == 0 )
    {
        NN_SDK_ASSERT(m_CacheOffsets != nullptr);
        NN_SDK_ASSERT_GREATER(m_CacheCount, 0);
        m_CacheOffsets.reset();
        m_CacheCount = 0;

        NN_SDK_ASSERT(m_Buffer != nullptr);
        NN_SDK_ASSERT_GREATER(m_BufferSize, static_cast<size_t>(0));
        m_pBufferManager->DeallocateBuffer(reinterpret_cast<uintptr_t>(m_Buffer), m_BufferSize);
        m_Buffer = nullptr;
        m_BufferSize = 0;
    }
}

}}}

