﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>
#include <nn/fssystem/dbm/fs_AllocationTableStorage.h>

namespace nn { namespace fssystem { namespace dbm {


NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTableStorage::UnallocatedIndex );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTableStorage::MinBlockSize );

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
AllocationTableStorage::AllocationTableStorage() NN_NOEXCEPT
    : m_pAllocTable(nullptr),
      m_Storage(),
      m_IndexTop(UnallocatedIndex),
      m_BlockSizeShift(0),
      m_SessionCountSeek(0),
      m_SessionIndex(0),
      m_SessionCountBlock(0),
      m_SessionNext(0)
{
}

/**
* @brief        ストレージをマウントします。
*
* @param[in]    pAllocationTable    アロケーションテーブル
* @param[in]    countManageBlock    管理用ブロックの数
* @param[in]    sizeBlock           ブロックのサイズ
* @param[in]    storage             実データ用ストレージ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       処理が正常に終了しました。
* @retval       上記以外            アロケーションテーブルからのデータ読み込みに失敗しました。
*
* @pre          offsetStorage >= 0
* @pre          sizeBlock >= MinBlockSize
*
* @details      ストレージをマウントします。
*/
Result AllocationTableStorage::Initialize(
            AllocationTable* pAllocationTable,
            uint32_t countManageBlock,
            uint32_t sizeBlock,
            fs::SubStorage storage
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocationTable);
    NN_SDK_REQUIRES_LESS_EQUAL(MinBlockSize, sizeBlock);

    m_pAllocTable = pAllocationTable;
    m_IndexTop = countManageBlock;

    // シフト数を計算します。2のべき乗
    uint32_t sizeShift = sizeBlock >> 1;
    m_BlockSizeShift = 0;
    do
    {
        sizeShift >>= 1;
        m_BlockSizeShift++;
    }
    while( sizeShift > 0 );
    NN_SDK_ASSERT_LESS(0U, m_BlockSizeShift);

    m_Storage = storage;

    return SeekToTop();
}

/**
* @brief        ストレージをアンマウントします。
*
* @details      ストレージをアンマウントします。
*/
void AllocationTableStorage::Finalize() NN_NOEXCEPT
{
    m_pAllocTable = nullptr;
    m_IndexTop = UnallocatedIndex;
    m_BlockSizeShift = 0;
}

/**
* @brief        ストレージの総バイトサイズを取得します。
*
* @param[out]   outValue        ストレージの総バイト数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に取得できました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          outValue が NULL ではない。
*
* @details      ストレージの総バイトサイズを取得します。
*/
Result AllocationTableStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    uint32_t index = m_IndexTop;
    uint32_t totalCount = 0;
    uint32_t count;

    while( index != AllocationTable::IndexEnd )
    {
        NN_RESULT_DO(m_pAllocTable->ReadNext(&index, &count, index));
        totalCount += count;
    }

    *outValue = GetBlockSize(totalCount);

    NN_RESULT_SUCCESS;
}

/**
* @brief        実ストレージの offset から size バイト読み込み buffer にコピーします。
*
* @param[in]    offset              オフセット
* @param[out]   outBuffer           読み込み先バッファ
* @param[in]    size                読み込むサイズ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   読み込みが終了しました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             offset が範囲外です。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          offset は 0 以上。
*
* @details      実ストレージの offset から size バイト読み込み buffer にコピーします。
*/
Result AllocationTableStorage::Read(
           int64_t offset,
           void* outBuffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(offset >= 0);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());

    uint8_t* buffer = static_cast<uint8_t*>(outBuffer);
    int64_t currentOffset = offset;

    // 初期済みかチェック
    NN_FSP_REQUIRES(UnallocatedIndex != m_IndexTop, nn::fs::ResultNotInitialized());

    int64_t storageSize;
    NN_RESULT_DO(m_Storage.GetSize(&storageSize));

    // 連続ブロック単位で読み込みます
    while( size != 0 )
    {
        uint32_t resultIndex;
        uint32_t resultBlockCount;
        int64_t resultOffset;

        // シーク位置を求めます
        NN_RESULT_DO(
            SeekTo(
                &resultIndex,
                &resultBlockCount,
                &resultOffset,
                currentOffset
            )
        );

        int64_t offsetRead = GetBlockSize(resultIndex) + resultOffset;
        int64_t sizeRead = GetBlockSize(resultBlockCount) - resultOffset;
        if( sizeRead > static_cast<int64_t>(size) )
        {
            sizeRead = size;
        }

        NN_FSP_REQUIRES(offsetRead >= 0, nn::fs::ResultInvalidAllocationTableBlock());
        NN_FSP_REQUIRES(sizeRead <= storageSize, nn::fs::ResultInvalidAllocationTableBlock());
        NN_FSP_REQUIRES(offsetRead <= storageSize - sizeRead, nn::fs::ResultInvalidAllocationTableBlock());

        // 実際にデータを読み込みます
        NN_RESULT_DO(
            m_Storage.Read(
                offsetRead,
                buffer,
                static_cast<size_t>(sizeRead)
            )
        );

        size -= static_cast<size_t>(sizeRead);
        buffer += sizeRead;
        currentOffset += sizeRead;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        実ストレージの offset 以降に buffer を size バイト分コピーします。
*
* @param[in]    offset              オフセット
* @param[in]    srcBuffer           書き込み元バッファ
* @param[in]    size                書き込むサイズ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にコピーしました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             offset が範囲外です。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          offset が 0 以上。
*
* @details      実ストレージの offset 以降に buffer を size バイト分コピーします。
*/
Result AllocationTableStorage::Write(
           int64_t offset,
           const void* srcBuffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(offset >= 0);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(srcBuffer != nullptr, nn::fs::ResultNullptrArgument());

    // 初期済みかチェック
    NN_FSP_REQUIRES(UnallocatedIndex != m_IndexTop, nn::fs::ResultNotInitialized());

    int64_t currentOffset = offset;
    const uint8_t* buffer = static_cast<const uint8_t*>(srcBuffer);
    size_t lastSize = size;

    int64_t storageSize;
    NN_RESULT_DO(m_Storage.GetSize(&storageSize));
    while( lastSize != 0 )
    {
        uint32_t resultIndex;
        uint32_t resultBlockCount;
        int64_t resultOffset;

        // シーク位置を求めます
        NN_RESULT_DO(
            SeekTo(
                &resultIndex,
                &resultBlockCount,
                &resultOffset,
                currentOffset
            )
        );

        int64_t offsetWrite = GetBlockSize(resultIndex) + resultOffset;
        int64_t sizeWrite = GetBlockSize(resultBlockCount) - resultOffset;
        if( sizeWrite > static_cast<int64_t>(lastSize) )
        {
            sizeWrite = lastSize;
        }

        NN_FSP_REQUIRES(offsetWrite >= 0, nn::fs::ResultInvalidAllocationTableBlock());
        NN_FSP_REQUIRES(sizeWrite <= storageSize, nn::fs::ResultInvalidAllocationTableBlock());
        NN_FSP_REQUIRES(offsetWrite <= storageSize - sizeWrite, nn::fs::ResultInvalidAllocationTableBlock());

        // 実際にデータを書き込みます。
        NN_RESULT_DO(
            m_Storage.Write(
                offsetWrite,
                buffer,
                static_cast<size_t>(sizeWrite)
            )
        );

        lastSize -= static_cast<size_t>(sizeWrite);
        buffer += sizeWrite;
        currentOffset += sizeWrite;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result AllocationTableStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(offset >= 0);

    int64_t currentOffset = offset;
    int64_t sizeRemain = size;

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    // 初期化済みかチェック
    NN_FSP_REQUIRES(UnallocatedIndex != m_IndexTop, nn::fs::ResultNotInitialized());

    int64_t storageSize = 0;
    NN_RESULT_DO(m_Storage.GetSize(&storageSize));

    // 連続ブロック単位で操作します
    while( sizeRemain != 0 )
    {
        uint32_t resultIndex;
        uint32_t resultBlockCount;
        int64_t resultOffset;

        // シーク位置を求めます
        NN_RESULT_DO(
            SeekTo(
                &resultIndex,
                &resultBlockCount,
                &resultOffset,
                currentOffset
            )
        );

        int64_t offsetOperate = GetBlockSize(resultIndex) + resultOffset;
        int64_t sizeOperate = GetBlockSize(resultBlockCount) - resultOffset;
        if( sizeOperate > sizeRemain )
        {
            sizeOperate = sizeRemain;
        }

        NN_FSP_REQUIRES(
            offsetOperate >= 0,
            nn::fs::ResultInvalidAllocationTableBlock());
        NN_FSP_REQUIRES(
            sizeOperate <= storageSize,
            nn::fs::ResultInvalidAllocationTableBlock());
        NN_FSP_REQUIRES(
            offsetOperate <= storageSize - sizeOperate,
            nn::fs::ResultInvalidAllocationTableBlock());

        // 実際に操作します
        NN_RESULT_DO(
            m_Storage.OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offsetOperate,
                static_cast<size_t>(sizeOperate),
                inBuffer,
                inBufferSize
            )
        );

        sizeRemain -= sizeOperate;
        currentOffset += sizeOperate;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        割り当て領域を増やします。
*
* @param[in]    counts                      確保するブロックの個数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess               正常に割り当て領域を増やしました。
* @retval       ResultAllocationTableFull   アロケーションテーブルに空きがありません。
* @retval       上記以外                    アロケーションテーブルでの読み書きに失敗しました。
*
* @details      割り当て領域を増やします。
*/
Result AllocationTableStorage::AllocateBlock(uint32_t counts) NN_NOEXCEPT
{
    uint32_t index;
    NN_RESULT_DO(m_pAllocTable->Allocate(&index, counts));
    if( UnallocatedIndex == m_IndexTop )
    {
        m_IndexTop = index;
    }
    else
    {
        NN_RESULT_DO(m_pAllocTable->Concat(m_IndexTop, index));
        if( m_SessionNext == AllocationTable::IndexEnd )
        {
            // 後ろにブロックが接続されたため
            // セッションを更新します
            m_SessionNext = index;
        }
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        offset 位置を含むブロックを取得します。
*
* @param[out]   outIndex        ブロックインデックス
* @param[out]   outBlockCount   該当ブロック数
* @param[out]   outOffset       ブロック内での位置
* @param[in]    offset          先頭からの位置
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       offset 位置を含むブロックを取得しました。
* @retval       ResultInvalidOffset offset 位置が範囲外でした。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @details      offset 位置を含むブロックを取得します。
*/
Result AllocationTableStorage::SeekTo(
           uint32_t* outIndex,
           uint32_t* outBlockCount,
           int64_t* outOffset,
           int64_t offset
       ) const NN_NOEXCEPT
{
    for( ; ; )
    {
        int64_t offsetRangeLo = GetBlockSize(m_SessionCountSeek);
        int64_t offsetRangeHi = GetBlockSize(m_SessionCountSeek + m_SessionCountBlock);

        // シークが不要なら終了
        {
            bool nextIsEnd = (m_SessionNext == AllocationTable::IndexEnd);
            bool isFinished = ((offset >= offsetRangeHi) && nextIsEnd);
            if( isFinished )
            {
                return nn::fs::ResultInvalidOffset();
            }
            bool isContain = (offsetRangeLo <= offset) && (offset < offsetRangeHi);
            if( isContain )
            {
                break;
            }
        }

        // 後ろにシークする場合
        if( offsetRangeHi <= offset )
        {
            // 1ブロック分、シークします
            m_SessionCountSeek += m_SessionCountBlock;
            m_SessionIndex = m_SessionNext;
            NN_RESULT_DO(
                m_pAllocTable->ReadNext(
                    &m_SessionNext,
                    &m_SessionCountBlock,
                    m_SessionIndex
                )
            );
        }
        else
        {
            // 先頭にシークします
            NN_SDK_ASSERT_LESS(offset, offsetRangeLo);
            NN_RESULT_DO(SeekToTop());
        }
    }

    *outIndex = m_SessionIndex;
    *outBlockCount = m_SessionCountBlock;
    *outOffset = offset - GetBlockSize(m_SessionCountSeek);

    NN_RESULT_SUCCESS;
}

/**
* @brief        シークセッションを先頭に戻します。
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       処理が正常に終了しました。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
*
* @details      シークセッションを先頭に戻します。
*/
Result AllocationTableStorage::SeekToTop() const NN_NOEXCEPT
{
    // シーク位置を0にリセットします
    m_SessionCountSeek = 0;
    m_SessionIndex = m_IndexTop;
    return m_pAllocTable->ReadNext(
               &m_SessionNext,
               &m_SessionCountBlock,
               m_SessionIndex
           );
}

}}}

