﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fssrv/fssrv_SpeedEmulationStorage.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/fssystem/fs_SpeedEmulationConfiguration.h>
#include <nn/fssystem/fs_StorageLayoutTypeSetter.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_TinyMt.h>

namespace nn { namespace fssrv {

namespace {
    nn::util::TinyMt g_TinyMt;
    bool g_IsTinyMtInitialized = false;
    nn::os::Mutex g_MutexTinyMt(false);

    nn::os::Mutex g_MutexStorage(false);

    struct DelayParameter
    {
        size_t lowerLimitAccessSize;
        int64_t picoSecondsPerBytes;
        int64_t nanoSecondsConst;
    };
    NN_STATIC_ASSERT(std::is_pod<DelayParameter>::value);

    enum class AccessPattern
    {
        SequentialRead,
        RandomRead,
        SequentialWrite,
        RandomWrite,
        Count
    };
    static const size_t AccessPatternCount = static_cast<size_t>(AccessPattern::Count);

    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=106352519
    // まとめ_速度エミュレーション用_20180118(1).xlsx
    const int DelayParameterArrayLength = 10;

    const DelayParameter MmcAverageDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {261193, 3407, 184092},
            {17948,  3381, 190883},
            {4097,   4894, 163729},
            {0,      0,    183775}
        },
        // AccessPattern_RandomRead
        {
            {1048985, 3432, 176718},
            {20993,   3302, 313086},
            {8193,    4527, 287370},
            {0,       6304, 272812}
        },
        // AccessPattern_SequentialWrite
        {
            {262140, 11926, 2899501},
            {131061, 22867, 31436},
            {32769,  19437, 480972},
            {17748,  36500, -78164},
            {0,      0,     569633}
        },
        // AccessPattern_RandomWrite
        {
            {4194096, 76340, -265444881},
            {262138,  12280, 3228903},
            {262033,  23330, 332289},
            {16013,   22740, 486888},
            {1024,    0,     851001},
            {0,       0,     624000}
        }
    };

    const DelayParameter MmcWorstDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {4204119, 6270, -11941520},
            {0,       3300, 544711},
        },
        // AccessPattern_RandomRead
        {
            {4074579, 3270, 1941000},
            {40679,   3590, 637135},
            {0,       4560, 597677},
        },
        // AccessPattern_SequentialWrite
        {
            {1606584, 12600,  10007416},
            {38709,   11880,  11164156},
            {0,       225250, 2905018},
        },
        // AccessPattern_RandomWrite
        {
            {4194315, 731320, -3002632110LL},
            {524371,  6110,   39126828},
            {131105,  24660,  29399753},
            {513,     40200,  27362390},
            {0,       0,      6826000},
        }
    };

    const DelayParameter SdCardAverageDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {2167, 42973, 689282},
            {0,    0,     782376}
        },
        // AccessPattern_RandomRead
        {
            {16384, 40914, 1200666},
            {11886, 21118, 1525000},
            {0,     0,     1776000}
        },
        // AccessPattern_SequentialWrite
        {
            {43507,  94137,  1946409},
            {17172,  104266, 1505735},
            {0,      0,      3296116}
        },
        // AccessPattern_RandomWrite
        {
            {1048568, 280737, -62325999},
            {262144,  186826, 36146000},
            {131072,  418266, -24524500},
            {65536,   98761,  17353600},
            {32769,   395682,  -2105400},
            {16385,   25849,  10013300},
            {8193,    382959, 4162400},
            {4096,    49952,  6890400},
            {1024,    0,      7095000},
            {0,       0,      6641800}
        }
    };

    const DelayParameter SdCardWorstDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {131076, 45842, 2041698},
            {23638,  56441, 652428},
            {0,      0,     1986536}
        },
        // AccessPattern_RandomRead
        {
            {246145, 45956, 2254000},
            {32768,  18608, 8985571},
            {0,      0,     9595304}
        },
        // AccessPattern_SequentialWrite
        {
            {1048580, 14830,    1520855142},
            {262144,  129525,   1400588333},
            {32955,   415237,   1325690666},
            {16838,   78275645, -1240178000},
            {0,       0,        77768576}
        },
        // AccessPattern_RandomWrite
        {
            {1048576, 2103711,  3615117857},
            {263985,  0,        5821017728},
            {33125,   19441198, 688848142},
            {1024,    0,        1332830608},
            {0,       0,        57142536}
        }
    };

    const DelayParameter GameCardAverageDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {0, 46320, 1141822}
        },
        // AccessPattern_RandomRead
        {
            {0, 46320, 1141822}
        },
        // AccessPattern_SequentialWrite
        {
            {0, 0, 0}
        },
        // AccessPattern_RandomWrite
        {
            {0, 0, 0}
        }
    };

    const DelayParameter GameCardWorstDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {0, 91770, 2372975}
        },
        // AccessPattern_RandomRead
        {
            {0, 91770, 2372975}
        },
        // AccessPattern_SequentialWrite
        {
            {0, 0, 0}
        },
        // AccessPattern_RandomWrite
        {
            {0, 0, 0}
        }
    };

    const DelayParameter UsbAverageDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {131076, 37970, -586874},
            {65536,  35930, -319480},
            {32768,  21995, 593754},
            {16384,  35571, 148896},
            {8192,   23154, 352336},
            {2049,   33381, 268557},
            {0,      80347, 172369}
        },
        // AccessPattern_RandomRead
        {
            {65531, 39446, 8338842},
            {0,     30784, 8906468}
        },
        // AccessPattern_SequentialWrite
        {
            {1048626, 28350,  -34120},
            {524310,  11025,  18133320},
            {262146,  8128,   19652244},
            {131072,  119838, -9632027},
            {65536,   24157,  2909068},
            {32768,   101344, -2149447},
            {8192,    27408,  273276},
            {2049,    40651,  164795},
            {0,       63174,  118667}
        },
        // AccessPattern_RandomWrite
        {
            {262130, 29248, 3747854},
            {105394, 35434, 2126324},
            {32768,  0,     5860841},
            {16049,  94925, 2750347},
            {0,      0,     4273797}
        }
    };

    const DelayParameter UsbWorstDelayParameterTable[AccessPatternCount][DelayParameterArrayLength] =
    {
        // AccessPattern_SequentialRead
        {
            {5045006, 41379, -378622},
            {4097,    29485, 523704},
            {0,       49189, 442996}
        },
        // AccessPattern_RandomRead
        {
            {0, 40895, 9070438}
        },
        // AccessPattern_SequentialWrite
        {
            {1048607, 31350, 177879},
            {262143,  13059, 19357936},
            {65534,   90945, -1059296},
            {18576,   79121, -284423},
            {0,       43427, 378628}
        },
        // AccessPattern_RandomWrite
        {
            {524232, 32259, 3915411},
            {32768,  29820, 5194011},
            {16785,  97925, 2962347},
            {0,      0,     4605984}
        },
    };

    void WaitNanoSeconds(int64_t time) NN_NOEXCEPT
    {
        if( time <= 0 )
        {
            return;
        }

        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(time));
    }

    int64_t GetDelayNanoSecondsFromDelayParameterTable(
        const DelayParameter* delayParameterTable,
        AccessPattern accessPattern,
        size_t accessSize
    ) NN_NOEXCEPT
    {
        for( int i = 0; i < DelayParameterArrayLength; ++i )
        {
            const DelayParameter& delayParameter
                = delayParameterTable[static_cast<int>(accessPattern) * DelayParameterArrayLength + i];
            if( accessSize >= delayParameter.lowerLimitAccessSize )
            {
                return (accessSize * delayParameter.picoSecondsPerBytes / 1000) + delayParameter.nanoSecondsConst;
            }
        }

        NN_SDK_ASSERT(false, "DelayParameter is invalid: AccessPattern: %d, size: %d",
            accessPattern, accessSize);
        return 0;
    }

    int64_t GetDelayNanoSeconds(
        bool isAverage, int storageFlag,
        AccessPattern accessPattern, size_t accessSize) NN_NOEXCEPT
    {
        int64_t delayNanoSeconds = 0;
        if( storageFlag & nn::fssystem::StorageFlag_Mmc )
        {
            delayNanoSeconds = std::max(delayNanoSeconds,
                GetDelayNanoSecondsFromDelayParameterTable(
                    isAverage ? &MmcAverageDelayParameterTable[0][0] : &MmcWorstDelayParameterTable[0][0],
                    accessPattern, accessSize));
        }
        if( storageFlag & nn::fssystem::StorageFlag_SdCard )
        {
            delayNanoSeconds = std::max(delayNanoSeconds,
                GetDelayNanoSecondsFromDelayParameterTable(
                    isAverage ? &SdCardAverageDelayParameterTable[0][0] : &SdCardWorstDelayParameterTable[0][0],
                    accessPattern, accessSize));
        }
        if( storageFlag & nn::fssystem::StorageFlag_GameCard )
        {
            delayNanoSeconds = std::max(delayNanoSeconds,
                GetDelayNanoSecondsFromDelayParameterTable(
                    isAverage ? &GameCardAverageDelayParameterTable[0][0] : &GameCardWorstDelayParameterTable[0][0],
                    accessPattern, accessSize));
        }
        if( storageFlag & nn::fssystem::StorageFlag_Usb )
        {
            delayNanoSeconds = std::max(delayNanoSeconds,
                GetDelayNanoSecondsFromDelayParameterTable(
                    isAverage ? &UsbAverageDelayParameterTable[0][0] : &UsbWorstDelayParameterTable[0][0],
                    accessPattern, accessSize));
        }
        return delayNanoSeconds;
    }

    int64_t GetDelayNanoSeconds(
        nn::fs::SpeedEmulationMode mode, int storageFlag,
        AccessPattern accessPattern, size_t size) NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_SDK_ASSERT_NOT_EQUAL(0, storageFlag);
#endif
        if( mode == nn::fs::SpeedEmulationMode::Random )
        {
            const int64_t delayNanoSeconds = GetDelayNanoSeconds(true, storageFlag, accessPattern, size);

            std::lock_guard<nn::os::Mutex> scopedLock(g_MutexTinyMt);

            if( !g_IsTinyMtInitialized )
            {
                const auto now = nn::os::GetSystemTick().GetInt64Value();
                g_TinyMt.Initialize(reinterpret_cast<const Bit32*>(&now), sizeof(now) / sizeof(Bit32));
                g_IsTinyMtInitialized = true;
            }
            const uint32_t randomValue = g_TinyMt.GenerateRandomN(64);

            if( randomValue != 0 )
            {
                // 63/64 の確率で 0 ～ Slower のランダム値
                return static_cast<int64_t>(g_TinyMt.GenerateRandomF64() * (delayNanoSeconds + 1));
            }
            else
            {
                // 1/64 の確率で Slower ～ Worst のランダム値
                const int64_t delayWorst = GetDelayNanoSeconds(false, storageFlag, accessPattern, size);

                return delayNanoSeconds + static_cast<int64_t>(g_TinyMt.GenerateRandomF64() * (delayWorst - delayNanoSeconds + 1));
            }
        }
        else if( mode == nn::fs::SpeedEmulationMode::Slower )
        {
            return GetDelayNanoSeconds(true, storageFlag, accessPattern, size);
        }

        return 0;
    }

    int32_t GetSpeedEmulationTypeFlag(int32_t storageFlag) NN_NOEXCEPT
    {
        int32_t speedEmulationTypeFlag = 0;
        if( storageFlag & nn::fssystem::StorageFlag_Mmc )
        {
            speedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);
        }
        if( storageFlag & nn::fssystem::StorageFlag_SdCard )
        {
            speedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled);
        }
        if( storageFlag & nn::fssystem::StorageFlag_GameCard )
        {
            speedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::GameCardStorageContextEnabled);
        }
        if( storageFlag & nn::fssystem::StorageFlag_Usb )
        {
            speedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);
        }

        return speedEmulationTypeFlag;
    }
}

SpeedEmulationStorage::SpeedEmulationStorage(nn::fs::IStorage* pBaseStorage) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage)
    , m_IsPreviousRead(true)
    , m_PreviousEnd(-1)
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
}

Result SpeedEmulationStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    const auto storageFlag = nn::fssystem::GetServiceContextReadOnly().GetStorageFlag();
    const nn::fs::SpeedEmulationMode mode = nn::fssystem::SpeedEmulationConfiguration::GetSpeedEmulationMode();
    if( mode == nn::fs::SpeedEmulationMode::None || mode == nn::fs::SpeedEmulationMode::Faster )
    {
        return m_pBaseStorage->Read(offset, buffer, size);
    }

    std::lock_guard<nn::os::Mutex> scopedLock(g_MutexStorage);

    nn::os::Tick tickBegin = nn::os::GetSystemTick();

    const AccessPattern accessPattern = IsSequentialAccess(true, offset)
        ? AccessPattern::SequentialRead : AccessPattern::RandomRead;

    const auto delayNanoSeconds = GetDelayNanoSeconds(mode, storageFlag, accessPattern, size);

    m_IsPreviousRead = true;
    m_PreviousEnd = offset + size;

    const auto result = m_pBaseStorage->Read(offset, buffer, size);

    nn::os::Tick tickEnd = nn::os::GetSystemTick();

    WaitNanoSeconds(delayNanoSeconds - nn::os::ConvertToTimeSpan(tickEnd - tickBegin).GetNanoSeconds());

    return result;
}

Result SpeedEmulationStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    const auto storageFlag = nn::fssystem::GetServiceContextReadOnly().GetStorageFlag();
    const nn::fs::SpeedEmulationMode mode = nn::fssystem::SpeedEmulationConfiguration::GetSpeedEmulationMode();
    if( mode == nn::fs::SpeedEmulationMode::None || mode == nn::fs::SpeedEmulationMode::Faster )
    {
        return m_pBaseStorage->Write(offset, buffer, size);
    }

    std::lock_guard<nn::os::Mutex> scopedLock(g_MutexStorage);

    const nn::os::Tick tickBegin = nn::os::GetSystemTick();

    const AccessPattern accessPattern = IsSequentialAccess(false, offset)
        ? AccessPattern::SequentialWrite : AccessPattern::RandomWrite;

    const auto delayNanoSeconds = GetDelayNanoSeconds(mode, storageFlag, accessPattern, size);

    m_IsPreviousRead = false;
    m_PreviousEnd = offset + size;

    const auto result = m_pBaseStorage->Write(offset, buffer, size);

    const nn::os::Tick tickEnd = nn::os::GetSystemTick();

    WaitNanoSeconds(delayNanoSeconds - nn::os::ConvertToTimeSpan(tickEnd - tickBegin).GetNanoSeconds());

    return result;
}

Result SpeedEmulationStorage::OperateRange(
    void* outBuffer,
    size_t outBufferSize,
    nn::fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    if( operationId == nn::fs::OperationId::QueryRange )
    {
        NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
        NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

        NN_RESULT_DO(m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize));

        nn::fs::QueryRangeInfo info;
        info.Clear();
        info.speedEmulationTypeFlag = GetSpeedEmulationTypeFlag(nn::fssystem::GetServiceContextReadOnly().GetStorageFlag());
        if( nn::fssystem::SpeedEmulationConfiguration::GetSpeedEmulationMode() != nn::fs::SpeedEmulationMode::None )
        {
            info.speedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
        }
        reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Merge(info);

        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize));
        NN_RESULT_SUCCESS;
    }
}

bool SpeedEmulationStorage::IsSequentialAccess(bool isRead, int64_t offset) const NN_NOEXCEPT
{
    if( m_IsPreviousRead != isRead )
    {
        return false;
    }

    if( m_PreviousEnd == offset )
    {
        return true;
    }

    return false;
}

}}

