﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssrv/fssrv_SaveDataIndexer.h>
#include <nn/fssrv/fssrv_SaveDataIndexerTypes.h>
#include <nn/fssrv/fssrv_SaveDataInfoReaderImpl.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include "detail/fssrv_Trace.h"

namespace nn { namespace fssrv {

    SaveDataIndexerLite::SaveDataIndexerLite() NN_NOEXCEPT
        : m_Mutex(false)
        , m_LastPublishedId(0x4000000000000000ULL)
    {
        std::memset(&m_Value, 0, sizeof(m_Value));
    }

    Result SaveDataIndexerLite::Publish(nn::fs::SaveDataId* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && key == m_Key.value())
        {
            return fs::ResultAlreadyExists();
        }
        else
        {
            // 既に1エントリ保持していた場合は失う
            m_Key.emplace(key);
            std::memset(&m_Value, 0, sizeof(m_Value));
            *outValue = m_LastPublishedId;
            m_Value.id = m_LastPublishedId;
            m_LastPublishedId++;
            NN_RESULT_SUCCESS;
        }

    }

    Result SaveDataIndexerLite::PutStaticSaveDataIdIndex(const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && key == m_Key.value())
        {
            return fs::ResultAlreadyExists();
        }
        else
        {
            // 既に1エントリ保持していた場合は失う
            m_Key.emplace(key);
            std::memset(&m_Value, 0, sizeof(m_Value));
            NN_RESULT_SUCCESS;
        }
    }

    bool SaveDataIndexerLite::IsRemainedReservedOnly() NN_NOEXCEPT
    {
        return false;
    }

    Result SaveDataIndexerLite::Delete(nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        if (m_Key != util::nullopt && id == m_Value.id)
        {
            m_Key = util::nullopt;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    Result SaveDataIndexerLite::Commit() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        // TODO: 対応
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexerLite::Rollback() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        // TODO: 対応
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexerLite::Reset() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        m_Key = util::nullopt;
        NN_RESULT_SUCCESS;
    }

    Result SaveDataIndexerLite::Get(SaveDataIndexerValue* outValue, const SaveDataIndexerKey& key) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && key == m_Key.value())
        {
            *outValue = m_Value;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    Result SaveDataIndexerLite::GetKey(SaveDataIndexerKey* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && id == m_Value.id)
        {
            *outValue = m_Key.value();
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    Result SaveDataIndexerLite::GetValue(SaveDataIndexerValue* outValue, nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && id == m_Value.id)
        {
            *outValue = m_Value;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    Result SaveDataIndexerLite::SetValue(const SaveDataIndexerKey& key, const SaveDataIndexerValue& value) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && m_Key.value() == key)
        {
            m_Value = value;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    int SaveDataIndexerLite::GetIndexCount() NN_NOEXCEPT
    {
        return 1;
    }

    // 1エントリのみ保持し返す reader
    class SaveDataIndexerLiteInfoReader : public SaveDataInfoReaderImpl
    {
    public:
        SaveDataIndexerLiteInfoReader(const SaveDataIndexerKey& key, const SaveDataIndexerValue& value)
            : m_Read(false)
        {
            memset(&m_Info, 0, sizeof(nn::fs::SaveDataInfo));
            m_Info.saveDataId = value.id;
            m_Info.saveDataSpaceId = value.spaceId;
            m_Info.saveDataSize = value.size;

            m_Info.systemSaveDataId = key.staticSaveDataId;
            m_Info.applicationId.value = key.programId.value;
            m_Info.saveDataType = key.type;
            m_Info.saveDataUserId = key.userId;
        }

        // 空
        SaveDataIndexerLiteInfoReader()
            : m_Read(true)
        {
            memset(&m_Info, 0, sizeof(nn::fs::SaveDataInfo));
        }

        virtual nn::Result Read(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outEntries) NN_NOEXCEPT NN_OVERRIDE
        {
            if (m_Read || outEntries.GetSize() == 0)
            {
                outValue.Set(0);
                NN_RESULT_SUCCESS;
            }
            else
            {
                auto info = reinterpret_cast<nn::fs::SaveDataInfo*>(outEntries.GetPointerUnsafe());
                *info = m_Info;
                outValue.Set(1);
                m_Read = true;
                NN_RESULT_SUCCESS;
            }
        }

    private:
        bool m_Read;
        fs::SaveDataInfo m_Info;
    };

    Result SaveDataIndexerLite::OpenSaveDataInfoReader(std::shared_ptr<SaveDataInfoReaderImpl>* outValue) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt)
        {
            auto reader = fssystem::AllocateShared<SaveDataIndexerLiteInfoReader>(m_Key.value(), m_Value);
            NN_RESULT_THROW_UNLESS(reader, nn::fs::ResultAllocationMemoryFailedInSaveDataIndexerA());
            *outValue = std::move(reader);
            NN_RESULT_SUCCESS;
        }
        else
        {
            auto reader = fssystem::AllocateShared<SaveDataIndexerLiteInfoReader>();
            NN_RESULT_THROW_UNLESS(reader, nn::fs::ResultAllocationMemoryFailedInSaveDataIndexerA());
            *outValue = std::move(reader);
            NN_RESULT_SUCCESS;
        }
    }

    Result SaveDataIndexerLite::SetSpaceId(nn::fs::SaveDataId id, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        if (m_Key != util::nullopt && id == m_Value.id)
        {
            m_Value.spaceId = spaceId;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    Result SaveDataIndexerLite::SetSize(nn::fs::SaveDataId id, int64_t size) NN_NOEXCEPT
    {
        if (m_Key != util::nullopt && id == m_Value.id)
        {
            m_Value.size = size;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

    Result SaveDataIndexerLite::SetState(nn::fs::SaveDataId id, SaveDataState state) NN_NOEXCEPT
    {
        if (m_Key != util::nullopt && id == m_Value.id)
        {
            m_Value.state = state;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return fs::ResultTargetNotFound();
        }
    }

}}
