﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssrv/fssrv_FileSystemProxyServerSessionResourceManager.h>
#include <nn/fssrv/fssrv_DeferredProcessQueue.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/sf/sf_ContextControl.h>

namespace nn { namespace fssrv {

NN_DEFINE_STATIC_CONSTANT(const int FileSystemProxyServerSessionResourceManager::RealtimeSessionCount);
NN_DEFINE_STATIC_CONSTANT(const int FileSystemProxyServerSessionResourceManager::BackgroundSessionCount);
NN_DEFINE_STATIC_CONSTANT(const int FileSystemProxyServerSessionResourceManager::OtherSessionCount);
NN_DEFINE_STATIC_CONSTANT(const int FileSystemProxyServerSessionResourceManager::SessionCount);
NN_STATIC_ASSERT(FileSystemProxyServerSessionResourceManager::SessionCount == FileSystemProxyServerActiveSessionCount);

Result FileSystemProxyServerSessionResourceManager::Acquire(fs::PriorityRaw priority) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        priority == nn::fs::PriorityRaw_Realtime
        || priority == nn::fs::PriorityRaw_Normal
        || priority == nn::fs::PriorityRaw_Low
        || priority == nn::fs::PriorityRaw_Background
    );

    auto pContext = fssystem::GetServiceContext();

    int prioritySessionType;
    if( !pContext->GetAcquiredPrioritySessionType(&prioritySessionType) )
    {
        FileSystemProxyServerSessionType sessionType;
        if( OnAcquire(&sessionType, priority) )
        {
            pContext->SetAcquiredPrioritySessionType(static_cast<int>(sessionType));
        }
        else
        {
            pContext->SetDeferredProcessPrioritySessionType(static_cast<int>(sessionType));
            NN_RESULT_THROW(sf::DeferProcess());
        }
    }

    NN_RESULT_SUCCESS;
}

bool FileSystemProxyServerSessionResourceManager::Acquire(FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        sessionType == FileSystemProxyServerSessionType::Any
        || sessionType == FileSystemProxyServerSessionType::Realtime
        || sessionType == FileSystemProxyServerSessionType::Other
        || sessionType == FileSystemProxyServerSessionType::Background
    );

    FileSystemProxyServerSessionType acquiredSessionType;
    if( OnAcquire(&acquiredSessionType, sessionType) )
    {
        fssystem::GetServiceContext()->SetAcquiredPrioritySessionType(static_cast<int>(acquiredSessionType));
        return true;
    }
    return false;
}

void FileSystemProxyServerSessionResourceManager::Release() NN_NOEXCEPT
{
    auto pContext = fssystem::GetServiceContext();

    int prioritySessionType;
    if( pContext->GetAcquiredPrioritySessionType(&prioritySessionType) )
    {
        OnRelease(static_cast<FileSystemProxyServerSessionType>(prioritySessionType));
        pContext->ResetAcquiredPrioritySessionType();
    }
}

bool FileSystemProxyServerSessionResourceManager::OnAcquire(FileSystemProxyServerSessionType* outValue, fs::PriorityRaw priority) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES(
        priority == nn::fs::PriorityRaw_Realtime
        || priority == nn::fs::PriorityRaw_Normal
        || priority == nn::fs::PriorityRaw_Low
        || priority == nn::fs::PriorityRaw_Background
    );

    // Acquire できなかった場合の値を設定しておく
    FileSystemProxyServerSessionType entrySessionType = FileSystemProxyServerSessionType::Other;
    switch( priority )
    {
    case fs::PriorityRaw_Realtime:
        entrySessionType = FileSystemProxyServerSessionType::Realtime;
        break;

    case fs::PriorityRaw_Background:
        entrySessionType = FileSystemProxyServerSessionType::Background;
        break;

    case fs::PriorityRaw_Low:
    case fs::PriorityRaw_Normal:
        entrySessionType = FileSystemProxyServerSessionType::Other;
        break;

    default:
        NN_SDK_REQUIRES(false);
        break;
    }
    return OnAcquire(outValue, entrySessionType);
}

bool FileSystemProxyServerSessionResourceManager::OnAcquire(FileSystemProxyServerSessionType* outValue, FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES(
        sessionType == FileSystemProxyServerSessionType::Any
        || sessionType == FileSystemProxyServerSessionType::Realtime
        || sessionType == FileSystemProxyServerSessionType::Other
        || sessionType == FileSystemProxyServerSessionType::Background
    );

    *outValue = sessionType;

    // Any は特にカウンタを操作しない
    if( sessionType == FileSystemProxyServerSessionType::Any )
    {
        return true;
    }

    // Realtime は専用のセッションを使う。
    if( sessionType == FileSystemProxyServerSessionType::Realtime )
    {
        for( ; ; )
        {
            auto currentValue = m_SessionUsedByRealtimeCount.load();
            if( currentValue >= RealtimeSessionCount )
            {
                return false;
            }
            if( m_SessionUsedByRealtimeCount.compare_exchange_weak(currentValue, currentValue + 1) )
            {
                *outValue = FileSystemProxyServerSessionType::Realtime;
                return true;
            }
        }
    }

    // Other は Other か Background のセッションを使う。
    // まず、Other が使用可能かどうかを調べ、次に Background 可能かどうかを調べる。
    if( sessionType == FileSystemProxyServerSessionType::Other )
    {
        for( ; ; )
        {
            auto currentValue = m_SessionUsedByOtherCount.load();
            if( currentValue >= OtherSessionCount )
            {
                // Background のリクエストが存在しなければ Background の空きを調べる
                if( m_pDeferredProcessQueueForPriority->HasAny(FileSystemProxyServerSessionType::Background) )
                {
                    return false;
                }
                else
                {
                    break;
                }
            }
            if( m_SessionUsedByOtherCount.compare_exchange_weak(currentValue, currentValue + 1) )
            {
                *outValue = FileSystemProxyServerSessionType::Other;
                return true;
            }
        }
    }

    // Background が使用可能かどうかを調べる。Realtime 以外の優先度で到達しうる。
    {
        for( ; ; )
        {
            auto currentValue = m_SessionUsedByBackgroundCount.load();
            if( currentValue >= BackgroundSessionCount )
            {
                return false;
            }
            if( m_SessionUsedByBackgroundCount.compare_exchange_weak(currentValue, currentValue + 1) )
            {
                *outValue = FileSystemProxyServerSessionType::Background;
                return true;
            }
        }
    }
}

void FileSystemProxyServerSessionResourceManager::OnRelease(FileSystemProxyServerSessionType sessionType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        sessionType == FileSystemProxyServerSessionType::Any
        || sessionType == FileSystemProxyServerSessionType::Realtime
        || sessionType == FileSystemProxyServerSessionType::Other
        || sessionType == FileSystemProxyServerSessionType::Background
    );

    switch( sessionType )
    {
    case FileSystemProxyServerSessionType::Any:
        break;

    case FileSystemProxyServerSessionType::Realtime:
        NN_SDK_REQUIRES(m_SessionUsedByRealtimeCount > 0);
        --m_SessionUsedByRealtimeCount;
        break;

    case FileSystemProxyServerSessionType::Background:
        NN_SDK_REQUIRES(m_SessionUsedByBackgroundCount > 0);
        --m_SessionUsedByBackgroundCount;
        break;

    case FileSystemProxyServerSessionType::Other:
        NN_SDK_REQUIRES(m_SessionUsedByOtherCount > 0);
        --m_SessionUsedByOtherCount;
        break;

    default:
        NN_SDK_REQUIRES(false);
        break;
    }
}

}}
