﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/os.h>
#include <nn/os/os_SdkThreadLocalStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Endian.h>

#include <nn/time.h>
#if !defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/fat/fat_FatFileSystem.h>
#endif /* !NN_BUILD_CONFIG_OS_WIN32 */

// fs
#include <nn/fs_Base.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_DeviceSimulation.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/fs/fs_CloudBackupWorkStorage.h>
#include <nn/fs/detail/fs_FileSystemProxyTypes.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_FileStorage.h>

#include <nn/fssrv/fssrv_FileSystemProxyImpl.h>
#include <nn/fssrv/fssrv_FileSystemProxyServer.h>
#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/fssrv/fssrv_MemoryReport.h>
#include <nn/fssrv/fssrv_ProgramRegistryImpl.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/fssrv/fssrv_SaveDataIndexerTypes.h>
#include <nn/fssrv/fssrv_SaveDataInfoReaderImpl.h>
#include <nn/fssrv/fssrv_SaveDataInfoFilter.h>

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_SaveDataFileSystem.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/fssystem/fs_StorageLayoutTypeSetter.h>
#include <nn/fssystem/fs_Utility.h>

// lr
#include <nn/lr/lr_Result.h>

// fssrv local
#include "detail/fssrv_AccessFailureDetectionEventManager.h"
#include "detail/fssrv_AccessFailureDetectionEventNotifier.h"
#include "detail/fssrv_AccessLogSdCardWriter.h"
#include "detail/fssrv_AllocatorForServiceFrameWork.h"
#include "detail/fssrv_DeviceOperator.h"
#include "detail/fssrv_DeviceDetectionEventNotifier.h"
#include "detail/fssrv_FileSystemInterfaceAdapter.h"
#include "detail/fssrv_FileSystemProxyCoreImpl.h"
#include "detail/fssrv_GameCardDetectionEventManager.h"
#include "detail/fssrv_DeviceEventSimulator.h"
#include "detail/fssrv_LocationResolverSet.h"
#include "detail/fssrv_MultiCommitManager.h"
#include "detail/fssrv_ProgramInfo.h"
#include "detail/fssrv_RuntimeConfiguration.h"
#include "detail/fssrv_SaveDataTransferManager.h"
#include "detail/fssrv_SaveDataTransferVersion2.h"
#include "detail/fssrv_SaveDataTransferPorterManager.h"
#include "detail/fssrv_SdCardDetectionEventManager.h"
#include "detail/fssrv_StorageInterfaceAdapter.h"
#include "detail/fssrv_SystemDataUpdateEventManager.h"
#include "detail/fssrv_SystemDataUpdateEventNotifier.h"
#include "detail/fssrv_PathNormalizer.h"
#include "detail/fssrv_Trace.h"

#define NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(condition, flagName, processId) \
{ \
    if (!(condition)) \
    { \
        NN_SDK_LOG("[fs] ResultPermissionDenied (%s) was returned to process %d\n", flagName, processId); \
        return nn::fs::ResultPermissionDenied(); \
    } \
}

using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nn::fssystem;
using namespace nn::fssrv::detail;

namespace nn { namespace fssrv {

    namespace
    {
        const SaveDataSpaceId InvalidSaveDataSpaceId = static_cast<SaveDataSpaceId>(127);

        const auto StorageFlag_ReadOnlyDataInAnyStorage = nn::fssystem::StorageFlag_Mmc
                                                        | nn::fssystem::StorageFlag_SdCard
                                                        | nn::fssystem::StorageFlag_GameCard
                                                        | nn::fssystem::StorageFlag_Usb;

        const auto StorageFlag_ReadWriteDataInAnyStorage = nn::fssystem::StorageFlag_Mmc
                                                         | nn::fssystem::StorageFlag_SdCard
                                                         | nn::fssystem::StorageFlag_Usb;

        nn::Bit64 g_InternalProgramIdValueMinForSpeedEmulation = 0ULL;
        nn::Bit64 g_InternalProgramIdValueMaxForSpeedEmulation = 0ULL;

        int GetStorageFlag(nn::Bit64 programIdValue) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(g_InternalProgramIdValueMinForSpeedEmulation, 0ULL);
            NN_SDK_REQUIRES_NOT_EQUAL(g_InternalProgramIdValueMaxForSpeedEmulation, 0ULL);

            if ((programIdValue >= g_InternalProgramIdValueMinForSpeedEmulation) && (programIdValue <= g_InternalProgramIdValueMaxForSpeedEmulation))
            {
                return nn::fssystem::StorageFlag_Mmc;
            }
            else
            {
                return StorageFlag_ReadOnlyDataInAnyStorage;
            }
        }

        bool IsStaticSaveDataIdValueRange(nn::fs::SaveDataId id)
        {
            return ((id & 0x8000000000000000ULL) != 0);
        }

        typedef nn::sf::ObjectFactory<AllocatorForServiceFrameWork::Policy> FileSystemFactory;

        util::optional<detail::FileSystemProxyCoreImpl> g_FileSystemProxyCoreImpl;

        nn::fssrv::detail::LocationResolverSet g_LocationResolverSet;

        nn::fssrv::detail::SaveDataPorterManager g_PorterManager;

        Result GetAccesibilityForOpenBisFileSystem(Accessibility* outAccessibility, const ProgramInfo* pProgramInfo, nn::fs::BisPartitionId id) NN_NOEXCEPT
        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            switch (id)
            {
            case nn::fs::BisPartitionId::CalibrationFile:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountBisCalibrationFile);
                }
                break;
            case nn::fs::BisPartitionId::SafeMode:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountBisSafeMode);
                }
                break;
            case nn::fs::BisPartitionId::System:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountBisSystem);
                }
                break;
            case nn::fs::BisPartitionId::User:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountBisUser);
                }
                break;
            case nn::fs::BisPartitionId::SystemProperPartition:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountBisSystemProperPartition);
                }
                break;
            default:
                return nn::fs::ResultInvalidArgument();
            }
            NN_RESULT_SUCCESS;
        }

        Result GetAccesibilityForOpenBisPartition(Accessibility* outAccessibility, const ProgramInfo* pProgramInfo, nn::fs::BisPartitionId id) NN_NOEXCEPT
        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            switch (id)
            {
            case nn::fs::BisPartitionId::BootPartition1Root:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootPartition1Root);
                }
                break;
            case nn::fs::BisPartitionId::BootPartition2Root:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootPartition2Root);
                }
                break;
            case nn::fs::BisPartitionId::UserDataRoot:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionUserDataRoot);
                }
                break;
            case nn::fs::BisPartitionId::BootConfigAndPackage2Part1:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part1);
                }
                break;
            case nn::fs::BisPartitionId::BootConfigAndPackage2Part2:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part2);
                }
                break;
            case nn::fs::BisPartitionId::BootConfigAndPackage2Part3:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part3);
                }
                break;
            case nn::fs::BisPartitionId::BootConfigAndPackage2Part4:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part4);
                }
                break;
            case nn::fs::BisPartitionId::BootConfigAndPackage2Part5:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part5);
                }
                break;
            case nn::fs::BisPartitionId::BootConfigAndPackage2Part6:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionBootConfigAndPackage2Part6);
                }
                break;
            case nn::fs::BisPartitionId::CalibrationBinary:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionCalibrationBinary);
                }
                break;
            case nn::fs::BisPartitionId::CalibrationFile:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionCalibrationFile);
                }
                break;
            case nn::fs::BisPartitionId::SafeMode:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionSafeMode);
                }
                break;
            case nn::fs::BisPartitionId::System:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionSystem);
                }
                break;
            case nn::fs::BisPartitionId::User:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionUser);
                }
                break;
            case nn::fs::BisPartitionId::SystemProperEncryption:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionSystemProperEncryption);
                }
                break;
            case nn::fs::BisPartitionId::SystemProperPartition:
                {
                    *outAccessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenBisPartitionSystemProperPartition);
                }
                break;
            default:
                return nn::fs::ResultInvalidArgument();
            }
            NN_RESULT_SUCCESS;
        }

        const size_t DefaultSaveDataSize = 32 * 1024 * 1024;
        const size_t DefaultSaveDataJournalSize = 16 * 1024 * 1024;

        uint32_t g_GlobalAccessLogMode = 0;

        // TORIAEZU: 固定値
        const int DefaultSaveDataEntryOpenCountMax = 256;
        const int DefaultAddOnContentOpenCountMax  = 128;
        const int DefaultRomMountCountMax  = 10;
        const int DefaultSaveDataMountCountMax  = 10;

        // 永続化される SpaceId は本来の値で扱う
        SaveDataSpaceId ConvertToRealSpaceId(SaveDataSpaceId id)
        {
            if( id == SaveDataSpaceId::ProperSystem || id == SaveDataSpaceId::SafeMode )
            {
                return SaveDataSpaceId::System;
            }
            else
            {
                return id;
            }
        }

        inline bool IsHostFs(const char* path) NN_NOEXCEPT
        {
            const auto mountName = HostRootFileSystemMountName;
            return strncmp(path, mountName, strnlen(mountName, fs::MountNameLengthMax)) == 0;
        }

        std::underlying_type<PathNormalizer::Option>::type GetNormalizeOption(
            const char* path) NN_NOEXCEPT
        {
            return (IsHostFs(path) ? PathNormalizer::PreserveUnc : PathNormalizer::None)
                | PathNormalizer::PreserveTailSeparator
                | PathNormalizer::WithMountName;
        }

        void ModifySaveDataExtraData(nn::fs::SaveDataExtraData* pTargetExtraData, const nn::fs::SaveDataExtraData& extraDataValue, const nn::fs::SaveDataExtraData& extraDataMask)
        {
            char* pTarget = reinterpret_cast<char*>(pTargetExtraData);
            const char* pValue = reinterpret_cast<const char*>(&extraDataValue);
            const char* pMask = reinterpret_cast<const char*>(&extraDataMask);
            for (int i = 0; i < static_cast<int>(sizeof(nn::fs::SaveDataExtraData)); i++)
            {
                pTarget[i] = (pTarget[i] & ~pMask[i]) | (pValue[i] & pMask[i]);
            }
        }

        int DecidePossibleStorageFlag(nn::fs::SaveDataType type, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
        {
            if( type == nn::fs::SaveDataType::Cache || type == nn::fs::SaveDataType::Bcat )
            {
                return StorageFlag_ReadWriteDataInAnyStorage;
            }

            if( (type == nn::fs::SaveDataType::System || type == nn::fs::SaveDataType::SystemBcat)
                && (spaceId == nn::fs::SaveDataSpaceId::SdSystem || spaceId == nn::fs::SaveDataSpaceId::SdUser) )
            {
                return nn::fssystem::StorageFlag_SdCard | nn::fssystem::StorageFlag_Usb;
            }

            return nn::fssystem::StorageFlag_Mmc;
        }
    }

    // FileSystemProxyImpl
    FileSystemProxyImpl::FileSystemProxyImpl() NN_NOEXCEPT
      : m_Impl(&g_FileSystemProxyCoreImpl.value()),
        m_ProcessId(0xFFFFFFFFFFFFFFFF),
        m_SaveDataSize(DefaultSaveDataSize),
        m_SaveDataJournalSize(DefaultSaveDataJournalSize),
        m_EnableAutoSaveDataCreation(true),
        m_SaveDataEntryOpenCountSemaphore(DefaultSaveDataEntryOpenCountMax, DefaultSaveDataEntryOpenCountMax),
        m_AddOnContentOpenCountSemaphore(DefaultAddOnContentOpenCountMax, DefaultAddOnContentOpenCountMax),
        m_RomMountCountSemaphore(DefaultRomMountCountMax, DefaultRomMountCountMax),
        m_SaveDataMountCountSemaphore(DefaultSaveDataMountCountMax, DefaultSaveDataMountCountMax)
    {
        memset(m_SaveDataRootPathBuffer, 0x00, sizeof(m_SaveDataRootPathBuffer));
    }

    FileSystemProxyImpl::~FileSystemProxyImpl() NN_NOEXCEPT
    {
    }

    Result FileSystemProxyImpl::GetProgramInfoByProcessId(std::shared_ptr<ProgramInfo>* outProgramInfo, Bit64 processId) NN_NOEXCEPT
    {
        ProgramRegistryImpl programRegistry;
        return programRegistry.GetProgramInfo(outProgramInfo, processId);
    }

    Result FileSystemProxyImpl::GetProgramInfoByProgramId(std::shared_ptr<ProgramInfo>* outProgramInfo, Bit64 programIdValue) NN_NOEXCEPT
    {
        ProgramRegistryImpl programRegistry;
        return programRegistry.GetProgramInfoByProgramId(outProgramInfo, programIdValue);
    }

    Result FileSystemProxyImpl::GetProgramInfo(std::shared_ptr<ProgramInfo>* outProgramInfo) NN_NOEXCEPT
    {
        ProgramRegistryImpl programRegistry;
        return programRegistry.GetProgramInfo(outProgramInfo, m_ProcessId);
    }

    Result FileSystemProxyImpl::OpenFileSystemWithId(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, const nn::fssrv::sf::FspPath& path, nn::Bit64 programId, std::uint32_t type) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%s %016llx %u",
            path.str,
            programId,
            type
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = StorageFlag_ReadOnlyDataInAnyStorage;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            Accessibility accessibility;

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto pAccessControl = pProgramInfo->GetAccessControl();
            switch (type)
            {
            case FileSystemProxyType::FileSystemProxyType_Control:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountContentControl);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountContentControl", m_ProcessId);
                break;
            case FileSystemProxyType::FileSystemProxyType_Manual:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountContentManual);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountContentManual", m_ProcessId);
                break;
            case FileSystemProxyType::FileSystemProxyType_Meta:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountContentMeta);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountContentMeta", m_ProcessId);
                break;
            case FileSystemProxyType::FileSystemProxyType_Data:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountContentData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountContentData", m_ProcessId);
                break;
            case FileSystemProxyType::FileSystemProxyType_Logo:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountLogo);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountLogo", m_ProcessId);
                break;
            case FileSystemProxyType::FileSystemProxyType_Package:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountApplicationPackage);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountApplicationPackage", m_ProcessId);
                break;
            default:
                return ResultInvalidArgument();
            }

            // Meta の時だけ例外的にチェックを飛ばす。TODO: 削除
            auto checkProgramId = programId;
            if (type == FileSystemProxyType::FileSystemProxyType_Meta)
            {
                checkProgramId = CheckThroughId;
            }
            else
            {
                NN_FSP_REQUIRES(checkProgramId != CheckThroughId, nn::fs::ResultInvalidArgument());
            }

            bool canReadPrivateDataContent = false;
            {
                auto accessibilityForData = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountSystemDataPrivate);
                canReadPrivateDataContent = accessibilityForData.CanRead();
            }

            PathNormalizer normalizedPath(path.str, GetNormalizeOption(path.str));
            NN_RESULT_DO(normalizedPath.GetResult());

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenFileSystem(&fileSystem, normalizedPath.GetPath(), static_cast<FileSystemProxyType>(type), canReadPrivateDataContent, checkProgramId));

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenFileSystemWithPatch(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, nn::ncm::ProgramId programId, std::uint32_t type) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%016llx %u",
            programId.value,
            type
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = StorageFlag_ReadOnlyDataInAnyStorage;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            Accessibility accessibility;

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            // HtmlDocument のみパッチに対応
            auto pAccessControl = pProgramInfo->GetAccessControl();
            switch (type)
            {
            case FileSystemProxyType::FileSystemProxyType_Manual:
                accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountContentManual);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountContentManual", m_ProcessId);
                break;
            case FileSystemProxyType::FileSystemProxyType_Control:
            case FileSystemProxyType::FileSystemProxyType_Meta:
            case FileSystemProxyType::FileSystemProxyType_Data:
            case FileSystemProxyType::FileSystemProxyType_Logo:
            case FileSystemProxyType::FileSystemProxyType_Package:
                return ResultNotImplemented();
            default:
                return ResultInvalidArgument();
            }

            // パッチと元 HtmlDocument のパスを解決
            std::shared_ptr<ProgramInfo> pApplicationProgramInfo;
            NN_RESULT_DO(GetProgramInfoByProgramId(&pApplicationProgramInfo, programId.value));

            nn::lr::Path originalPath;
            nn::ncm::ApplicationId applicationId = { programId.value };
            Result originalResult = g_LocationResolverSet.ResolveApplicationHtmlDocumentPath(&originalPath, applicationId, pApplicationProgramInfo->GetStorageId());
            if (!nn::lr::ResultHtmlDocumentNotFound::Includes(originalResult))
            {
                NN_RESULT_DO(originalResult);
            }

            nn::util::optional<PathNormalizer> normalizedOriginalPath;
            if (originalResult.IsSuccess())
            {
                normalizedOriginalPath.emplace(originalPath.string, GetNormalizeOption(originalPath.string));
                NN_RESULT_DO(normalizedOriginalPath->GetResult());
            }

            nn::lr::Path patchPath;
            Result patchResult = g_LocationResolverSet.ResolveRegisteredHtmlDocumentPath(&patchPath, programId.value);

            // ファイルシステムを開いて返す
            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            if (nn::lr::ResultHtmlDocumentNotFound::Includes(patchResult))
            {
                NN_RESULT_DO(originalResult);

                NN_RESULT_DO(m_Impl->OpenFileSystem(&fileSystem, normalizedOriginalPath->GetPath(), static_cast<FileSystemProxyType>(type), programId.value));
            }
            else
            {
                const char* originalPathString;
                if (normalizedOriginalPath)
                {
                    originalPathString = normalizedOriginalPath->GetPath();
                }
                else
                {
                    // 元の HtmlDocument の不在を許す
                    originalPathString = nullptr;
                }

                PathNormalizer normalizedPatchPath(patchPath.string, GetNormalizeOption(patchPath.string));
                NN_RESULT_DO(normalizedPatchPath.GetResult());

                NN_RESULT_DO(patchResult);
                NN_RESULT_DO(m_Impl->OpenFileSystemWithPatch(&fileSystem, originalPathString, normalizedPatchPath.GetPath(), static_cast<FileSystemProxyType>(type), programId.value));
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, false, true);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenCodeFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, const nn::fssrv::sf::FspPath& path, nn::ncm::ProgramId programId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%s %016llx",
            path.str,
            programId.value
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = GetStorageFlag(programId.value);
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            NN_RESULT_THROW_UNLESS(detail::IsInitialProgram(m_ProcessId), nn::fs::ResultPermissionDenied());

            PathNormalizer normalizedPath(path.str, GetNormalizeOption(path.str));
            NN_RESULT_DO(normalizedPath.GetResult());

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenFileSystem(&fileSystem, normalizedPath.GetPath(), FileSystemProxyType::FileSystemProxyType_Code, programId.value));

            std::shared_ptr<nn::fs::fsa::IFileSystem> typeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(typeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::SetCurrentProcess(nn::Bit64 processId) NN_NOEXCEPT
    {
        m_ProcessId = processId;
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::GetFreeSpaceSizeForSaveData(nn::sf::Out<std::int64_t> outValue, std::uint8_t saveDataSpaceId) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(m_Impl->OpenSaveDataDirectoryFileSystem(&pFileSystem, static_cast<SaveDataSpaceId>(saveDataSpaceId), "", false));
        int64_t freeSpaceSize = 0;
        NN_RESULT_DO(pFileSystem->GetFreeSpaceSize(&freeSpaceSize, "/"));
        outValue.Set(freeSpaceSize);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenDataFileSystemCore(std::shared_ptr<fs::fsa::IFileSystem>* outValue, nn::Bit64 programIdValue, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d",
            storageId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                programIdValue
            );

            const int storageFlag = GetStorageFlag(programIdValue);
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            nn::lr::Path programPath;
            NN_RESULT_DO(g_LocationResolverSet.ResolveProgramPath(&programPath, programIdValue, storageId));

            PathNormalizer normalizedProgramPath(programPath.string, GetNormalizeOption(programPath.string));
            NN_RESULT_DO(normalizedProgramPath.GetResult());

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenDataFileSystem(&fileSystem, normalizedProgramPath.GetPath(), FileSystemProxyType::FileSystemProxyType_Rom, programIdValue));

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            *outValue = std::move(pTypeSetFileSystem);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenDataFileSystemByCurrentProcess(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        std::shared_ptr<fs::fsa::IFileSystem> fileSystem;
        NN_RESULT_DO(OpenDataFileSystemCore(&fileSystem, pProgramInfo->GetProgramIdValue(), pProgramInfo->GetStorageId()));
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(fileSystem), this, false);
        NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pFileSystem));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenDataFileSystemByProgramId(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, nn::ncm::ProgramId programId) NN_NOEXCEPT
    {
        // TODO: 起動していないプログラムへの対応
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfoByProgramId(&pProgramInfo, programId.value));

        std::shared_ptr<fs::fsa::IFileSystem> fileSystem;
        NN_RESULT_DO(OpenDataFileSystemCore(&fileSystem, programId.value, pProgramInfo->GetStorageId()));

        // 存在確認の後で権限チェック
        {
            std::shared_ptr<ProgramInfo> pProperProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProperProgramInfo));
            auto pAccessControl = pProperProgramInfo->GetAccessControl();
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(pAccessControl->HasContentOwnerId(programId.value), "MountOthersRom", m_ProcessId);
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(fileSystem), this, false);
        NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pFileSystem));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenDataStorageCore(std::shared_ptr<fs::IStorage>* outValue, NcaDigest* pOutDigest, nn::Bit64 programIdValue, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d",
            storageId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = GetStorageFlag(programIdValue);
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                programIdValue
            );

            nn::lr::Path programPath;
            NN_RESULT_DO(g_LocationResolverSet.ResolveProgramPath(&programPath, programIdValue, storageId));

            PathNormalizer normalizedProgramPath(programPath.string, GetNormalizeOption(programPath.string));
            NN_RESULT_DO(normalizedProgramPath.GetResult());

            nn::lr::Path patchPath;
            Result patchResult = g_LocationResolverSet.ResolveRegisteredProgramPath(&patchPath, programIdValue);

            std::shared_ptr<fs::IStorage> storage;

            if( nn::lr::ResultProgramNotFound::Includes(patchResult) )
            {
                NN_RESULT_DO(m_Impl->OpenDataStorage(&storage, pOutDigest, normalizedProgramPath.GetPath(), FileSystemProxyType::FileSystemProxyType_Rom, programIdValue));
            }
            else
            {
                NN_RESULT_DO(patchResult);

                PathNormalizer normalizedPatchPath(patchPath.string, GetNormalizeOption(patchPath.string));
                NN_RESULT_DO(normalizedPatchPath.GetResult());

                NN_RESULT_DO(m_Impl->OpenStorageWithPatch(&storage, pOutDigest, normalizedProgramPath.GetPath(), normalizedPatchPath.GetPath(), FileSystemProxyType::FileSystemProxyType_Rom, programIdValue));
            }

            std::shared_ptr<nn::fs::IStorage> typeSetStorage = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetStorage>(std::move(storage), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetStorage, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            *outValue = std::move(typeSetStorage);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenDataStorageByCurrentProcess(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        std::unique_ptr<NcaDigest> digest(new NcaDigest());
        NN_RESULT_THROW_UNLESS(digest != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

        // ROM はプロセスごとにマウント数制限
        std::unique_lock<fssystem::SemaphoreAdaptor> mountCountSemaphore;
        NN_RESULT_DO(TryAcquireRomMountCountSemaphore(&mountCountSemaphore));

        // プロセスの storageId が HOST の場合のみ指定されているストレージへリダイレクト
        nn::ncm::StorageId storageId = pProgramInfo->GetStorageId();
        nn::ncm::StorageId redirectTarget = detail::GetDataStorageRedirectTarget();
        if ( redirectTarget != nn::ncm::StorageId::None && storageId == nn::ncm::StorageId::Host )
        {
            storageId = redirectTarget;
        }

        std::shared_ptr<fs::IStorage> storage;
        NN_RESULT_DO(OpenDataStorageCore(&storage, digest.get(), pProgramInfo->GetProgramIdValue(), storageId));
        nn::sf::SharedPointer<nn::fssrv::sf::IStorage> pStorage = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IStorage, StorageInterfaceAdapter>(std::move(storage), this, std::move(mountCountSemaphore), std::move(digest), pProgramInfo->GetProgramIdValue(), storageId);
        NN_RESULT_THROW_UNLESS(pStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pStorage));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenDataStorageByProgramId(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>> outValue, nn::ncm::ProgramId programId) NN_NOEXCEPT
    {
        // TODO: 起動していないプログラムへの対応
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfoByProgramId(&pProgramInfo, programId.value));

        std::unique_ptr<NcaDigest> digest(new NcaDigest());
        NN_RESULT_THROW_UNLESS(digest != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

        // ROM はプロセスごとにマウント数制限
        std::unique_lock<fssystem::SemaphoreAdaptor> mountCountSemaphore;
        NN_RESULT_DO(TryAcquireRomMountCountSemaphore(&mountCountSemaphore));

        std::shared_ptr<fs::IStorage> storage;
        NN_RESULT_DO(OpenDataStorageCore(&storage, digest.get(), programId.value, pProgramInfo->GetStorageId()));

        // 存在確認の後で権限チェック
        {
            std::shared_ptr<ProgramInfo> pProperProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProperProgramInfo));
            auto pAccessControl = pProperProgramInfo->GetAccessControl();
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(pAccessControl->HasContentOwnerId(programId.value), "MountOthersRom", m_ProcessId);
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IStorage> pStorage = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IStorage, StorageInterfaceAdapter>(std::move(storage), this, std::move(mountCountSemaphore), std::move(digest), programId.value, pProgramInfo->GetStorageId());
        NN_RESULT_THROW_UNLESS(pStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pStorage));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenDataStorageByDataId(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>> outValue, nn::ncm::DataId dataId, nn::Bit8 storageId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%llx %d",
            dataId.value,
            storageId
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const auto ncmStorageId = static_cast<nn::ncm::StorageId>(storageId);
            const int storageFlag = GetStorageFlag(dataId.value);
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            bool isAoc = (ncmStorageId == nn::ncm::StorageId::None);

            // AoC はプロセスごとにマウント数制限
            std::unique_lock<fssystem::SemaphoreAdaptor> openCountSemaphore;
            if (isAoc)
            {
                NN_RESULT_DO(TryAcquireAddOnContentOpenCountSemaphore(&openCountSemaphore));
            }

            nn::lr::Path systemDataPath;
            if (isAoc)
            {
                NN_RESULT_DO(g_LocationResolverSet.ResolveAddOnContentPath(&systemDataPath, dataId));
            }
            else
            {
                NN_RESULT_DO(g_LocationResolverSet.ResolveDataPath(&systemDataPath, dataId, ncmStorageId));
            }

            bool canReadPrivateDataContent = false;
            {
                std::shared_ptr<ProgramInfo> pProgramInfo;
                NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

                NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                    m_ProcessId,
                    pProgramInfo->GetProgramIdValue()
                );

                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountSystemDataPrivate);
                canReadPrivateDataContent = accessibility.CanRead();
            }

            PathNormalizer normalizedSystemDataPath(systemDataPath.string, GetNormalizeOption(systemDataPath.string));
            NN_RESULT_DO(normalizedSystemDataPath.GetResult());

            std::shared_ptr<fs::IStorage> storage;
            NN_RESULT_DO(m_Impl->OpenDataStorage(&storage, nullptr, normalizedSystemDataPath.GetPath(), FileSystemProxyType::FileSystemProxyType_Data, dataId.value, canReadPrivateDataContent));

            std::shared_ptr<nn::fs::IStorage> typeSetStorage = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetStorage>(std::move(storage), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetStorage, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IStorage> pStorage = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IStorage, StorageInterfaceAdapter>(std::move(typeSetStorage), this, std::move(openCountSemaphore), isAoc);
            NN_RESULT_THROW_UNLESS(pStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pStorage));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    // 旧ライブラリ (NX Addon 0.12 (Generic 0.17) 以前) の DualMount から呼び出されるため残しています。
    // （旧ライブラリはプリインストール版 Ocean, ローンチアプリで使用されています。）
    Result FileSystemProxyImpl::OpenPatchDataStorageByCurrentProcess(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>> outValue) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        return nn::fs::ResultTargetNotFound();
    }

    Result FileSystemProxyImpl::RegisterSaveDataFileSystemAtomicDeletion(nn::sf::InBuffer saveDataIdArray) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::DeleteSaveData);
                canCall &= pAccessControl->CanCall(AccessControl::OperationType::DeleteSystemSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "RegisterSaveDataFileSystemAtomicDeletion", m_ProcessId);
            }

            bool success = false;
            SaveDataIndexerAccessor accessor;
            NN_RESULT_DO(accessor.Initialize(nn::fs::SaveDataSpaceId::System));

            auto pSaveDataIdArray = reinterpret_cast<const SaveDataId*>(saveDataIdArray.GetPointerUnsafe());
            auto saveDataIdArrayCount = static_cast<int>(saveDataIdArray.GetSize() / sizeof(SaveDataId));

            NN_UTIL_SCOPE_EXIT
            {
                if( !success )
                {
                    auto result = accessor.GetInterface()->Rollback();
                    if (result.IsFailure())
                    {
                        NN_SDK_LOG("[fs] Error : Failed to rollback save data indexer.\n");
                    }
                }
            };

            NN_FS_SCOPED_TRACE_APPEND_LOG(" ids:");
            for(auto i = 0; i < saveDataIdArrayCount; i++)
            {
                NN_RESULT_DO(accessor.GetInterface()->SetState(pSaveDataIdArray[i], SaveDataState::DeletionRegistered));
                NN_FS_SCOPED_TRACE_APPEND_LOG(" %d", pSaveDataIdArray[i]);
            }

            NN_RESULT_DO(accessor.GetInterface()->Commit());
            success = true;
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::DeleteSaveDataFileSystemCore(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            saveDataSpaceId,
            saveDataId
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_TRY(m_Impl->DeleteAllSaveDataMetas(saveDataId, static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId)))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    // NotFound は無視して継続
                }
            NN_RESULT_END_TRY

            NN_RESULT_TRY(m_Impl->DeleteSaveDataFileSystem(static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, m_SaveDataRootPathBuffer))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    // NotFound は無視して継続
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::GetAccesibilityForSaveData(detail::Accessibility* outValue, const detail::ProgramInfo* pProgramInfo, nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId spaceId, bool isTemporaryFs) NN_NOEXCEPT
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(m_Impl->ReadSaveDataFileSystemExtraData(&extraData, spaceId, saveDataId, m_SaveDataRootPathBuffer, isTemporaryFs));

        if (extraData.ownerId == 0 && extraData.availableSize == 0 && extraData.journalSize == 0)
        {
            // ディレクトリ版セーブデータは権限無しで開ける
            *outValue = Accessibility::MakeAccessibility(true, true);
            NN_RESULT_SUCCESS;
        }

        // 所有者 ID がプログラム ID と等しいか、ACI に所有者 ID が含まれていたらアクセス可能
        if (extraData.ownerId == pProgramInfo->GetProgramIdValue())
        {
            *outValue = Accessibility::MakeAccessibility(true, true);
        }
        else
        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            *outValue = pAccessControl->GetAccessibilitySaveDataOwnedBy(extraData.ownerId);
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::DeleteSaveDataFileSystem(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%llu",
            saveDataId
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);
            return DeleteSaveDataFileSystemCommon(static_cast<uint8_t>(SaveDataSpaceId::System), saveDataId);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::DeleteSaveDataFileSystemBySaveDataSpaceId(std::uint8_t spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        NN_RESULT_DO(DeleteSaveDataFileSystemBySaveDataSpaceIdCore(spaceId, saveDataId));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::DeleteSaveDataFileSystemBySaveDataSpaceIdCore(std::uint8_t spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            spaceId,
            saveDataId
        );

        if (saveDataId != IndexerSaveDataId) // SaveDataIndexer 用 ID
        {
            SaveDataIndexerAccessor accessor;
            NN_RESULT_DO(accessor.Initialize(static_cast<nn::fs::SaveDataSpaceId>(spaceId)));
            SaveDataIndexerValue value;
            NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));
            if (value.spaceId != ConvertToRealSpaceId(static_cast<nn::fs::SaveDataSpaceId>(spaceId)))
            {
                return nn::fs::ResultTargetNotFound();
            }
        }

        return DeleteSaveDataFileSystemCommon(spaceId, saveDataId);
    }

    Result FileSystemProxyImpl::DeleteSaveDataFileSystemCommon(std::uint8_t spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            spaceId,
            saveDataId
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fs::SaveDataSpaceId targetSpaceId;
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            SaveDataIndexerAccessor accessor;

            if (saveDataId != IndexerSaveDataId) // SaveDataIndexer 用 ID
            {
                NN_RESULT_DO(accessor.Initialize(static_cast<SaveDataSpaceId>(spaceId)));

                if (static_cast<SaveDataSpaceId>(spaceId) == SaveDataSpaceId::ProperSystem || static_cast<SaveDataSpaceId>(spaceId) == SaveDataSpaceId::SafeMode)
                {
                    targetSpaceId = static_cast<SaveDataSpaceId>(spaceId);
                }
                else
                {
                    SaveDataIndexerValue value;
                    NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));
                    targetSpaceId = value.spaceId;
                }

                SaveDataIndexerKey tmpKey;
                NN_RESULT_DO(accessor.GetInterface()->GetKey(&tmpKey, saveDataId));

                auto pAccessControl = pProgramInfo->GetAccessControl();
                if ((tmpKey.type == nn::fs::SaveDataType::System) || (tmpKey.type == nn::fs::SaveDataType::SystemBcat))
                {
                    auto canCall = pAccessControl->CanCall(AccessControl::OperationType::DeleteSystemSaveData);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "DeleteSystemSaveData", m_ProcessId);
                }
                else
                {
                    auto canCall = pAccessControl->CanCall(AccessControl::OperationType::DeleteSaveData);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "DeleteSaveData", m_ProcessId);
                }

                NN_RESULT_DO(
                    accessor.GetInterface()->SetState(saveDataId, SaveDataState::Deleting)
                );
                NN_RESULT_DO(accessor.GetInterface()->Commit());
            }
            else
            {
                // SaveDataIndexer 用セーブデータの削除時は、アクセス権限チェックだけ行い、データベースは操作しない
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(detail::IsCurrentProcess(m_ProcessId), "DeleteSaveDataIndexer", m_ProcessId);
                targetSpaceId = static_cast<SaveDataSpaceId>(spaceId);
            }

            NN_RESULT_DO(DeleteSaveDataFileSystemCore(targetSpaceId, saveDataId));

            if (saveDataId != IndexerSaveDataId)
            {
                NN_RESULT_DO(accessor.GetInterface()->Delete(saveDataId));
                NN_RESULT_DO(accessor.GetInterface()->Commit());
            }

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::SwapSaveDataKeyAndState(std::uint8_t spaceId, nn::fs::SaveDataId lhsSaveDataId, nn::fs::SaveDataId rhsSaveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu %llu",
            spaceId,
            lhsSaveDataId,
            rhsSaveDataId
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            SaveDataIndexerAccessor accessor;

            NN_RESULT_DO(accessor.Initialize(static_cast<SaveDataSpaceId>(spaceId)));

            SaveDataIndexerKey lhsKey;
            SaveDataIndexerValue lhsValue;
            NN_RESULT_DO(accessor.GetInterface()->GetKey(&lhsKey, lhsSaveDataId));
            NN_RESULT_DO(accessor.GetInterface()->GetValue(&lhsValue, lhsSaveDataId));

            SaveDataIndexerKey rhsKey;
            SaveDataIndexerValue rhsValue;
            NN_RESULT_DO(accessor.GetInterface()->GetKey(&rhsKey, rhsSaveDataId));
            NN_RESULT_DO(accessor.GetInterface()->GetValue(&rhsValue, rhsSaveDataId));

            // state を swap
            {
                SaveDataState tmp;
                tmp = rhsValue.state;
                rhsValue.state = lhsValue.state;
                lhsValue.state = tmp;
            }

            // key を swap
            NN_RESULT_DO(accessor.GetInterface()->SetValue(rhsKey, lhsValue));
            NN_RESULT_DO(accessor.GetInterface()->SetValue(lhsKey, rhsValue));

            NN_RESULT_DO(accessor.GetInterface()->Commit());
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }


    Result FileSystemProxyImpl::GetSaveDataInfo(nn::fs::SaveDataInfo* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataAttribute saveDataAttribute) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        SaveDataIndexerAccessor accessor;
        NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));

        SaveDataIndexerValue value;
        NN_RESULT_DO(accessor.GetInterface()->Get(&value, saveDataAttribute));

        fs::SaveDataInfo info;
        GenerateSaveDataInfo(&info, saveDataAttribute, value);

        *outValue = info;
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::DeleteSaveDataFileSystemBySaveDataAttribute(std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %d %016llx %016llx %016llx-%016llx",
            saveDataSpaceId,
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.userId._data[0],
            attribute.userId._data[1]
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            SaveDataInfo info;
            NN_RESULT_DO(GetSaveDataInfo(&info, static_cast<SaveDataSpaceId>(saveDataSpaceId), attribute));
            // 権限チェックは DeleteSaveDataFileSystemBySaveDataSpaceIdCore 内で行う
            NN_RESULT_DO(DeleteSaveDataFileSystemBySaveDataSpaceIdCore(saveDataSpaceId, info.saveDataId));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::FinalizeSaveDataCreation(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(saveDataId != IndexerSaveDataId, fs::ResultInvalidArgument());

        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        SaveDataIndexerAccessor accessor;
        NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
        SaveDataIndexerKey key;
        NN_RESULT_DO(accessor.GetInterface()->GetKey(&key, saveDataId));

        // user id, rank 更新
        SaveDataExtraData extraData;
        NN_RESULT_DO(m_Impl->ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId, m_SaveDataRootPathBuffer, false));
        extraData.attribute.userId = key.userId;
        extraData.attribute.rank = SaveDataRank::Primary;
        NN_RESULT_DO(m_Impl->WriteSaveDataFileSystemExtraData(saveDataSpaceId, saveDataId, extraData, m_SaveDataRootPathBuffer, key.type, false));

        NN_RESULT_DO(accessor.GetInterface()->SetState(saveDataId, SaveDataState::None));
        NN_RESULT_DO(accessor.GetInterface()->Commit());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CancelSaveDataCreation(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        NN_FSP_REQUIRES(saveDataId != IndexerSaveDataId, fs::ResultInvalidArgument());
        NN_RESULT_DO(DeleteSaveDataFileSystemCore(saveDataSpaceId, saveDataId));
        SaveDataIndexerAccessor accessor;
        NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));

        SaveDataIndexerValue value;
        NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));
        if (value.spaceId != ConvertToRealSpaceId(saveDataSpaceId))
        {
            return nn::fs::ResultTargetNotFound();
        }
        NN_RESULT_DO(accessor.GetInterface()->Delete(saveDataId));
        NN_RESULT_DO(accessor.GetInterface()->Commit());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::UpdateSaveDataMacForDebug(std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(saveDataId != IndexerSaveDataId, fs::ResultInvalidArgument());

#if 0
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        NN_RESULT_DO(m_Impl->UpdateSaveDataFileSystemMac(static_cast<SaveDataSpaceId>(saveDataSpaceId), static_cast<SaveDataId>(saveDataId)));
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(saveDataSpaceId);
        return ResultNotImplemented();
#endif
    }

    Result FileSystemProxyImpl::OpenSaveDataFile(std::shared_ptr<fs::fsa::IFile>* outValue, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        const int storageFlag = StorageFlag_ReadWriteDataInAnyStorage;
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

        std::shared_ptr<nn::fs::fsa::IFile> pFile;
        NN_RESULT_DO(m_Impl->OpenSaveDataFile(&pFile, spaceId, saveDataId, mode));

        std::shared_ptr<nn::fs::fsa::IFile> pTypeSetFile = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFile>(std::move(pFile), storageFlag);
        NN_RESULT_THROW_UNLESS(pTypeSetFile, nn::fs::ResultAllocationMemoryFailedAllocateShared());

        *outValue = std::move(pTypeSetFile);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CreateSaveDataFileSystemCore(const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const nn::fs::SaveDataMetaInfo& metaInfo, const nn::fs::SaveDataHashSalt& hashSalt) NN_NOEXCEPT
    {
        return CreateSaveDataFileSystemCore(attribute, creationInfo, metaInfo, hashSalt, false);
    }

    Result FileSystemProxyImpl::CheckSaveDataFile(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        NN_RESULT_DO(m_Impl->RecoverSaveDataFileSystemMasterHeader(saveDataSpaceId, saveDataId));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CreateSaveDataFileSystemCore(const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const nn::fs::SaveDataMetaInfo& metaInfo, const nn::fs::SaveDataHashSalt& hashSalt, bool isImportable) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %016llx %016llx %d %016llx-%016llx, %016llx %016llx %016llx %016llx %08x %d, %08x %d",
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.index,
            attribute.userId._data[0],
            attribute.userId._data[1],
            creationInfo.size,
            creationInfo.journalSize,
            creationInfo.blockSize,
            creationInfo.ownerId,
            creationInfo.flags,
            creationInfo.spaceId,
            metaInfo.size,
            metaInfo.type
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(DecidePossibleStorageFlag(attribute.type, creationInfo.spaceId));

            std::uint64_t saveDataId;
            SaveDataIndexerAccessor accessor;
            bool isCreating = false;
            bool isAccessorInitialized = false;

            NN_UTIL_SCOPE_EXIT
            {
                if( isCreating )
                {
                    // 作成失敗時のロールバック処理
                    auto result = DeleteSaveDataFileSystemCore(creationInfo.spaceId, saveDataId);
                    if (result.IsFailure())
                    {
                        NN_SDK_LOG("[fs] Error : Failed to rollback save data creation. (%x)\n", result.GetInnerValueForDebug());
                    }

                    if (isAccessorInitialized && saveDataId != IndexerSaveDataId)
                    {
                        SaveDataIndexerValue value;
                        result = accessor.GetInterface()->GetValue(&value, saveDataId);
                        if (result.IsSuccess() && (value.spaceId == creationInfo.spaceId))
                        {
                            result = accessor.GetInterface()->Delete(saveDataId);
                            if (result.IsFailure() && !ResultTargetNotFound::Includes(result))
                            {
                                NN_SDK_LOG("[fs] Error : Failed to rollback save data creation. (%x)\n", result.GetInnerValueForDebug());
                            }

                            result = accessor.GetInterface()->Commit();
                            if (result.IsFailure())
                            {
                                NN_SDK_LOG("[fs] Error : Failed to rollback save data creation. (%x)\n", result.GetInnerValueForDebug());
                            }
                        }
                    }
                }
            };

            bool isExists = false;

            if (attribute.staticSaveDataId == IndexerSaveDataId)
            {
                // SaveDataIndexer 用 ID は登録せずに作成
                saveDataId = attribute.staticSaveDataId;

                if (m_Impl->IsSaveDataEntityExists(&isExists, creationInfo.spaceId, saveDataId).IsSuccess() && isExists)
                {
                    return nn::fs::ResultPathAlreadyExists();
                }

                isCreating = true;
            }
            else
            {
                NN_RESULT_DO(accessor.Initialize(creationInfo.spaceId));
                isAccessorInitialized = true;

                auto key = attribute;
                bool isStaticSaveDataId = (attribute.staticSaveDataId != InvalidSystemSaveDataId && attribute.userId == InvalidUserId);
                bool isSystemSaveData = (attribute.type == SaveDataType::System || attribute.type == SaveDataType::SystemBcat);

                Result result;
                if( isStaticSaveDataId )
                {
                    saveDataId = attribute.staticSaveDataId;
                    result = accessor.GetInterface()->PutStaticSaveDataIdIndex(key);
                }
                else
                {
                    // システムセーブ以外はインデクサの予約領域を使えないようにしておく
                    if (!isSystemSaveData && accessor.GetInterface()->IsRemainedReservedOnly())
                    {
                        return nn::kvdb::ResultOutOfMaxKeySize();
                    }
                    result = accessor.GetInterface()->Publish(&saveDataId, key);
                }

                if( result.IsSuccess() )
                {
                    isCreating = true;
                    NN_RESULT_DO(
                        accessor.GetInterface()->SetState(saveDataId, SaveDataState::Deleting)
                    );
                    NN_RESULT_DO(
                        accessor.GetInterface()->SetSpaceId(
                            saveDataId, ConvertToRealSpaceId(creationInfo.spaceId)
                        )
                    );
                    int64_t saveDataSize = 0;
                    {
                        NN_RESULT_DO(QuerySaveDataTotalSize(&saveDataSize, creationInfo.size, creationInfo.journalSize));
                    }
                    NN_RESULT_DO(accessor.GetInterface()->SetSize(saveDataId, saveDataSize));

                    auto resultCommit = accessor.GetInterface()->Commit();
                    if (resultCommit.IsFailure())
                    {
                        NN_SDK_LOG("[fs] Error : Failed to commit save data indexer.\n");
                        return resultCommit;
                    }
                }
                else if( nn::fs::ResultAlreadyExists::Includes(result) )
                {
                    return nn::fs::ResultPathAlreadyExists();
                }
                else
                {
                    return result;
                }
            }

            NN_RESULT_TRY(m_Impl->CreateSaveDataFileSystem(saveDataId, attribute, creationInfo, m_SaveDataRootPathBuffer, hashSalt, false))
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    // Indexer にはなかったのに実在した場合、改めて作り直す
                    NN_RESULT_DO(DeleteSaveDataFileSystemCore(creationInfo.spaceId, saveDataId));
                    NN_RESULT_DO(m_Impl->CreateSaveDataFileSystem(saveDataId, attribute, creationInfo, m_SaveDataRootPathBuffer, hashSalt, false));
                }
            NN_RESULT_END_TRY;

            if (metaInfo.type != SaveDataMetaType::None)
            {
                NN_RESULT_DO(m_Impl->CreateSaveDataMeta(saveDataId, creationInfo.spaceId, metaInfo.type, static_cast<size_t>(metaInfo.size)));
                if (metaInfo.type == SaveDataMetaType::Thumbnail)
                {
                    std::unique_ptr<nn::fs::fsa::IFile> file;
                    NN_RESULT_DO(m_Impl->OpenSaveDataMeta(&file, saveDataId, creationInfo.spaceId, metaInfo.type));

                    char hash[crypto::Sha256Generator::HashSize] = {0}; // ハッシュゼロ埋めはファイル全体のゼロ埋めを意味する
                    NN_RESULT_DO(file->Write(0, hash, sizeof(hash), WriteOption::MakeValue(WriteOptionFlag_Flush)));
                }
            }

            if (isImportable)
            {
                // インポート用セーブデータは FinalizeSaveDataCreation で確定する
                isCreating = false;
                NN_RESULT_SUCCESS;
            }

            if (attribute.staticSaveDataId != IndexerSaveDataId)
            {
                NN_RESULT_DO(accessor.GetInterface()->SetState(saveDataId, SaveDataState::None));
                NN_RESULT_DO(accessor.GetInterface()->Commit());
            }

            isCreating = false;
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    } // NOLINT(impl/function_size)

    // TODO: 複数の SaveDataMetaInfo の受け付け
    Result FileSystemProxyImpl::CreateSaveDataFileSystem(const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const nn::fs::SaveDataMetaInfo& metaInfo) NN_NOEXCEPT
    {
        nn::fs::SaveDataHashSalt hashSalt;
        NN_RESULT_DO(CreateSaveDataFileSystemWithHashSaltImpl(attribute, creationInfo, metaInfo, hashSalt));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CreateSaveDataFileSystemWithHashSalt(const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const nn::fs::SaveDataMetaInfo& metaInfo, const nn::fs::HashSalt& salt) NN_NOEXCEPT
    {
        nn::fs::SaveDataHashSalt hashSalt(salt);
        NN_RESULT_DO(CreateSaveDataFileSystemWithHashSaltImpl(attribute, creationInfo, metaInfo, hashSalt));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CreateSaveDataFileSystemWithHashSaltImpl(const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo, const nn::fs::SaveDataMetaInfo& metaInfo, const nn::fs::SaveDataHashSalt& hashSalt) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %016llx %016llx %d %016llx-%016llx, %016llx %016llx %016llx %016llx %08x %d, %08x %d",
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.index,
            attribute.userId._data[0],
            attribute.userId._data[1],
            creationInfo.size,
            creationInfo.journalSize,
            creationInfo.blockSize,
            creationInfo.ownerId,
            creationInfo.flags,
            creationInfo.spaceId,
            metaInfo.size,
            metaInfo.type
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(DecidePossibleStorageFlag(attribute.type, creationInfo.spaceId));

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto tmpAttribute = attribute;
            auto tmpCreationInfo = creationInfo;

            if (hashSalt)
            {
                // HashSalt 指定がある時は専用権限でチェック
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CreateSaveDataWithHashSalt);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CreateSaveDataWithHashSalt", m_ProcessId);
            }
            else if (tmpAttribute.type == SaveDataType::Account && tmpAttribute.userId == InvalidUserId)
            {
                // MountSaveDataForDebug 用のセーブデータは Debug 権限で作成可能
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CreateSaveData) || pAccessControl->CanCall(AccessControl::OperationType::DebugSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CreateSaveData|DebugSaveData|Debug", m_ProcessId);

                if (tmpAttribute.programId.value == 0)
                {
                    tmpAttribute.programId.value = pProgramInfo->GetProgramIdValue();
                }
                if (tmpCreationInfo.ownerId == 0)
                {
                    tmpCreationInfo.ownerId = pProgramInfo->GetProgramIdValue();
                }
            }
            else
            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CreateSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CreateSaveData", m_ProcessId);
            }

            return CreateSaveDataFileSystemCore(tmpAttribute, tmpCreationInfo, metaInfo, hashSalt);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::CreateSaveDataFileSystemBySystemSaveDataId(const nn::fs::SaveDataAttribute& attribute, const nn::fs::SaveDataCreationInfo& creationInfo) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %016llx %016llx %d %016llx-%016llx, %016llx %016llx %016llx %016llx %08x %d",
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.index,
            attribute.userId._data[0],
            attribute.userId._data[1],
            creationInfo.size,
            creationInfo.journalSize,
            creationInfo.blockSize,
            creationInfo.ownerId,
            creationInfo.flags,
            creationInfo.spaceId
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(DecidePossibleStorageFlag(attribute.type, creationInfo.spaceId));

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto systemSaveDataId = attribute.staticSaveDataId;

            // StaticSaveDataId の値域外はエラー
            NN_FSP_REQUIRES(IsStaticSaveDataIdValueRange(systemSaveDataId), nn::fs::ResultInvalidArgument());

            auto tmpCreationInfo = creationInfo;
            if (tmpCreationInfo.ownerId == 0)
            {
                tmpCreationInfo.ownerId = pProgramInfo->GetProgramIdValue();
            }

            auto pAccessControl = pProgramInfo->GetAccessControl();
            if (tmpCreationInfo.ownerId != pProgramInfo->GetProgramIdValue())
            {
                auto accessibility = pAccessControl->GetAccessibilitySaveDataOwnedBy(tmpCreationInfo.ownerId);
                if (!accessibility.CanWrite())
                {
                    // 所有者でないシステムセーブデータの作成には CreateOthersSystemSaveData 権限が必要
                    auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CreateOthersSystemSaveData);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CreateOthersSystemSaveData", m_ProcessId);
                }
            }
            else
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CreateSystemSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CreateSystemSaveData", m_ProcessId);
            }

            SaveDataMetaInfo metaInfo;
            metaInfo.type = SaveDataMetaType::None;

            if (attribute.type == SaveDataType::SystemBcat)
            {
                tmpCreationInfo.ownerId = BcatOwnerId;
            }

            nn::fs::SaveDataHashSalt hashSalt;

            return CreateSaveDataFileSystemCore(attribute, tmpCreationInfo, metaInfo, hashSalt);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    nn::Result FileSystemProxyImpl::ExtendSaveDataFileSystem(
        std::uint8_t spaceId,
        nn::fs::SaveDataId saveDataId,
        int64_t saveDataAvailableSize,
        int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        NN_FS_SCOPED_TRACE("%d %llu %lld %lld by %llu %016llx",
            spaceId,
            saveDataId,
            saveDataAvailableSize,
            saveDataJournalSize,
            m_ProcessId,
            pProgramInfo->GetProgramIdValue()
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            const auto saveDataSpaceId = static_cast<SaveDataSpaceId>(spaceId);
            const auto pAccessControl = pProgramInfo->GetAccessControl();

            // 権限チェック
            switch( saveDataSpaceId )
            {
            case SaveDataSpaceId::System:
            case SaveDataSpaceId::SdSystem:
            case SaveDataSpaceId::ProperSystem:
            case SaveDataSpaceId::SafeMode:
                {
                    const auto operationType = AccessControl::OperationType::ExtendSystemSaveData;
                    const auto canCall = pAccessControl->CanCall(operationType);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(
                        canCall,
                        "ExtendSystemSaveData",
                        m_ProcessId);
                }
                break;

            case SaveDataSpaceId::User:
            case SaveDataSpaceId::SdUser:
                {
                    const auto operationType = AccessControl::OperationType::ExtendSaveData;
                    const auto canCall = pAccessControl->CanCall(operationType);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(
                        canCall,
                        "ExtendSaveData",
                        m_ProcessId);
                }
                break;

            default:
                NN_RESULT_THROW(nn::fs::ResultInvalidSaveDataSpaceId());
            };

            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));

                SaveDataIndexerValue value;
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));
                switch( value.state )
                {
                case nn::fssrv::SaveDataState::None:
                    break;
                case nn::fssrv::SaveDataState::Deleting:
                case nn::fssrv::SaveDataState::DeletionRegistered:
                case nn::fssrv::SaveDataState::Extending:
                    NN_RESULT_THROW(nn::fs::ResultTargetLocked());
                case nn::fssrv::SaveDataState::Corrupted:
                    NN_RESULT_THROW(nn::fs::ResultSaveDataCorrupted());
                default:
                    NN_RESULT_THROW(nn::fs::ResultInvalidSaveDataState());
                }

                NN_RESULT_DO(
                    accessor.GetInterface()->SetState(saveDataId, SaveDataState::Extending)
                );

                NN_RESULT_DO(accessor.GetInterface()->Commit());
            }

            // 拡張する
            int64_t extendedTotalSize = 0;
            nn::Result extendResult = m_Impl->StartExtendSaveDataFileSystem(
                &extendedTotalSize,
                saveDataId,
                saveDataSpaceId,
                saveDataAvailableSize,
                saveDataJournalSize,
                m_SaveDataRootPathBuffer);
            if( extendResult.IsFailure() )
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));

                // 拡張に失敗した場合は拡張前の状態に戻す
                SaveDataIndexerValue value;
                nn::Result resultGetValue = accessor.GetInterface()->GetValue(&value, saveDataId);
                if( resultGetValue.IsSuccess() )
                {
                    m_Impl->RevertExtendSaveDataFileSystem(
                        saveDataId,
                        saveDataSpaceId,
                        value.size,
                        m_SaveDataRootPathBuffer);
                }

                nn::Result resultSetState = accessor.GetInterface()->SetState(saveDataId, SaveDataState::None);
                if( resultSetState.IsSuccess() )
                {
                    (void)accessor.GetInterface()->Commit();
                }
                NN_RESULT_THROW(extendResult);
            }

            // Size を更新する
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
                const auto result = accessor.GetInterface()->SetSize(saveDataId, extendedTotalSize);
                if( result.IsFailure() )
                {
                    NN_SDK_LOG("[fs] Failed to set size of save data %016llx (%x)\n", saveDataId, result.GetInnerValueForDebug());
                }

                NN_RESULT_DO(accessor.GetInterface()->Commit());
            }

            // 拡張を終了する
            NN_RESULT_DO(m_Impl->FinishExtendSaveDataFileSystem(saveDataId, saveDataSpaceId));

            // SaveDataState を更新する
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
                NN_RESULT_DO(accessor.GetInterface()->SetState(saveDataId, SaveDataState::None));
                NN_RESULT_DO(accessor.GetInterface()->Commit());
            }

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    } // NOLINT(impl/function_size)

    Result FileSystemProxyImpl::OpenSaveDataFileSystemCore(std::shared_ptr<nn::fs::fsa::IFileSystem> *pOutFileSystem, nn::fs::SaveDataId* pOutSaveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute, bool isReadOnly) NN_NOEXCEPT
    {
        SaveDataIndexerAccessor accessor;
        SaveDataId saveDataId;
        bool isStaticSaveDataId = (attribute.staticSaveDataId != InvalidSystemSaveDataId && attribute.userId == InvalidUserId);

        if (isStaticSaveDataId)
        {
            // 非アカウントシステムセーブは Indexer を見に行かない
            saveDataId = attribute.staticSaveDataId;
        }
        else
        {
            // アカウントユーザセーブ＆アカウントシステムセーブ
            SaveDataIndexerValue value;
            auto key = attribute;
            NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
            NN_RESULT_DO(accessor.GetInterface()->Get(&value, key));
            if (value.spaceId != ConvertToRealSpaceId(saveDataSpaceId))
            {
                return ResultTargetNotFound();
            }

            // 拡張中はマウント不可
            if (value.state == SaveDataState::Extending)
            {
                return ResultSaveDataExtending();
            }

            saveDataId = value.id;
        }

        auto DeleteMissingEntryFromIndexer = [&]() -> Result {
            if (saveDataId != IndexerSaveDataId)
            {
                if (isStaticSaveDataId)
                {
                    // 非アカウントシステムセーブの場合、ここで初めて Indexer を触る
                    SaveDataIndexerValue value;
                    auto key = attribute;
                    NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
                    NN_RESULT_DO(accessor.GetInterface()->Get(&value, key));
                    if (value.spaceId != ConvertToRealSpaceId(saveDataSpaceId))
                    {
                        return ResultTargetNotFound();
                    }
                }
                // Indexer にあるが実在しない場合、Indexer の登録を削除する
                accessor.GetInterface()->Delete(saveDataId);
                accessor.GetInterface()->Commit();
            }
            NN_RESULT_SUCCESS;
        };

        NN_RESULT_TRY(m_Impl->OpenSaveDataFileSystem(pOutFileSystem, saveDataSpaceId, saveDataId, m_SaveDataRootPathBuffer, isReadOnly, attribute.type))
            NN_RESULT_CATCH(ResultPathNotFound)
            {
                NN_RESULT_DO(DeleteMissingEntryFromIndexer());
                NN_RESULT_THROW(ResultTargetNotFound());
            }
            NN_RESULT_CATCH(ResultTargetNotFound)
            {
                // Indexer にあるが実在しない場合
                NN_RESULT_DO(DeleteMissingEntryFromIndexer());
                NN_RESULT_THROW(ResultTargetNotFound());
            }
        NN_RESULT_END_TRY;

        *pOutSaveDataId = saveDataId;
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenUserSaveDataFileSystemCore(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute, const detail::ProgramInfo* pProgramInfo, bool isReadOnly) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %d %016llx %016llx %016llx-%016llx by %llu %llu %016llx %u %d",
            saveDataSpaceId,
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.userId._data[0],
            attribute.userId._data[1],
            m_ProcessId,
            pProgramInfo->GetProcessId(),
            pProgramInfo->GetProgramIdValue(),
            pProgramInfo->GetStorageId(),
            isReadOnly
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const nn::fs::SaveDataSpaceId spaceId = static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId);
            const int storageFlag = DecidePossibleStorageFlag(attribute.type, spaceId);
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            // セーブデータはプロセスごとにマウント数制限
            std::unique_lock<fssystem::SemaphoreAdaptor> mountCountSemaphore;
            NN_RESULT_DO(TryAcquireSaveDataMountCountSemaphore(&mountCountSemaphore));

            nn::fs::SaveDataId saveDataId;
            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(OpenSaveDataFileSystemCore(&fileSystem, &saveDataId, spaceId, attribute, isReadOnly));

            // ownerId アクセス権チェック
            {
                Accessibility accessibility;
                const bool isTemporaryFs = (attribute.type == SaveDataType::Temporary);
                NN_RESULT_DO(
                    GetAccesibilityForSaveData(
                        &accessibility,
                        pProgramInfo,
                        saveDataId,
                        spaceId,
                        isTemporaryFs
                    )
                );
                if (!accessibility.CanRead() && !accessibility.CanWrite())
                {
                    auto pAccessControl = pProgramInfo->GetAccessControl();
                    accessibility = pAccessControl->GetAccessibilityFor(
                                        AccessControl::AccessibilityType::MountOthersSaveData);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(
                        accessibility.CanRead() && accessibility.CanWrite(),
                        "MountOthersSaveData",
                        m_ProcessId
                    );
                }
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, std::move(mountCountSemaphore), true);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenUserSaveDataFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute, bool isReadOnly) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %d %d",
            saveDataSpaceId,
            attribute.type,
            isReadOnly
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" %016llx %016llx %016llx-%016llx by %llu %016llx",
                attribute.programId.value,
                attribute.staticSaveDataId,
                attribute.userId._data[0],
                attribute.userId._data[1],
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto pAccessControl = pProgramInfo->GetAccessControl();

            // 本体セーブデータの利用は権限が必要
            if (attribute.type == nn::fs::SaveDataType::Device)
            {
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountDeviceSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountDeviceSaveData", m_ProcessId);
            }

            // Debug 用セーブデータの利用には権限が必要
            if (attribute.type == nn::fs::SaveDataType::Account && attribute.userId == nn::fs::InvalidUserId)
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::DebugSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "MountSaveDataForDebug|Debug", m_ProcessId);
            }

            nn::fs::SaveDataAttribute tmpAttribute;
            if (attribute.programId.value == 0)
            {
                tmpAttribute = nn::fs::SaveDataAttribute::Make(pProgramInfo->GetProgramId(), attribute.type, attribute.userId, attribute.staticSaveDataId, attribute.index);
            }
            else
            {
                tmpAttribute = attribute;
            }
            // キャッシュストレージだったら分岐
            std::uint8_t spaceId;
            if (attribute.type == nn::fs::SaveDataType::Cache)
            {
                nn::fs::SaveDataSpaceId cacheStorageSpaceId;
                NN_RESULT_DO(GetCacheStorageSpaceId(&cacheStorageSpaceId, tmpAttribute.programId.value));
                spaceId = static_cast<uint8_t>(cacheStorageSpaceId);
            }
            else
            {
                spaceId = saveDataSpaceId;
            }
            return OpenUserSaveDataFileSystemCore(outValue, spaceId, tmpAttribute, pProgramInfo.get(), isReadOnly);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenSaveDataFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute) NN_NOEXCEPT
    {
        return OpenUserSaveDataFileSystem(outValue, saveDataSpaceId, attribute, false);
    }

    Result FileSystemProxyImpl::OpenReadOnlySaveDataFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute) NN_NOEXCEPT
    {
        return OpenUserSaveDataFileSystem(outValue, saveDataSpaceId, attribute, true);
    }

    Result FileSystemProxyImpl::OpenSaveDataFileSystemBySystemSaveDataId(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute) NN_NOEXCEPT
    {
        auto systemSaveDataId = attribute.staticSaveDataId;
        NN_FS_SCOPED_TRACE("%d %d %016llx %016llx %016llx-%016llx",
            saveDataSpaceId,
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.userId._data[0],
            attribute.userId._data[1]
        );
        NN_FSP_REQUIRES(IsStaticSaveDataIdValueRange(systemSaveDataId), nn::fs::ResultInvalidArgument());

        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
            m_ProcessId,
            pProgramInfo->GetProgramIdValue()
        );

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const nn::fs::SaveDataSpaceId spaceId = static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId);
            const int storageFlag = DecidePossibleStorageFlag(attribute.type, spaceId);
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            auto pAccessControl = pProgramInfo->GetAccessControl();
            {
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountSystemSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountSystemSaveData", m_ProcessId);
            }

            nn::fs::SaveDataId saveDataId;
            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(OpenSaveDataFileSystemCore(&fileSystem, &saveDataId, spaceId, attribute, false));

            // ownerId アクセス権チェック
            {
                Accessibility accessibility;
                const bool IsTemporaryFs = (attribute.type == SaveDataType::Temporary);
                NN_RESULT_DO(GetAccesibilityForSaveData(&accessibility, pProgramInfo.get(), saveDataId, spaceId, IsTemporaryFs));
                if (!accessibility.CanRead() && !accessibility.CanWrite())
                {
                    accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountOthersSystemSaveData);
                    NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountOthersSystemSaveData", m_ProcessId);
                }
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, true);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::ReadSaveDataFileSystemExtraDataCore(nn::fs::SaveDataExtraData* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, bool isTemporaryFs) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        return m_Impl->ReadSaveDataFileSystemExtraData(outValue, saveDataSpaceId, saveDataId, m_SaveDataRootPathBuffer, isTemporaryFs);
    }

    Result FileSystemProxyImpl::ReadSaveDataFileSystemExtraData(const nn::sf::OutBuffer& buffer, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        return ReadSaveDataFileSystemExtraDataBySaveDataSpaceId(buffer, static_cast<uint8_t>(InvalidSaveDataSpaceId), saveDataId);
    }

    Result FileSystemProxyImpl::ReadSaveDataFileSystemExtraDataBySaveDataSpaceId(const nn::sf::OutBuffer& buffer, std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            bool isTemporaryFs;

            nn::fs::SaveDataSpaceId resolvedSpaceId;
            if( static_cast<SaveDataSpaceId>(saveDataSpaceId) == InvalidSaveDataSpaceId )
            {
                // 存在確認と SpaceId の取得
                SaveDataIndexerAccessor accessor;

                // TORIAEZU: safe モード起動時、アプリセーブについては常に本来側にリダイレクトさせるため、User を指定する
                if (IsStaticSaveDataIdValueRange(saveDataId))
                {
                    NN_RESULT_DO(accessor.Initialize(nn::fs::SaveDataSpaceId::System));
                }
                else
                {
                    NN_RESULT_DO(accessor.Initialize(nn::fs::SaveDataSpaceId::User));
                }

                SaveDataIndexerValue value;
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));
                resolvedSpaceId = value.spaceId;

                SaveDataIndexerKey key;
                NN_RESULT_DO(accessor.GetInterface()->GetKey(&key, saveDataId));
                isTemporaryFs = (key.type == SaveDataType::Temporary);
            }
            else
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(static_cast<SaveDataSpaceId>(saveDataSpaceId)));

                SaveDataIndexerValue value;
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId)); // 存在確認

                resolvedSpaceId = static_cast<SaveDataSpaceId>(saveDataSpaceId);

                SaveDataIndexerKey key;
                NN_RESULT_DO(accessor.GetInterface()->GetKey(&key, saveDataId));
                isTemporaryFs = (key.type == SaveDataType::Temporary);
            }

            // 対象がシステムセーブの場合は ownerId アクセス権があれば権限不要
            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::ReadSaveDataFileSystemExtraData);
                if (IsStaticSaveDataIdValueRange(saveDataId))
                {
                    Accessibility accessibility;
                    NN_RESULT_DO(GetAccesibilityForSaveData(&accessibility, pProgramInfo.get(), saveDataId, resolvedSpaceId, isTemporaryFs));
                    canCall |= accessibility.CanRead();
                }
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "ReadSaveDataFileSystemExtraData", m_ProcessId);
            }

            NN_FSP_REQUIRES(buffer.GetSize() == sizeof(nn::fs::SaveDataExtraData), nn::fs::ResultInvalidArgument());
            auto pExtraData = reinterpret_cast<nn::fs::SaveDataExtraData*>(buffer.GetPointerUnsafe());
            NN_RESULT_DO(ReadSaveDataFileSystemExtraDataCore(pExtraData, resolvedSpaceId, saveDataId, isTemporaryFs));

#if defined(NN_BUILD_CONFIG_OS_WIN32)
            if( pExtraData->ownerId == 0 && pExtraData->availableSize == 0 && pExtraData->journalSize == 0 )
            {
                // ディレクトリ版セーブデータは pExtraData が設定されないので Indexer から情報を取得する
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(resolvedSpaceId));
                SaveDataIndexerKey tmpKey;
                NN_RESULT_DO(accessor.GetInterface()->GetKey(&tmpKey, saveDataId));
                pExtraData->attribute = tmpKey;
            }
#endif

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::WriteSaveDataFileSystemExtraDataCore(nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId, const nn::fs::SaveDataExtraData& extraData, nn::fs::SaveDataType saveDataType, bool updateTimeStampAndHash) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        return m_Impl->WriteSaveDataFileSystemExtraData(saveDataSpaceId, saveDataId, extraData, m_SaveDataRootPathBuffer, saveDataType, updateTimeStampAndHash);
    }

    Result FileSystemProxyImpl::WriteSaveDataFileSystemExtraDataWithMaskCore(nn::fs::SaveDataId saveDataId, nn::fs::SaveDataSpaceId saveDataSpaceId, const nn::fs::SaveDataExtraData& extraDataValue, const nn::fs::SaveDataExtraData& extraDataMask) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            nn::fs::SaveDataType saveDataType;
            {
                SaveDataIndexerAccessor accessor; // ReadModifyWrite のためのロックを兼ねる
                NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
                SaveDataIndexerKey key;
                NN_RESULT_DO(accessor.GetInterface()->GetKey(&key, saveDataId));
                saveDataType = key.type;

                {
                    auto pAccessControl = pProgramInfo->GetAccessControl();

                    if (extraDataMask.flags != 0)
                    {
                        auto canCall =
                            pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataAll) ||
                            pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataFlags);
                        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "WriteSaveDataFileSystemExtraDataAll|Flags", m_ProcessId);
                    }

                    if (extraDataMask.timeStamp != 0)
                    {
                        auto canCall =
                            pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataAll) ||
                            pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataTimeStamp);
                        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "WriteSaveDataFileSystemExtraDataAll|TimeStamp", m_ProcessId);
                    }

                    if (extraDataMask.commitId != 0)
                    {
                        auto canCall =
                            pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataAll) ||
                            pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataCommitId);
                        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "WriteSaveDataFileSystemExtraDataAll|CommitId", m_ProcessId);
                    }

                    {
                        SaveDataExtraData extraDataMaskZero;
                        memset(&extraDataMaskZero, 0, sizeof(extraDataMaskZero));

                        SaveDataExtraData extraDataMaskOtherThanFlags = extraDataMask;
                        {
                            extraDataMaskOtherThanFlags.flags = 0;
                            extraDataMaskOtherThanFlags.timeStamp = 0;
                            extraDataMaskOtherThanFlags.commitId = 0;
                        }

                        // 上記フィールド以外の書き換えには All 権限が必要
                        if (memcmp(&extraDataMaskOtherThanFlags, &extraDataMaskZero, sizeof(SaveDataExtraData)) != 0)
                        {
                            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::WriteSaveDataFileSystemExtraDataAll);
                            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "WriteSaveDataFileSystemExtraDataAll", m_ProcessId);
                        }
                    }
                }

                SaveDataExtraData extraData;
                NN_RESULT_DO(m_Impl->ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId, m_SaveDataRootPathBuffer, false));
                ModifySaveDataExtraData(&extraData, extraDataValue, extraDataMask);
                NN_RESULT_DO(m_Impl->WriteSaveDataFileSystemExtraData(saveDataSpaceId, saveDataId, extraData, m_SaveDataRootPathBuffer, saveDataType, false));
            }

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::WriteSaveDataFileSystemExtraData(std::uint64_t saveDataId, std::uint8_t saveDataSpaceId, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(buffer.GetSize() == sizeof(nn::fs::SaveDataExtraData), nn::fs::ResultInvalidArgument());
        auto pExtraData = reinterpret_cast<const nn::fs::SaveDataExtraData*>(buffer.GetPointerUnsafe());

        // マスク指定なしの WriteSaveDataFileSystemExtraData は flag のみマスク指定とみなす
        SaveDataExtraData extraDataMask;
        memset(&extraDataMask, 0, sizeof(extraDataMask));
        extraDataMask.flags = 0xFFFFFFFF;

        return WriteSaveDataFileSystemExtraDataWithMaskCore(saveDataId, static_cast<SaveDataSpaceId>(saveDataSpaceId), *pExtraData, extraDataMask);
    }

    Result FileSystemProxyImpl::WriteSaveDataFileSystemExtraDataWithMask(std::uint64_t saveDataId, std::uint8_t saveDataSpaceId, const nn::sf::InBuffer& extraDataValue, const nn::sf::InBuffer& extraDataMask) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(extraDataValue.GetSize() == sizeof(nn::fs::SaveDataExtraData), nn::fs::ResultInvalidArgument());
        NN_FSP_REQUIRES(extraDataMask.GetSize()  == sizeof(nn::fs::SaveDataExtraData), nn::fs::ResultInvalidArgument());

        return WriteSaveDataFileSystemExtraDataWithMaskCore(
            saveDataId,
            static_cast<SaveDataSpaceId>(saveDataSpaceId),
            *reinterpret_cast<const nn::fs::SaveDataExtraData*>(extraDataValue.GetPointerUnsafe()),
            *reinterpret_cast<const nn::fs::SaveDataExtraData*>(extraDataMask.GetPointerUnsafe())
            );
    }

    Result FileSystemProxyImpl::OpenImageDirectoryFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint32_t storageId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%u",
            storageId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const auto imageDirectoryId = static_cast<nn::fs::ImageDirectoryId>(storageId);

            const int storageFlag = StorageFlag_ReadWriteDataInAnyStorage;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountImageAndVideoStorage);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountImageAndVideoStorage", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenImageDirectoryFileSystem(&fileSystem, imageDirectoryId));

            std::shared_ptr<nn::fs::fsa::IFileSystem> typeSetFileSystem = nn::fssystem::AllocateShared <nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(typeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::SetBisRootForHost(std::uint32_t id, const nn::fssrv::sf::FspPath& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%u %s",
            id,
            path.str
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            return m_Impl->SetBisRootForHost(static_cast<nn::fs::BisPartitionId>(id), path.str);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenBisFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, const nn::fssrv::sf::FspPath& path, std::uint32_t id) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%s %u",
            path.str,
            id
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = nn::fssystem::StorageFlag_Mmc;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                Accessibility accessibility;
                NN_RESULT_DO(GetAccesibilityForOpenBisFileSystem(&accessibility, pProgramInfo.get(), static_cast<nn::fs::BisPartitionId>(id)));
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountBis", m_ProcessId);
            }

            // normalizedPath の寿命に注意
            PathNormalizer normalizedPath(path.str, PathNormalizer::AcceptEmpty);
            NN_RESULT_DO(normalizedPath.GetResult());

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenBisFileSystem(&fileSystem, normalizedPath.GetPath(), static_cast<nn::fs::BisPartitionId>(id)));

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenBisStorage(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>> outValue, std::uint32_t id) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%u",
            id
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = nn::fssystem::StorageFlag_Mmc;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                Accessibility accessibility;
                NN_RESULT_DO(GetAccesibilityForOpenBisPartition(&accessibility, pProgramInfo.get(), static_cast<nn::fs::BisPartitionId>(id)));
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "OpenBisStorage", m_ProcessId);
            }

            std::shared_ptr<fs::IStorage> storage;
            NN_RESULT_DO(m_Impl->OpenBisStorage(&storage, static_cast<nn::fs::BisPartitionId>(id)));

            std::shared_ptr<fs::IStorage> pTypeSetStorage = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetStorage>(std::move(storage), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetStorage, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IStorage> pStorage = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IStorage, StorageInterfaceAdapter>(std::move(pTypeSetStorage));
            NN_RESULT_THROW_UNLESS(pStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pStorage));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::InvalidateBisCache() NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::InvalidateBisCache);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "InvalidateBisCache", m_ProcessId);
        }

        NN_RESULT_DO(m_Impl->InvalidateBisCache());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenHostFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, const nn::fssrv::sf::FspPath& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%s",
            path.str
        );
        // 解析用ログを無効化
        NN_FS_SCOPED_TRACE_SUPPRESS();

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountHost);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountHost|Debug", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenHostFileSystem(&fileSystem, path.str));
            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(fileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenSdCardFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = nn::fssystem::StorageFlag_SdCard;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountSdCard);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountSdCard|Debug", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenSdCardProxyFileSystem(&fileSystem));

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::FormatSdCardFileSystem() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_SdCard);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::FormatSdCard);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "FormatSdCard", m_ProcessId);
            }

            NN_RESULT_DO(m_Impl->FormatSdCardProxyFileSystem());
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::FormatSdCardDryRun() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_SdCard);

            NN_RESULT_DO(m_Impl->FormatSdCardDryRun());
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::IsExFatSupported(nn::sf::Out<bool> outValue) NN_NOEXCEPT
    {
        outValue.Set(m_Impl->IsExFatSupported());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenGameCardStorage(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>> outValue, std::uint32_t handle, std::uint32_t partition) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%08x %u",
            handle,
            partition
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
            }

            std::shared_ptr<fs::IStorage> storage;
            NN_RESULT_DO(m_Impl->OpenGameCardPartition(&storage, static_cast<nn::fs::GameCardHandle>(handle), static_cast<nn::fs::GameCardPartitionRaw>(partition)));
            nn::sf::SharedPointer<nn::fssrv::sf::IStorage> pStorage = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IStorage, StorageInterfaceAdapter>(std::move(storage));
            NN_RESULT_THROW_UNLESS(pStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pStorage));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> pDeviceOperator = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IDeviceOperator, DeviceOperator>(pProgramInfo->GetAccessControl(), m_ProcessId);
        NN_RESULT_THROW_UNLESS(pDeviceOperator != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pDeviceOperator));
        NN_RESULT_SUCCESS;
    }

    namespace
    {
        SdCardDetectionEventManager* GetSdCardDetectionEventManager() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(SdCardDetectionEventManager, g_SdCardDetectionEventManager, );
            return &g_SdCardDetectionEventManager;
        }

        GameCardDetectionEventManager* GetGameCardDetectionEventManager() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(GameCardDetectionEventManager, g_GameCardDetectionEventManager, );
            return &g_GameCardDetectionEventManager;
        }

        SystemDataUpdateEventManager* GetSystemDataUpdateEventManager() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(SystemDataUpdateEventManager, g_SystemDataUpdateEventManager, );
            return &g_SystemDataUpdateEventManager;
        }
    }

    Result FileSystemProxyImpl::OpenSdCardDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSdCardDetectionEventNotifier);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSdCardDetectionEventNotifier", m_ProcessId);
        }

        std::unique_ptr<DeviceDetectionEventNotifier> pDeviceDetectionEventNotifier;
        NN_RESULT_DO(GetSdCardDetectionEventManager()->CreateNotifier(&pDeviceDetectionEventNotifier));
        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> pEventNotifier = FileSystemFactory::CreateShared<nn::fssrv::sf::IEventNotifier>(std::move(pDeviceDetectionEventNotifier));
        NN_RESULT_THROW_UNLESS(pEventNotifier != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pEventNotifier));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenGameCardDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenGameCardDetectionEventNotifier);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenGameCardDetectionEventNotifier", m_ProcessId);
        }

        std::unique_ptr<DeviceDetectionEventNotifier> pDeviceDetectionEventNotifier;
        NN_RESULT_DO(GetGameCardDetectionEventManager()->CreateNotifier(&pDeviceDetectionEventNotifier));
        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> pEventNotifier = FileSystemFactory::CreateShared<nn::fssrv::sf::IEventNotifier>(std::move(pDeviceDetectionEventNotifier));
        NN_RESULT_THROW_UNLESS(pEventNotifier != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pEventNotifier));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::SimulateDeviceDetectionEvent(std::uint32_t deviceType, std::uint32_t simulatingDetectionMode, bool isWithEvent) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::SimulateDevice);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SimulateDevice", m_ProcessId);
        }

        fs::SimulatingDeviceType device = static_cast<fs::SimulatingDeviceType>(deviceType);
        fs::SimulatingDeviceDetectionMode detectionMode = static_cast<fs::SimulatingDeviceDetectionMode>(simulatingDetectionMode);

        switch (device)
        {
        case fs::SimulatingDeviceType::GameCard:
            nn::fssrv::detail::GetGameCardEventSimulator().SetDetectionSimulationMode(detectionMode);
            if (isWithEvent)
            {
                GetGameCardDetectionEventManager()->SimulateEvent();
            }
            break;
        case fs::SimulatingDeviceType::SdCard:
            // GetSdCardEventSimulator()...
            if (isWithEvent)
            {
                GetSdCardDetectionEventManager()->SimulateEvent();
            }
            break;
        case fs::SimulatingDeviceType::eMMC:
            return fs::ResultNotImplemented();
        default:
            return fs::ResultInvalidArgument();
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSystemDataUpdateEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSystemDataUpdateEventNotifier);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSystemDataUpdateEventNotifier", m_ProcessId);
        }

        std::unique_ptr<SystemDataUpdateEventNotifier> pSystemDataUpdateEventNotifier;
        NN_RESULT_DO(GetSystemDataUpdateEventManager()->CreateNotifier(&pSystemDataUpdateEventNotifier));
        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> pEventNotifier = FileSystemFactory::CreateShared<nn::fssrv::sf::IEventNotifier>(std::move(pSystemDataUpdateEventNotifier));
        NN_RESULT_THROW_UNLESS(pEventNotifier != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pEventNotifier));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::NotifySystemDataUpdateEvent() NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::NotifySystemDataUpdateEvent);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "NotifySystemDataUpdateEvent", m_ProcessId);
        }

        NN_RESULT_DO(GetSystemDataUpdateEventManager()->NotifySystemDataUpdateEvent());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataInfoReader(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>> outValue) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto pAccessControl = pProgramInfo->GetAccessControl();

            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataInfoReader);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataInfoReader", m_ProcessId);
            }
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataInfoReaderForSystem);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataInfoReaderForSystem", m_ProcessId);
            }

            std::shared_ptr<SaveDataInfoReaderImpl> reader;
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(nn::fs::SaveDataSpaceId::System));
                NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&reader));
            }
            nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> pSaveDataInfoReader = FileSystemFactory::CreateShared<nn::fssrv::sf::ISaveDataInfoReader>(std::move(reader));
            NN_RESULT_THROW_UNLESS(pSaveDataInfoReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pSaveDataInfoReader));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

namespace {

    Result CheckOpenSaveDataInfoReaderAccessControl(std::shared_ptr<ProgramInfo> pProgramInfo, nn::Bit64 processId, nn::fs::SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_UNUSED(processId);
#endif
        NN_SDK_ASSERT_NOT_NULL(pProgramInfo);
        auto pAccessControl = pProgramInfo->GetAccessControl();

        switch (saveDataSpaceId)
        {
        case SaveDataSpaceId::System:
        case SaveDataSpaceId::SdSystem:
        case SaveDataSpaceId::ProperSystem:
        case SaveDataSpaceId::SafeMode:
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataInfoReaderForSystem);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataInfoReaderForSystem", processId);
            }
            break;
        case SaveDataSpaceId::User:
        case SaveDataSpaceId::SdUser:
        case SaveDataSpaceId::Temporary:
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataInfoReader);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataInfoReader", processId);
            }
            break;
        default:
            NN_RESULT_THROW(nn::fs::ResultInvalidSaveDataSpaceId());
        };

        NN_RESULT_SUCCESS;
    }

}

    Result FileSystemProxyImpl::OpenSaveDataInfoReaderBySaveDataSpaceId(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>> outValue, std::uint8_t saveDataSpaceId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d",
            saveDataSpaceId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto spaceId = static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId);
            NN_RESULT_DO(CheckOpenSaveDataInfoReaderAccessControl(pProgramInfo, m_ProcessId, spaceId));

            std::unique_ptr<SaveDataInfoFilterReader> filterReader;
            {
                std::shared_ptr<SaveDataInfoReaderImpl> reader;
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(spaceId));
                NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&reader));

                SaveDataInfoFilter filter(ConvertToRealSpaceId(spaceId), util::nullopt, util::nullopt, util::nullopt, util::nullopt, util::nullopt, 0);
                filterReader.reset(new SaveDataInfoFilterReader(std::move(reader), filter));
                NN_RESULT_THROW_UNLESS(filterReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            }
            nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> pSaveDataInfoReader = FileSystemFactory::CreateShared<nn::fssrv::sf::ISaveDataInfoReader>(std::move(filterReader));
            NN_RESULT_THROW_UNLESS(pSaveDataInfoReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pSaveDataInfoReader));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenSaveDataInfoReaderWithFilter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>> outValue, std::uint8_t saveDataSpaceId, const SaveDataFilter& filter) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d",
            saveDataSpaceId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataInfoReaderForInternal);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataInfoReaderForInternal", m_ProcessId);
            }

            auto spaceId = static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId);
            NN_RESULT_DO(CheckOpenSaveDataInfoReaderAccessControl(pProgramInfo, m_ProcessId, spaceId));

            std::unique_ptr<SaveDataInfoFilterReader> filterReader;
            {
                std::shared_ptr<SaveDataInfoReaderImpl> reader;
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(spaceId));
                NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&reader));

                filterReader.reset(new SaveDataInfoFilterReader(std::move(reader), SaveDataInfoFilter(ConvertToRealSpaceId(spaceId), filter)));
                NN_RESULT_THROW_UNLESS(filterReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            }
            nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> pSaveDataInfoReader = FileSystemFactory::CreateShared<nn::fssrv::sf::ISaveDataInfoReader>(std::move(filterReader));
            NN_RESULT_THROW_UNLESS(pSaveDataInfoReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pSaveDataInfoReader));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::FindSaveDataWithFilterImpl(int64_t* outValue, nn::fs::SaveDataInfo* outInfoBuffer, nn::fs::SaveDataSpaceId saveDataSpaceId, const SaveDataInfoFilter& filter) NN_NOEXCEPT
    {
        std::shared_ptr<SaveDataInfoReaderImpl> reader;
        SaveDataIndexerAccessor accessor;
        NN_RESULT_DO(accessor.Initialize(saveDataSpaceId));
        NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&reader));

        std::unique_ptr<SaveDataInfoFilterReader> filterReader(new SaveDataInfoFilterReader(std::move(reader), filter));
        NN_RESULT_THROW_UNLESS(filterReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());

        NN_RESULT_DO(filterReader->Read(nn::sf::Out<int64_t>(outValue), nn::sf::OutBuffer(reinterpret_cast<char*>(outInfoBuffer), sizeof(SaveDataInfo))));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::FindSaveDataWithFilter(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outEntries, std::uint8_t saveDataSpaceId, const SaveDataFilter& filter) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(outEntries.GetSize() == sizeof(nn::fs::SaveDataInfo), nn::fs::ResultInvalidArgument());

        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        nn::fs::SaveDataSpaceId spaceId = static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId);
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
        NN_RESULT_DO(CheckOpenSaveDataInfoReaderAccessControl(pProgramInfo, m_ProcessId, spaceId));

        SaveDataInfoFilter infoFilter(ConvertToRealSpaceId(spaceId), filter);
        int64_t outCount;
        NN_RESULT_DO(FindSaveDataWithFilterImpl(&outCount, reinterpret_cast<nn::fs::SaveDataInfo*>(outEntries.GetPointerUnsafe()), spaceId, infoFilter));

        outValue.Set(outCount);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CreateEmptyThumbnailFile(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Impl->CreateSaveDataMeta(saveDataId, spaceId, fs::SaveDataMetaType::Thumbnail, fs::detail::ThumbnailFileSize));
        std::unique_ptr<nn::fs::fsa::IFile> file;
        NN_RESULT_DO(m_Impl->OpenSaveDataMeta(&file, saveDataId, spaceId, fs::SaveDataMetaType::Thumbnail));

        char hash[crypto::Sha256Generator::HashSize] = { 0 };
        NN_RESULT_DO(file->Write(0, hash, sizeof(hash), WriteOption::MakeValue(WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataInternalStorageFileSystemCore(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        const int storageFlag = StorageFlag_ReadWriteDataInAnyStorage;
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

        // Indexer チェック
        nn::fs::SaveDataId fsSaveDataId = static_cast<fs::SaveDataId>(saveDataId);
        {
            SaveDataIndexerAccessor accessor;
            NN_RESULT_DO(accessor.Initialize(static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId)));
            SaveDataIndexerValue value;
            NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, fsSaveDataId));
            if (value.spaceId != static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId))
            {
                NN_RESULT_THROW(nn::fs::ResultTargetNotFound());
            }

            // 拡張中はマウント不可
            if (value.state == SaveDataState::Extending)
            {
                NN_RESULT_THROW(nn::fs::ResultSaveDataExtending());
            }

            saveDataId = value.id;
        }
        const bool isTemporaryFs = false;

        std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
        NN_RESULT_DO(m_Impl->OpenSaveDataInternalStorageFileSystem(&fileSystem, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, m_SaveDataRootPathBuffer, isTemporaryFs));

        std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
        NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

        *outValue = std::move(pTypeSetFileSystem);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataInternalStorageFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %lld",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            // 権限チェック
            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountSaveDataInternalStorage);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountSaveDataInternalStorage", m_ProcessId);
            }

            // セーブデータはプロセスごとにマウント数制限
            std::unique_lock<fssystem::SemaphoreAdaptor> mountCountSemaphore;
            NN_RESULT_DO(TryAcquireSaveDataMountCountSemaphore(&mountCountSemaphore));

            // Indexer チェック
            nn::fs::SaveDataId fsSaveDataId = static_cast<fs::SaveDataId>(saveDataId);
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId)));
                SaveDataIndexerValue value;
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, fsSaveDataId));
                if( value.spaceId != ConvertToRealSpaceId(static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId)) )
                {
                    NN_RESULT_THROW(nn::fs::ResultTargetNotFound());
                }

                // 拡張中はマウント不可
                if( value.state == SaveDataState::Extending )
                {
                    NN_RESULT_THROW(nn::fs::ResultSaveDataExtending());
                }

                saveDataId = value.id;
            }

            std::shared_ptr<fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(OpenSaveDataInternalStorageFileSystemCore(&fileSystem, static_cast<SaveDataSpaceId>(saveDataSpaceId), static_cast<SaveDataId>(saveDataId)));

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(fileSystem), this, std::move(mountCountSemaphore), true);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT;
    }

    Result FileSystemProxyImpl::QuerySaveDataInternalStorageTotalSize(nn::sf::Out<int64_t> outValue, std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %lld",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::QuerySaveDataInternalStorageTotalSize);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "QuerySaveDataInternalStorageTotalSize", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> internalStorageFs;
            NN_RESULT_DO(OpenSaveDataInternalStorageFileSystemCore(&internalStorageFs, static_cast<SaveDataSpaceId>(saveDataSpaceId), static_cast<SaveDataId>(saveDataId)));

            const char* const InternalStorageFileNameArray[fssystem::save::SaveDataFileSystemCore::InternalStorageFileCount] =
            {
                fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableControlArea,
                fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta,
                fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree,
            };

            int64_t totalSize = 0;
            for (int i = 0; i < fssystem::save::SaveDataFileSystemCore::InternalStorageFileCount; i++)
            {
                char path[fs::EntryNameLengthMax + 1];
                util::SNPrintf(path, sizeof(path), "/%s", InternalStorageFileNameArray[i]);
                int64_t size;
                std::unique_ptr<fs::fsa::IFile> file;
                NN_RESULT_DO(internalStorageFs->OpenFile(&file, path, fs::OpenMode_Read));
                NN_RESULT_DO(file->GetSize(&size));
                totalSize += size;
            }
            *outValue = totalSize;
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT;
    }

    Result FileSystemProxyImpl::GetSaveDataCommitId(nn::sf::Out<int64_t> outValue, std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %lld",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::GetSaveDataCommitId);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "GetSaveDataCommitId", m_ProcessId);
            }

            SaveDataExtraData extraData;
            NN_RESULT_DO(ReadSaveDataFileSystemExtraDataBySaveDataSpaceId(nn::sf::OutBuffer(reinterpret_cast<char*>(&extraData), sizeof(extraData)), saveDataSpaceId, saveDataId));

            if (extraData.commitId == 0)
            {
                char hash[crypto::Sha256Generator::HashSize];
                crypto::GenerateSha256Hash(hash, sizeof(hash), &extraData, sizeof(extraData));
                int64_t commitId;
                commitId = util::LoadLittleEndian<int64_t>(reinterpret_cast<int64_t*>(&hash));
                outValue.Set(commitId);
                NN_RESULT_SUCCESS;
            }
            else
            {
                outValue.Set(extraData.commitId);
                NN_RESULT_SUCCESS;
            }
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT;
    }

    Result FileSystemProxyImpl::OpenSaveDataInfoReaderOnlyCacheStorage(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>> outValue) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        fs::SaveDataSpaceId spaceId = fs::SaveDataSpaceId::User;
        NN_RESULT_TRY(GetCacheStorageSpaceId(&spaceId))
            NN_RESULT_CATCH(ResultTargetNotFound)
            {
                // 無視する
            }
        NN_RESULT_END_TRY
        NN_RESULT_DO(OpenSaveDataInfoReaderOnlyCacheStorage(outValue, static_cast<uint8_t>(spaceId)));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataInfoReaderOnlyCacheStorage(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>> outValue, std::uint8_t saveDataSpaceId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d",
            saveDataSpaceId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            auto spaceId = static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId);
            if (spaceId != nn::fs::SaveDataSpaceId::User && spaceId != nn::fs::SaveDataSpaceId::SdUser)
            {
                return nn::fs::ResultInvalidSaveDataSpaceId();
            }

            std::unique_ptr<SaveDataInfoFilterReader> filterReader;
            {
                std::shared_ptr<SaveDataInfoReaderImpl> reader;
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(spaceId));
                NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&reader));

                SaveDataInfoFilter filter(ConvertToRealSpaceId(spaceId), pProgramInfo->GetProgramIdValue(), nn::fs::SaveDataType::Cache, util::nullopt, util::nullopt, util::nullopt, 0);
                filterReader.reset(new SaveDataInfoFilterReader(std::move(reader), filter));
                NN_RESULT_THROW_UNLESS(filterReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            }
            nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> pSaveDataInfoReader = FileSystemFactory::CreateShared<nn::fssrv::sf::ISaveDataInfoReader>(std::move(filterReader));
            NN_RESULT_THROW_UNLESS(pSaveDataInfoReader != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pSaveDataInfoReader));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenSaveDataMetaFileRaw(std::shared_ptr<fs::fsa::IFile>* outValue, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId, fs::SaveDataMetaType metaType, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        const int storageFlag = nn::fssystem::StorageFlag_Mmc;
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

        // TODO: OpenSaveDataMetaFile() と共通化

        std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
        NN_RESULT_DO(m_Impl->OpenSaveDataMetaDirectoryFileSystem(&fileSystem, spaceId, saveDataId));

        std::shared_ptr<nn::fs::fsa::IFileSystem> typeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
        NN_RESULT_THROW_UNLESS(typeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

        char path[1 + 8 + 1 + 4 + 1];
        util::SNPrintf(path, sizeof(path), "/%08x.meta", metaType);

        // W/A for SIGLO-71646
        if( metaType == fs::SaveDataMetaType::Thumbnail )
        {
            DirectoryEntryType type;
            NN_RESULT_TRY(typeSetFileSystem->GetEntryType(&type, path))
                NN_RESULT_CATCH(ResultPathNotFound)
                {
                    // サムネイルファイルが無い場合は再作成する
                    (void)(CreateEmptyThumbnailFile(spaceId, saveDataId));
                }
                NN_RESULT_CATCH_ALL
                {
                    // エラーは無視する
                }
            NN_RESULT_END_TRY
        }

        std::unique_ptr<nn::fs::fsa::IFile> file;
        NN_RESULT_DO(typeSetFileSystem->OpenFile(&file, path, mode));

        // TODO: file -> fileSystem の shared_ptr

        *outValue = std::move(file);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataMetaFile(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFile>> outValue, std::uint8_t saveDataSpaceId, const nn::fs::SaveDataAttribute& attribute, std::uint32_t metaType) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %d %016llx %016llx %016llx-%016llx %x",
            saveDataSpaceId,
            attribute.type,
            attribute.programId.value,
            attribute.staticSaveDataId,
            attribute.userId._data[0],
            attribute.userId._data[1],
            metaType
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = DecidePossibleStorageFlag(attribute.type, static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId));
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataMetaFile);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataMetaFile", m_ProcessId);
            }

            nn::fs::SaveDataId saveDataId;
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(static_cast<SaveDataSpaceId>(saveDataSpaceId)));
                SaveDataIndexerValue value;
                auto key = attribute;
                NN_RESULT_DO(accessor.GetInterface()->Get(&value, key));
                saveDataId = value.id;
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> tmpFileSystem;
            NN_RESULT_DO(m_Impl->OpenSaveDataMetaDirectoryFileSystem(&tmpFileSystem, static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId));

            std::shared_ptr<nn::fs::fsa::IFileSystem> pTypeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(tmpFileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(pTypeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            auto fileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(pTypeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(fileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());

            nn::fssrv::sf::Path sfPath;
            util::SNPrintf(sfPath.str, sizeof(sfPath.str), "/%08x.meta", metaType);

            // W/A for SIGLO-71646
            if (static_cast<fs::SaveDataMetaType>(metaType) == fs::SaveDataMetaType::Thumbnail)
            {
                uint32_t type;
                NN_RESULT_TRY(fileSystem->GetEntryType(nn::sf::Out<uint32_t>(&type), sfPath))
                    NN_RESULT_CATCH(ResultPathNotFound)
                    {
                        // サムネイルファイルが無い場合は再作成する
                        (void)(CreateEmptyThumbnailFile(static_cast<SaveDataSpaceId>(saveDataSpaceId), saveDataId));
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        // エラーは無視する
                    }
                NN_RESULT_END_TRY
            }

            nn::sf::SharedPointer<nn::fssrv::sf::IFile> file;
            NN_RESULT_DO(fileSystem->OpenFile(&file, sfPath, static_cast<uint32_t>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

            outValue.Set(file);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::GetCacheStorageSpaceId(fs::SaveDataSpaceId* pOutSpaceId) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
        return GetCacheStorageSpaceId(pOutSpaceId, pProgramInfo->GetProgramIdValue());
    }

    Result FileSystemProxyImpl::GetCacheStorageSpaceId(fs::SaveDataSpaceId* pOutSpaceId, nn::Bit64 programIdValue) NN_NOEXCEPT
    {
        auto isCacheStorageExisting = [=](bool* outIsExisting, SaveDataSpaceId spaceId) NN_NOEXCEPT -> Result
        {
            SaveDataInfoFilter filter(ConvertToRealSpaceId(spaceId), programIdValue, SaveDataType::Cache, util::nullopt, util::nullopt, util::nullopt, 0);
            fs::SaveDataInfo info = {};
            int64_t count = 0;
            NN_RESULT_DO(FindSaveDataWithFilterImpl(&count, &info, spaceId, filter));

            *outIsExisting = (count != 0);
            NN_RESULT_SUCCESS;
        };

        // SD にアクセス可能なら SD にアプリ ID が同一のキャッシュストレージがあるかどうかを確認する
        if (m_Impl->IsSdCardAccessible())
        {
            bool isExistInSd = false;
            NN_RESULT_DO(isCacheStorageExisting(&isExistInSd, SaveDataSpaceId::SdUser));
            if (isExistInSd)
            {
                *pOutSpaceId = SaveDataSpaceId::SdUser;
                NN_RESULT_SUCCESS;
            }
        }

        // NAND にアプリ ID が同一のキャッシュストレージがあるかどうか確認
        bool isExistInNand = false;
        NN_RESULT_DO(isCacheStorageExisting(&isExistInNand, SaveDataSpaceId::User));
        if (isExistInNand)
        {
            *pOutSpaceId = SaveDataSpaceId::User;
            NN_RESULT_SUCCESS;
        }

        // どちらにもない場合は TargetNotFound
        NN_RESULT_THROW(fs::ResultTargetNotFound());
    }

    Result FileSystemProxyImpl::FindCacheStorage(nn::fs::SaveDataInfo* pOutValue, fs::SaveDataSpaceId* pOutSpaceId, uint16_t index) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetCacheStorageSpaceId(pOutSpaceId));

        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        SaveDataInfoFilter filter(ConvertToRealSpaceId(*pOutSpaceId), pProgramInfo->GetProgramIdValue(), SaveDataType::Cache, util::nullopt, util::nullopt, index, 0);
        nn::fs::SaveDataInfo info;
        int64_t outCount = 0;
        NN_RESULT_DO(FindSaveDataWithFilterImpl(&outCount, &info, *pOutSpaceId, filter));
        if (outCount == 0)
        {
            NN_RESULT_THROW(fs::ResultTargetNotFound());
        }
        *pOutValue = info;
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::DeleteCacheStorage(std::uint16_t index) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        nn::fs::SaveDataInfo info;
        SaveDataSpaceId spaceId;
        NN_RESULT_DO(FindCacheStorage(&info, &spaceId, index));
        // 呼び出し元アプリの権限でなくFSプロセスの権限で呼ぶ必要があるのでIPC経由で呼ぶ
        NN_RESULT_DO(DeleteSaveData(spaceId, info.saveDataId));
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::GetCacheStorageSize(nn::sf::Out<std::int64_t> outSize, nn::sf::Out<std::int64_t> outJournalSize, std::uint16_t index) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        nn::fs::SaveDataInfo info;
        SaveDataSpaceId spaceId;
        NN_RESULT_DO(FindCacheStorage(&info, &spaceId, index));
        SaveDataExtraData extraData;
        NN_RESULT_DO(m_Impl->ReadSaveDataFileSystemExtraData(&extraData, spaceId, info.saveDataId, m_SaveDataRootPathBuffer, false));
        outSize.Set(extraData.availableSize);
        outJournalSize.Set(extraData.journalSize);
        NN_RESULT_SUCCESS;
    }

namespace
{
    nn::fssrv::SaveDataTransferCryptoConfiguration* g_SaveDataTransferCryptoConfiguration;
}

    Result FileSystemProxyImpl::OpenSaveDataTransferManager(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManager>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
        NN_SDK_ASSERT_NOT_NULL(g_SaveDataTransferCryptoConfiguration);

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataTransferManager);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataTransferManager", m_ProcessId);
        }

        nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManager> pSaveDataTransferManager = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataTransferManager, SaveDataTransferManager>(*g_SaveDataTransferCryptoConfiguration, this);
        NN_RESULT_THROW_UNLESS(pSaveDataTransferManager != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pSaveDataTransferManager));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataTransferManagerVersion2(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManagerWithDivision>> outValue) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
        NN_SDK_ASSERT_NOT_NULL(g_SaveDataTransferCryptoConfiguration);

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataTransferManagerVersion2);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataTransferManagerVersion2", m_ProcessId);
        }

        nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManagerWithDivision> manager = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataTransferManagerWithDivision, fssrv::detail::SaveDataTransferManagerVersion2>(*g_SaveDataTransferCryptoConfiguration, this, &g_PorterManager);
        NN_RESULT_THROW_UNLESS(manager != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(manager));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::OpenSaveDataTransferProhibiter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferProhibiter>> outValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
        NN_SDK_ASSERT_NOT_NULL(g_SaveDataTransferCryptoConfiguration);

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenSaveDataTransferProhibiter);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenSaveDataTransferProhibiter", m_ProcessId);
        }

        auto prohibiter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataTransferProhibiter, fssrv::detail::SaveDataPorterProhibiter>(&g_PorterManager, applicationId);
        NN_RESULT_THROW_UNLESS(prohibiter != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());

        g_PorterManager.RegisterProhibiter(&prohibiter.GetImpl());

        outValue.Set(std::move(prohibiter));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::ListAccessibleSaveDataOwnerId(nn::sf::Out<std::int32_t> outValue, const nn::sf::OutBuffer& outBuffer, nn::ncm::ProgramId programId, std::int32_t offset, std::int32_t count) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(offset >= 0, nn::fs::ResultInvalidOffset());

        ncm::ApplicationId* pOwnerIds = nullptr;

        // count が 0 の場合は個数のみ取得
        if (count != 0)
        {
            pOwnerIds = reinterpret_cast<ncm::ApplicationId*>(outBuffer.GetPointerUnsafe());
            NN_FSP_REQUIRES(pOwnerIds != nullptr, nn::fs::ResultNullptrArgument());
            NN_FSP_REQUIRES(outBuffer.GetSize() >= sizeof(ncm::ApplicationId) * count, nn::fs::ResultInvalidSize());
        }

        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::ListAccessibleSaveDataOwnerId);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "ListAccessibleSaveDataOwnerId", m_ProcessId);
        }

        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            // 起動しているプログラム限定
            NN_RESULT_DO(GetProgramInfoByProgramId(&pProgramInfo, programId.value));

            auto pAccessControl = pProgramInfo->GetAccessControl();
            pAccessControl->ListSaveDataOwnedId(outValue.GetPointer(), pOwnerIds, offset, count);
            NN_RESULT_SUCCESS;
        }
    }

    // TODO: 削除
    Result FileSystemProxyImpl::SetSaveDataSize(std::int64_t size, std::int64_t journalSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(size >= 0 && journalSize >= 0, nn::fs::ResultInvalidSize());
        m_SaveDataSize = size;
        m_SaveDataJournalSize = journalSize;
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::SetSaveDataRootPath(const nn::fssrv::sf::FspPath& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%s",
            path.str
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::DebugSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetSaveDataRootPath|Debug", m_ProcessId);
            }

            if(strnlen(path.str, EntryNameLengthMax + 1) > EntryNameLengthMax)
            {
                return ResultTooLongPath();
            }
            if(path.str[0] == '\0')
            {
                m_SaveDataRootPathBuffer[0] = '\0';
            }
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            strncpy(m_SaveDataRootPathBuffer, path.str, strnlen(path.str, EntryNameLengthMax));
#else
            util::SNPrintf(m_SaveDataRootPathBuffer, EntryNameLengthMax, "/%s", path.str);
#endif
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenContentStorageFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint32_t storageId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%u",
            storageId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const auto contentStorageId = static_cast<nn::fs::ContentStorageId>(storageId);
            const int storageFlag = (contentStorageId == nn::fs::ContentStorageId::System)
                    ? nn::fssystem::StorageFlag_Mmc
                    : StorageFlag_ReadOnlyDataInAnyStorage;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountContentStorage);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountContentStorage", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenContentStorageFileSystem(&fileSystem, contentStorageId));

            std::shared_ptr<nn::fs::fsa::IFileSystem> typeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(typeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenCloudBackupWorkStorageFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint32_t storageId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%u",
            storageId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const auto workStorageId = static_cast<nn::fs::CloudBackupWorkStorageId>(storageId);

            const int storageFlag = StorageFlag_ReadWriteDataInAnyStorage;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountCloudBackupWorkStorage);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead() && accessibility.CanWrite(), "MountCloudBackupWorkStorage", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenCloudBackupWorkStorageFileSystem(&fileSystem, workStorageId));

            std::shared_ptr<nn::fs::fsa::IFileSystem> typeSetFileSystem = nn::fssystem::AllocateShared <nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(typeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::OpenGameCardFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue, std::uint32_t handle, std::uint32_t partition) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%08x %u",
            handle,
            partition
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx",
                m_ProcessId,
                pProgramInfo->GetProgramIdValue()
            );

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountGameCard);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountGameCard", m_ProcessId);
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenGameCardFileSystem(&fileSystem, static_cast<nn::fs::GameCardHandle>(handle), static_cast<nn::fs::GameCardPartition>(partition)));

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(fileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::IsArchivedProgram(nn::sf::Out<bool> outValue, nn::Bit64 processId) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfoByProcessId(&pProgramInfo, processId));

        nn::lr::Path programPath;
        NN_RESULT_DO(g_LocationResolverSet.ResolveProgramPath(&programPath, pProgramInfo->GetProgramIdValue(), pProgramInfo->GetStorageId()));

        if (strnlen(programPath.string, sizeof(programPath.string)) > 4)
        {
            if (strncmp(programPath.string + strnlen(programPath.string, sizeof(programPath.string)) - 4, ".nca", 4) == 0)
            {
                outValue.Set(true);
                NN_RESULT_SUCCESS;
            }
        }

        outValue.Set(false);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::QuerySaveDataTotalSize(nn::sf::Out<int64_t> outValue, std::int64_t saveDataSize, std::int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%lld %lld",
            saveDataSize,
            saveDataJournalSize
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(saveDataSize >= 0 && saveDataJournalSize >= 0, nn::fs::ResultInvalidSize());

            int64_t totalSize;
            const int32_t BlockSize = 16 * 1024;
            NN_RESULT_DO(m_Impl->QuerySaveDataTotalSize(&totalSize, BlockSize, saveDataSize, saveDataJournalSize));

            outValue.Set(totalSize);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::SetCurrentPosixTimeWithTimeDifference(std::int64_t posixTime, std::int32_t timeDifference) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::SetCurrentPosixTime);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetCurrentPosixTime", m_ProcessId);
        }

        detail::SetCurrentPosixTime(posixTime, timeDifference);
        NN_RESULT_SUCCESS;
    }

    namespace {

        //! アプリ、パッチ、AOC を lr から探す
        Result ResolveProgramPath(nn::lr::Path* pProgramPath, nn::ncm::ProgramId programId, nn::ncm::StorageId storageId)
        {
            if (g_LocationResolverSet.ResolveProgramPath(pProgramPath, programId.value, storageId).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }

            ncm::DataId dataId = { programId.value };
            if (g_LocationResolverSet.ResolveDataPath(pProgramPath, dataId, storageId).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }

            return ResultTargetNotFound();
        }
    }

    Result FileSystemProxyImpl::GetRightsId(nn::sf::Out<RightsId> outValue, nn::ncm::ProgramId programId, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadOnlyDataInAnyStorage);

        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::GetRightsId);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "GetRightsId", m_ProcessId);
        }

        nn::lr::Path programPath;
        NN_RESULT_DO(ResolveProgramPath(&programPath, programId, storageId));

        PathNormalizer normalizedProgramPath(programPath.string, GetNormalizeOption(programPath.string));
        NN_RESULT_DO(normalizedProgramPath.GetResult());

        RightsId rightsId;
        uint8_t keyGenerationUnused;
        NN_RESULT_DO(m_Impl->GetRightsId(&rightsId, &keyGenerationUnused, normalizedProgramPath.GetPath(), programId));

        outValue.Set(rightsId);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::GetRightsIdByPath(nn::sf::Out<RightsId> outValue, const nn::fssrv::sf::FspPath& path) NN_NOEXCEPT
    {
        uint8_t keyGenerationUnused;
        return GetRightsIdAndKeyGenerationByPath(outValue, nn::sf::Out<uint8_t>(&keyGenerationUnused), path);
    }

    Result FileSystemProxyImpl::GetRightsIdAndKeyGenerationByPath(nn::sf::Out<fs::RightsId> outRightsId, nn::sf::Out<std::uint8_t> outKeyGeneration, const nn::fssrv::sf::FspPath& path) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadOnlyDataInAnyStorage);

        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::GetRightsId);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "GetRightsId", m_ProcessId);
        }

        PathNormalizer normalizedPath(path.str, GetNormalizeOption(path.str));
        NN_RESULT_DO(normalizedPath.GetResult());

        RightsId rightsId;
        uint8_t keyGeneration;
        const nn::ncm::ProgramId CheckThroughProgramId = { CheckThroughId };
        NN_RESULT_DO(m_Impl->GetRightsId(&rightsId, &keyGeneration, normalizedPath.GetPath(), CheckThroughProgramId));

        outRightsId.Set(rightsId);
        outKeyGeneration.Set(keyGeneration);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::RegisterExternalKey(const RightsId& rightsId, const nn::spl::AccessKey& accessKey) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::RegisterExternalKey);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "RegisterExternalKey", m_ProcessId);
        }

        NN_RESULT_DO(m_Impl->RegisterExternalKey(rightsId, accessKey));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::UnregisterAllExternalKey() NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::RegisterExternalKey);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "RegisterExternalKey", m_ProcessId);
        }

        NN_RESULT_DO(m_Impl->UnregisterAllExternalKey());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::SetSdCardEncryptionSeed(const EncryptionSeed& seed) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::SetEncryptionSeed);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetEncryptionSeed", m_ProcessId);
        }

        NN_RESULT_DO(m_Impl->SetSdCardEncryptionSeed(seed));
        InvalidateSdCardSaveDataIndexer();
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::GetAndClearErrorInfo(nn::sf::Out<fs::FileSystemProxyErrorInfo> outValue) NN_NOEXCEPT
    {
        FileSystemProxyErrorInfo info;
        m_Impl->GetAndClearErrorInfo(&info);

        SaveDataIndexerAccessor accessor;
        NN_RESULT_DO(accessor.Initialize(SaveDataSpaceId::User));
        info.saveDataIndexCount = static_cast<uint32_t>(accessor.GetInterface()->GetIndexCount());

        outValue.Set(info);
        NN_RESULT_SUCCESS;
    }

    void FileSystemProxyImpl::IncrementRomFsRemountForDataCorruptionCount() NN_NOEXCEPT
    {
        m_Impl->IncrementRomFsRemountForDataCorruptionCount();
    }

    void FileSystemProxyImpl::IncrementRomFsUnrecoverableDataCorruptionByRemountCount() NN_NOEXCEPT
    {
        m_Impl->IncrementRomFsUnrecoverableDataCorruptionByRemountCount();
    }

    void FileSystemProxyImpl::IncrementRomFsRecoveredByInvalidateCacheCount() NN_NOEXCEPT
    {
        m_Impl->IncrementRomFsRecoveredByInvalidateCacheCount();
    }

    Result FileSystemProxyImpl::VerifySaveDataFileSystemBySaveDataSpaceId(std::uint8_t saveDataSpaceId, std::uint64_t saveDataId, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::VerifySaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "VerifySaveData", m_ProcessId);
            }

            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            NN_UTIL_SCOPE_EXIT{
                memset(workBuffer.GetPointerUnsafe(), 0x00, workBuffer.GetSize());
            };

            SaveDataIndexerValue value;
            nn::fs::SaveDataType saveDataType;
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(static_cast<SaveDataSpaceId>(saveDataSpaceId)));
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));

                SaveDataIndexerKey key;
                NN_RESULT_DO(accessor.GetInterface()->GetKey(&key, saveDataId));
                saveDataType = key.type;
            }

            std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenSaveDataFileSystem(&fileSystem, value.spaceId, saveDataId, m_SaveDataRootPathBuffer, false, saveDataType));

            NN_RESULT_DO(VerifyDirectoryRecursive(fileSystem.get(), workBuffer.GetPointerUnsafe(), workBuffer.GetSize()));

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::VerifySaveDataFileSystem(std::uint64_t saveDataId, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        return VerifySaveDataFileSystemBySaveDataSpaceId(static_cast<std::uint8_t>(nn::fs::SaveDataSpaceId::System), saveDataId, workBuffer);
    }

    Result FileSystemProxyImpl::CorruptSaveDataFileSystemBySaveDataSpaceId(std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%d %llu",
            saveDataSpaceId,
            saveDataId
        );
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            auto pAccessControl = pProgramInfo->GetAccessControl();
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CorruptSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CorruptSaveData|Debug", m_ProcessId);
            }

            SaveDataIndexerValue value;
            {
                SaveDataIndexerAccessor accessor;
                NN_RESULT_DO(accessor.Initialize(static_cast<SaveDataSpaceId>(saveDataSpaceId)));
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, saveDataId));
            }

            if ((value.spaceId != SaveDataSpaceId::User) && (value.spaceId != SaveDataSpaceId::SdUser))
            {
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::CorruptSystemSaveData);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CorruptSaveData(System)", m_ProcessId);
            }

            NN_RESULT_DO(m_Impl->CorruptSaveDataFileSystem(value.spaceId, saveDataId, m_SaveDataRootPathBuffer));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemProxyImpl::CorruptSaveDataFileSystem(std::uint64_t saveDataId) NN_NOEXCEPT
    {
        return CorruptSaveDataFileSystemBySaveDataSpaceId(static_cast<std::uint8_t>(nn::fs::SaveDataSpaceId::System), saveDataId);
    }

    Result FileSystemProxyImpl::CreatePaddingFile(std::int64_t paddingSize) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

        NN_RESULT_THROW_UNLESS(paddingSize >= 0, nn::fs::ResultInvalidSize());
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::FillBis);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "CreatePaddingFile|Debug", m_ProcessId);
        }

        NN_RESULT_DO(m_Impl->CreatePaddingFile(paddingSize));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::DeleteAllPaddingFiles() NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::FillBis);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "DeleteAllPaddingFiles|Debug", m_ProcessId);
        }

        NN_RESULT_DO(m_Impl->DeleteAllPaddingFiles());
        NN_RESULT_SUCCESS;
    }

    // TODO: 削除
    Result FileSystemProxyImpl::DisableAutoSaveDataCreation() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN32)
        m_EnableAutoSaveDataCreation = false;
#endif
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::SetGlobalAccessLogMode(uint32_t accessLogMode) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::SetGlobalAccessLogMode);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetGlobalAccessLogMode", m_ProcessId);
        }

        g_GlobalAccessLogMode = accessLogMode;
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::GetGlobalAccessLogMode(nn::sf::Out<uint32_t> outValue) NN_NOEXCEPT
    {
        outValue.Set(g_GlobalAccessLogMode);
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::OutputAccessLogToSdCard(const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        if( (g_GlobalAccessLogMode & nn::fs::AccessLogMode_SdCard) != 0 )
        {
            detail::AccessLogSdCardWriter::AppendLog(buffer.GetPointerUnsafe(), buffer.GetSize(), m_ProcessId);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::RegisterUpdatePartition() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx", m_ProcessId, pProgramInfo->GetProgramIdValue());

            {
                auto pAccessControl = pProgramInfo->GetAccessControl();
                auto canCall = pAccessControl->CanCall(AccessControl::OperationType::RegisterUpdatePartition);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "RegisterUpdatePartition", m_ProcessId);
            }

            auto targetProgramId = pProgramInfo->GetProgramIdValue();

            NN_FS_SCOPED_TRACE_APPEND_LOG(" on %016llx", targetProgramId);

            nn::lr::Path programPath;
            NN_RESULT_DO(g_LocationResolverSet.ResolveProgramPath(&programPath, targetProgramId, pProgramInfo->GetStorageId()));

            PathNormalizer normalizedProgramPath(programPath.string, GetNormalizeOption(programPath.string));
            NN_RESULT_DO(normalizedProgramPath.GetResult());

            NN_RESULT_DO(m_Impl->RegisterUpdatePartition(targetProgramId, normalizedProgramPath.GetPath()));

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    nn::Result FileSystemProxyImpl::OpenRegisteredUpdatePartition(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>> outValue) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE();
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            const int storageFlag = StorageFlag_ReadOnlyDataInAnyStorage;
            nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(storageFlag);

            std::shared_ptr<ProgramInfo> pProperProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProperProgramInfo));

            NN_FS_SCOPED_TRACE_APPEND_LOG(" by %llu %016llx", m_ProcessId, pProperProgramInfo->GetProgramIdValue());

            {
                auto pAccessControl = pProperProgramInfo->GetAccessControl();
                auto accessibility = pAccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::MountRegisteredUpdatePartition);
                NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanRead(), "MountRegisteredUpdatePartition", m_ProcessId);
            }

            std::shared_ptr<fs::fsa::IFileSystem> fileSystem;
            NN_RESULT_DO(m_Impl->OpenRegisteredUpdatePartition(&fileSystem));

            std::shared_ptr<fs::fsa::IFileSystem> typeSetFileSystem = nn::fssystem::AllocateShared<nn::fssystem::StorageLayoutTypeSetFileSystem>(std::move(fileSystem), storageFlag);
            NN_RESULT_THROW_UNLESS(typeSetFileSystem, nn::fs::ResultAllocationMemoryFailedAllocateShared());

            nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> pFileSystem = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFileSystem, FileSystemInterfaceAdapter>(std::move(typeSetFileSystem), this, false);
            NN_RESULT_THROW_UNLESS(pFileSystem != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());

            outValue.Set(std::move(pFileSystem));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    nn::Result FileSystemProxyImpl::GetAndClearMemoryReportInfo(nn::sf::Out<nn::fs::MemoryReportInfo> outValue) NN_NOEXCEPT
    {
        nn::fs::MemoryReportInfo memoryReportInfo;
        NN_RESULT_DO(nn::fssrv::GetMemoryReportInfo(&memoryReportInfo));

        outValue.Set(std::move(memoryReportInfo));
        NN_RESULT_SUCCESS;
    }

    nn::Result FileSystemProxyImpl::OverrideSaveDataTransferTokenSignVerificationKey(const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OverrideSaveDataTransferTokenSignVerificationKey);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OverrideSaveDataTransferTokenSignVerificationKey", m_ProcessId);
        }

        NN_FSP_REQUIRES(buffer.GetSize() == g_SaveDataTransferCryptoConfiguration->TokenSignKeyModulusSize, nn::fs::ResultInvalidSize());

        memcpy(g_SaveDataTransferCryptoConfiguration->tokenSignKeyModulus, buffer.GetPointerUnsafe(), g_SaveDataTransferCryptoConfiguration->TokenSignKeyModulusSize);

        NN_RESULT_SUCCESS;
    }


    Result FileSystemProxyImpl::CleanUpSaveData() NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

        SaveDataIndexerAccessor accessor;
        NN_RESULT_DO(accessor.Initialize(nn::fs::SaveDataSpaceId::System));

        NN_UTIL_SCOPE_EXIT
        {
            accessor.GetInterface()->Commit();
        };

        std::shared_ptr<SaveDataInfoReaderImpl> pReader;
        NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&pReader));

        int64_t readCount;
        SaveDataInfo info;
        while(NN_STATIC_CONDITION(true))
        {
            NN_RESULT_DO(pReader->Read(nn::sf::Out<int64_t>(&readCount), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(SaveDataInfo))));

            if( readCount == 0 )
            {
                break;
            }

            SaveDataIndexerValue value;
            NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, info.saveDataId));

            if( value.state == SaveDataState::Deleting || value.state == SaveDataState::DeletionRegistered )
            {
                auto result = DeleteSaveDataFileSystemCore(value.spaceId, info.saveDataId);
                if( result.IsSuccess() )
                {
                    NN_RESULT_DO(accessor.GetInterface()->Delete(info.saveDataId));
                }
                else
                {
                    NN_SDK_LOG("[fs] Failed to delete save data %016llx (%x)\n", info.saveDataId, result.GetInnerValueForDebug());
                    // 無視して継続
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CompleteSaveDataExtension() NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

        const SaveDataSpaceId spaces[] = {
            SaveDataSpaceId::System,
            SaveDataSpaceId::User
        };

        for( const auto spaceId : spaces )
        {
            SaveDataIndexerAccessor accessor;
            NN_RESULT_DO(accessor.Initialize(spaceId));

            std::shared_ptr<SaveDataInfoReaderImpl> pReader;
            NN_RESULT_DO(accessor.GetInterface()->OpenSaveDataInfoReader(&pReader));

            int64_t readCount;
            SaveDataInfo info;
            while( NN_STATIC_CONDITION(true) )
            {
                NN_RESULT_DO(pReader->Read(nn::sf::Out<int64_t>(&readCount), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(SaveDataInfo))));

                if( readCount == 0 )
                {
                    break;
                }

                SaveDataIndexerValue value;
                NN_RESULT_DO(accessor.GetInterface()->GetValue(&value, info.saveDataId));

                if (value.state == SaveDataState::Extending && info.saveDataType != SaveDataType::Temporary)
                {
                    // 拡張処理を再開する
                    int64_t extendedTotalSize = 0;
                    auto resultExtend = m_Impl->ResumeExtendSaveDataFileSystem(
                        &extendedTotalSize,
                        info.saveDataId,
                        info.saveDataSpaceId,
                        m_SaveDataRootPathBuffer);

                    if( resultExtend.IsSuccess() )
                    {
                        // Size を更新する
                        NN_RESULT_DO(accessor.GetInterface()->SetSize(
                            info.saveDataId,
                            extendedTotalSize));
                        NN_RESULT_DO(accessor.GetInterface()->Commit());
                     }

                    // 拡張を終了する
                    (void)m_Impl->FinishExtendSaveDataFileSystem(
                        info.saveDataId,
                        info.saveDataSpaceId);

                    // SaveDataState を更新する
                    {
                        NN_RESULT_DO(accessor.GetInterface()->SetState(
                            info.saveDataId,
                            SaveDataState::None));
                        NN_RESULT_DO(accessor.GetInterface()->Commit());
                    }
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::CleanUpTemporaryStorage() NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

        std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;
        NN_RESULT_DO(m_Impl->OpenSaveDataDirectoryFileSystem(
            &pFileSystem, SaveDataSpaceId::Temporary, "", false
        ));
        NN_RESULT_DO(pFileSystem->CleanDirectoryRecursively("/"));

        ResetTemporaryStorageIndexer();

        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::FixSaveData() NN_NOEXCEPT
    {
        Result lastFailedResult = ResultSuccess();

        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

        // ncm の システムセーブデータの ownerId を修正する
        {
            const SystemSaveDataId NcmSystemSaveDataIdArray[] = {
                0x8000000000000120ULL,
                0x8000000000000121ULL,
            };

            auto FixNcmSystemDataOwnerId = [&](SystemSaveDataId id) -> Result {
                const Bit64 CorrectNcmOwnerId = 0;
                SaveDataExtraData extraData;
                NN_RESULT_DO(m_Impl->ReadSaveDataFileSystemExtraData(&extraData, SaveDataSpaceId::System, id, m_SaveDataRootPathBuffer, false));
                if (extraData.ownerId != CorrectNcmOwnerId)
                {
                    NN_SDK_LOG("[fs] Fix incorrect ownerId of ncm system data: %llx -> %llx\n", extraData.ownerId, CorrectNcmOwnerId);
                    extraData.ownerId = CorrectNcmOwnerId;
                    NN_RESULT_DO(m_Impl->WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, id, extraData, m_SaveDataRootPathBuffer, SaveDataType::System, false));
                }
                NN_RESULT_SUCCESS;
            };

            for (auto id : NcmSystemSaveDataIdArray)
            {
                auto result = FixNcmSystemDataOwnerId(id);
                if (result.IsFailure())
                {
                    lastFailedResult = result;
                }
            }
        }

        return lastFailedResult;
    }

    Result FileSystemProxyImpl::SetSdCardAccessibility(bool isAccessible) NN_NOEXCEPT
    {
        std::shared_ptr<ProgramInfo> pProgramInfo;
        NN_RESULT_DO(GetProgramInfo(&pProgramInfo));

        {
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::SetSdCardAccessibility);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetSdCardAccessibility", m_ProcessId);
        }

        m_Impl->SetSdCardAccessibility(isAccessible);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::IsSdCardAccessible(nn::sf::Out<bool> outValue) NN_NOEXCEPT
    {
        outValue.Set(m_Impl->IsSdCardAccessible());
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::IsSignedSystemPartitionOnSdCardValid(nn::sf::Out<bool> outValue) NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);
        outValue.Set(m_Impl->IsSignedSystemPartitionOnSdCardValid());
        NN_RESULT_SUCCESS;
    }

    namespace {
        detail::AccessFailureDetectionEventManager* GetAccessFailureDetectionEventManager() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(detail::AccessFailureDetectionEventManager, g_AccessFailureDetectionEventManager, );
            return &g_AccessFailureDetectionEventManager;
        }

        os::Mutex g_OpenAccessFailureDetectionNotifierLock(false);
    }

    Result FileSystemProxyImpl::OpenAccessFailureDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>> outValue, nn::Bit64 processId) NN_NOEXCEPT
    {
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::OpenAccessFailureDetectionEventNotifier);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "OpenAccessFailureDetectionEventNotifier", m_ProcessId);
        }

        std::lock_guard<os::Mutex> scopedLock(g_OpenAccessFailureDetectionNotifierLock);

        std::unique_ptr<detail::AccessFailureDetectionEventNotifier> pAccessFailureDetectionEventNotifier;
        NN_RESULT_DO(GetAccessFailureDetectionEventManager()->CreateNotifier(&pAccessFailureDetectionEventNotifier, processId));
        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> pEventNotifier = FileSystemFactory::CreateShared<nn::fssrv::sf::IEventNotifier>(std::move(pAccessFailureDetectionEventNotifier));
        NN_RESULT_THROW_UNLESS(pEventNotifier != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pEventNotifier));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::GetAccessFailureDetectionEvent(nn::sf::Out<nn::sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::GetAccessFailureDetectionEvent);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "GetAccessFailureDetectionEvent", m_ProcessId);
        }

        auto handle = GetAccessFailureDetectionEventManager()->GetEvent();
        outValue.Set(nn::sf::NativeHandle(handle, false));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::IsAccessFailureDetected(nn::sf::Out<bool> outValue, nn::Bit64 processId) NN_NOEXCEPT
    {
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::IsAccessFailureDetected);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "IsAccessFailureDetected", m_ProcessId);
        }

        outValue.Set(GetAccessFailureDetectionEventManager()->IsAccessFailureDetectionNotified(processId));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::ResolveAccessFailure(nn::Bit64 processId) NN_NOEXCEPT
    {
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::ResolveAccessFailure);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "ResolveAccessFailure", m_ProcessId);
        }

        std::lock_guard<os::Mutex> scopedLock(g_OpenAccessFailureDetectionNotifierLock);
        GetAccessFailureDetectionEventManager()->ResetAccessFailureDetection(processId);
        fssystem::GetServiceContext()->RequestToInvokeDeferred(processId);
        NN_RESULT_SUCCESS;
    }

    Result FileSystemProxyImpl::AbandonAccessFailure(nn::Bit64 processId) NN_NOEXCEPT
    {
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::AbandonAccessFailure);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "AbandonAccessFailure", m_ProcessId);
        }

        std::lock_guard<os::Mutex> scopedLock(g_OpenAccessFailureDetectionNotifierLock);
        GetAccessFailureDetectionEventManager()->DisableAccessFailureDetection(processId);
        fssystem::GetServiceContext()->RequestToInvokeDeferred(processId);
        NN_RESULT_SUCCESS;
    }

    bool FileSystemProxyImpl::IsAccessFailureDetectionObserved() NN_NOEXCEPT
    {
        auto ret = GetAccessFailureDetectionEventManager()->IsAccessFailureDetectionObserved(m_ProcessId);
        if (ret)
        {
            fssystem::GetServiceContext()->SetDeferredProcessProcessId(m_ProcessId);
        }
        return ret;
    }

    Result FileSystemProxyImpl::SetDataStorageRedirectTarget(nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        {
            std::shared_ptr<ProgramInfo> pProgramInfo;
            NN_RESULT_DO(GetProgramInfo(&pProgramInfo));
            auto pAccessControl = pProgramInfo->GetAccessControl();
            auto canCall = pAccessControl->CanCall(AccessControl::OperationType::SetDataStorageRedirectTarget);
            NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetDataStorageRedirectTarget|Debug", m_ProcessId);
        }

        if (storageId == ncm::StorageId::None || storageId == ncm::StorageId::BuiltInUser)
        {
            detail::SetDataStorageRedirectTarget(storageId);
            NN_RESULT_SUCCESS;
        }
        else
        {
            return nn::fs::ResultInvalidArgument();
        }
    }

    Result FileSystemProxyImpl::OpenMultiCommitManager(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IMultiCommitManager>> outValue) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IMultiCommitManager> pCommitManager = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IMultiCommitManager, MultiCommitManager>();
        NN_RESULT_THROW_UNLESS(pCommitManager != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemProxyImplA());
        outValue.Set(std::move(pCommitManager));
        NN_RESULT_SUCCESS;
    }

namespace {

    Result TryAcquireCountSemaphore(std::unique_lock<fssystem::SemaphoreAdaptor>* pOutValue, fssystem::SemaphoreAdaptor* pCountSemaphore)
    {
        std::unique_lock<SemaphoreAdaptor> uniqueLock(*pCountSemaphore, std::defer_lock);
        NN_RESULT_THROW_UNLESS(uniqueLock.try_lock(), ResultOpenCountLimit());
        *pOutValue = std::move(uniqueLock);
        NN_RESULT_SUCCESS;
    }
}

    Result FileSystemProxyImpl::TryAcquireSaveDataEntryOpenCountSemaphore(std::unique_lock<fssystem::SemaphoreAdaptor>* pOutValue) NN_NOEXCEPT
    {
        return TryAcquireCountSemaphore(pOutValue, &m_SaveDataEntryOpenCountSemaphore);
    }

    Result FileSystemProxyImpl::TryAcquireAddOnContentOpenCountSemaphore(std::unique_lock<fssystem::SemaphoreAdaptor>* pOutValue) NN_NOEXCEPT
    {
        return TryAcquireCountSemaphore(pOutValue, &m_AddOnContentOpenCountSemaphore);
    }

    Result FileSystemProxyImpl::TryAcquireRomMountCountSemaphore(std::unique_lock<fssystem::SemaphoreAdaptor>* pOutValue) NN_NOEXCEPT
    {
        return TryAcquireCountSemaphore(pOutValue, &m_RomMountCountSemaphore);
    }

    Result FileSystemProxyImpl::TryAcquireSaveDataMountCountSemaphore(std::unique_lock<fssystem::SemaphoreAdaptor>* pOutValue) NN_NOEXCEPT
    {
        return TryAcquireCountSemaphore(pOutValue, &m_SaveDataMountCountSemaphore);
    }

}}

namespace nn { namespace fssrv {

    void InitializeFileSystemProxy(
        fscreator::FileSystemCreatorInterfaces* pFileSystemCreatorInterfaces,
        fssystem::IBufferManager* pBufferManager,
        bool isDebugFunctionEnabled,
        SaveDataTransferCryptoConfiguration* pSaveDataTransferCryptoConfiguration,
        InternalProgramIdRangeForSpeedEmulation* pInternalProgramIdRangeForSpeedEmulation,
        GenerateRandomFunction generateRandom
    ) NN_NOEXCEPT
    {
        g_FileSystemProxyCoreImpl.emplace(pFileSystemCreatorInterfaces, pBufferManager, generateRandom);
        SetDebugFlagEnabled(isDebugFunctionEnabled);
        g_SaveDataTransferCryptoConfiguration = pSaveDataTransferCryptoConfiguration;
        g_InternalProgramIdValueMinForSpeedEmulation = pInternalProgramIdRangeForSpeedEmulation->programIdValueMin;
        g_InternalProgramIdValueMaxForSpeedEmulation = pInternalProgramIdRangeForSpeedEmulation->programIdValueMax;

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
        nn::fat::SetCurrentCalendarTimeGetter(GetCurrentCalendarTime);
#endif /* !NN_BUILD_CONFIG_OS_WIN32 */
    }

    nn::fssrv::detail::FileSystemProxyCoreImpl* GetFileSystemProxyCoreImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_FileSystemProxyCoreImpl != util::nullopt);
        return &g_FileSystemProxyCoreImpl.value();
    }

    Result GetCurrentCalendarTime(nn::time::CalendarTime *pOutCalendarTime) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_UNUSED(pOutCalendarTime);
        return ResultNotInitialized();
#else /* !NN_BUILD_CONFIG_OS_WIN32 */
        int64_t posix_time = 0;
        int32_t time_diff = 0;
        NN_RESULT_DO(GetCurrentPosixTime(&posix_time, &time_diff));

        nn::time::PosixTime time = {posix_time + time_diff};
        *pOutCalendarTime = nn::time::ToCalendarTimeInUtc(time);
        NN_RESULT_SUCCESS;
#endif /* !NN_BUILD_CONFIG_OS_WIN32 */
    }

    void NotifyProcessDeferred(nn::Bit64 processId) NN_NOEXCEPT
    {
        GetAccessFailureDetectionEventManager()->NotifyAccessFailureDetection(processId);
    }

    Result FlushFatCache() NN_NOEXCEPT
    {
        nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(StorageFlag_ReadWriteDataInAnyStorage);

        NN_RESULT_DO(FlushFatCacheImpl());
        NN_RESULT_SUCCESS;
    };
}}
