﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/fscreator/fssrv_SdCardProxyFileSystemCreator.h>
#include "../detail/fssrv_FatFileSystemCacheManager.h"
#include "../detail/fssrv_SdmmcStorageService.h"

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/fs_ProxyFileSystem.h>

namespace nn { namespace fssrv { namespace fscreator {


Result SdCardProxyFileSystemCreator::OpenSdCardFileSystem(std::shared_ptr<nn::fs::fsa::IFileSystem>* pOutFileSystem, bool isUncache) NN_NOEXCEPT
{
    std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

    {
        auto scopedLock = m_pFatFileSystemCacheManager->GetScopedLock();

        if (isUncache)
        {
            m_pFatFileSystemCacheManager->UnsetCache(detail::FatDriveId_SdCard);
        }

        pFileSystem = m_pFatFileSystemCacheManager->GetCache(detail::FatDriveId_SdCard);
        if (pFileSystem == nullptr)
        {
            std::shared_ptr<fs::IStorage> sdStorage;
            NN_RESULT_DO(m_pSdStorageCreator->Create(&sdStorage));

            std::shared_ptr<nn::fs::fsa::IFileSystem> fatFs;
            nn::fat::FatAttribute attrs = { false, true, true };
            std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetter(new nn::fat::FatErrorInfoSetter(&m_pErrorInfo->fatFsError, m_pErrorInfoMutex, detail::FatDriveId_SdCard));
            NN_RESULT_THROW_UNLESS(pFatErrorInfoSetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
            NN_RESULT_DO(m_pFatFsCreator->Create(&fatFs, std::move(sdStorage), &attrs, std::move(pFatErrorInfoSetter), fs::ResultInvalidFatFormatSd(), fs::ResultUsableSpaceNotEnoughSdCard()));

            m_pFatFileSystemCacheManager->SetCache(fatFs, detail::FatDriveId_SdCard);

            pFileSystem = std::move(fatFs);
        }
    }
    *pOutFileSystem = std::move(pFileSystem);
    NN_RESULT_SUCCESS;
}

Result SdCardProxyFileSystemCreator::Create(std::shared_ptr<fs::fsa::IFileSystem>* pOutFileSystem) NN_NOEXCEPT
{
    std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem;

    std::lock_guard<os::Mutex> scopedLock(g_ProxySdFatFileSystemCacheMutex);
    bool isUncache = false;

    if (g_ProxySdFatFileSystemCache != nullptr)
    {
        if (nn::fssrv::detail::IsSdCardRemovedAfterOpen())
        {
            // FatFileSystem のキャッシュを破棄するフラグ
            isUncache = true;

            g_ProxySdFatFileSystemCache->Invalidate();
            g_ProxySdFatFileSystemCache.reset();
        }
        else
        {
            *pOutFileSystem = g_ProxySdFatFileSystemCache;
            NN_RESULT_SUCCESS;
        }
    }

    // FatFileSystem のキャッシュを破棄する場合は再生成 (Deactivate/Activate を含む) される
    // 以前に生成した SD カードの ProxyFileSystem のキャッシュからは、無効な FatFileSystem のキャッシュが参照される状態になる
    NN_RESULT_DO(OpenSdCardFileSystem(&pFileSystem, isUncache));

    std::shared_ptr<fssystem::ProxyFileSystem> proxyFs = fssystem::AllocateShared<fssystem::ProxyFileSystem>(pFileSystem.get(), fs::ResultSdCardFileSystemInvalidatedByRemoved());
    NN_RESULT_THROW_UNLESS(proxyFs, fs::ResultAllocationMemoryFailedInFileSystemProxyCoreImplC());

    g_ProxySdFatFileSystemCache = proxyFs;

    *pOutFileSystem = std::move(proxyFs);

    NN_RESULT_SUCCESS;

}

Result SdCardProxyFileSystemCreator::Format(bool isUncache) NN_NOEXCEPT
{
    {
        auto scopedLock = m_pFatFileSystemCacheManager->GetScopedLock();

        if (isUncache)
        {
            m_pFatFileSystemCacheManager->UnsetCache(detail::FatDriveId_SdCard);
        }

        std::shared_ptr<fs::IStorage> sdStorage;
        NN_RESULT_DO(m_pSdStorageCreator->Create(&sdStorage));

        uint32_t numSectors;
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardProtectedAreaNumSectors(&numSectors));

        // Format処理によりFS Infoは規格準拠になるため、重複を防ぐためにisFsInfoNormalizedはfalseを指定
        nn::fat::FatAttribute attrs = { false, false, true };
        nn::fat::FatFormatParam param = { true, numSectors, nn::fs::ResultPortSdCardWriteVerifyError() };
        std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetter(new nn::fat::FatErrorInfoSetter(&m_pErrorInfo->fatFsError, m_pErrorInfoMutex, detail::FatDriveId_SdCard));
        NN_RESULT_THROW_UNLESS(pFatErrorInfoSetter != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        NN_RESULT_DO(m_pFatFsCreator->Format(std::move(sdStorage), &attrs, &param, std::move(pFatErrorInfoSetter), fs::ResultInvalidFatFormatSd(), fs::ResultUsableSpaceNotEnoughSdCard()));
    }
    NN_RESULT_SUCCESS;
}


Result SdCardProxyFileSystemCreator::Format() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(g_ProxySdFatFileSystemCacheMutex);
    bool isUncache = false;

    if (g_ProxySdFatFileSystemCache != nullptr)
    {
        // FatFileSystem のキャッシュを破棄するフラグ
        isUncache = true;

        g_ProxySdFatFileSystemCache->Invalidate();
        g_ProxySdFatFileSystemCache.reset();
    }

    // FatFileSystem のキャッシュを破棄する場合は再生成 (Deactivate/Activate を含む) される
    // 以前に生成した SD カードの ProxyFileSystem のキャッシュからは、無効な FatFileSystem のキャッシュが参照される状態になる
    NN_RESULT_DO(Format(isUncache));
    NN_RESULT_SUCCESS;
}

void SdCardProxyFileSystemCreator::Initialize(fs::FileSystemProxyErrorInfo* pErrorInfo, os::Mutex* pErrorInfoMutex) NN_NOEXCEPT
{
    m_pErrorInfo = pErrorInfo;
    m_pErrorInfoMutex = pErrorInfoMutex;
}



}}}
