﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <memory>
#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Allocator.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fssrv/fscreator/fssrv_GameCardStorageCreator.h>
#include <nn/gc/gc_Result.h>
#include "../detail/fssrv_GameCardManager.h"

namespace nn { namespace fssrv { namespace fscreator {


    using namespace nn::fs;
    using namespace nn::fssystem;
    using namespace nn::fssrv::detail;

    namespace {

        //! @brief  下層 IStorage の所有権を持つ SubStorage
        // TODO: shared_ptr で置き換え
        class DelegatedSubStorage : public SubStorage
        {
            NN_DISALLOW_COPY(DelegatedSubStorage);
            NN_DISALLOW_MOVE(DelegatedSubStorage);

        public:
            DelegatedSubStorage(std::unique_ptr<fs::IStorage>&& baseStorage, int64_t offset, int64_t size)
                : SubStorage(baseStorage.get(), offset, size),
                m_BaseStorage(std::move(baseStorage))
            {
            }
        private:
            std::unique_ptr<fs::IStorage> m_BaseStorage;
        };


        class ReadOnlyGameCardStorage : public fs::IStorage, public fs::detail::Newable
        {
            friend ReadOnlyGameCardStorage* MakeNormalModeReadOnlyGameCardStorage(GameCardHandle) NN_NOEXCEPT;
            friend ReadOnlyGameCardStorage* MakeSecureModeReadOnlyGameCardStorage(GameCardHandle, char (&)[gc::GcCardDeviceIdSize], char (&)[gc::GcCardImageHashSize]) NN_NOEXCEPT;

        public:
            virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
            {
                TryUpdateHandle();
                return ReadGameCard(m_Handle, offset, buffer, size);
            }

            virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_UNUSED(offset);
                NN_UNUSED(buffer);
                NN_UNUSED(size);
                return ResultUnsupportedOperation();
            }

            virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
            {
                TryUpdateHandle();
                gc::GameCardStatus status;
                NN_RESULT_DO(GetGameCardStatus(&status, m_Handle));
                *outValue = status.cardSize;
                NN_RESULT_SUCCESS;
            }

            virtual Result OperateRange(
                void* outBuffer,
                size_t outBufferSize,
                OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_UNUSED(offset);
                NN_UNUSED(size);
                NN_UNUSED(inBuffer);
                NN_UNUSED(inBufferSize);

                switch( operationId )
                {
                case OperationId::Invalidate:
                    NN_RESULT_SUCCESS;
                case OperationId::QueryRange:
                    NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
                    NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

                    reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
                    NN_RESULT_SUCCESS;
                default:
                    return nn::fs::ResultUnsupportedOperation();
                }
            }

            void TryUpdateHandle() NN_NOEXCEPT
            {
                // ハンドルが無効になっていた場合でも、セキュアモードかつ同一カードと判定できたらハンドルを自動更新する

                if (!m_IsSecureMode)
                {
                    return;
                }

                if (!IsGameCardSecureMode())
                {
                    // ゲームカードがしかるべき確認を経てセキュアモードに入れられるまではハンドルは更新しない
                    return;
                }

                // 毎回呼び出される際の処理負荷軽減のため、ここではハンドルの再取得はしない
                // IsInvalidHandle はカード再挿入直後は GameCardManager 側のハンドルも最新になっていないため true が返る
                // ただしセキュアモードに再度入る過程で GameCardManager 側のハンドルが更新されているはず
                if (!IsInvalidHandle(m_Handle))
                {
                    // 既にハンドルが最新のものだった場合
                    return;
                }

                char currentCardDeviceId[gc::GcCardDeviceIdSize];
                if (GetGameCardDeviceId(currentCardDeviceId, sizeof(currentCardDeviceId)).IsFailure())
                {
                    return;
                }

                char currentCardImageHash[gc::GcCardImageHashSize];
                if (GetGameCardImageHash(currentCardImageHash, sizeof(currentCardImageHash)).IsFailure())
                {
                    return;
                }

                if (!crypto::IsSameBytes(currentCardDeviceId, m_CardDeviceId, gc::GcCardDeviceIdSize) ||
                    !crypto::IsSameBytes(currentCardImageHash, m_CardImageHash, gc::GcCardImageHashSize))
                {
                    return;
                }

                GameCardHandle newHandle = 0;
                if (GetGameCardHandle(&newHandle).IsFailure())
                {
                    return;
                }

                m_Handle = newHandle;
            }

        private:
            NN_IMPLICIT ReadOnlyGameCardStorage(GameCardHandle handle) NN_NOEXCEPT
                : m_Handle(handle),
                  m_IsSecureMode(false),
                  m_CardDeviceId(),
                  m_CardImageHash()
            {
            }

            ReadOnlyGameCardStorage(GameCardHandle handle, bool isSecureMode, char (&cardDeviceId)[gc::GcCardDeviceIdSize], char (&cardImageHash)[gc::GcCardImageHashSize]) NN_NOEXCEPT
                : m_Handle(handle),
                  m_IsSecureMode(isSecureMode)
            {
                memcpy(m_CardDeviceId, cardDeviceId, gc::GcCardDeviceIdSize);
                memcpy(m_CardImageHash, cardImageHash, gc::GcCardImageHashSize);
            }

            GameCardHandle m_Handle;
            const bool m_IsSecureMode;
            char m_CardDeviceId[gc::GcCardDeviceIdSize];
            char m_CardImageHash[gc::GcCardImageHashSize];
        };

        ReadOnlyGameCardStorage* MakeNormalModeReadOnlyGameCardStorage(GameCardHandle handle) NN_NOEXCEPT
        {
            return new ReadOnlyGameCardStorage(handle);
        }

        ReadOnlyGameCardStorage* MakeSecureModeReadOnlyGameCardStorage(GameCardHandle handle, char (&cardDeviceId)[gc::GcCardDeviceIdSize], char (&cardImageHash)[gc::GcCardImageHashSize]) NN_NOEXCEPT
        {
            return new ReadOnlyGameCardStorage(handle, true, cardDeviceId, cardImageHash);
        }

        class WriteOnlyGameCardStorage : public fs::IStorage, public fs::detail::Newable
        {
        public:
            explicit WriteOnlyGameCardStorage(GameCardHandle handle) NN_NOEXCEPT
                : m_Handle(handle)
            {
            }

            virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_UNUSED(offset);
                NN_UNUSED(buffer);
                NN_UNUSED(size);
                return ResultUnsupportedOperation();
            }

            virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
            {
                return WriteGameCard(m_Handle, offset, buffer, size);
            }

            virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
            {
                const int64_t KeyAreaSize = 0x8 * 512;
                int64_t gameCardSize = 0;
                NN_RESULT_DO(GetGameCardRawSize(&gameCardSize));
                *outValue = gameCardSize + KeyAreaSize;

                NN_RESULT_SUCCESS;
            }

        private:
            GameCardHandle m_Handle;
        };
    }


    GameCardStorageCreator::GameCardStorageCreator(nn::MemoryResource* pAllocator) NN_NOEXCEPT
        : m_Allocator(pAllocator)
    {
    }

    GameCardStorageCreator::~GameCardStorageCreator() NN_NOEXCEPT
    {
    }

    Result GameCardStorageCreator::CreateReadOnly(GameCardHandle handle, std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsInvalidHandle(handle), fs::ResultGameCardFsCheckHandleInCreateReadOnlyFailure());

        GameCardHandle newHandle = 0;
        NN_RESULT_DO(EnsureGameCardNormalMode(&newHandle));

        NN_RESULT_THROW_UNLESS(handle == newHandle, fs::ResultGameCardFsCheckHandleInCreateReadOnlyFailure());

        std::unique_ptr<fs::IStorage> pRoStorage(MakeNormalModeReadOnlyGameCardStorage(newHandle));
        NN_RESULT_THROW_UNLESS(pRoStorage != nullptr, ResultAllocationMemoryFailedInGameCardStorageCreatorA());

        gc::GameCardStatus status;
        NN_RESULT_DO(GetGameCardStatus(&status, handle));
        int64_t secureAreaOffset = status.normalAreaSize;

        std::shared_ptr<fs::IStorage> pNewStorage = fssystem::AllocateShared<DelegatedSubStorage>(std::move(pRoStorage), 0, secureAreaOffset);
        NN_RESULT_THROW_UNLESS(pNewStorage != nullptr, ResultAllocationMemoryFailedInGameCardStorageCreatorB());

        *outValue = std::move(pNewStorage);
        NN_RESULT_SUCCESS;
    }

    Result GameCardStorageCreator::CreateSecureReadOnly(GameCardHandle handle, std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsInvalidHandle(handle), fs::ResultGameCardFsCheckHandleInCreateSecureReadOnlyFailure());

        GameCardHandle newHandle = 0;
        NN_RESULT_DO(EnsureGameCardSecureMode(&newHandle));

        NN_RESULT_THROW_UNLESS(handle == newHandle, fs::ResultGameCardFsCheckHandleInCreateSecureReadOnlyFailure());

        char currentCardDeviceId[gc::GcCardDeviceIdSize];
        NN_RESULT_DO(GetGameCardDeviceId(currentCardDeviceId, sizeof(currentCardDeviceId)));

        char currentCardImageHash[gc::GcCardImageHashSize];
        NN_RESULT_DO(GetGameCardImageHash(currentCardImageHash, sizeof(currentCardImageHash)));

        std::unique_ptr<fs::IStorage> pRoStorage(MakeSecureModeReadOnlyGameCardStorage(newHandle, currentCardDeviceId, currentCardImageHash));
        NN_RESULT_THROW_UNLESS(pRoStorage != nullptr, ResultAllocationMemoryFailedInGameCardStorageCreatorC());

        gc::GameCardStatus status;
        NN_RESULT_DO(GetGameCardStatus(&status, handle));
        int64_t secureAreaOffset = status.normalAreaSize;
        int64_t secureAreaSize   = status.secureAreaSize;

        int64_t size;
        NN_RESULT_DO(pRoStorage->GetSize(&size));

        std::shared_ptr<fs::IStorage> pNewStorage = fssystem::AllocateShared<DelegatedSubStorage>(std::move(pRoStorage), secureAreaOffset, secureAreaSize);
        NN_RESULT_THROW_UNLESS(pNewStorage != nullptr, ResultAllocationMemoryFailedInGameCardStorageCreatorD());

        *outValue = std::move(pNewStorage);
        NN_RESULT_SUCCESS;
    }

    Result GameCardStorageCreator::CreateWriteOnly(GameCardHandle handle, std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT
    {
        GameCardHandle newHandle = 0;
        NN_RESULT_DO(EnsureGameCardWriteMode(&newHandle));

        // TODO: handle チェック
        NN_UNUSED(handle);

        std::unique_ptr<fs::IStorage> pWoStorage(new WriteOnlyGameCardStorage(newHandle));
        NN_RESULT_THROW_UNLESS(pWoStorage != nullptr, ResultAllocationMemoryFailedInGameCardStorageCreatorE());

        int64_t size;
        NN_RESULT_DO(pWoStorage->GetSize(&size));

        std::shared_ptr<fs::IStorage> pNewStorage = fssystem::AllocateShared<DelegatedSubStorage>(std::move(pWoStorage), 0, size);
        NN_RESULT_THROW_UNLESS(pNewStorage != nullptr, ResultAllocationMemoryFailedInGameCardStorageCreatorF());

        *outValue = std::move(pNewStorage);
        NN_RESULT_SUCCESS;
    }


}}}
