﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fssrv/fssrv_FileSystemProxyImpl.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ContextControl.h>
#include <nn/util/util_Optional.h>

#include "fssrv_StorageInterfaceAdapter.h"
#include "fssrv_Trace.h"

namespace nn { namespace fssrv { namespace detail {
    StorageInterfaceAdapter::StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface) NN_NOEXCEPT
        : m_Storage(std::move(storageInterface), fssystem::GetRegisteredThreadPool())
        , m_DataStorageContext()
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
    }
    StorageInterfaceAdapter::~StorageInterfaceAdapter() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
    }

    StorageInterfaceAdapter::StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxy, true)
        , m_Storage(std::move(storageInterface), fssystem::GetRegisteredThreadPool())
        , m_OpenCountSemaphore(std::move(openCountSemaphore))
        , m_DataStorageContext()
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
    }

    StorageInterfaceAdapter::StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore, std::unique_ptr<fssystem::NcaDigest>&& digest, nn::Bit64 programIdValue, nn::ncm::StorageId storageId) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxy, true)
        , m_Storage(std::move(storageInterface), fssystem::GetRegisteredThreadPool())
        , m_OpenCountSemaphore(std::move(openCountSemaphore))
        , m_DataStorageContext(std::move(digest), programIdValue, storageId)
        , m_DeepRetryEnabled(true)
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
    }

    StorageInterfaceAdapter::StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore, bool enableDeepRetry) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxy, true)
        , m_Storage(std::move(storageInterface), fssystem::GetRegisteredThreadPool())
        , m_OpenCountSemaphore(std::move(openCountSemaphore))
        , m_DataStorageContext()
        , m_DeepRetryEnabled(enableDeepRetry)
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
    }

    StorageInterfaceAdapter::StorageInterfaceAdapter(std::shared_ptr<fs::IStorage>&& storageInterface, FileSystemProxyImpl* pFileSystemProxy) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxy, true)
        , m_Storage(std::move(storageInterface), fssystem::GetRegisteredThreadPool())
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
    }

    Result StorageInterfaceAdapter::InvalidateCacheOnStorage(bool doRemountStorage) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_DeepRetryEnabled);
        std::lock_guard<os::ReaderWriterLock> scopedWriterLock(m_InvalidateCacheLock);

        if (!doRemountStorage || !m_DataStorageContext.IsValid())
        {
            // エラーが起きていてもリトライに失敗するだけなので無視する
            m_Storage.OperateRange(nullptr, 0, fs::OperationId::Invalidate, 0, std::numeric_limits<int64_t>::max(), nullptr, 0);
            NN_RESULT_SUCCESS;
        }
        else
        {
            // storage をリマウント
            std::shared_ptr<fs::IStorage> remountStorage;
            fssystem::NcaDigest digest;

            NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_pFileSystemProxyImpl->OpenDataStorageCore(&remountStorage, &digest, m_DataStorageContext.GetProgramIdValue(), m_DataStorageContext.GetStorageId()));
            // nca の一致確認
            NN_RESULT_THROW_UNLESS(nn::crypto::IsSameBytes(m_DataStorageContext.GetDigest(), &digest, sizeof(fssystem::NcaDigest)), fs::ResultNcaDigestInconsistent());

            m_Storage.SetBaseStorage(std::move(remountStorage));
            NN_RESULT_SUCCESS;
        }
    }

    Result StorageInterfaceAdapter::Read(std::int64_t offset, const nn::sf::OutBuffer& buffer, std::int64_t size) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %lld %lld", m_Storage.GetBaseStorage(), offset, size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(offset >= 0, nn::fs::ResultInvalidOffset());
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());

            auto pBuffer = buffer.GetPointerUnsafe();

            // TODO: remount 削除時、FileSystemInterfaceAdapter のロジックと共通化
            if (m_DeepRetryEnabled)
            {
                const int RetryCountMax = 2;
                Result result;

                for (int retryCount = 0; retryCount < RetryCountMax + 1; retryCount++)
                {
                    if (retryCount != 0)
                    {
                        NN_DETAIL_FS_WARN_ON_DATA_CORRUPTED("DataCorrupted (retry count: %d), 0x%08x at %s() - %d %lld\n", retryCount - 1, result.GetInnerValueForDebug(), __FUNCTION__, __LINE__);
                    }

                    auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();

                    result = m_Storage.Read(offset, pBuffer, static_cast<size_t>(size));
                    if (result.IsSuccess())
                    {
                        if (retryCount == 1)
                        {
                            m_pFileSystemProxyImpl->IncrementRomFsRecoveredByInvalidateCacheCount();
                        }
                        else if (retryCount == 2)
                        {
                            m_pFileSystemProxyImpl->IncrementRomFsRemountForDataCorruptionCount();
                        }
                        NN_RESULT_SUCCESS;
                    }

                    auto isDataCorrupted = fs::ResultDataCorrupted::Includes(result);
                    auto isAccessFailed = fs::ResultGameCardAccessFailed::Includes(result);

                    if (!isDataCorrupted && !isAccessFailed)
                    {
                        // retry 不能なエラー扱いなので即時返す
                        return result;
                    }
                    // access failed 検知時にはアクセスエラー検知の対象であればキャッシュを無効化した上で処理を後回しにする
                    // アクセスエラー検知の対象でなければエラーを即時返す
                    else if (isAccessFailed)
                    {
                        if (m_pFileSystemProxyImpl->IsAccessFailureDetectionObserved())
                        {
                            scopedReaderLock = util::nullopt;
                            InvalidateCacheOnStorage(false);
                            return nn::sf::DeferProcess();
                        }
                        return result;
                    }
                    // corruption 検知時には retry する
                    else
                    {
                        scopedReaderLock = util::nullopt;

                        // ラスト 1 回は storage をリマウントできる場合はする TODO: 不要とわかり次第削除
                        // remount に失敗した時はアクセスエラー検知の対象であれば処理を後回しにする。
                        // アクセスエラー検知の対象でなければエラーを即時返す
                        NN_RESULT_TRY(InvalidateCacheOnStorage(retryCount == RetryCountMax - 1))
                            NN_RESULT_CATCH_ALL
                            {
                                if (m_pFileSystemProxyImpl->IsAccessFailureDetectionObserved())
                                {
                                    return nn::sf::DeferProcess();
                                }
                                NN_RESULT_THROW(result);
                            }
                        NN_RESULT_END_TRY;

                        if (retryCount >= RetryCountMax)
                        {
                            // remount できたがリトライに失敗したカウント
                            m_pFileSystemProxyImpl->IncrementRomFsUnrecoverableDataCorruptionByRemountCount();

                            // retry してもストレージアクセスが一向にできないときアクセスエラー検知の対象であれば処理を後回しにする。
                            // アクセスエラー検知の対象でなければエラーを即時返す。
                            if (m_pFileSystemProxyImpl->IsAccessFailureDetectionObserved())
                            {
                                return nn::sf::DeferProcess();
                            }
                            return result;
                        }

                        continue;
                    }
                }

                // ここには到達しない
                NN_ABORT("Must be unreachable.");
            }
            else
            {
                NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_Storage.Read(offset, pBuffer, static_cast<size_t>(size)));
            }
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result StorageInterfaceAdapter::Write(std::int64_t offset, const nn::sf::InBuffer& buffer, std::int64_t size) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %lld %lld", m_Storage.GetBaseStorage(), offset, size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(offset >= 0, nn::fs::ResultInvalidOffset());
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());

            fssystem::ScopedThreadPriorityChangerByAccessPriority scopedPriorityChanger(fssystem::ScopedThreadPriorityChangerByAccessPriority::AccessMode::Write);

            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            auto pBuffer = buffer.GetPointerUnsafe();
            return m_Storage.Write(offset, pBuffer, static_cast<size_t>(size));
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result StorageInterfaceAdapter::Flush() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            return m_Storage.Flush();
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result StorageInterfaceAdapter::SetSize(std::int64_t size) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %lld", m_Storage.GetBaseStorage(), size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());
            return m_Storage.SetSize(size);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result StorageInterfaceAdapter::GetSize(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_Storage.GetBaseStorage());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            std::int64_t size;
            NN_RESULT_DO(m_Storage.GetSize(&size));
            outValue.Set(size);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result StorageInterfaceAdapter::OperateRange(
        nn::sf::Out<nn::fs::StorageQueryRangeInfo> outValue,
        std::int32_t operationId,
        int64_t offset,
        int64_t size) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %d %lld %lld", m_Storage.GetBaseStorage(), operationId, offset, size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(outValue.GetPointer() != nullptr, nn::fs::ResultNullptrArgument());

            // プロセスの情報が混入しないように、返却する値は必ず Clear してから設定します。
            outValue->Clear();

            // デバッグ用の operationId のみで動作するようにします。
            if( operationId == static_cast<int32_t>(nn::fs::OperationId::QueryRange) )
            {
                auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();

                nn::fs::StorageQueryRangeInfo info;
                NN_RESULT_DO(m_Storage.OperateRange(
                    &info,
                    sizeof(info),
                    nn::fs::OperationId::QueryRange,
                    offset,
                    size,
                    nullptr,
                    0));

                outValue->Merge(info);
            }

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    util::optional<SharedLock<os::ReaderWriterLock>> StorageInterfaceAdapter::AcquireReaderLockForCacheInvalidation() NN_NOEXCEPT
    {
        util::optional<SharedLock<os::ReaderWriterLock>> lock;
        if (m_DeepRetryEnabled)
        {
            lock.emplace(m_InvalidateCacheLock);
        }

        return lock;
    }

}}}
