﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/fssrv/detail/fssrv_ISaveDataTransferCoreInterface.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/fs_ConcatenationFile.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>

#include "fssrv_AllocatorForServiceFrameWork.h"
#include "fssrv_SaveDataTransferManager.h"
#include "fssrv_SaveDataExporter.h"
#include "fssrv_SaveDataImporter.h"

namespace nn { namespace fssrv { namespace detail {


namespace {
    typedef nn::sf::ObjectFactory<AllocatorForServiceFrameWork::Policy> FileSystemFactory;

    const uint32_t InitialDataVersion = 1;

}

    SaveDataTransferManager::SaveDataTransferManager(const SaveDataTransferCryptoConfiguration& configuration, ISaveDataTransferCoreInterface* pImpl) NN_NOEXCEPT
        : m_pImpl(pImpl, true)
        , m_Configuration(configuration)
        , m_IsTokenSet(false)
        , m_ExporterCount(0)
        , m_ImporterCount(0)
    {
        m_Configuration.pGenerateRandom(m_Challenge.data, Challenge::Size);
    }

    Result SaveDataTransferManager::GetChallenge(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(buffer.GetSize() == Challenge::Size, fs::ResultInvalidSize());
        auto pDst = buffer.GetPointerUnsafe();
        NN_FSP_REQUIRES(pDst != nullptr, fs::ResultNullptrArgument());
        std::memcpy(pDst, m_Challenge.data, Challenge::Size);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManager::SetToken(const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(!m_IsTokenSet, fs::ResultPreconditionViolation());
        NN_FSP_REQUIRES(buffer.GetSize() == sizeof(SaveDataTransferToken), fs::ResultInvalidSize());

        auto pSrc = buffer.GetPointerUnsafe();
        NN_FSP_REQUIRES(pSrc != nullptr, fs::ResultNullptrArgument());

        auto pSrcToken = reinterpret_cast<const SaveDataTransferToken*>(pSrc);

        SaveDataTransferToken token;
        memcpy(&token, pSrcToken, sizeof(SaveDataTransferToken));

        char mac[crypto::Aes128GcmDecryptor::MacSize];
        char key[crypto::Aes128GcmDecryptor::BlockSize];

        m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransferToken, nullptr, 0);

        // AES-GCM 検証
        auto decryptedSize = crypto::DecryptAes128Gcm(
            &token.encryptedArea,        sizeof(token.encryptedArea),
            &mac,                        sizeof(mac),
            key,                         sizeof(key),
            token.tokenIv,               sizeof(token.tokenIv),
            &token.encryptedArea,        sizeof(token.encryptedArea),
            nullptr,                     0
        );

        NN_RESULT_THROW_UNLESS(decryptedSize == sizeof(token.encryptedArea), fs::ResultUnknown());

        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(mac, token.tokenMac, crypto::Aes128GcmDecryptor::MacSize), fs::ResultSaveDataTransferTokenMacVerificationFailed());

        // サーバ署名検証
        const char TokenSignKeyPublicExponent[] = { 0x01, 0x00, 0x01 };
        NN_RESULT_THROW_UNLESS(
            crypto::VerifyRsa2048PssSha256(
                token.encryptedArea.sign, sizeof(token.encryptedArea.sign),
                m_Configuration.tokenSignKeyModulus, sizeof(m_Configuration.tokenSignKeyModulus),
                TokenSignKeyPublicExponent, sizeof(TokenSignKeyPublicExponent),
                &token.encryptedArea.signedArea, sizeof(token.encryptedArea.signedArea)
            ),
            fs::ResultSaveDataTransferTokenSignatureVerificationFailed()
        );

        // challenge 検証
        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(m_Challenge.data, token.encryptedArea.signedArea.challenge, sizeof(m_Challenge.data)), fs::ResultSaveDataTransferTokenChallengeVerificationFailed());

        memcpy(m_TransferAesKeySeed, token.encryptedArea.signedArea.keySeed, sizeof(m_TransferAesKeySeed));
        memcpy(m_TransferAesIv,      token.encryptedArea.signedArea.iv,      sizeof(m_TransferAesIv));

        m_IsTokenSet = true;
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManager::OpenSaveDataExporter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataExporter>> outValue, std::uint8_t saveDataSpaceId, std::uint64_t saveDataId) NN_NOEXCEPT
    {
        int index = m_ExporterCount;
        m_ExporterCount++;

        NN_FSP_REQUIRES(m_IsTokenSet, fs::ResultPreconditionViolation());

        NN_RESULT_DO(m_pImpl->CheckSaveDataFile(static_cast<nn::fs::SaveDataId>(saveDataId), static_cast<nn::fs::SaveDataSpaceId>(saveDataSpaceId)));

        auto pSaveDataExtraData = std::unique_ptr<fs::SaveDataExtraData>(new fs::SaveDataExtraData());
        NN_RESULT_THROW_UNLESS(pSaveDataExtraData != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        NN_RESULT_DO(m_pImpl->ReadSaveDataFileSystemExtraDataCore(pSaveDataExtraData.get(), static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), saveDataId, false));

        fs::SaveDataInfo info;
        NN_RESULT_DO(m_pImpl->GetSaveDataInfo(&info, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), pSaveDataExtraData->attribute));

        std::shared_ptr<fs::fsa::IFile> file;
        std::unique_ptr<SaveDataMacUpdater> unusedMacUpdater;
        NN_RESULT_DO(OpenSaveDataConcatenationFile(&file, &unusedMacUpdater, info, fs::OpenMode_Read));

        char key[crypto::Aes128GcmEncryptor::BlockSize];
        m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransfer, m_TransferAesKeySeed, sizeof(m_TransferAesKeySeed));

        char iv[crypto::AesEncryptor128::BlockSize];
        memcpy(iv, m_TransferAesIv, sizeof(m_TransferAesIv));
        fssystem::AddCounter(iv, sizeof(iv), index * 2);

        std::unique_ptr<crypto::Aes128GcmEncryptor> initialDataEncryptor(new crypto::Aes128GcmEncryptor());
        NN_RESULT_THROW_UNLESS(initialDataEncryptor != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        initialDataEncryptor->Initialize(key, sizeof(key), iv, sizeof(iv));

        fssystem::AddCounter(iv, sizeof(iv), 1);
        std::unique_ptr<crypto::Aes128GcmEncryptor> encryptor(new crypto::Aes128GcmEncryptor());
        NN_RESULT_THROW_UNLESS(encryptor != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        encryptor->Initialize(key, sizeof(key), iv, sizeof(iv));

        int64_t fileSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(file->GetSize(&fileSize));
        auto pExporter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataExporter, SaveDataExporter>(std::move(pSaveDataExtraData), info, std::move(file), fileSize, std::move(initialDataEncryptor), std::move(encryptor), InitialDataVersion);
        NN_RESULT_THROW_UNLESS(pExporter != nullptr, nn::fs::ResultAllocationMemoryFailedInSaveDataTransferManagerA());
        outValue.Set(pExporter);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManager::OpenSaveDataImporter(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataImporter>> outValue, nn::sf::Out<std::int64_t> outRequiredSize, const nn::sf::InBuffer& buffer, const nn::fs::UserId& userId, std::uint8_t saveDataSpaceId) NN_NOEXCEPT
    {
        int index = m_ImporterCount;
        m_ImporterCount++;

        NN_FSP_REQUIRES(m_IsTokenSet, fs::ResultPreconditionViolation());

        NN_FSP_REQUIRES(buffer.GetSize() >= sizeof(fs::InitialData), fs::ResultInvalidSize());
        auto pEncryptedInitialData = reinterpret_cast<const fs::InitialData*>(buffer.GetPointerUnsafe());
        NN_FSP_REQUIRES(pEncryptedInitialData != nullptr, fs::ResultNullptrArgument());

        char key[crypto::Aes128GcmEncryptor::BlockSize];
        m_Configuration.pGenerateAesKey(key, sizeof(key), SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransfer, m_TransferAesKeySeed, sizeof(m_TransferAesKeySeed));

        char iv[crypto::AesEncryptor128::BlockSize];
        memcpy(iv, m_TransferAesIv, sizeof(m_TransferAesIv));
        fssystem::AddCounter(iv, sizeof(iv), index * 2);

        char mac[crypto::Aes128GcmDecryptor::MacSize];
        fs::InitialData initialData;

        auto decryptedSize = crypto::DecryptAes128Gcm(
            &initialData.encryptedArea, sizeof(initialData.encryptedArea),
            mac, sizeof(mac),
            key, sizeof(key),
            iv,  sizeof(iv),
            &pEncryptedInitialData->encryptedArea, sizeof(pEncryptedInitialData->encryptedArea),
            nullptr, 0
        );
        NN_RESULT_THROW_UNLESS(decryptedSize == sizeof(initialData.encryptedArea), fs::ResultUnknown());
        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(mac, pEncryptedInitialData->mac, sizeof(mac)), fs::ResultSaveDataTransferInitialDataMacVerificationFailed());

        NN_RESULT_THROW_UNLESS(initialData.encryptedArea.version == InitialDataVersion, fs::ResultSaveDataTransferInitialDataVersionVerificationFailed());

        // セーブデータの実体作成
        fs::SaveDataInfo info;
        {
            const auto& extraData = initialData.encryptedArea.extraData;

            fs::SaveDataAttribute attribute = initialData.encryptedArea.extraData.attribute;
            {
                attribute.userId = userId; // userId だけ付け替える
            }

            NN_FSP_REQUIRES(attribute.staticSaveDataId != IndexerSaveDataId, fs::ResultInvalidArgument());

            fs::SaveDataCreationInfo creationInfo;
            {
                creationInfo.size           = extraData.availableSize;
                creationInfo.journalSize    = extraData.journalSize;
                creationInfo.blockSize      = fs::detail::DefaultSaveDataBlockSize;
                creationInfo.ownerId        = extraData.ownerId;
                creationInfo.flags          = extraData.flags;
                creationInfo.spaceId        = static_cast<fs::SaveDataSpaceId>(saveDataSpaceId);
                creationInfo.isPseudoSaveFs = false;
            }

            fs::SaveDataMetaInfo metaInfo;
            if (attribute.type == fs::SaveDataType::Account)
            {
                metaInfo.type = fs::SaveDataMetaType::Thumbnail;
                metaInfo.size = fs::detail::ThumbnailFileSize;
            }
            else
            {
                metaInfo.type = fs::SaveDataMetaType::None;
                metaInfo.size = 0;
            }

            nn::fs::SaveDataHashSalt hashSalt;
            NN_RESULT_TRY(m_pImpl->CreateSaveDataFileSystemCore(attribute, creationInfo, metaInfo, hashSalt, true))
                NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
                {
                    const int64_t MarginSizePerFile = 16 * 1024;

                    int64_t totalSize = 0;
                    m_pImpl->QuerySaveDataTotalSize(&totalSize, creationInfo.size, creationInfo.journalSize);
                    int64_t freeSpaceSize = 0;
                    m_pImpl->GetFreeSpaceSizeForSaveData(&freeSpaceSize, static_cast<uint8_t>(creationInfo.spaceId));

                    int64_t requiredSize = totalSize + MarginSizePerFile;
                    requiredSize += metaInfo.size != 0 ? metaInfo.size + MarginSizePerFile : 0;

                    outRequiredSize.Set(requiredSize - freeSpaceSize);

                    // W/A 失敗時も成功を返す
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY;

            NN_RESULT_DO(m_pImpl->GetSaveDataInfo(&info, static_cast<fs::SaveDataSpaceId>(saveDataSpaceId), attribute));

            NN_RESULT_DO(m_pImpl->WriteSaveDataFileSystemExtraDataCore(creationInfo.spaceId, info.saveDataId, extraData, attribute.type, false));
        }

        std::shared_ptr<fs::fsa::IFile> file;
        std::unique_ptr<SaveDataMacUpdater> macUpdater;
        NN_RESULT_DO(OpenSaveDataConcatenationFile(&file, &macUpdater, info, fs::OpenMode_Write));

        fssystem::AddCounter(iv, sizeof(iv), 1);
        std::unique_ptr<crypto::Aes128GcmDecryptor> decryptor(new crypto::Aes128GcmDecryptor());
        NN_RESULT_THROW_UNLESS(decryptor != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        decryptor->Initialize(key, sizeof(key), iv, sizeof(iv));

        int64_t fileSize;
        NN_RESULT_DO(file->GetSize(&fileSize));
        auto pImporter = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::ISaveDataImporter, SaveDataImporter>(m_pImpl, info, std::move(file), fileSize, std::move(macUpdater), std::move(decryptor));
        NN_RESULT_THROW_UNLESS(pImporter != nullptr, nn::fs::ResultAllocationMemoryFailedInSaveDataTransferManagerB());
        outValue.Set(pImporter);
        NN_RESULT_SUCCESS;

    }

    // SaveDataInternalFileSystem の寿命管理用アダプタ
    class SaveDataInternalFile : public fs::fsa::IFile
    {
    public:
        SaveDataInternalFile(std::shared_ptr<fs::fsa::IFile> file, std::shared_ptr<fs::fsa::IFileSystem> fileSystem)
            : m_FileSystem(std::move(fileSystem))
            , m_File(std::move(file))
        {
        }

    private:
        virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_File->Read(outValue, offset, buffer, size, option);
        }

        virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_File->Write(offset, buffer, size, option);
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_File->Flush();
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_File->SetSize(size);
        }

        virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_File->GetSize(outValue);
        }

        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            return OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
        }

    private:
        const std::shared_ptr<fs::fsa::IFileSystem> m_FileSystem;
        const std::shared_ptr<fs::fsa::IFile>       m_File;
    };


    Result SaveDataTransferManager::OpenSaveDataConcatenationFile(std::shared_ptr<fs::fsa::IFile>* outFile, std::unique_ptr<SaveDataMacUpdater>* outMacUpdater, const nn::fs::SaveDataInfo& info, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        std::shared_ptr<fs::fsa::IFileSystem> saveDataInternalFs;
        NN_RESULT_DO(m_pImpl->OpenSaveDataInternalStorageFileSystemCore(&saveDataInternalFs, info.saveDataSpaceId, info.saveDataId));

        const char* const InternalStorageFileNameArray[] =
        {
            fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableControlArea,
            fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta,
            fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree,
        };

        std::shared_ptr<fs::fsa::IFile> concatenatedSaveDataInternalFile;
        {
            std::shared_ptr<fs::fsa::IFile> internalFile[fssystem::save::SaveDataFileSystemCore::InternalStorageFileCount];
            for (int i = 0; i < fssystem::save::SaveDataFileSystemCore::InternalStorageFileCount; i++)
            {
                std::unique_ptr<fs::fsa::IFile> file;
                char path[fs::EntryNameLengthMax + 1];
                util::SNPrintf(path, sizeof(path), "/%s", InternalStorageFileNameArray[i]);
                NN_ABORT_UNLESS_RESULT_SUCCESS(saveDataInternalFs->OpenFile(&file, path, static_cast<fs::OpenMode>(fs::OpenMode_Write | fs::OpenMode_Read)));
                internalFile[i] = std::move(file);
            }

            NN_STATIC_ASSERT(fssystem::save::SaveDataFileSystemCore::InternalStorageFileCount == 3);
            std::array<std::shared_ptr<fs::fsa::IFile>, fssystem::save::SaveDataFileSystemCore::InternalStorageFileCount> fileArray{
                {
                    internalFile[0],
                    internalFile[1],
                    internalFile[2],
                }
            };

            concatenatedSaveDataInternalFile = fssystem::AllocateShared<fssystem::ConcatenationFile<3>>(std::move(fileArray));
            NN_RESULT_THROW_UNLESS(concatenatedSaveDataInternalFile != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());
        }

        std::unique_ptr<SaveDataMacUpdater> macUpdater(new SaveDataMacUpdater(saveDataInternalFs));
        NN_RESULT_THROW_UNLESS(macUpdater != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

        auto concatenatedSaveDataInternalFileWithFs = fssystem::AllocateShared<SaveDataInternalFile>(std::move(concatenatedSaveDataInternalFile), std::move(saveDataInternalFs));
        NN_RESULT_THROW_UNLESS(concatenatedSaveDataInternalFileWithFs != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

        std::shared_ptr<fs::fsa::IFile> file;
        if (info.saveDataType == fs::SaveDataType::Account)
        {
            std::shared_ptr<fs::fsa::IFile> thumbnailFile;
            NN_RESULT_DO(m_pImpl->OpenSaveDataMetaFileRaw(&thumbnailFile, info.saveDataSpaceId, info.saveDataId, fs::SaveDataMetaType::Thumbnail, mode));

            std::array<std::shared_ptr<fs::fsa::IFile>, 2> fileArray{
                {
                    concatenatedSaveDataInternalFileWithFs,
                    thumbnailFile,
                }
            };
            file = fssystem::AllocateShared<fssystem::ConcatenationFile<2>>(std::move(fileArray));

            NN_RESULT_THROW_UNLESS(file != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());
        }
        else
        {
            file = std::move(concatenatedSaveDataInternalFileWithFs);
        }

        *outFile = std::move(file);
        *outMacUpdater = std::move(macUpdater);
        NN_RESULT_SUCCESS;
    }



}}}
