﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_Utility.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>

#include "fssrv_SaveDataTransferVersion2.h"
#include "fssrv_SaveDataTransferStream.h"
#include "fssrv_SaveDataTransferPorterCore.h"
#include "fssrv_SaveDataTransferChunkPorter.h"
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace fssrv { namespace detail {


class SaveDataChunkEncryptor : public AesGcmSource::IEncryptor, public fs::detail::Newable
{
public:
    SaveDataChunkEncryptor(const SaveDataTransferCryptoConfiguration& configuration, const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage) NN_NOEXCEPT
        : m_Configuration(configuration)
        , m_KeySeedPackage(keySeedPackage)
    {
    }

public:
    virtual void   Initialize(AesGcmStreamHeader* pOutHeader) NN_NOEXCEPT NN_OVERRIDE
    {
        char key[decltype(m_Encryptor)::BlockSize];
        AesGcmStreamHeader header;

        // TODO: m_Configuration から keyGeneration を指定
        header.keyGeneration = 0;
        NN_UNUSED(header.keyGeneration);
        m_Configuration.pGenerateAesKey(key, sizeof(key), fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpData, m_KeySeedPackage.keySeed, sizeof(m_KeySeedPackage.keySeed));

        // TODO: IV を乱数にする
        //m_Configuration.pGenerateRandom(header.iv, sizeof(header.iv));
        memset(header.iv, 0, sizeof(header.iv));

        m_Encryptor.Initialize(key, sizeof(key), header.iv, sizeof(header.iv));
        memcpy(pOutHeader, &header, sizeof(AesGcmStreamHeader));
    }

    virtual size_t Update(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Encryptor.Update(pDst, dstSize, pSrc, srcSize);
    }

    virtual void   GetMac(void* pMac, size_t macSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Encryptor.GetMac(pMac, macSize);
    }

private:
    const SaveDataTransferCryptoConfiguration& m_Configuration;
    const SaveDataTransferManagerVersion2::KeySeedPackage::Content& m_KeySeedPackage;

    crypto::Aes128GcmEncryptor m_Encryptor;
};

class SaveDataChunkDecryptor : public AesGcmSink::IDecryptor, public fs::detail::Newable
{
public:
    SaveDataChunkDecryptor(const SaveDataTransferCryptoConfiguration& configuration, const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage) NN_NOEXCEPT
        : m_Configuration(configuration)
        , m_KeySeedPackage(keySeedPackage)
    {
    }

public:
    virtual void   Initialize(const AesGcmStreamHeader& header) NN_NOEXCEPT NN_OVERRIDE
    {
        char key[decltype(m_Decryptor)::BlockSize];

        // TODO: keyGeneration を指定
        NN_UNUSED(header.keyGeneration);
        m_Configuration.pGenerateAesKey(key, sizeof(key), fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpData, m_KeySeedPackage.keySeed, sizeof(m_KeySeedPackage.keySeed));
        m_Decryptor.Initialize(key, sizeof(key), header.iv, sizeof(header.iv));
    }

    virtual size_t Update(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Decryptor.Update(pDst, dstSize, pSrc, srcSize);
    }

    virtual void   GetMac(void* pMac, size_t macSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Decryptor.GetMac(pMac, macSize);
    }

private:
    const SaveDataTransferCryptoConfiguration& m_Configuration;
    const SaveDataTransferManagerVersion2::KeySeedPackage::Content& m_KeySeedPackage;
    crypto::Aes128GcmDecryptor m_Decryptor;
};



SaveDataChunkExporter::SaveDataChunkExporter(SaveDataPorterCore* pParentPorter, fs::SaveDataChunkId id, const SaveDataTransferCryptoConfiguration& configuration, const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage) NN_NOEXCEPT
    : m_pParentPorter(pParentPorter)
    , m_Id(id)
{
    m_Encryptor = fssystem::AllocateShared<SaveDataChunkEncryptor>(configuration, keySeedPackage);
}

Result SaveDataChunkExporter::Initialize(std::shared_ptr<fs::IStorage> storage, bool compressionEnabled) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Encryptor != nullptr, nn::fs::ResultAllocationMemoryFailed());

    auto storageStream = fssystem::AllocateShared<StorageStream>(std::move(storage));
    NN_RESULT_THROW_UNLESS(storageStream != nullptr, nn::fs::ResultAllocationMemoryFailed());

    std::shared_ptr<ISource> source;
    if (compressionEnabled)
    {
        auto compressionSource = fssystem::AllocateShared<CompressionSource>(std::move(storageStream));
        NN_RESULT_THROW_UNLESS(compressionSource != nullptr, nn::fs::ResultAllocationMemoryFailed());
        NN_RESULT_DO(compressionSource->Initialize());

        source = std::move(compressionSource);
    }
    else
    {
        source = std::move(storageStream);
    }

    auto aesGcmSource = fssystem::AllocateShared<AesGcmSource>(std::move(source), std::move(m_Encryptor));
    NN_RESULT_THROW_UNLESS(aesGcmSource != nullptr, nn::fs::ResultAllocationMemoryFailed());
    m_Source = std::move(aesGcmSource);

    NN_RESULT_SUCCESS;
}

Result SaveDataChunkExporter::Pull(nn::sf::Out<uint64_t> outValue, const nn::sf::OutBuffer& outBuffer, uint64_t size) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(outBuffer.GetSize() == size, fs::ResultInvalidSize());
    auto pBuffer = reinterpret_cast<char*>(outBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pBuffer != nullptr, fs::ResultNullptrArgument());

    size_t readSize;
    NN_RESULT_DO(m_Source->Pull(&readSize, pBuffer, static_cast<size_t>(size)));
    m_PulledSize += readSize;

    // TODO: PorterCore への報告方法再検討
    if (m_Id != fs::SaveDataChunkIdForInitialData && m_Source->IsEnd())
    {
        // TODO: 複数回報告の抑制
        char mac[AesGcmSource::MacSize];
        NN_RESULT_DO(m_Source->GetMac(mac, sizeof(mac)));
        NN_RESULT_DO(m_pParentPorter->ReportMac(m_Id, mac, sizeof(mac)));
        NN_RESULT_DO(m_pParentPorter->ReportSize(m_Id, m_PulledSize));
    }

    *outValue = static_cast<uint64_t>(readSize);
    NN_RESULT_SUCCESS;
}


Result SaveDataChunkExporter::GetRestRawDataSize(nn::sf::Out<int64_t> outValue) NN_NOEXCEPT
{
    int64_t size;
    NN_RESULT_DO(m_Source->GetRestRawDataSize(&size));
    outValue.Set(size);
    NN_RESULT_SUCCESS;
}


SaveDataChunkImporter::SaveDataChunkImporter(const SaveDataTransferCryptoConfiguration& configuration, const SaveDataTransferManagerVersion2::KeySeedPackage::Content& keySeedPackage) NN_NOEXCEPT
{
    m_Decryptor = fssystem::AllocateShared<SaveDataChunkDecryptor>(configuration, keySeedPackage);
}

Result SaveDataChunkImporter::Initialize(std::shared_ptr<fs::IStorage> storage, int64_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Decryptor != nullptr, nn::fs::ResultAllocationMemoryFailed());

    auto storageStream = fssystem::AllocateShared<StorageStream>(std::move(storage));
    NN_RESULT_THROW_UNLESS(storageStream != nullptr, nn::fs::ResultAllocationMemoryFailed());

    auto decompressionSink = fssystem::AllocateShared<DecompressionSink>(std::move(storageStream));
    NN_RESULT_THROW_UNLESS(decompressionSink != nullptr, nn::fs::ResultAllocationMemoryFailed());
    NN_RESULT_DO(decompressionSink->Initialize());

    auto aesGcmSink = fssystem::AllocateShared<AesGcmSink>(std::move(decompressionSink), size, std::move(m_Decryptor));
    NN_RESULT_THROW_UNLESS(aesGcmSink != nullptr, nn::fs::ResultAllocationMemoryFailed());

    m_Sink = std::move(aesGcmSink);

    NN_RESULT_SUCCESS;
}

Result SaveDataChunkImporter::Push(const nn::sf::InBuffer& inBuffer, uint64_t size) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(inBuffer.GetSize() == size, fs::ResultInvalidSize());
    auto pBuffer = reinterpret_cast<const char*>(inBuffer.GetPointerUnsafe());
    NN_FSP_REQUIRES(pBuffer != nullptr, fs::ResultNullptrArgument());

    // ChunkImporter が有効でない場合は DataCorrupted を返す
    NN_RESULT_THROW_UNLESS(m_IsValid, fs::ResultSaveDataTransferDataCorrupted(););

    NN_RESULT_TRY(m_Sink->Push(pBuffer, static_cast<size_t>(size)))
        NN_RESULT_CATCH_ALL
        {
            // ChunkImporter を無効にする
            m_IsValid = false;
            return NN_RESULT_CURRENT_RESULT;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}


}}}
