﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/fssrv/detail/fssrv_ISaveDataTransferCoreInterface.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

#include "fssrv_SaveDataImporter.h"

namespace nn { namespace fssrv { namespace detail {

    SaveDataMacUpdater::SaveDataMacUpdater(std::shared_ptr<fs::fsa::IFileSystem> fileSystem)
        : m_FileSystem(std::move(fileSystem))
    {
    }

    Result SaveDataMacUpdater::UpdateMac() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_FileSystem->Commit());
        NN_RESULT_SUCCESS;
    }


    SaveDataImporter::SaveDataImporter(
        nn::sf::SharedPointer<ISaveDataTransferCoreInterface> pImpl,
        const fs::SaveDataInfo& info,
        std::shared_ptr<fs::fsa::IFile>&& file,
        int64_t fileSize,
        std::unique_ptr<SaveDataMacUpdater>&& macUpdater,
        std::unique_ptr<crypto::Aes128GcmDecryptor>&& decryptor) NN_NOEXCEPT
        : m_pImpl(pImpl)
        , m_SaveDataInfo(info)
        , m_IsFinalized(false)
        , m_Pushed(0)
        , m_Length(fileSize + decltype(m_pDecryptor)::element_type::MacSize)
        , m_pFile(std::move(file))
        , m_FileOffset(0)
        , m_pDecryptor(std::move(decryptor))
        , m_MacUpdater(std::move(macUpdater))
    {
    }

    SaveDataImporter::~SaveDataImporter() NN_NOEXCEPT
    {
        if (!m_IsFinalized)
        {
            m_pFile.reset();
            m_MacUpdater.reset();
            auto result = m_pImpl->CancelSaveDataCreation(m_SaveDataInfo.saveDataId, m_SaveDataInfo.saveDataSpaceId);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[fs] Error : Failed to cancel save data importing. (%x)\n", result.GetInnerValueForDebug());
            }
        }
    }

    Result SaveDataImporter::GetSaveDataInfo(nn::sf::Out<nn::fs::SaveDataInfo> outValue) NN_NOEXCEPT
    {
        outValue.Set(m_SaveDataInfo);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataImporter::GetRestSize(nn::sf::Out<std::uint64_t> outValue) NN_NOEXCEPT
    {
        outValue.Set(m_Length - m_Pushed);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataImporter::Push(const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        const auto MacSize = decltype(m_pDecryptor)::element_type::MacSize;

        if (buffer.GetSize() == 0)
        {
            NN_RESULT_SUCCESS;
        }

        NN_FSP_REQUIRES(!m_IsFinalized, nn::fs::ResultPreconditionViolation());

        NN_FSP_REQUIRES(buffer.GetSize() <= (m_Length - m_Pushed), fs::ResultInvalidSize());
        auto pSrc = buffer.GetPointerUnsafe();
        NN_FSP_REQUIRES(pSrc != nullptr, fs::ResultNullptrArgument());

        auto restBufferSize = std::min(static_cast<uint64_t>(buffer.GetSize()), m_Length - m_Pushed);

        // 実体
        if (m_Length - m_Pushed > MacSize)
        {

            fssystem::PooledBuffer pooledBuffer;
            {
                const size_t IdealWorkBufferSize = 512 * 1024;
                const size_t MinWorkBufferSize = 1024;
                pooledBuffer.Allocate(IdealWorkBufferSize, MinWorkBufferSize);
            }

            const size_t WorkBufferSize = pooledBuffer.GetSize();
            auto* workBuffer = pooledBuffer.GetBuffer();

            NN_SDK_REQUIRES_NOT_NULL(workBuffer);

            auto restFileSize = m_Length - m_Pushed - MacSize;
            auto tryWriteSize = static_cast<size_t>(std::min(static_cast<uint64_t>(restBufferSize), restFileSize));

            auto restDecryptSize = tryWriteSize;
            while(restDecryptSize > 0)
            {
                auto decryptSize = std::min(restDecryptSize, WorkBufferSize);
                m_pDecryptor->Update(workBuffer, WorkBufferSize, pSrc, decryptSize);
                NN_RESULT_DO(m_pFile->Write(m_FileOffset, workBuffer, decryptSize, nn::fs::WriteOption()));

                m_FileOffset    += decryptSize;
                pSrc            += decryptSize;
                restDecryptSize -= decryptSize;
            }

            m_Pushed       += tryWriteSize;
            restBufferSize -= tryWriteSize;
        }

        NN_SDK_ASSERT(m_Length - m_Pushed >= 0);

        // mac
        if (m_Length - m_Pushed <= MacSize)
        {
            auto macOffset = m_Pushed - (m_Length - MacSize);

            auto writeMacSize = static_cast<size_t>(std::min(restBufferSize, m_Length - m_Pushed));
            memcpy(m_Mac + macOffset, pSrc, writeMacSize);
            m_Pushed += writeMacSize;
        }

        NN_RESULT_SUCCESS;
    }

    Result SaveDataImporter::Finalize() NN_NOEXCEPT
    {
        const auto MacSize = decltype(m_pDecryptor)::element_type::MacSize;

        NN_FSP_REQUIRES(m_Pushed == m_Length, nn::fs::ResultPreconditionViolation());

        NN_RESULT_DO(m_pFile->Flush());

        char actualMac[MacSize];
        m_pDecryptor->GetMac(&actualMac, sizeof(actualMac));

        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(actualMac, m_Mac, crypto::Aes128GcmDecryptor::MacSize), fs::ResultSaveDataTransferImportMacVerificationFailed());

        m_pFile.reset();
        NN_RESULT_DO(m_MacUpdater->UpdateMac());
        m_MacUpdater.reset();
        NN_RESULT_DO(m_pImpl->FinalizeSaveDataCreation(m_SaveDataInfo.saveDataId, m_SaveDataInfo.saveDataSpaceId));
        m_IsFinalized = true;
        NN_RESULT_SUCCESS;
    }

}}}
