﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/fssystem/fs_AllocatorUtility.h>

#include "fssrv_AccessControl.h"

namespace nn { namespace fssrv { namespace detail {

    class ProgramInfo : public nn::fs::detail::Newable
    {
    public:
        ProgramInfo(uint64_t processId, uint64_t programId, uint8_t storageId, const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize) NN_NOEXCEPT
            : m_ProcessId(processId),
              m_AccessControl(fsAccessControlData, dataSize, fsAccessControlDesc, descSize)
        {
            m_ProgramId.value = programId;
            m_StorageId = static_cast<nn::ncm::StorageId>(storageId);
        }

        bool Contains(uint64_t processId) const NN_NOEXCEPT
        {
            return m_ProcessId == processId;
        }
        const AccessControl* GetAccessControl() const NN_NOEXCEPT
        {
            return &m_AccessControl;
        }

        uint64_t GetProcessId() const NN_NOEXCEPT
        {
            return m_ProcessId;
        }

        nn::ncm::ProgramId GetProgramId() const NN_NOEXCEPT
        {
            return m_ProgramId;
        }

        nn::Bit64 GetProgramIdValue() const NN_NOEXCEPT
        {
            return m_ProgramId.value;
        }
        nn::ncm::StorageId GetStorageId() const NN_NOEXCEPT
        {
            return m_StorageId;
        }

        static std::shared_ptr<ProgramInfo> GetProgramInfoForInitialProcess() NN_NOEXCEPT;

    private:
        ProgramInfo(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize) NN_NOEXCEPT // for initial process
            : m_ProcessId(0),
              m_ProgramId({0}),
              m_StorageId(static_cast<nn::ncm::StorageId>(0)),
              m_AccessControl(fsAccessControlData, dataSize, fsAccessControlDesc, descSize, 0xFFFFFFFFFFFFFFFF)
        {
        }

        uint64_t m_ProcessId;
        nn::ncm::ProgramId m_ProgramId;
        nn::ncm::StorageId m_StorageId;
        AccessControl m_AccessControl;
    };

    struct ProgramInfoNode : public util::IntrusiveListBaseNode<ProgramInfoNode>, public nn::fs::detail::Newable
    {
    public:
        std::shared_ptr<ProgramInfo> pProgramInfo;
    };

    bool IsInitialProgram(Bit64 processId) NN_NOEXCEPT;
    bool IsCurrentProcess(Bit64 processId) NN_NOEXCEPT;

}}}
