﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fssystem/fs_PathTool.h>
#include "fssrv_PathNormalizer.h"

namespace nn { namespace fssrv { namespace detail {

    nn::Result PathNormalizer::Normalize(
        const char** outPath,
        BufferType* outBuffer,
        const char* path,
        bool isUncPreserved,
        bool isTailSeparatorPreserved,
        bool isMountNameContained) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outPath);
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        *outPath = nullptr;
        *outBuffer = BufferType();

        // パス本体の開始位置を探索する
        auto pPathBody = path;
        if( isMountNameContained )
        {
            for( ; pPathBody < path + nn::fs::MountNameLengthMax + 1; ++pPathBody )
            {
                if( *pPathBody == '\0' )
                {
                    break;
                }
                else if( *pPathBody == ':' )
                {
                    ++pPathBody;
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(
                (path < pPathBody - 1) && (*(pPathBody - 1) == ':'),
                nn::fs::ResultInvalidPath());
        }

        // 正規化済みかどうか判定する
        auto isNormalized = false;
        NN_RESULT_DO(nn::fssystem::PathTool::IsNormalized(&isNormalized, pPathBody));

        if( isNormalized )
        {
            // 正規化済みならそのまま出力する
            *outPath = path;
        }
        else
        {
            // 正規化されていなければバッファを確保して正規化する
            auto buffer = nn::fs::detail::MakeUnique<char[]>(nn::fs::EntryNameLengthMax + 1);
            NN_RESULT_THROW_UNLESS(
                buffer != nullptr,
                nn::fs::ResultAllocationMemoryFailedInPathNormalizerA());

            // マウント名は単にコピーする
            const auto mountNameLength = pPathBody - path;
            memcpy(buffer.get(), path, mountNameLength);

            // マウント名に続くパス部分を正規化する
            size_t normalizedLength = 0;
            NN_RESULT_DO(nn::fssystem::PathTool::Normalize(
                buffer.get() + mountNameLength,
                &normalizedLength,
                pPathBody,
                nn::fs::EntryNameLengthMax + 1 - mountNameLength,
                isUncPreserved));

            // 必要なら末尾のセパレータを残す
            if( isTailSeparatorPreserved )
            {
                const auto unnormalizedLength = strnlen(path, nn::fs::EntryNameLengthMax);
                if( nn::fssystem::PathTool::IsSeparator(path[unnormalizedLength - 1]) )
                {
                    NN_SDK_ASSERT_LESS(
                        mountNameLength + normalizedLength,
                        static_cast<size_t>(nn::fs::EntryNameLengthMax));
                    buffer[mountNameLength + normalizedLength] = '/';
                    buffer[mountNameLength + normalizedLength + 1] = '\0';
                }
            }

            // 正規化に成功したので結果を出力する
            *outPath = buffer.get();
            *outBuffer = std::move(buffer);
        }

        NN_RESULT_SUCCESS;
    }

}}}
