﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/detail/fs_Log.h>

#include "fssrv_AccessControl.h"
#include "fssrv_DeviceOperator.h"
#include "fssrv_SdmmcStorageService.h"
#include "fssrv_GameCardManager.h"
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_DeviceSimulation.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_SpeedEmulationConfiguration.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/result/result_HandlingUtility.h>

// SDMMC 制御の一時停止・再開の有効化（有効にしてコミットしないでください）
//#define NN_DETAIL_FS_SDMMC_CONTROL_ENABLE

#if defined(NN_DETAIL_FS_SDMMC_CONTROL_ENABLE)
#include "fssrv_DeviceBuffer.h"
#endif

#define NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(condition, flagName, processId) \
{ \
    if (!(condition)) \
    { \
        NN_SDK_LOG("[fs] ResultPermissionDenied (%s) was returned to process %d\n", flagName, processId); \
        return nn::fs::ResultPermissionDenied(); \
    } \
}

namespace nn { namespace fssrv { namespace detail {

    DeviceOperator::DeviceOperator(const AccessControl* accessControl, Bit64 processId) NN_NOEXCEPT : m_AccessControl(accessControl), m_ProcessId(processId)
    {
    }

    DeviceOperator::~DeviceOperator() NN_NOEXCEPT
    {
    }

    Result DeviceOperator::IsSdCardInserted(nn::sf::Out<bool> outValue) NN_NOEXCEPT
    {
        outValue.Set(nn::fssrv::detail::IsSdCardInserted());
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetSdCardCid(const nn::sf::OutBuffer& outBuffer, std::int64_t size) NN_NOEXCEPT
    {
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardCid(pBuffer, static_cast<size_t>(size)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetSdCardSpeedMode(nn::sf::Out<int64_t> outValue) NN_NOEXCEPT
    {
        nn::fs::SdCardSpeedMode speedMode;
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardSpeedMode(&speedMode));
        outValue.Set(static_cast<int64_t>(speedMode));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetSdCardUserAreaSize(nn::sf::Out<int64_t> outValue) NN_NOEXCEPT
    {
        int64_t size;
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardUserAreaSize(&size));
        outValue.Set(size);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetSdCardProtectedAreaSize(nn::sf::Out<int64_t> outValue) NN_NOEXCEPT
    {
        int64_t size;
        NN_RESULT_DO(nn::fssrv::detail::GetSdCardProtectedAreaSize(&size));
        outValue.Set(size);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetAndClearSdCardErrorInfo(nn::sf::Out<nn::fs::StorageErrorInfo> outStorageErrorInfo, nn::sf::Out<std::int64_t> outLogSize, const nn::sf::OutBuffer& outLogBuffer, std::int64_t logBufferSize) NN_NOEXCEPT
    {
        nn::fs::StorageErrorInfo storageErrorInfo;
        size_t logSize;
        NN_RESULT_DO(nn::fssrv::detail::GetAndClearSdCardErrorInfo(&storageErrorInfo, &logSize, outLogBuffer.GetPointerUnsafe(), static_cast<size_t>(logBufferSize)));
        outStorageErrorInfo.Set(storageErrorInfo);
        outLogSize.Set(logSize);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetMmcCid(const nn::sf::OutBuffer& outBuffer, std::int64_t size) NN_NOEXCEPT
    {
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetMmcCid(pBuffer, static_cast<size_t>(size)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetMmcSpeedMode(nn::sf::Out<int64_t> outValue) NN_NOEXCEPT
    {
        nn::fs::MmcSpeedMode speedMode;
        NN_RESULT_DO(nn::fssrv::detail::GetMmcSpeedMode(&speedMode));
        outValue.Set(static_cast<int64_t>(speedMode));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::EraseMmc(std::uint32_t id) NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::EraseMmc);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "EraseMmc", m_ProcessId);
        NN_RESULT_DO(nn::fssrv::detail::EraseMmc(static_cast<nn::fs::MmcPartition>(id)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetMmcPartitionSize(nn::sf::Out<int64_t> outValue, std::uint32_t id) NN_NOEXCEPT
    {
        int64_t mmcPartitionSize;
        NN_RESULT_DO(nn::fssrv::detail::GetMmcPartitionSize(&mmcPartitionSize, static_cast<nn::fs::MmcPartition>(id)));
        outValue.Set(mmcPartitionSize);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetMmcPatrolCount(nn::sf::Out<uint32_t> outValue) NN_NOEXCEPT
    {
        uint32_t mmcPatrolCount;
        NN_RESULT_DO(nn::fssrv::detail::GetMmcPatrolCount(&mmcPatrolCount));
        outValue.Set(mmcPatrolCount);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetAndClearMmcErrorInfo(nn::sf::Out<nn::fs::StorageErrorInfo> outStorageErrorInfo, nn::sf::Out<std::int64_t> outLogSize, const nn::sf::OutBuffer& outLogBuffer, std::int64_t logBufferSize) NN_NOEXCEPT
    {
        nn::fs::StorageErrorInfo storageErrorInfo;
        size_t logSize;
        NN_RESULT_DO(nn::fssrv::detail::GetAndClearMmcErrorInfo(&storageErrorInfo, &logSize, outLogBuffer.GetPointerUnsafe(), static_cast<size_t>(logBufferSize)));
        outStorageErrorInfo.Set(storageErrorInfo);
        outLogSize.Set(logSize);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetMmcExtendedCsd(const nn::sf::OutBuffer& outBuffer, std::int64_t size) NN_NOEXCEPT
    {
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetMmcExtendedCsd(pBuffer, static_cast<size_t>(size)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::SuspendMmcPatrol() NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::ControlMmcPatrol);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "ControlMmcPatrol", m_ProcessId);
        NN_RESULT_DO(nn::fssrv::detail::SuspendMmcPatrol());
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::ResumeMmcPatrol() NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::ControlMmcPatrol);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "ControlMmcPatrol", m_ProcessId);
        NN_RESULT_DO(nn::fssrv::detail::ResumeMmcPatrol());
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::IsGameCardInserted(nn::sf::Out<bool> outValue) NN_NOEXCEPT
    {
        outValue.Set(nn::fssrv::detail::IsGameCardInserted());
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::EraseGameCard(std::uint32_t sizeId, std::uint64_t normalAreaSize) NN_NOEXCEPT
    {
        auto accessibility = m_AccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
        NN_RESULT_DO(nn::fssrv::detail::EraseGameCard(static_cast<nn::fs::GameCardSize>(sizeId), normalAreaSize));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardHandle(nn::sf::Out<uint32_t> outValue) NN_NOEXCEPT
    {
        // GetGameCardHandle のみ GC ASIC とのセッション構築結果を取得・チェック
        NN_RESULT_DO(nn::fssrv::detail::GetInitializationResult());
        NN_RESULT_THROW_UNLESS(nn::fssrv::detail::IsGameCardInserted(), nn::fs::ResultGameCardFsGetHandleFailure());
        nn::fs::GameCardHandle handle;
        NN_RESULT_DO((nn::fssrv::detail::GetGameCardHandle(&handle)));
        outValue.Set(handle);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardUpdatePartitionInfo(nn::sf::Out<uint32_t> outVersion, nn::sf::Out<uint64_t> outId, std::uint32_t handle) NN_NOEXCEPT
    {
        nn::gc::GameCardStatus gameCardStatus;
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardStatus(&gameCardStatus, static_cast<nn::fs::GameCardHandle>(handle)));
        outVersion.Set(gameCardStatus.cupVersion);
        outId.Set(gameCardStatus.cupId);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::FinalizeGameCardDriver() NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::FinalizeGameCardDriver);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "FinalizeGameCardDriver", m_ProcessId);

        nn::fssrv::detail::FinalizeGcLibrary();
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardAttribute(nn::sf::Out<uint8_t> outValue, std::uint32_t handle) NN_NOEXCEPT
    {
        nn::gc::GameCardStatus gameCardStatus;
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardStatus(&gameCardStatus, static_cast<nn::fs::GameCardHandle>(handle)));
        outValue.Set(gameCardStatus.flags);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardDeviceCertificate(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize, std::uint32_t handle) NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::GetGameCardDeviceCertificate);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "GetGameCardDeviceCertificate", m_ProcessId);

        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardDeviceCertificate(pBuffer, static_cast<size_t>(outBufferSize), static_cast<nn::fs::GameCardHandle>(handle)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardAsicInfo(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize, const nn::sf::InBuffer& inBuffer, std::int64_t inBufferSize) NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::GetGameCardAsicInfo);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "GetGameCardAsicInfo", m_ProcessId);

        NN_FSP_REQUIRES(outBufferSize == sizeof(nn::gc::RmaInformation), nn::fs::ResultInvalidArgument());
        nn::gc::RmaInformation rmaInfo;
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardAsicInfo(&rmaInfo, inBuffer.GetPointerUnsafe(), static_cast<size_t>(inBufferSize)));
        memcpy(outBuffer.GetPointerUnsafe(), &rmaInfo, sizeof(nn::gc::RmaInformation));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardIdSet(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(outBufferSize == sizeof(nn::gc::GameCardIdSet), nn::fs::ResultInvalidArgument());
        nn::gc::GameCardIdSet gcIdSet;
        memset(&gcIdSet, 0, sizeof(nn::gc::GameCardIdSet));
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardIdSet(&gcIdSet));
        memcpy(outBuffer.GetPointerUnsafe(), &gcIdSet, sizeof(nn::gc::GameCardIdSet));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::WriteToGameCardDirectly(std::int64_t offset, const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize) NN_NOEXCEPT
    {
        auto accessibility = m_AccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::WriteToGameCardDirectly(static_cast<size_t>(offset), pBuffer, static_cast<size_t>(outBufferSize)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::SetVerifyWriteEnalbleFlag(bool flag) NN_NOEXCEPT
    {
        // フラグ設定のみなので権限管理不要
        nn::fssrv::detail::SetVerifyWriteEnalbleFlag(flag);
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardImageHash(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize, std::uint32_t handle) NN_NOEXCEPT
    {
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardImageHash(pBuffer, static_cast<size_t>(outBufferSize), static_cast<nn::fs::GameCardHandle>(handle)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::SetSpeedEmulationMode(std::int32_t mode) NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::SetSpeedEmulationMode);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SetSpeedEmulationMode", m_ProcessId);

        nn::fs::SpeedEmulationMode emulationMode = static_cast<nn::fs::SpeedEmulationMode>(mode);
        nn::fssystem::SpeedEmulationConfiguration::SetSpeedEmulationMode(emulationMode);

        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetSpeedEmulationMode(nn::sf::Out<std::int32_t> outValue) NN_NOEXCEPT
    {
        const nn::fs::SpeedEmulationMode mode
            = nn::fssystem::SpeedEmulationConfiguration::GetSpeedEmulationMode();

        outValue.Set(static_cast<std::int32_t>(mode));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardDeviceIdForProdCard(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize, const nn::sf::InBuffer& inBuffer, std::int64_t inBufferSize) NN_NOEXCEPT
    {
        auto accessibility = m_AccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
        auto pOutBuffer = outBuffer.GetPointerUnsafe();
        auto pInBuffer  = inBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardDeviceIdForProdCard(pOutBuffer, static_cast<size_t>(outBufferSize), pInBuffer, static_cast<size_t>(inBufferSize)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::EraseAndWriteParamDirectly(const nn::sf::InBuffer& inBuffer, std::int64_t inBufferSize) NN_NOEXCEPT
    {
        auto accessibility = m_AccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
        NN_RESULT_DO(nn::fssrv::detail::EraseAndWriteParamDirectly(inBuffer.GetPointerUnsafe(), static_cast<size_t>(inBufferSize)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::ReadParamDirectly(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize) NN_NOEXCEPT
    {
        auto accessibility = m_AccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::ReadParamDirectly(pBuffer, static_cast<size_t>(outBufferSize)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::ForceEraseGameCard() NN_NOEXCEPT
    {
        auto accessibility = m_AccessControl->GetAccessibilityFor(AccessControl::AccessibilityType::OpenGameCardStorage);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(accessibility.CanWrite(), "OpenGameCardStorage", m_ProcessId);
        NN_RESULT_DO(nn::fssrv::detail::ForceEraseGameCard());
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardErrorInfo(nn::sf::Out<nn::fs::GameCardErrorInfo> outValue) NN_NOEXCEPT
    {
        nn::fs::GameCardErrorInfo gameCardErrorInfo;
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardErrorInfo(&gameCardErrorInfo));
        *outValue = gameCardErrorInfo;
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardErrorReportInfo(nn::sf::Out<nn::fs::GameCardErrorReportInfo> outValue) NN_NOEXCEPT
    {
        nn::fs::GameCardErrorReportInfo gameCardErrorReportInfo;
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardErrorReportInfo(&gameCardErrorReportInfo));
        *outValue = gameCardErrorReportInfo;
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetGameCardDeviceId(const nn::sf::OutBuffer& outBuffer, std::int64_t outBufferSize) NN_NOEXCEPT
    {
        auto pBuffer = outBuffer.GetPointerUnsafe();
        NN_RESULT_DO(nn::fssrv::detail::GetGameCardDeviceId(pBuffer, static_cast<size_t>(outBufferSize)));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::GetSdmmcConnectionStatus(nn::sf::Out<int32_t> outSdmmcSpeedMode, nn::sf::Out<int32_t> outSdmmcBusWidth, std::int32_t sdmmcPort) NN_NOEXCEPT
    {
        // 情報取得のみで制御ポートに作用は起こらないため、権限チェックは行わない
        nn::fs::SdmmcSpeedMode sdmmcSpeedMode = nn::fs::SdmmcSpeedMode_Unknown;
        nn::fs::SdmmcBusWidth sdmmcBusWidth = nn::fs::SdmmcBusWidth_Unknown;
        NN_RESULT_DO(nn::fssrv::detail::GetSdmmcConnectionStatus(&sdmmcSpeedMode, &sdmmcBusWidth, static_cast<nn::fs::SdmmcPort>(sdmmcPort)));
        outSdmmcSpeedMode.Set(static_cast<int32_t>(sdmmcSpeedMode));
        outSdmmcBusWidth.Set(static_cast<int32_t>(sdmmcBusWidth));
        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::SuspendSdmmcControl() NN_NOEXCEPT
    {
#if defined(NN_DETAIL_FS_SDMMC_CONTROL_ENABLE)
        nn::fssrv::detail::FinalizeGcLibrary();
        nn::fssrv::detail::SuspendSdmmcControl();
        nn::fssrv::detail::FinalizeDeviceBuffer();
        NN_RESULT_SUCCESS;
#else
        NN_RESULT_THROW(nn::fs::ResultNotImplemented());
#endif
    }

    Result DeviceOperator::ResumeSdmmcControl() NN_NOEXCEPT
    {
#if defined(NN_DETAIL_FS_SDMMC_CONTROL_ENABLE)
        nn::fssrv::detail::InitializeDeviceBuffer();
        nn::fssrv::detail::ResumeSdmmcControl();
        // GcLibrary は必要なタイミングで内部的に初期化される
        NN_RESULT_SUCCESS;
#else
        NN_RESULT_THROW(nn::fs::ResultNotImplemented());
#endif
    }

    Result DeviceOperator::SetDeviceSimulationEvent(std::uint32_t deviceType, std::uint32_t targetOperationFlag, std::uint32_t failureEventType, std::uint32_t respondingFailureResultValue, bool isPersistent) NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::SimulateDevice);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SimulateDevice", m_ProcessId);

        Result respondingFailureResult = result::detail::ConstructResult(respondingFailureResultValue);
        fs::SimulatingDeviceAccessFailureEventType simulatingDeviceEventType = static_cast<fs::SimulatingDeviceAccessFailureEventType>(failureEventType);

        switch (static_cast<fs::SimulatingDeviceType>(deviceType))
        {
        case fs::SimulatingDeviceType::GameCard:
            nn::fssrv::detail::GetGameCardEventSimulator().SetDeviceEvent(targetOperationFlag, simulatingDeviceEventType, respondingFailureResult, isPersistent);
            break;
        case fs::SimulatingDeviceType::SdCard:
        case fs::SimulatingDeviceType::eMMC:
            return fs::ResultNotImplemented();
        default:
            return fs::ResultInvalidArgument();
        }

        NN_RESULT_SUCCESS;
    }

    Result DeviceOperator::ClearDeviceSimulationEvent(std::uint32_t deviceType) NN_NOEXCEPT
    {
        auto canCall = m_AccessControl->CanCall(AccessControl::OperationType::SimulateDevice);
        NN_FSP_RESULT_THROW_PERMISSION_DENIED_UNLESS(canCall, "SimulateDevice", m_ProcessId);

        switch (static_cast<fs::SimulatingDeviceType>(deviceType))
        {
        case fs::SimulatingDeviceType::GameCard:
            nn::fssrv::detail::GetGameCardEventSimulator().ClearDeviceEvent();
            break;
        case fs::SimulatingDeviceType::SdCard:
        case fs::SimulatingDeviceType::eMMC:
            return fs::ResultNotImplemented();
        default:
            return fs::ResultInvalidArgument();
        }

        NN_RESULT_SUCCESS;
    }

}}}
