﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include "fssrv_BuiltInStorageService.h"


using namespace nn::fs;

namespace nn { namespace fssrv { namespace detail {



struct GptHeader
{
    char     signature[8];
    uint32_t revision;
    uint32_t headerSize;
    uint32_t headerCrc32;
    uint32_t reserved;
    int64_t  myLba;
    int64_t  alternateLba;
    int64_t  firstUsableLba;
    int64_t  lastUsableLba;
    Guid     diskGuid;
    int64_t  partitionEntryLba;
    int32_t  numberOfPartitionEntries;
    int32_t  sizeOfPartitionEntry;
    uint32_t partitionEntryArrayCrc32;
    uint8_t  reserved2[512 - 92];
};
NN_STATIC_ASSERT(std::is_pod<GptHeader>::value);

struct GptPartitionEntry
{
    Guid     partitionTypeGuid;
    Guid     uniquePartitionGuid;
    int64_t  startingLba;
    int64_t  endingLba;
    uint64_t attribute;
    char16_t partitionName[36];
};
NN_STATIC_ASSERT(std::is_pod<GptPartitionEntry>::value);

static_assert(sizeof(GptHeader) == 512, "");
static_assert(sizeof(GptPartitionEntry) == 128, "");

const int SectorSize = nn::sdmmc::SectorSize;
const char GptHeaderSignature[8] = {0x45, 0x46, 0x49, 0x20, 0x50, 0x41, 0x52, 0x54};



Result ReadPrimaryGptHeader(GptHeader *pOut, IStorage* pStorage) NN_NOEXCEPT
{
    static_assert(sizeof(GptHeader) == SectorSize, "sizeof(GptHeader) is invalid.");
    NN_RESULT_DO(pStorage->Read(SectorSize * 1, pOut, sizeof(GptHeader)));

    NN_RESULT_THROW_UNLESS(nn::crypto::IsSameBytes(pOut->signature, GptHeaderSignature, sizeof(GptHeaderSignature)), nn::fs::ResultGptHeaderSignatureVerificationFailed());

    // TODO: CRC の検証
    NN_RESULT_SUCCESS;
}

Result ReadGptPartitionEntry(int* pOutCount, GptPartitionEntry *pOutArray, int outArrayCount, int offsetIndex, const GptHeader &gptHeader, IStorage *pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES(util::is_aligned( outArrayCount * sizeof(GptPartitionEntry), SectorSize));
    NN_SDK_REQUIRES(util::is_aligned( offsetIndex   * sizeof(GptPartitionEntry), SectorSize));

    if( outArrayCount == 0 )
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }
    if( pOutArray == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }

    if( gptHeader.numberOfPartitionEntries < offsetIndex )
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }

    auto readCount = std::min(outArrayCount, gptHeader.numberOfPartitionEntries - offsetIndex);
    auto offset = gptHeader.partitionEntryLba * SectorSize + offsetIndex * sizeof(GptPartitionEntry);

    *pOutCount = readCount;

    return pStorage->Read(offset, pOutArray, nn::util::align_up(sizeof(GptPartitionEntry) * readCount, SectorSize));
}


// TORIAEZU: baseStorage は破棄されないものと仮定
// TODO: shared_ptr で baseStorage を pin する
Result OpenGptPartition(std::unique_ptr<IStorage>* outValue, IStorage* baseStorage, const Guid* pTargetGuid) NN_NOEXCEPT
{
    // TODO: パーティション情報や pNandStorage をキャッシュする等
    GptHeader gptHeader;
    NN_RESULT_DO(ReadPrimaryGptHeader(&gptHeader, baseStorage));

    const int EntryCount = 4;
    int validEntryCount;
    GptPartitionEntry gptEntryArray[EntryCount];

    int offsetIndex = 0;
    while( NN_STATIC_CONDITION(true) )
    {

        NN_RESULT_DO(ReadGptPartitionEntry(&validEntryCount, gptEntryArray, EntryCount, offsetIndex, gptHeader, baseStorage));

        for(auto i = 0; i < validEntryCount; ++i)
        {
            if( memcmp(pTargetGuid, &gptEntryArray[i].partitionTypeGuid, sizeof(Guid)) == 0 )
            {
                int64_t offset = gptEntryArray[i].startingLba * SectorSize;
                int64_t size   = (gptEntryArray[i].endingLba + 1 - gptEntryArray[i].startingLba) * SectorSize;

                std::unique_ptr<IStorage> bis(new SubStorage(baseStorage, offset, size));

                if( bis == nullptr )
                {
                    return nn::fs::ResultAllocationMemoryFailedInBuiltInStorageServiceA();
                }

                *outValue = std::move(bis);
                NN_RESULT_SUCCESS;
            }
        }

        if( validEntryCount < EntryCount )
        {
            return nn::fs::ResultPartitionNotFound();
        }

        offsetIndex += validEntryCount;
    }
}




}}}
