﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/os.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fsa/fs_IFile.h>

#include "fssrv_AccessLogSdCardWriter.h"

namespace nn { namespace fssrv { namespace detail {

namespace
{

const size_t WorkBufferSize = 1024 * 16;
const char AccessLogMountName[] = "$FsAccessLog";
const char AccessLogFilePath[] = "$FsAccessLog:/FsAccessLog.txt";

const char AccessLogStartMarker[] = "FS_ACCESS: { start_tag: true }\n";
const char AccessLogEndMarker[] = "FS_ACCESS: { end_tag: true }\n";

class AccessLogSdCardWriterImpl
{
public:
    AccessLogSdCardWriterImpl() NN_NOEXCEPT;
    ~AccessLogSdCardWriterImpl() NN_NOEXCEPT;

public:
    void AppendBuffer(const char* buffer, size_t length) NN_NOEXCEPT;
    void AppendProcessId(nn::Bit64 processId) NN_NOEXCEPT;

    void Flush() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

private:
    bool AllocateWorkBuffer() NN_NOEXCEPT;
    void DeallocateWorkBuffer() NN_NOEXCEPT;

    bool SetUp() NN_NOEXCEPT;
    bool WriteStartMarker() NN_NOEXCEPT;
    void TearDown(bool writeEndMarker) NN_NOEXCEPT;

private:
    void Write(const void* buffer, size_t size) NN_NOEXCEPT;
    void WriteWorkBuffer() NN_NOEXCEPT;
    void FlushBuffer() NN_NOEXCEPT;

private:
    bool m_IsLogging;
    bool m_IsTriedSetUp;

    nn::fs::FileHandle m_Handle;

    char* m_WorkBuffer;
    int64_t m_WriteOffset;
    size_t m_WriteSize;
};

AccessLogSdCardWriterImpl g_AccessLogSdCardWriter;
nn::os::Mutex g_Mutex(false);


AccessLogSdCardWriterImpl::AccessLogSdCardWriterImpl() NN_NOEXCEPT
    : m_IsLogging(false)
    , m_IsTriedSetUp(false)
    , m_WorkBuffer(nullptr)
    , m_WriteOffset(0)
    , m_WriteSize(0)
{
}

AccessLogSdCardWriterImpl::~AccessLogSdCardWriterImpl() NN_NOEXCEPT
{
    DeallocateWorkBuffer();
}

void AccessLogSdCardWriterImpl::AppendBuffer(const char* buffer, size_t length) NN_NOEXCEPT
{
    if( SetUp() && AllocateWorkBuffer() )
    {
        if( length > WorkBufferSize )
        {
            // ワークバッファよりも大きいバッファを書き込みたい場合
            // 溜まったバッファを書き出し
            WriteWorkBuffer();

            // ワークバッファを使わずに直接書き込み
            Write(buffer, length);
            return;
        }

        if( (m_WriteSize + length) > WorkBufferSize )
        {
            // ワークバッファに余裕がない場合は、ファイルに書き込み
            WriteWorkBuffer();
        }

        // ワークバッファに書き込み
        memcpy(m_WorkBuffer + m_WriteSize, buffer, length);
        m_WriteSize += length;
    }
}

void AccessLogSdCardWriterImpl::AppendProcessId(nn::Bit64 processId) NN_NOEXCEPT
{
    // 0x + processId の表示幅 + () + スペース + NULL文字
    char buffer[2 + 16 + 2 + 1 + 1];
    const int length = nn::util::SNPrintf(buffer, sizeof(buffer), "(0x%016llX) ", processId);
    if( static_cast<size_t>(length) >= sizeof(buffer) )
    {
        AppendBuffer(buffer, sizeof(buffer) - 1);
    }
    else
    {
        AppendBuffer(buffer, length);
    }
}

void AccessLogSdCardWriterImpl::Flush() NN_NOEXCEPT
{
    FlushBuffer();
    TearDown(true);

    // 次の AppendLog() までの間に SD カードの交換を許す
    m_IsTriedSetUp = false;
}

void AccessLogSdCardWriterImpl::Finalize() NN_NOEXCEPT
{
    FlushBuffer();
    TearDown(true);

    // 以降の SetUp を禁止する
    m_IsTriedSetUp = true;
}

bool AccessLogSdCardWriterImpl::AllocateWorkBuffer() NN_NOEXCEPT
{
    if( m_WorkBuffer != nullptr )
    {
        return true;
    }

    m_WorkBuffer = static_cast<char*>(fssystem::Allocate(WorkBufferSize));
    return m_WorkBuffer != nullptr;
}

void AccessLogSdCardWriterImpl::DeallocateWorkBuffer() NN_NOEXCEPT
{
    if( m_WorkBuffer != nullptr )
    {
        fssystem::Deallocate(m_WorkBuffer, WorkBufferSize);
        m_WorkBuffer = nullptr;
    }
}

bool AccessLogSdCardWriterImpl::SetUp() NN_NOEXCEPT
{
    // SetUp 済みなら現在の状態を即時返す
    if( m_IsTriedSetUp )
    {
        return m_IsLogging;
    }
    m_IsTriedSetUp = true;

    if( fs::MountSdCard(AccessLogMountName).IsFailure() )
    {
        return false;
    }

    // 書き込みファイルの作成を試みる
    nn::Result result = fs::CreateFile(AccessLogFilePath, 0);
    if( result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        // 作成成功、もしくは既にファイルがあった場合ファイルオープンする
        if( fs::OpenFile(&m_Handle, AccessLogFilePath, fs::OpenMode_Read | fs::OpenMode_Write | fs::OpenMode_AllowAppend).IsSuccess() )
        {
            // ファイルオープンに成功したら、ファイル情報をメンバ変数に読み込み、ファイルにマーカーを書き込む
            if( WriteStartMarker() )
            {
                // セットアップ完了
                m_IsLogging = true;
                return true;
            }

            fs::CloseFile(m_Handle);
        }
    }

    fs::Unmount(AccessLogMountName);
    return false;
}

bool AccessLogSdCardWriterImpl::WriteStartMarker() NN_NOEXCEPT
{
    if( fs::GetFileSize(&m_WriteOffset, m_Handle).IsFailure() )
    {
        // サイズの取得に失敗したらセットアップ失敗
        return false;
    }

    // アクセスログ開始地点であることがわかるように書き込む
    if( fs::WriteFile(
            m_Handle,
            m_WriteOffset,
            AccessLogStartMarker,
            sizeof(AccessLogStartMarker) - 1,
            fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)
        ).IsFailure() )
    {
        return false;
    }
    m_WriteOffset += sizeof(AccessLogStartMarker) - 1;

    return true;
}

void AccessLogSdCardWriterImpl::TearDown(bool writeEndMarker) NN_NOEXCEPT
{
    if( m_IsLogging )
    {
        if( writeEndMarker )
        {
            // AccessLogEndMarker を追加で書き込む
            // 書き込みに失敗した場合はログが不完全なものになる
            (void)fs::WriteFile(
                      m_Handle,
                      m_WriteOffset,
                      AccessLogEndMarker,
                      sizeof(AccessLogEndMarker) - 1,
                      fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)
                  );
        }
        fs::CloseFile(m_Handle);
        fs::Unmount(AccessLogMountName);
        m_IsLogging = false;
    }
}

void AccessLogSdCardWriterImpl::Write(const void* buffer, size_t size) NN_NOEXCEPT
{
    if( m_IsLogging )
    {
        // 書き込み前に offset のチェックをする
        if( m_WriteOffset + static_cast<int64_t>(size) < m_WriteOffset )
        {
            // offset の限度を超えた場合は、TearDown
            TearDown(true);
            return;
        }

        // 実行速度に影響を与えないように Flush せずに書き込む
        if( fs::WriteFile(m_Handle, m_WriteOffset, buffer, size, fs::WriteOption()).IsSuccess() )
        {
            m_WriteOffset += size;
        }
        else
        {
            // 書き込みに失敗したら TearDown
            TearDown(false);
        }
    }
}

void AccessLogSdCardWriterImpl::WriteWorkBuffer() NN_NOEXCEPT
{
    if( (m_WorkBuffer != nullptr) && (m_WriteSize > 0) )
    {
        Write(m_WorkBuffer, m_WriteSize);
        m_WriteSize = 0;
    }
}

void AccessLogSdCardWriterImpl::FlushBuffer() NN_NOEXCEPT
{
    WriteWorkBuffer();

    if( m_IsLogging )
    {
        if( fs::FlushFile(m_Handle).IsFailure() )
        {
            // 失敗したら TearDown
            TearDown(false);
        }
    }
}

}

void AccessLogSdCardWriter::Flush() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_Mutex);

    g_AccessLogSdCardWriter.Flush();
}

void AccessLogSdCardWriter::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_Mutex);

    g_AccessLogSdCardWriter.Finalize();
}

void AccessLogSdCardWriter::AppendLog(const char* buffer, size_t length, nn::Bit64 processId) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_Mutex);

    if( (buffer != nullptr) && (length > 0) )
    {
        g_AccessLogSdCardWriter.AppendProcessId(processId);
        g_AccessLogSdCardWriter.AppendBuffer(buffer, length);
    }
}

}}}
