﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fssrv/sf/fssrv_ISaveDataTransfer.h>
#include "fs_FileSystemProxyServiceObject.h"

#include <nn/fs.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>

using namespace nn;
using namespace nn::fs;

namespace {

    class SaveDataChunkIterator : public ISaveDataChunkIterator, public fs::detail::Newable
    {
    public:
        explicit SaveDataChunkIterator(sf::SharedPointer<fssrv::sf::ISaveDataChunkIterator>&& interface) NN_NOEXCEPT
            : m_Interface(std::move(interface))
        {
        }

        virtual ~SaveDataChunkIterator() NN_NOEXCEPT NN_OVERRIDE {}

        virtual const SaveDataChunkId GetId() const NN_NOEXCEPT NN_OVERRIDE
        {
            uint32_t id;
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetId(sf::Out<uint32_t>(&id)));
            return static_cast<SaveDataChunkId>(id);
        }

        virtual void Next() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->Next());
        }

        virtual bool IsEnd() const NN_NOEXCEPT NN_OVERRIDE
        {
            bool isEnd;
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->IsEnd(sf::Out<bool>(&isEnd)));
            return isEnd;
        }

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataChunkIterator> m_Interface;
    };


    class SaveDataChunkExporter : public ISaveDataChunkExporter, public fs::detail::Newable
    {
    public:
        explicit SaveDataChunkExporter(sf::SharedPointer<fssrv::sf::ISaveDataChunkExporter>&& interface) NN_NOEXCEPT
            : m_Interface(std::move(interface))
        {
        }

        virtual ~SaveDataChunkExporter() NN_NOEXCEPT NN_OVERRIDE {}

        virtual Result Pull(size_t* outValue, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            uint64_t pulledSize;
            NN_FS_RESULT_DO(m_Interface->Pull(sf::Out<uint64_t>(&pulledSize), sf::OutBuffer(static_cast<char*>(buffer), size), size));
            *outValue = static_cast<size_t>(pulledSize);
            NN_RESULT_SUCCESS;
        }

        virtual int64_t GetRestRawDataSize() NN_NOEXCEPT NN_OVERRIDE
        {
            int64_t restSize;
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetRestRawDataSize(sf::Out<int64_t>(&restSize)));
            return restSize;
        }

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataChunkExporter> m_Interface;
    };


    class SaveDataChunkImporter : public ISaveDataChunkImporter, public fs::detail::Newable
    {
    public:
        explicit SaveDataChunkImporter(sf::SharedPointer<fssrv::sf::ISaveDataChunkImporter>&& interface) NN_NOEXCEPT
            : m_Interface(std::move(interface))
        {
        }

        virtual ~SaveDataChunkImporter() NN_NOEXCEPT NN_OVERRIDE {}

        virtual Result Push(const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_RESULT_DO(m_Interface->Push(sf::InBuffer(static_cast<const char*>(buffer), size), size));
            NN_RESULT_SUCCESS;
        }

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataChunkImporter> m_Interface;
    };


    class SaveDataExporterVersion2 : public ISaveDataDivisionExporter, public fs::detail::Newable
    {
    public:

        explicit SaveDataExporterVersion2(sf::SharedPointer<fssrv::sf::ISaveDataDivisionExporter>&& interface) NN_NOEXCEPT
            : m_Interface(std::move(interface))
        {
        }

        virtual ~SaveDataExporterVersion2() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void SetDivisionCount(int count) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->SetDivisionCount(count));
        }

        virtual Result OpenSaveDataDiffChunkIterator(std::unique_ptr<ISaveDataChunkIterator>* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            sf::SharedPointer<fssrv::sf::ISaveDataChunkIterator> iteratorObject;
            NN_FS_RESULT_DO(m_Interface->OpenSaveDataDiffChunkIterator(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataChunkIterator>>(&iteratorObject)));
            std::unique_ptr<SaveDataChunkIterator> iterator(new SaveDataChunkIterator(std::move(iteratorObject)));
            NN_FS_RESULT_THROW_UNLESS(iterator != nullptr, ResultAllocationMemoryFailedNew());
            *outValue = std::move(iterator);
            NN_RESULT_SUCCESS;
        }

        virtual Result OpenSaveDataChunkExporter(std::unique_ptr<ISaveDataChunkExporter>* outValue, SaveDataChunkId id) NN_NOEXCEPT NN_OVERRIDE
        {
            sf::SharedPointer<fssrv::sf::ISaveDataChunkExporter> exporterObject;
            NN_FS_RESULT_DO(m_Interface->OpenSaveDataChunkExporter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataChunkExporter>>(&exporterObject), id));
            std::unique_ptr<SaveDataChunkExporter> exporter(new SaveDataChunkExporter(std::move(exporterObject)));
            NN_FS_RESULT_THROW_UNLESS(exporter != nullptr, ResultAllocationMemoryFailedNew());
            *outValue = std::move(exporter);
            NN_RESULT_SUCCESS;
        }

        virtual Result FinalizeFullExport(KeySeed* pOutKeySeed, InitialDataMac* pOutInitialDataMac) NN_NOEXCEPT NN_OVERRIDE
        {
            KeySeed keySeed;
            InitialDataMac initialDataMac;
            NN_FS_RESULT_DO(m_Interface->FinalizeFullExport(sf::Out<KeySeed>(&keySeed), sf::Out<InitialDataMac>(&initialDataMac)));
            *pOutKeySeed = keySeed;
            *pOutInitialDataMac = initialDataMac;
            NN_RESULT_SUCCESS;
        }

        virtual Result FinalizeDiffExport(InitialDataMac* pOutInitialDataMac) NN_NOEXCEPT NN_OVERRIDE
        {
            InitialDataMac initialDataMac;
            NN_FS_RESULT_DO(m_Interface->FinalizeDiffExport(sf::Out<InitialDataMac>(&initialDataMac)));
            *pOutInitialDataMac = initialDataMac;
            NN_RESULT_SUCCESS;
        }

        virtual Result CancelExport() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_RESULT_DO(m_Interface->CancelExport());
            NN_RESULT_SUCCESS;
        }

        virtual Result GetImportInitialDataAad(InitialDataAad* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            InitialDataAad initialDataAad;
            NN_FS_RESULT_DO(m_Interface->GetImportInitialDataAad(sf::Out<InitialDataAad>(&initialDataAad)));
            *pOutValue = initialDataAad;
            NN_RESULT_SUCCESS;
        }

        virtual Result SetExportInitialDataAad(const InitialDataAad& aad) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_RESULT_DO(m_Interface->SetExportInitialDataAad(aad));
            NN_RESULT_SUCCESS;
        }

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionExporter> m_Interface;

    };

    class SaveDataImporterVersion2 : public ISaveDataDivisionImporter, public fs::detail::Newable
    {
    public:
        explicit SaveDataImporterVersion2(sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter>&& interface)
            : m_Interface(std::move(interface))
        {
        }

        virtual ~SaveDataImporterVersion2() NN_NOEXCEPT NN_OVERRIDE {}

        virtual Result InitializeImport(int64_t* outRestSize, int64_t processSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_RESULT_DO(m_Interface->InitializeImport(sf::Out<int64_t>(outRestSize), processSize));
            NN_RESULT_SUCCESS;
        }

        // obsolete
        virtual Result InitializeImport(int64_t* outRequiredSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outRequiredSize);
            int64_t restSize;
            do
            {
                const int64_t ProcessSize = 512 * 1024 * 1024;
                NN_FS_RESULT_DO(InitializeImport(&restSize, ProcessSize));
            } while (restSize > 0);
            NN_RESULT_SUCCESS;
        }

        virtual Result FinalizeImport() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_RESULT_DO(m_Interface->FinalizeImport());
            NN_RESULT_SUCCESS;
        }

        virtual Result CancelImport() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_FS_RESULT_DO(m_Interface->CancelImport());
            NN_RESULT_SUCCESS;
        }

        virtual Result OpenSaveDataDiffChunkIterator(std::unique_ptr<ISaveDataChunkIterator>* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            sf::SharedPointer<fssrv::sf::ISaveDataChunkIterator> iteratorObject;
            NN_FS_RESULT_DO(m_Interface->OpenSaveDataDiffChunkIterator(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataChunkIterator>>(&iteratorObject)));
            std::unique_ptr<SaveDataChunkIterator> iterator(new SaveDataChunkIterator(std::move(iteratorObject)));
            NN_FS_RESULT_THROW_UNLESS(iterator != nullptr, ResultAllocationMemoryFailedNew());
            *outValue = std::move(iterator);
            NN_RESULT_SUCCESS;
        }

        virtual Result OpenSaveDataChunkImporter(std::unique_ptr<ISaveDataChunkImporter>* outValue, SaveDataChunkId id) NN_NOEXCEPT NN_OVERRIDE
        {
            sf::SharedPointer<fssrv::sf::ISaveDataChunkImporter> importerObject;
            NN_FS_RESULT_DO(m_Interface->OpenSaveDataChunkImporter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataChunkImporter>>(&importerObject), id));
            std::unique_ptr<SaveDataChunkImporter> importer(new SaveDataChunkImporter(std::move(importerObject)));
            NN_FS_RESULT_THROW_UNLESS(importer != nullptr, ResultAllocationMemoryFailedNew());
            *outValue = std::move(importer);
            NN_RESULT_SUCCESS;
        }

        virtual Result GetImportInitialDataAad(InitialDataAad* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            InitialDataAad initialDataAad;
            NN_FS_RESULT_DO(m_Interface->GetImportInitialDataAad(sf::Out<InitialDataAad>(&initialDataAad)));
            *pOutValue = initialDataAad;
            NN_RESULT_SUCCESS;
        }

    private:
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter> m_Interface;

    };

} // namespace

namespace nn { namespace fs {

    SaveDataTransferManagerVersion2::SaveDataTransferManagerVersion2() NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->OpenSaveDataTransferManagerVersion2(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataTransferManagerWithDivision>>(&m_Interface)));
    }

    Result SaveDataTransferManagerVersion2::GetChallenge(Challenge* outValue) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->GetChallenge(sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(Challenge))));
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManagerVersion2::SetKeySeedPackage(const Token& token) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->SetKeySeedPackage(sf::InBuffer(reinterpret_cast<const char*>(&token), sizeof(Token))));
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManagerVersion2::OpenSaveDataFullExporter(std::unique_ptr<ISaveDataDivisionExporter>* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionExporter> pExporterInterface;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataExporter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataDivisionExporter>>(&pExporterInterface), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
        std::unique_ptr<SaveDataExporterVersion2> pExporter(new SaveDataExporterVersion2(std::move(pExporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pExporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pExporter);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManagerVersion2::OpenSaveDataDiffExporter(std::unique_ptr<ISaveDataDivisionExporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionExporter> pExporterInterface;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataExporterForDiffExport(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataDivisionExporter>>(&pExporterInterface), sf::InBuffer(reinterpret_cast<const char*>(&initialData), sizeof(InitialDataVersion2)), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
        std::unique_ptr<SaveDataExporterVersion2> pExporter(new SaveDataExporterVersion2(std::move(pExporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pExporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pExporter);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManagerVersion2::OpenSaveDataFullImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, const fs::UserId& userId, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter> pImporterInterface;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataImporter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter>>(&pImporterInterface), sf::InBuffer(reinterpret_cast<const char*>(&initialData), sizeof(InitialDataVersion2)), userId, static_cast<uint8_t>(saveDataSpaceId)));
        std::unique_ptr<SaveDataImporterVersion2> pImporter(new SaveDataImporterVersion2(std::move(pImporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pImporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pImporter);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManagerVersion2::OpenSaveDataDiffImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter> pImporterInterface;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataImporterForDiffImport(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter>>(&pImporterInterface), sf::InBuffer(reinterpret_cast<const char*>(&initialData), sizeof(InitialDataVersion2)), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
        std::unique_ptr<SaveDataImporterVersion2> pImporter(new SaveDataImporterVersion2(std::move(pImporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pImporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pImporter);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManagerVersion2::OpenSaveDataDuplicateDiffImporter(std::unique_ptr<ISaveDataDivisionImporter>* outValue, const InitialDataVersion2& initialData, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter> pImporterInterface;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataImporterForDuplicateDiffImport(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataDivisionImporter>>(&pImporterInterface), sf::InBuffer(reinterpret_cast<const char*>(&initialData), sizeof(InitialDataVersion2)), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
        std::unique_ptr<SaveDataImporterVersion2> pImporter(new SaveDataImporterVersion2(std::move(pImporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pImporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pImporter);
        NN_RESULT_SUCCESS;
    }

    Result OpenSaveDataExportProhibiter(std::unique_ptr<SaveDataExportProhibiter>* outValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(OpenSaveDataTransferProhibiterForCloudBackUp(outValue, applicationId));
        NN_RESULT_SUCCESS;
    }

    Result OpenSaveDataExportProhibiter(std::unique_ptr<SaveDataExportProhibiter> outValue[], nn::ncm::ApplicationId applicationIdList[], int count) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(OpenSaveDataTransferProhibiterForCloudBackUp(outValue, applicationIdList, count));
        NN_RESULT_SUCCESS;
    }

    Result OpenSaveDataTransferProhibiterForCloudBackUp(std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp>* outValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();

        sf::SharedPointer<fssrv::sf::ISaveDataTransferProhibiter> pInterface;
        NN_FS_RESULT_DO(fileSystemProxy->OpenSaveDataTransferProhibiter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataTransferProhibiter>>(&pInterface), applicationId));

        std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> pProhibiter(new SaveDataTransferProhibiterForCloudBackUp(std::move(pInterface)));
        NN_FS_RESULT_THROW_UNLESS(pProhibiter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pProhibiter);
        NN_RESULT_SUCCESS;
    }

    Result OpenSaveDataTransferProhibiterForCloudBackUp(std::unique_ptr<SaveDataTransferProhibiterForCloudBackUp> outValue[], nn::ncm::ApplicationId applicationIdList[], int count) NN_NOEXCEPT
    {
        for (int i = 0; i < count; i++)
        {
            NN_RESULT_DO(OpenSaveDataTransferProhibiterForCloudBackUp(&outValue[i], applicationIdList[i]));
        }
        NN_RESULT_SUCCESS;
    }

    Result ListApplicationAccessibleSaveDataOwnerId(int* outValue, nn::ncm::ApplicationId outList[], nn::ncm::ApplicationId applicationId, int programIndex, int offset, int count) NN_NOEXCEPT
    {
        NN_FS_RESULT_THROW_UNLESS(outList != nullptr, ResultNullptrArgument());
        if (count == 0)
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }
        sf::SharedPointer<fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_FS_RESULT_DO(fileSystemProxy->ListAccessibleSaveDataOwnerId(sf::Out<int32_t>(outValue), sf::OutBuffer(reinterpret_cast<char*>(outList), sizeof(nn::ncm::ApplicationId) * count), { applicationId.value + programIndex }, offset, count));
        NN_RESULT_SUCCESS;
    }

    Result GetCountOfApplicationAccessibleSaveDataOwnerId(int* outValue, nn::ncm::ApplicationId applicationId, int programIndex) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::ncm::ApplicationId tmpAppId = { 0 };
        NN_FS_RESULT_DO(fileSystemProxy->ListAccessibleSaveDataOwnerId(sf::Out<int32_t>(outValue), sf::OutBuffer(reinterpret_cast<char*>(&tmpAppId), sizeof(nn::ncm::ApplicationId)), { applicationId.value + programIndex }, 0, 0));
        NN_RESULT_SUCCESS;
    }

}} // nn::fs
