﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fssrv/sf/fssrv_ISaveDataTransfer.h>

#include "fs_FileSystemProxyServiceObject.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

namespace
{
    const int64_t MarginSizePerFile                    = 16 * 1024; // クラスタサイズ切り上げ用マージン
    const int64_t MarginForSaveDataVersionUpFromV4ToV5 = 64 * 1024; // v5 化用マージン
    const int64_t MarginForSaveDataDirectory           = (16 * 1024 * 2); // save ディレクトリ、saveMeta ディレクトリのディレクトリクラスタ伸長用マージン
}

    SaveDataTransferManager::SaveDataTransferManager() NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->OpenSaveDataTransferManager(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataTransferManager>>(&m_Interface)));
    }

    Result SaveDataTransferManager::GetChallenge(void* outValue, size_t size) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->GetChallenge(sf::OutBuffer(static_cast<char*>(outValue), size)));
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManager::SetToken(const void* saveDataTransferToken, size_t size) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->SetToken(sf::InBuffer(static_cast<const char*>(saveDataTransferToken), size)));
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManager::OpenSaveDataExporter(std::unique_ptr<SaveDataExporter>* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataExporter> pExporterInterface;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataExporter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataExporter>>(&pExporterInterface), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
        std::unique_ptr<SaveDataExporter> pExporter(new SaveDataExporter(std::move(pExporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pExporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pExporter);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataTransferManager::OpenSaveDataImporter(std::unique_ptr<SaveDataImporter>* outValue, int64_t* outRequiredSize, const void* exportInitialData, size_t exportInitialDataSize, const fs::UserId& userId, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::ISaveDataImporter> pImporterInterface;
        int64_t requiredSize = 0;
        NN_FS_RESULT_DO(m_Interface->OpenSaveDataImporter(sf::Out<sf::SharedPointer<fssrv::sf::ISaveDataImporter>>(&pImporterInterface), sf::Out<int64_t>(&requiredSize), sf::InBuffer(static_cast<const char*>(exportInitialData), exportInitialDataSize), userId, static_cast<uint8_t>(saveDataSpaceId)));
        if (requiredSize != 0)
        {
            *outRequiredSize = requiredSize;
            return ResultUsableSpaceNotEnoughForSaveData();
        }
        std::unique_ptr<SaveDataImporter> pImporter(new SaveDataImporter(std::move(pImporterInterface)));
        NN_FS_RESULT_THROW_UNLESS(pImporter != nullptr, ResultAllocationMemoryFailedNew());
        *outValue = std::move(pImporter);
        NN_RESULT_SUCCESS;
    }

    int64_t SaveDataTransferSizeCalculator::QuerySaveDataExportSize(const SaveDataInfo& info) NN_NOEXCEPT
    {
        int64_t size;
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(QuerySaveDataInternalStorageTotalSize(&size, info.saveDataSpaceId, info.saveDataId));
        return size + (info.saveDataType == SaveDataType::Account ? detail::ThumbnailFileSize : 0) + crypto::Aes128GcmEncryptor::MacSize;
    }

    int64_t SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(const SaveDataInfo& info) NN_NOEXCEPT
    {
        return info.saveDataSize + MarginSizePerFile
            + (info.saveDataType == SaveDataType::Account ? detail::ThumbnailFileSize + MarginSizePerFile : 0)
            + MarginForSaveDataVersionUpFromV4ToV5
            + MarginForSaveDataDirectory;
    }

    int64_t SaveDataTransferSizeCalculator::QuerySaveDataExportSize(const SaveDataInfo* info, uint32_t count) NN_NOEXCEPT
    {
        int64_t retSize = 0;
        for (uint32_t i = 0; i < count; i++)
        {
            retSize += QuerySaveDataExportSize(info[i]);
        }
        return retSize;
    }

    int64_t SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(const SaveDataInfo* info, uint32_t count) NN_NOEXCEPT
    {
        int64_t retSize = 0;
        for (uint32_t i = 0; i < count; i++)
        {
            retSize += QuerySaveDataRequiredSizeForImport(info[i]);
        }
        return retSize;
    }

    int64_t SaveDataTransferSizeCalculator::GetFreeSpaceSize(SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        sf::SharedPointer<fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        int64_t freeSpaceSize = 0;
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->GetFreeSpaceSizeForSaveData(sf::Out<int64_t>(&freeSpaceSize), static_cast<uint8_t>(saveDataSpaceId)));
        return freeSpaceSize;
    }

    SaveDataExporter::SaveDataExporter(nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataExporter>&& interface) NN_NOEXCEPT : m_Interface(std::move(interface))
    {
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetSaveDataInfo(sf::Out<SaveDataInfo>(&m_SaveDataInfo)));
    }

    const SaveDataInfo& SaveDataExporter::GetSaveDataInfo() NN_NOEXCEPT
    {
        return m_SaveDataInfo;
    }

    Result SaveDataExporter::PullInitialData(void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->PullInitialData(sf::OutBuffer(static_cast<char*>(buffer), size)));
        NN_RESULT_SUCCESS;
    }

    Result SaveDataExporter::Pull(size_t* outPulledSize, void* buffer, size_t size) NN_NOEXCEPT
    {
        uint64_t pulledSize;
        NN_FS_RESULT_DO(m_Interface->Pull(sf::Out<uint64_t>(&pulledSize), sf::OutBuffer(static_cast<char*>(buffer), size)));
        *outPulledSize = static_cast<size_t>(pulledSize);
        NN_RESULT_SUCCESS;
    }

    int64_t SaveDataExporter::GetRestSize() NN_NOEXCEPT
    {
        uint64_t restSize;
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetRestSize(sf::Out<uint64_t>(&restSize)));
        return static_cast<int64_t>(restSize);
    }

    SaveDataImporter::SaveDataImporter(nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataImporter>&& interface) NN_NOEXCEPT : m_Interface(std::move(interface))
    {
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetSaveDataInfo(sf::Out<SaveDataInfo>(&m_SaveDataInfo)));
    }

    Result SaveDataImporter::Finalize() NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->Finalize());
        NN_RESULT_SUCCESS;
    }

    const SaveDataInfo& SaveDataImporter::GetSaveDataInfo() NN_NOEXCEPT
    {
        return m_SaveDataInfo;
    }

    Result SaveDataImporter::Push(const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_FS_RESULT_DO(m_Interface->Push(sf::InBuffer(static_cast<const char*>(buffer), size)));
        NN_RESULT_SUCCESS;
    }

    size_t SaveDataImporter::GetRestSize() NN_NOEXCEPT
    {
        uint64_t restSize;
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetRestSize(sf::Out<uint64_t>(&restSize)));
        return static_cast<size_t>(restSize);
    }


}}

