﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <shared_mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include <nn/os/os_ReaderWriterLock.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fs_FileDataCache.h>
#include <nn/fs/fs_PathTool.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include "../fsa/fs_FileAccessor.h"
#include "../fsa/fs_FileSystemAccessor.h"
#include "../fsa/fs_MountUtility.h"
#include "fs_PathBasedFileDataCache.h"
#include "fs_FileDataCacheAccessor.h"

namespace nn { namespace fs {

namespace detail {

namespace {

    PathBasedFileDataCache& GetPathBasedFileDataCacheImpl() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(PathBasedFileDataCache, s_PathBasedFileDataCache);
        return s_PathBasedFileDataCache;
    }

    Result CalculateHashByDjb2a(FilePathHash* outValue, const char* path) NN_NOEXCEPT
    {
        std::memset(outValue->data, 0, FilePathHashSize);

        uint32_t hash = 5381UL;
        int count = 0;
        uint8_t c = path[0];

        while (c != '\0')
        {
            hash = (((hash << 5) + hash) ^ c) & 0xFFFFFFFF; /* hash * 33 ^ c */
            if (count == EntryNameLengthMax)
            {
                break;
            }
            c = path[++count];
        }

        std::memcpy(outValue->data, &hash, sizeof(hash));
        NN_RESULT_SUCCESS;
    }

    Result GetNormalizedPath(std::unique_ptr<char[], Deleter>* outPath, size_t *outLength, const char* path) NN_NOEXCEPT
    {
        int pathLength = util::Strnlen(path, fs::EntryNameLengthMax);
        size_t pathBufferSize = util::align_up(pathLength + 1, 8);
        auto normalizedPath = detail::MakeUnique<char[]>(pathBufferSize);
        NN_RESULT_THROW_UNLESS(normalizedPath != nullptr, ResultAllocationMemoryFailedNew());
        size_t normalizedLength = 0;
        NN_RESULT_DO(PathTool::Normalize(normalizedPath.get(), &normalizedLength, path, pathBufferSize));
        *outPath = std::move(normalizedPath);
        *outLength = normalizedLength;
        NN_RESULT_SUCCESS;
    }

}

    PathBasedFileDataCache::PathBasedFileDataCache() NN_NOEXCEPT
        : m_CalculateHashFunc(CalculateHashByDjb2a)
    {
    }

    PathBasedFileDataCache::~PathBasedFileDataCache() NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedLock(m_ListLock);
        while (!m_List.empty())
        {
            auto entry = &(*m_List.rbegin());
            m_List.erase(m_List.iterator_to(*entry));
            delete entry;
        }
    }

    Result PathBasedFileDataCache::Read(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, size_t* outValue, int64_t offset, void* pBuffer, size_t size, const ReadOption& option, FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT
    {
        // nn::fs::fsa::IFile::Read
        NN_FSP_REQUIRES(outValue != nullptr, ResultNullptrArgument());
        if( size == 0 )
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }
        NN_FSP_REQUIRES(pBuffer != nullptr, ResultNullptrArgument());
        NN_FSP_REQUIRES(offset >= 0, ResultOutOfRange());
        int64_t castedSize = static_cast<int64_t>(size);
        NN_FSP_REQUIRES(castedSize >= 0, ResultOutOfRange());
        NN_FSP_REQUIRES((INT64_MAX - offset) >= castedSize, ResultOutOfRange());

        // nn::fs::Fsa::IFile::DryRead
        NN_FSP_REQUIRES((fileOpenMode & static_cast<int>(nn::fs::OpenMode::OpenMode_Read)) != 0, nn::fs::ResultInvalidOperationForOpenMode());

        {
            std::shared_lock<os::ReaderWriterLock> scopedLock(m_ListLock);
            for (auto& entry : m_List)
            {
                if (entry.hash == hash && entry.hashIndex == hashIndex && entry.pFileSystemAccessor == pFileSystemAccessor)
                {
                    NN_FSP_REQUIRES(offset <= static_cast<int64_t>(entry.fileSize), ResultOutOfRange());

                    int64_t restSize = std::max(static_cast<int64_t>(entry.fileSize) - offset, static_cast<int64_t>(0));
                    int64_t readSize = std::min(static_cast<int64_t>(size), restSize);
                    auto pSrc = reinterpret_cast<char*>(entry.pBuffer) + offset;

                    if (entry.modifyLock.TryAcquireReadLock())
                    {
                        NN_UTIL_SCOPE_EXIT
                        {
                            entry.modifyLock.ReleaseReadLock();
                        };

                        std::memcpy(pBuffer, pSrc, static_cast<size_t>(readSize));
                        *outValue = static_cast<size_t>(readSize);
                        pAccessResult->SetFileDataCacheUsed(true);
                        NN_RESULT_SUCCESS;
                    }
                    else
                    {
                        break;
                    }
                }
            }
        }

        pAccessResult->SetFileDataCacheUsed(false);
        return pFile->Read(outValue, offset, pBuffer, size, option);
    }

    Result PathBasedFileDataCache::Write(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, int64_t offset, const void* pBuffer, size_t size, const WriteOption& option) NN_NOEXCEPT
    {
        auto writeImpl = [&]() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(pFile->Write(offset, pBuffer, size, option));

            if (size == 0)
            {
                NN_RESULT_SUCCESS;
            }

            NN_FSP_REQUIRES((fileOpenMode & static_cast<int>(nn::fs::OpenMode::OpenMode_Write)) != 0, ResultInvalidOperationForOpenMode());

            std::shared_lock<os::ReaderWriterLock> scopedLock(m_ListLock);
            for (auto& entry : m_List)
            {
                if (entry.hash == hash && entry.hashIndex == hashIndex && entry.pFileSystemAccessor == pFileSystemAccessor)
                {
                    NN_FSP_REQUIRES(offset <= static_cast<int64_t>(entry.fileSize), ResultOutOfRange());
                    NN_FSP_REQUIRES(offset + static_cast<int64_t>(size) <= static_cast<int64_t>(entry.fileSize), ResultOutOfRange());
                    auto pDst = reinterpret_cast<char*>(entry.pBuffer) + offset;

                    std::lock_guard<os::ReaderWriterLock> writeLock(entry.modifyLock);
                    std::memcpy(pDst, pBuffer, size);
                    NN_RESULT_SUCCESS;
                }
            }

            NN_RESULT_SUCCESS;
        };

        NN_RESULT_TRY(writeImpl())
            NN_RESULT_CATCH_ALL
            {
                // Write に失敗した場合 FileAccessor が操作を受け付けなくなるが、念のためキャッシュは Invalidate しておく
                Remove(pFileSystemAccessor, hash, hashIndex);
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    bool PathBasedFileDataCache::Add(std::unique_ptr<PathBasedFileDataCacheEntry>&& newEntry) NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedLock(m_ListLock);
        for (auto& entry : m_List)
        {
            if (entry.hash == newEntry->hash && entry.hashIndex == newEntry->hashIndex && entry.pFileSystemAccessor == newEntry->pFileSystemAccessor)
            {
                return false;
            }
        }
        m_List.push_back(*(newEntry.release()));
        return true;
    }

    UniqueLock PathBasedFileDataCache::Lock() NN_NOEXCEPT
    {
        return UniqueLock(m_ListLock);
    }

    UniqueLock PathBasedFileDataCache::Lock(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT
    {
        std::shared_lock<os::ReaderWriterLock> scopedLock(m_ListLock);
        for (auto& entry : m_List)
        {
            if (entry.hash == hash && entry.pFileSystemAccessor == pFileSystemAccessor && util::Strncmp(path, &(entry.path[0]), entry.pathLength) == 0)
            {
                return UniqueLock(entry.modifyLock);
            }
        }
        return UniqueLock();
    }

    void PathBasedFileDataCache::Remove(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT
    {
        auto scopedLock = std::move(lock);
        for (auto it = m_List.begin(); it != m_List.end(); )
        {
            auto& entry = *it;
            if (entry.pFileSystemAccessor == pFileSystemAccessor)
            {
                it = m_List.erase(it);
                {
                    std::lock_guard<os::ReaderWriterLock> writeLock(entry.modifyLock);
                }
                delete &entry;
            }
            else
            {
                it++;
            }
        }
    }

    void PathBasedFileDataCache::Remove(FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT
    {
        return Remove(Lock(), pFileSystemAccessor);
    }

    bool PathBasedFileDataCache::Remove(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex) NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedLock(m_ListLock);
        for (auto& entry : m_List)
        {
            if (entry.hash == hash && entry.hashIndex == hashIndex && entry.pFileSystemAccessor == pFileSystemAccessor)
            {
                m_List.erase(m_List.iterator_to(entry));
                {
                    std::lock_guard<os::ReaderWriterLock> writeLock(entry.modifyLock);
                }
                delete &entry;
                return true;
            }
        }
        return false;
    }

    bool PathBasedFileDataCache::Remove(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedLock(m_ListLock);
        for (auto& entry : m_List)
        {
            if (entry.hash == hash && entry.pFileSystemAccessor == pFileSystemAccessor && util::Strncmp(path, &(entry.path[0]), entry.pathLength) == 0)
            {
                m_List.erase(m_List.iterator_to(entry));
                if (lock.Get() == &(entry.modifyLock))
                {
                    lock.Detach();
                }
                delete &entry;
                return true;
            }
        }
        return false;
    }

    bool PathBasedFileDataCache::Remove(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedLock(m_ListLock);
        for (auto& entry : m_List)
        {
            if (entry.hash == hash && entry.pFileSystemAccessor == pFileSystemAccessor && util::Strncmp(path, &(entry.path[0]), entry.pathLength) == 0)
            {
                m_List.erase(m_List.iterator_to(entry));
                {
                    std::lock_guard<os::ReaderWriterLock> writeLock(entry.modifyLock);
                }
                delete &entry;
                return true;
            }
        }
        return false;
    }

    // path が完全一致するエントリがあった場合はその hashIndex を格納し true を返す。
    // 無ければ hash が一致した中で最大の hashIndex に 1 加算した値を格納し false を返す。
    bool PathBasedFileDataCache::FindHashIndexOrNextImpl(int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT
    {
        int hashIndexMax = 0;
        for (auto& entry : m_List)
        {
            if (entry.hash == hash && entry.pFileSystemAccessor == pFileSystemAccessor)
            {
                if (util::Strncmp(path, &(entry.path[0]), entry.pathLength) == 0)
                {
                    *outHashIndex = entry.hashIndex;
                    return true;
                }
                else
                {
                    hashIndexMax = std::max(entry.hashIndex, hashIndexMax);
                }
            }
        }
        *outHashIndex = hashIndexMax + 1;
        return false;
    }

    bool PathBasedFileDataCache::FindHashIndexOrNext(int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedLock(m_ListLock);
        return FindHashIndexOrNextImpl(outHashIndex, pFileSystemAccessor, hash, path);
    }

    bool PathBasedFileDataCache::FindHashIndex(int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, const char* path) NN_NOEXCEPT
    {
        std::shared_lock<os::ReaderWriterLock> scopedLock(m_ListLock);
        return FindHashIndexOrNextImpl(outHashIndex, pFileSystemAccessor, hash, path);
    }

    void PathBasedFileDataCache::SetDefaultCalculateHashFunc() NN_NOEXCEPT // for test
    {
        m_CalculateHashFunc = CalculateHashByDjb2a;
    }

    void PathBasedFileDataCache::SetCalculateHashFunc(CalculateHashFunc func) NN_NOEXCEPT // for test
    {
        m_CalculateHashFunc = func;
    }

    Result GetFilePathHash(FilePathHash* outValue, const char* path) NN_NOEXCEPT
    {
        // TODO: 各 FS の実装にしたい
        bool isNormalized = false;
        NN_RESULT_DO(PathTool::IsNormalized(&isNormalized, path));
        NN_RESULT_THROW_UNLESS(isNormalized, ResultNotNormalized()); // 実装ミス以外で返らない想定
        return GetPathBasedFileDataCacheImpl().CalculateHash(outValue, path);
    }

    void SetDefaultCalculateHashFunc() NN_NOEXCEPT // for test
    {
        return GetPathBasedFileDataCacheImpl().SetDefaultCalculateHashFunc();
    }

    void SetCalculateHashFunc(CalculateHashFunc func) NN_NOEXCEPT // for test
    {
        return GetPathBasedFileDataCacheImpl().SetCalculateHashFunc(func);
    }

    UniqueLock LockPathBasedFileDataCacheEntries() NN_NOEXCEPT
    {
        return GetPathBasedFileDataCacheImpl().Lock();
    }

    void InvalidatePathBasedFileDataCacheEntries(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT
    {
        GetPathBasedFileDataCacheImpl().Remove(std::move(lock), pFileSystemAccessor);
    }

    void InvalidatePathBasedFileDataCacheEntries(FileSystemAccessor* pFileSystemAccessor) NN_NOEXCEPT
    {
        GetPathBasedFileDataCacheImpl().Remove(pFileSystemAccessor);
    }

    void InvalidatePathBasedFileDataCacheEntry(FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex) NN_NOEXCEPT
    {
        GetPathBasedFileDataCacheImpl().Remove(pFileSystemAccessor, hash, hashIndex);
    }

    UniqueLock LockPathBasedFileDataCacheEntry(FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT
    {
        FilePathHash hash;
        size_t copiedPathLength = 0;
        std::unique_ptr<char[], Deleter> copiedPath;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetNormalizedPath(&copiedPath, &copiedPathLength, path));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetFilePathHash(&hash, &(copiedPath[0])));
        return GetPathBasedFileDataCacheImpl().Lock(pFileSystemAccessor, hash, &(copiedPath[0]));
    }

    void InvalidatePathBasedFileDataCacheEntry(UniqueLock&& lock, FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT
    {
        FilePathHash hash;
        size_t copiedPathLength = 0;
        std::unique_ptr<char[], Deleter> copiedPath;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetNormalizedPath(&copiedPath, &copiedPathLength, path));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetFilePathHash(&hash, &(copiedPath[0])));
        GetPathBasedFileDataCacheImpl().Remove(std::move(lock), pFileSystemAccessor, hash, &(copiedPath[0]));
    }

    void InvalidatePathBasedFileDataCacheEntry(FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT
    {
        FilePathHash hash;
        size_t copiedPathLength = 0;
        std::unique_ptr<char[], Deleter> copiedPath;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetNormalizedPath(&copiedPath, &copiedPathLength, path));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetFilePathHash(&hash, &(copiedPath[0])));
        GetPathBasedFileDataCacheImpl().Remove(pFileSystemAccessor, hash, &(copiedPath[0]));
    }

    bool FindPathBasedFileDataCacheEntry(FilePathHash* outValue, int* outHashIndex, FileSystemAccessor* pFileSystemAccessor, const char* path) NN_NOEXCEPT
    {
        size_t copiedPathLength = 0;
        std::unique_ptr<char[], Deleter> copiedPath;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetNormalizedPath(&copiedPath, &copiedPathLength, path));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetFilePathHash(outValue, &(copiedPath[0])));
        return GetPathBasedFileDataCacheImpl().FindHashIndex(outHashIndex, pFileSystemAccessor, *outValue, &(copiedPath[0]));
    }

    Result ReadViaPathBasedFileDataCache(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, size_t* outValue, int64_t offset, void* pBuffer, size_t size, const ReadOption& option, FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT
    {
        return GetPathBasedFileDataCacheImpl().Read(pFile, fileOpenMode, pFileSystemAccessor, hash, hashIndex, outValue, offset, pBuffer, size, option, pAccessResult);
    }

    Result WriteViaPathBasedFileDataCache(fsa::IFile* pFile, int fileOpenMode, FileSystemAccessor* pFileSystemAccessor, const FilePathHash& hash, int hashIndex, int64_t offset, const void* pBuffer, size_t size, const WriteOption& option) NN_NOEXCEPT
    {
        return GetPathBasedFileDataCacheImpl().Write(pFile, fileOpenMode, pFileSystemAccessor, hash, hashIndex, offset, pBuffer, size, option);
    }

}

Result EnableIndividualFileDataCache(const char* path, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
        detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    int64_t fileSize = 0;
    auto enableImpl = [&]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_THROW_UNLESS(pBuffer != nullptr, ResultNullptrArgument());

        if (!system->IsPathBasedFileDataCacheAttachable())
        {
            NN_RESULT_SUCCESS;
        }

        std::unique_ptr<detail::PathBasedFileDataCacheEntry> entry(new detail::PathBasedFileDataCacheEntry());
        NN_RESULT_THROW_UNLESS(entry != nullptr, ResultAllocationMemoryFailedNew());

        auto subPath = detail::GetSubPath(path);

        entry->pBuffer = pBuffer;
        entry->pFileSystemAccessor = system;

        {
            std::unique_ptr<detail::FileAccessor> file;
            NN_RESULT_DO(system->OpenFile(&file, subPath, OpenMode::OpenMode_Read));

            {
                int64_t size = 0;
                NN_RESULT_DO(file->GetSize(&size));
                fileSize = size;
            }
            NN_RESULT_THROW_UNLESS(fileSize <= static_cast<int64_t>(bufferSize), ResultInvalidSize());

            entry->fileSize = static_cast<size_t>(fileSize);

            size_t readSize;
            NN_RESULT_DO(file->ReadWithoutCacheAccessLog(&readSize, 0, entry->pBuffer, entry->fileSize, ReadOption::MakeValue(0)));
        }

        {
            size_t copiedPathLength = 0;
            std::unique_ptr<char[], detail::Deleter> copiedPath;
            NN_RESULT_DO(detail::GetNormalizedPath(&copiedPath, &copiedPathLength, subPath));
            entry->pathLength = static_cast<int>(copiedPathLength);
            entry->path = std::move(copiedPath);
        }

        NN_RESULT_DO(GetFilePathHash(&(entry->hash), &(entry->path[0])));

        if (detail::GetPathBasedFileDataCacheImpl().FindHashIndexOrNext(&(entry->hashIndex), entry->pFileSystemAccessor, entry->hash, &(entry->path[0])))
        {
            return fs::ResultIndividualFileDataCacheAlreadyEnabled();
        }

        if (detail::GetPathBasedFileDataCacheImpl().Add(std::move(entry)))
        {
            system->AttachPathBasedFileDataCache();
        }
        else
        {
            return fs::ResultIndividualFileDataCacheAlreadyEnabled();
        }

        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(
            enableImpl(),
            nullptr,
            system,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_ENABLEINDIVIDUALFILEDATACACHE(path, bufferSize, fileSize)
        )
    );
    NN_RESULT_SUCCESS;
}

void DisableIndividualFileDataCache(const char* path) NN_NOEXCEPT
{
    detail::FileSystemAccessor* system;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM_FOR_FINDFILESYSTEM(
            detail::FindFileSystem(&system, path), NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path));

    auto disableImpl = [&]() NN_NOEXCEPT -> Result
    {
        if (!system->IsPathBasedFileDataCacheAttachable())
        {
            NN_RESULT_SUCCESS;
        }

        size_t copiedPathLength = 0;
        std::unique_ptr<char[], detail::Deleter> copiedPath;
        NN_RESULT_DO(detail::GetNormalizedPath(&copiedPath, &copiedPathLength, detail::GetSubPath(path)));

        detail::FilePathHash hash;
        NN_RESULT_DO(GetFilePathHash(&hash, &(copiedPath[0])));

        detail::GetPathBasedFileDataCacheImpl().Remove(system, hash, &(copiedPath[0]));

        NN_RESULT_SUCCESS;
    };

    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_FILESYSTEM(
            disableImpl(),
            nullptr,
            system,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, path
        )
    );
}

}}
