﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/os/os_Mutex.h>
#include "fs_FileSystemCreatorInterfaces.h"
#include "fsa/fs_UserMountTable.h"

namespace nn { namespace fs {


namespace {

    bool g_UsedDefaultAllocator = false;

    void* DefaultAllocate(size_t size) NN_NOEXCEPT
    {
        g_UsedDefaultAllocator = true;
        auto p = malloc(size);
        // auto p = ::operator new(size, std::nothrow);
        if( p == nullptr )
        {
            NN_DETAIL_FS_ERROR(
                "[fs] malloc() returned nullptr. "
                "Refer an API reference of nn::fs::SetAllocator(). \n"
                );
        }
        return p;
    }

    void DefaultDeallocate(void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        free(p);
        // ::operator delete(p);
    }

    AllocateFunction s_AllocateFunction = DefaultAllocate;
    DeallocateFunction s_DeallocateFunction = DefaultDeallocate;

    const char* const ALLOCATE_ASSERT_MESSAGE NN_IS_UNUSED_MEMBER =
                          "nn::fs::SetAllocator() must be called before any filesystem operation.";

    const int RequiredAlignment = 8;
    const char* const ALIGNMENT_ABORT_MESSAGE NN_IS_UNUSED_MEMBER =
                          "nn::fs::SetAllocator() must set an allocator that returns 8-byte aligned addresses.";

    nn::os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
}

    void SetAllocator(AllocateFunction alloc, DeallocateFunction dealloc) NN_NOEXCEPT
    {
        auto SetAllocatorImpl = [=]() NN_NOEXCEPT -> Result
        {
            if( (s_AllocateFunction != DefaultAllocate) || (s_DeallocateFunction != DefaultDeallocate) )
            {
                NN_DETAIL_FS_ERROR("nn::fs::SetAllocator() cannot be called more than once");
            }
            NN_RESULT_THROW_UNLESS(s_AllocateFunction == DefaultAllocate,     ResultAllocatorAlreadyRegistered());
            NN_RESULT_THROW_UNLESS(s_DeallocateFunction == DefaultDeallocate, ResultAllocatorAlreadyRegistered());
            NN_RESULT_THROW_UNLESS(alloc != nullptr,   ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(dealloc != nullptr, ResultNullptrArgument());

            if( g_UsedDefaultAllocator )
            {
                NN_DETAIL_FS_ERROR(
                    "[fs] Default allocator has been used before nn::fs::SetAllocator(). "
                    "It's strongly recommended to invoke nn::fs::SetAllocator() in nninitStartup(). "
                );
            }
            NN_RESULT_THROW_UNLESS(!g_UsedDefaultAllocator, ResultDefaultAllocatorAlreadyUsed());

            s_AllocateFunction = alloc;
            s_DeallocateFunction = dealloc;
            NN_RESULT_SUCCESS;
        };
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(NN_DETAIL_FS_ACCESS_LOG(SetAllocatorImpl(), nullptr, ""));
    }

namespace detail {

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_AllocateFunction, ALLOCATE_ASSERT_MESSAGE);
        nn::os::LockMutex(&g_Mutex);
        void* p = s_AllocateFunction(size);
        if( !nn::util::is_aligned(reinterpret_cast<uintptr_t>(p), RequiredAlignment) )
        {
            NN_DETAIL_FS_ERROR("%s", ALIGNMENT_ABORT_MESSAGE);
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::ResultAllocatorAlignmentViolation() );
        }
        nn::os::UnlockMutex(&g_Mutex);
        return p;
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(s_DeallocateFunction, ALLOCATE_ASSERT_MESSAGE);
        if( p == nullptr )
        {
            return;
        }
        nn::os::LockMutex(&g_Mutex);
        s_DeallocateFunction(p, size);
        nn::os::UnlockMutex(&g_Mutex);
    }
}

}}
