﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_ReaderWriterLock.h>
#include <nn/os/os_Tick.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_BufferRegion.h>
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/detail/fs_LruFileDataCacheSystem.h>
#include <nn/fs/detail/fs_LruList.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>


// デバッグ用  !!! 1 のままコミットしないこと !!!
#define NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED   0  // NOLINT(preprocessor/const)
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
#if defined(NN_SDK_BUILD_RELEASE)
#define NN_DETAIL_ENABLE_SDK_LOG
#endif
#define NN_SDK_LOG_DEFAULT_MODULE_NAME "fs::fdc"
#include <nn/nn_SdkLog.h>
#define NN_FS_DETAIL_CACHE_SYSTEM_LOG NN_SDK_LOG
#else
#define NN_FS_DETAIL_CACHE_SYSTEM_LOG(...)
#endif

// 性能評価用  !!! 1 のままコミットしないこと !!!
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED  0  // NOLINT(preprocessor/const)
#if NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(tag) \
    os::Tick timeMeasurementStartTick_ ## tag = os::GetSystemTick()
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(tag) \
    os::Tick timeMeasurementEndTick_ ## tag = os::GetSystemTick()
#define NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(tag) \
    (timeMeasurementEndTick_ ## tag - timeMeasurementStartTick_ ## tag).ToTimeSpan()
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(expression) \
    (expression)
#else
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(tag)
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(tag)
#define NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(tag)
#define NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(expression)
#endif

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
#define NN_FS_DETAIL_CACHE_SYSTEM_ASSERT_CHECK  1  // NOLINT(preprocessor/const)
#else
#define NN_FS_DETAIL_CACHE_SYSTEM_ASSERT_CHECK  0  // NOLINT(preprocessor/const)
#endif


namespace nn { namespace fs { namespace detail {

namespace
{

class ScopedSharedLock
{
private:
    os::ReaderWriterLock* m_pReaderWriterLock;
public:
    explicit ScopedSharedLock(os::ReaderWriterLock* pReaderWriterLock) NN_NOEXCEPT
        : m_pReaderWriterLock(pReaderWriterLock)
    {
        m_pReaderWriterLock->AcquireReadLock();
    }
    ~ScopedSharedLock() NN_NOEXCEPT
    {
        m_pReaderWriterLock->ReleaseReadLock();
    }
};

class ScopedExclusiveLock
{
private:
    os::ReaderWriterLock* m_pReaderWriterLock;
public:
    explicit ScopedExclusiveLock(os::ReaderWriterLock* pReaderWriterLock) NN_NOEXCEPT
        : m_pReaderWriterLock(pReaderWriterLock)
    {
        m_pReaderWriterLock->AcquireWriteLock();
    }
    ~ScopedExclusiveLock() NN_NOEXCEPT
    {
        m_pReaderWriterLock->ReleaseWriteLock();
    }
};

class ScopedUpgradedExclusiveLock
{
private:
    os::ReaderWriterLock* m_pReaderWriterLock;
public:
    explicit ScopedUpgradedExclusiveLock(os::ReaderWriterLock* pReaderWriterLock) NN_NOEXCEPT
        : m_pReaderWriterLock(pReaderWriterLock)
    {
        NN_SDK_ASSERT(m_pReaderWriterLock->IsReadLockHeld());
        m_pReaderWriterLock->ReleaseReadLock();
        m_pReaderWriterLock->AcquireWriteLock();
    }
    ~ScopedUpgradedExclusiveLock() NN_NOEXCEPT
    {
        m_pReaderWriterLock->ReleaseWriteLock();
        m_pReaderWriterLock->AcquireReadLock();
    }
};

class ScopedExclusiveLockRelease
{
private:
    os::ReaderWriterLock* m_pReaderWriterLock;
public:
    explicit ScopedExclusiveLockRelease(os::ReaderWriterLock* pReaderWriterLock) NN_NOEXCEPT
        : m_pReaderWriterLock(pReaderWriterLock)
    {
        m_pReaderWriterLock->ReleaseWriteLock();
    }
    ~ScopedExclusiveLockRelease() NN_NOEXCEPT
    {
        m_pReaderWriterLock->AcquireWriteLock();
    }
};

class RegionAttributedBuffer
{
private:
    Bit8* m_pBuffer;
    BufferRegion m_Region;

public:
    RegionAttributedBuffer(void* pBuffer, int64_t regionOffset, size_t regionSize) NN_NOEXCEPT
        : m_pBuffer(reinterpret_cast<Bit8*>(pBuffer))
        , m_Region(regionOffset, regionSize)
    {
    }
    RegionAttributedBuffer(void* pBuffer, const BufferRegion& region) NN_NOEXCEPT
        : m_pBuffer(reinterpret_cast<Bit8*>(pBuffer))
        , m_Region(region)
    {
    }

    void* GetBuffer() const NN_NOEXCEPT { return m_pBuffer; }
    int64_t GetOffset() const NN_NOEXCEPT { return m_Region.offset; }
    size_t GetSize() const NN_NOEXCEPT { return m_Region.size; }
    const BufferRegion& GetRegion() const NN_NOEXCEPT { return m_Region; }

    Result Read(IStorage* pStorage) NN_NOEXCEPT
    {
        NN_FS_DETAIL_CACHE_SYSTEM_LOG("<<< --- readfile: offset = %11lld, size = %10zd\n", m_Region.offset, m_Region.size);
        return pStorage->Read(m_Region.offset, m_pBuffer, m_Region.size);
    }

    Result Read(IStorage* pStorage, const BufferRegion& readRegion) NN_NOEXCEPT
    {
        const BufferRegion intersection = BufferRegion::GetIntersection(m_Region, readRegion);
        if (intersection.size == 0)
        {
            NN_RESULT_SUCCESS;
        }

        NN_FS_DETAIL_CACHE_SYSTEM_LOG("<<< --- readfile: offset = %11lld, size = %10zd\n", intersection.offset, intersection.size);
        return pStorage->Read(intersection.offset, m_pBuffer + (intersection.offset - m_Region.offset), intersection.size);
    }

    size_t CopyFrom(const RegionAttributedBuffer& other) NN_NOEXCEPT
    {
        const BufferRegion intersection = BufferRegion::GetIntersection(m_Region, other.m_Region);
        if (intersection.size == 0)
        {
            return 0;
        }

        Bit8* pSource = other.m_pBuffer + (intersection.offset - other.m_Region.offset);
        Bit8* pDestination = m_pBuffer + (intersection.offset - m_Region.offset);
        std::memcpy(pDestination, pSource, intersection.size);
        return intersection.size;
    }

    size_t CopyTo(RegionAttributedBuffer& other) const NN_NOEXCEPT
    {
        return other.CopyFrom(*this);
    }
};

}  // namespace unnamed


// この変数の参照を使用している箇所があるため定義が必須
const size_t LruFileDataCacheSystem::MaxReadAheadSize;

class LruFileDataCacheSystem::LruFileDataCacheSystemAccessResult
{
private:
    FileDataCacheAccessResult* m_pBaseAccessResult;
    BufferRegion m_BaseBufferRegion;

#if NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED
    // cache hit:
    TimeSpan m_PageSearchTime;
    TimeSpan m_CacheCopyTime;
    TimeSpan m_PromotePageTime;
    // cache miss:
    TimeSpan m_ReservePageTime;
    TimeSpan m_FileReadTime;
    TimeSpan m_CacheStoreTime;
    TimeSpan m_CommitPageTime;
#endif

public:
    LruFileDataCacheSystemAccessResult(FileDataCacheAccessResult* pBaseAccessResult, const BufferRegion& baseBufferRegion) NN_NOEXCEPT
        : m_pBaseAccessResult(pBaseAccessResult)
        , m_BaseBufferRegion(baseBufferRegion)
#if NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED
        , m_PageSearchTime(0)
        , m_CacheCopyTime(0)
        , m_PromotePageTime(0)
        , m_ReservePageTime(0)
        , m_FileReadTime(0)
        , m_CacheStoreTime(0)
        , m_CommitPageTime(0)
#endif
    {
        m_pBaseAccessResult->SetFileDataCacheUsed(true);
    }

    void AddCacheFetchedRegion(const BufferRegion& region) NN_NOEXCEPT
    {
        // ファイル領域の外は記録しない
        BufferRegion bufferRegion = region.GetIntersection(m_BaseBufferRegion);

        // オフセットを IStorage ベースからファイルベースへ変換
        bufferRegion.offset -= m_BaseBufferRegion.offset;

        m_pBaseAccessResult->AddCacheFetchedRegion(bufferRegion);
    }

#if NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED
    const TimeSpan& GetPageSearchTime() const NN_NOEXCEPT
    {
        return m_PageSearchTime;
    }
    const TimeSpan& GetCacheCopyTime() const NN_NOEXCEPT
    {
        return m_CacheCopyTime;
    }
    const TimeSpan& GetPromotePageTime() const NN_NOEXCEPT
    {
        return m_PromotePageTime;
    }
    const TimeSpan& GetReservePageTime() const NN_NOEXCEPT
    {
        return m_ReservePageTime;
    }
    const TimeSpan& GetFileReadTime() const NN_NOEXCEPT
    {
        return m_FileReadTime;
    }
    const TimeSpan& GetCacheStoreTime() const NN_NOEXCEPT
    {
        return m_CacheStoreTime;
    }
    const TimeSpan& GetCommitPageTime() const NN_NOEXCEPT
    {
        return m_CommitPageTime;
    }

    void AddPageSearchTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_PageSearchTime += time;
    }
    void AddCacheCopyTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_CacheCopyTime += time;
    }
    void AddPromotePageTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_PromotePageTime += time;
    }
    void AddReservePageTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_ReservePageTime += time;
    }
    void AddFileReadTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_FileReadTime += time;
    }
    void AddCacheStoreTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_CacheStoreTime += time;
    }
    void AddCommitPageTime(const TimeSpan& time) NN_NOEXCEPT
    {
        m_CommitPageTime += time;
    }
#endif
};

LruFileDataCacheSystem::LruFileDataCacheSystem() NN_NOEXCEPT
    : m_pCache(nullptr)
    , m_pPages(nullptr)
    , m_TotalPageCount(0)
    , m_FreePageCount(0)
    , m_ReservedPageCount(0)
    , m_CommitEvent(os::EventClearMode_AutoClear)
    , m_LastReadOffset(0)
    , m_ReadAheadSize(0)
    , m_ReadAheadLock(false)
    , m_LruListLock(false)
    , m_CacheOperationLock()
{
}

LruFileDataCacheSystem::~LruFileDataCacheSystem() NN_NOEXCEPT
{
    Finalize();
}

Result LruFileDataCacheSystem::Initialize(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pBuffer);

    NN_SDK_ASSERT(m_pCache == nullptr);

    size_t pageCount;
    size_t bucketCount;
    NN_RESULT_DO(CalculateAvailableCount(&pageCount, &bucketCount, bufferSize));

    uintptr_t p = reinterpret_cast<uintptr_t>(pBuffer);
    const uintptr_t nodeStart = util::align_up(p, NN_ALIGNOF(LruListType::NodeType));
    p = nodeStart + sizeof(LruListType::NodeType) * pageCount;
    p = util::align_up(p, NN_ALIGNOF(LruListType::BucketType));
    p = p + sizeof(LruListType::BucketType) * bucketCount;
    m_LruList.Initialize(reinterpret_cast<void*>(nodeStart), p - nodeStart, pageCount, bucketCount);

    p = util::align_up(p, NN_ALIGNOF(Page));
    m_pPages = reinterpret_cast<Page*>(p);
    p += sizeof(Page) * pageCount;

    p = util::align_up(p, NN_ALIGNOF(PageRegion));
    m_pPageRegions = reinterpret_cast<PageRegion*>(p);
    p += sizeof(PageRegion) * pageCount;

    p = util::align_up(p, os::MemoryPageSize);
    m_pCache = reinterpret_cast<void*>(p);
    p += PageSize * pageCount;

    if (p > reinterpret_cast<uintptr_t>(pBuffer) + bufferSize)
    {
        m_pCache = nullptr;
        m_pPageRegions = nullptr;
        m_pPages = nullptr;
        m_LruList.Finalize();
        return ResultFileDataCacheMemorySizeTooSmall();
    }

    for (size_t i = 0; i < pageCount; i++)
    {
        m_pPages[i].address = reinterpret_cast<void*>(reinterpret_cast<Bit8*>(m_pCache) + i * PageSize);
        m_pPages[i].reversePointer = &m_pPages[i];
    }

    for (size_t i = 0; i < pageCount; i++)
    {
        new (&m_pPageRegions[i]) PageRegion();
        m_UnusedPageRegionList.push_back(m_pPageRegions[i]);
    }

    PageRegion& freeRegion = m_UnusedPageRegionList.front();
    m_UnusedPageRegionList.pop_front();
    freeRegion.index = 0;
    freeRegion.count = static_cast<int>(pageCount);
    m_FreePageRegionList.push_back(freeRegion);

    m_TotalPageCount = static_cast<int>(pageCount);
    m_FreePageCount = m_TotalPageCount;
    m_ReservedPageCount = 0;

    m_LastReadOffset = 0;
    m_ReadAheadSize = 0;

    NN_RESULT_SUCCESS;
}

void LruFileDataCacheSystem::Finalize() NN_NOEXCEPT
{
    if (m_pCache)
    {
        m_LastReadOffset = 0;
        m_ReadAheadSize = 0;

        m_ReservedPageRegionList.clear();
        m_FreePageRegionList.clear();
        m_UnusedPageRegionList.clear();
        m_ReservedPageCount = 0;
        m_FreePageCount = 0;
        m_TotalPageCount = 0;

        for (int i = 0; i < m_LruList.CountTotal(); i++)
        {
            m_pPageRegions[i].~PageRegion();
        }

        m_pCache = nullptr;
        m_pPageRegions = nullptr;
        m_pPages = nullptr;
        m_LruList.Finalize();
    }
}

Result LruFileDataCacheSystem::Read(
    IStorage* pStorage,
    int64_t offset,
    void* pBuffer,
    size_t size,
    int64_t offsetBase,
    int64_t offsetLimit,
    FileDataCacheAccessResult* pBaseAccessResult) NN_NOEXCEPT
{
    const bool EnableReadAhead = true;

    NN_SDK_ASSERT(offsetBase < 0 || offsetBase <= offset);
    NN_SDK_ASSERT(offsetLimit < 0 || offsetLimit >= offset);

    NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);

    LruFileDataCacheSystemAccessResult accessResult(pBaseAccessResult, BufferRegion(offsetBase, static_cast<size_t>(offsetLimit - offsetBase)));

    BufferRegion toCacheRegion = BufferRegion::Null;
    BufferRegion toCopyRegion = BufferRegion::Null;
    const int64_t alignedOffset = util::align_down(offset, PageSize);
    const BufferRegion requestedRegion = BufferRegion(offset, size);

    size_t readAheadSize = 0;
    bool readAheadAdjusted = false;

    // リードロックを獲得する
    ScopedSharedLock sharedLock(&m_CacheOperationLock);

    for (int64_t position = alignedOffset; position < offset + static_cast<int64_t>(size + readAheadSize); position += PageSize)
    {
        bool isCached;
        {
            std::lock_guard<os::Mutex> lk(m_LruListLock);
            isCached = m_LruList.ContainsKey(Key(pStorage, position));
        }
        if (isCached)
        {
            // キャッシュヒット

            // 次にキャッシュミスが起こるか、もしくは要求された領域の終端に到達したら実際にキャッシュからデータをコピーする
            // それまで「コピー予定の領域」として記録する
            if (toCopyRegion == BufferRegion::Null)
            {
                toCopyRegion = BufferRegion(position, PageSize);
            }
            else
            {
                NN_SDK_ASSERT(toCopyRegion.GetEndOffset() == position);
                toCopyRegion.size += PageSize;
            }
        }
        else
        {
            // キャッシュミス

            // 先読みが有効化されている場合は、どのくらい先読みを行うかのサイズを決める
            if (NN_STATIC_CONDITION(EnableReadAhead) && !readAheadAdjusted)
            {
                {
                    std::lock_guard<os::Mutex> lk(m_ReadAheadLock);

                    // シーケンシャルリードが発生している時、もしくはファイルの先頭から読み取りを開始する時に先読みをする
                    bool sequentialReadSeemsOccuring = requestedRegion.offset == m_LastReadOffset;
                    bool readingStartsFromFileHead = offsetBase >= 0 && offsetBase == offset;

                    if (sequentialReadSeemsOccuring || readingStartsFromFileHead)
                    {
                        if (m_ReadAheadSize == 0 || readingStartsFromFileHead)
                        {
                            // 最初は適当にページサイズ分
                            m_ReadAheadSize = PageSize;
                        }
                        else
                        {
                            // 徐々に増やす
                            m_ReadAheadSize *= 2;
                            if (m_ReadAheadSize > MaxReadAheadSize)
                            {
                                m_ReadAheadSize = util::align_up(MaxReadAheadSize, PageSize);
                            }
                        }

                        // 先読みサイズより要求サイズの方が大きい場合は、そちらを先読みサイズとして採用
                        readAheadSize = util::align_up(std::max(m_ReadAheadSize, std::min(size, MaxReadAheadSize)), PageSize);

                        // ファイルの終端を超えて先読みを行わない
                        if (offsetLimit >= 0)
                        {
                            if ((offset + static_cast<int64_t>(size + readAheadSize)) > offsetLimit)
                            {
                                int64_t s = util::align_up(offsetLimit, PageSize) - (offset + static_cast<int64_t>(size));
                                readAheadSize = s > 0 ? static_cast<size_t>(s) : 0;
                            }
                        }
                    }
                    else
                    {
                        m_ReadAheadSize = 0;
                        readAheadSize = 0;
                    }
                }
                readAheadAdjusted = true;
            }

            if (toCacheRegion == BufferRegion::Null)
            {
                // キャッシュ予定領域がない ---> 今見ているオフセットは要求された領域の先頭である or 直前までキャッシュヒットが続いていた

                // 直前までキャッシュヒットが続いていた場合 (コピー予定領域がある場合) は実際のコピーを行う
                if (toCopyRegion != BufferRegion::Null)
                {
                    DoCopy(pStorage, pBuffer, toCopyRegion, requestedRegion, &accessResult);
                    toCopyRegion = BufferRegion::Null;
                }

                // 次にサイズの大きなキャッシュヒットが起こるか、もしくは要求された領域の終端に到達したら実際にキャッシュミスしたデータをファイルから読む
                // それまで「キャッシュ予定の領域」として記録する
                toCacheRegion = BufferRegion(position, PageSize);
            }
            else
            {
                // キャッシュ予定領域がある ---> 直前までキャッシュミスが続いていた

                if (toCopyRegion == BufferRegion::Null)
                {
                    // コピー予定領域がないなら、単にキャッシュ予定領域を更新するだけ
                    NN_SDK_ASSERT(toCacheRegion.GetEndOffset() == position);
                    toCacheRegion.size += PageSize;
                }
                else
                {
                    // キャッシュ予定領域もコピー予定領域もあるのは、次のようなシーケンスの場合
                    //
                    //    1. キャッシュミスした   (toCacheRegion)
                    //    2. キャッシュヒットした (toCopyRegion)
                    //    3. キャッシュミスした   (toCacheRegion2)  <--- イマココ
                    //
                    // この時、2 つの方針が考えられる
                    //
                    //    A. 1 と 3 のキャッシュミス領域をそれぞれ読む
                    //    B. 1, 2, 3 の領域は連続しているので、キャッシュ済みの 2 の領域も含め全体を一括して読む (coalesced read)
                    //
                    // 2 の領域が十分小さい場合は B の方が IPC 回数が減るのでパフォーマンスが良いと考えられる

                    BufferRegion toCacheRegion2(position, PageSize);
                    NN_SDK_ASSERT(toCacheRegion.GetEndOffset() == toCopyRegion.offset);
                    NN_SDK_ASSERT(toCopyRegion.GetEndOffset() == toCacheRegion2.offset);

                    // A, B どちらの方針にするかを決める
                    if (ShouldPerformCoalescedRead(toCacheRegion, toCacheRegion2))
                    {
                        // 方針 B の場合
                        // キャッシュ予定領域を 1, 2, 3 の連続領域として更新する
                        // コピー予定領域はリセットする
                        toCacheRegion = BufferRegion::GetInclusion(toCacheRegion, toCacheRegion2);
                        toCopyRegion = BufferRegion::Null;
                    }
                    else
                    {
                        // 方針 A の場合
                        // まず 2 のキャッシュヒットした領域をコピーする
                        // 先にキャッシュミスしたデータを読んでしまうと、2 のコピー予定領域のデータが上書きされることがある
                        DoCopy(pStorage, pBuffer, toCopyRegion, requestedRegion, &accessResult);
                        toCopyRegion = BufferRegion::Null;

                        // 次に最初にキャッシュミスした 1 の領域をファイルから読む
                        NN_RESULT_DO(DoCache(pStorage, pBuffer, toCacheRegion, requestedRegion, &accessResult));

                        // 最後に 3 の領域を新たなキャッシュ予定領域とする
                        toCacheRegion = toCacheRegion2;
                    }
                }
            }
        }
    }
    if (toCopyRegion != BufferRegion::Null)
    {
        DoCopy(pStorage, pBuffer, toCopyRegion, requestedRegion, &accessResult);
    }
    if (toCacheRegion != BufferRegion::Null)
    {
        NN_RESULT_DO(DoCache(pStorage, pBuffer, toCacheRegion, requestedRegion, &accessResult));
    }

    {
        std::lock_guard<os::Mutex> lk(m_ReadAheadLock);
        m_LastReadOffset = requestedRegion.GetEndOffset();
    }

#if NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED
    NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "total read time    = %10lld ns., offset = %11lld, size = %10zd\n", NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0).GetNanoSeconds(), offset, size);
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|  cache hit\n");
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     |     search = %10lld ns.\n", accessResult.GetPageSearchTime().GetNanoSeconds());
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     | cache copy = %10lld ns.\n", accessResult.GetCacheCopyTime().GetNanoSeconds());
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     |    promote = %10lld ns.\n", accessResult.GetPromotePageTime().GetNanoSeconds());
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "| cache miss\n");
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     |    reserve = %10lld ns.\n", accessResult.GetReservePageTime().GetNanoSeconds());
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     |       read = %10lld ns.\n", accessResult.GetFileReadTime().GetNanoSeconds());
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     |       copy = %10lld ns.\n", accessResult.GetCacheStoreTime().GetNanoSeconds());
    NN_FS_DETAIL_CACHE_SYSTEM_LOG(
        "|     |     commit = %10lld ns.\n", accessResult.GetCommitPageTime().GetNanoSeconds());
#endif

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

void LruFileDataCacheSystem::Purge(IStorage* pStorage) NN_NOEXCEPT
{
    ScopedExclusiveLock exclusiveLock(&m_CacheOperationLock);

    // 指定された IStorage のキャッシュを破棄して空き領域にする
    m_LruList.Remove([&](Key key, Page* pPage) -> bool
    {
        if (key.pStorage == pStorage)
        {
            AddFreePage(pPage);
            return true;
        }
        return false;
    });

    // 空き領域のコンパクションは次回のキャッシュミス時に行う
}

Result LruFileDataCacheSystem::CalculateAvailableCount(size_t* pOutPageCount, size_t* pOutBucketCount, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPageCount);
    NN_SDK_ASSERT_NOT_NULL(pOutBucketCount);

    const size_t bitLengthOfMaxPageCountPerBucket = 5;

    const size_t sizePerPage = sizeof(LruListType::NodeType) + sizeof(Page) + sizeof(PageRegion) + PageSize;
    const size_t maxAlignmentAggregated = NN_ALIGNOF(LruListType::NodeType) + NN_ALIGNOF(LruListType::BucketType) + NN_ALIGNOF(Page) + NN_ALIGNOF(PageRegion) + os::MemoryPageSize;
    const size_t availableBufferSize = bufferSize - maxAlignmentAggregated;
    NN_RESULT_THROW_UNLESS(availableBufferSize <= bufferSize, ResultFileDataCacheMemorySizeTooSmall());

    const size_t tmpPageCount = availableBufferSize / sizePerPage;
    const size_t tmpBucketCount = (tmpPageCount > (1 << (bitLengthOfMaxPageCountPerBucket - 1))) ? util::ceilp2(tmpPageCount) >> bitLengthOfMaxPageCountPerBucket : 1;
    NN_RESULT_THROW_UNLESS(tmpPageCount > 0 && tmpBucketCount > 0, ResultFileDataCacheMemorySizeTooSmall());

    const size_t pageCount = (availableBufferSize - tmpBucketCount * sizeof(LruListType::BucketType)) / sizePerPage;
    const size_t bucketCount = (pageCount > (1 << (bitLengthOfMaxPageCountPerBucket - 1))) ? util::ceilp2(pageCount) >> bitLengthOfMaxPageCountPerBucket : 1;
    NN_RESULT_THROW_UNLESS(pageCount > 0 && bucketCount > 0, ResultFileDataCacheMemorySizeTooSmall());

    *pOutPageCount = pageCount;
    *pOutBucketCount = bucketCount;

    NN_RESULT_SUCCESS;
}

void LruFileDataCacheSystem::DoCopy(
    IStorage* pStorage,
    void* pBuffer,
    const BufferRegion& copyRegion,
    const BufferRegion& originalRequest,
    LruFileDataCacheSystemAccessResult* pAccessResult) NN_NOEXCEPT
{
#if !NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_ENABLED
    NN_UNUSED(pAccessResult);
#endif

    NN_FS_DETAIL_CACHE_SYSTEM_LOG(">>> cache hit:  offset = %11lld, size = %10zd\n", copyRegion.offset, copyRegion.size);

    NN_SDK_ASSERT(util::is_aligned(copyRegion.offset, PageSize));
    NN_SDK_ASSERT(util::is_aligned(copyRegion.size, PageSize));

    RegionAttributedBuffer originalRequestBuffer(pBuffer, originalRequest);
    for (int64_t position = copyRegion.offset; position < copyRegion.GetEndOffset(); position += PageSize)
    {
        LruList<Key, Page>::EntryNode* pNode = nullptr;
        {
            std::lock_guard<os::Mutex> lk(m_LruListLock);

            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            pNode = m_LruList.Get(Key(pStorage, position));
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddPageSearchTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        NN_SDK_ASSERT_NOT_NULL(pNode);

        RegionAttributedBuffer cacheRegionBuffer(pNode->Get()->address, position, PageSize);
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            originalRequestBuffer.CopyFrom(cacheRegionBuffer);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCacheCopyTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        {
            std::lock_guard<os::Mutex> lk(m_LruListLock);

            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            m_LruList.Promote(pNode);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddPromotePageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
    }
}

Result LruFileDataCacheSystem::DoCache(
    IStorage* pStorage,
    void* pBuffer,
    const BufferRegion& cacheRegion,
    const BufferRegion& originalRequest,
    LruFileDataCacheSystemAccessResult* pAccessResult) NN_NOEXCEPT
{
    // リードロックからライトロックにアップグレード
    ScopedUpgradedExclusiveLock exclusiveLock(&m_CacheOperationLock);

    // ライトロック獲得待ちの間にキャッシュが更新されている可能性があるので、再度キャッシュが必要な領域を確認する
    BufferRegion toCacheRegion = BufferRegion::Null;
    BufferRegion toCopyRegion = BufferRegion::Null;
    for (int64_t position = cacheRegion.offset; position < cacheRegion.GetEndOffset(); position += PageSize)
    {
        auto pNode = m_LruList.Get(Key(pStorage, position));
        if (pNode)
        {
            if (toCopyRegion == BufferRegion::Null)
            {
                toCopyRegion = BufferRegion(position, PageSize);
            }
            else
            {
                NN_SDK_ASSERT(toCopyRegion.GetEndOffset() == position);
                toCopyRegion.size += PageSize;
            }
        }
        else
        {
            if (toCacheRegion == BufferRegion::Null)
            {
                if (toCopyRegion != BufferRegion::Null)
                {
                    DoCopy(pStorage, pBuffer, toCopyRegion, originalRequest, pAccessResult);
                    toCopyRegion = BufferRegion::Null;
                }
                toCacheRegion = BufferRegion(position, PageSize);
            }
            else
            {
                if (toCopyRegion == BufferRegion::Null)
                {
                    NN_SDK_ASSERT(toCacheRegion.GetEndOffset() == position);
                    toCacheRegion.size += PageSize;
                }
                else
                {
                    BufferRegion toCacheRegion2(position, PageSize);
                    NN_SDK_ASSERT(toCacheRegion.GetEndOffset() == toCopyRegion.offset);
                    NN_SDK_ASSERT(toCopyRegion.GetEndOffset() == toCacheRegion2.offset);
                    if (ShouldPerformCoalescedRead(toCacheRegion, toCacheRegion2))
                    {
                        toCacheRegion = BufferRegion::GetInclusion(toCacheRegion, toCacheRegion2);
                        toCopyRegion = BufferRegion::Null;
                    }
                    else
                    {
                        DoCopy(pStorage, pBuffer, toCopyRegion, originalRequest, pAccessResult);
                        toCopyRegion = BufferRegion::Null;

                        NN_RESULT_DO(DoCacheForce(pStorage, pBuffer, toCacheRegion, originalRequest, pAccessResult));
                        toCacheRegion = toCacheRegion2;
                    }
                }
            }
        }
    }
    if (toCopyRegion != BufferRegion::Null)
    {
        DoCopy(pStorage, pBuffer, toCopyRegion, originalRequest, pAccessResult);
    }
    if (toCacheRegion != BufferRegion::Null)
    {
        NN_RESULT_DO(DoCacheForce(pStorage, pBuffer, toCacheRegion, originalRequest, pAccessResult));
    }

    NN_RESULT_SUCCESS;
}

Result LruFileDataCacheSystem::DoCacheForce(
    IStorage* pStorage,
    void* pBuffer,
    const BufferRegion& cacheRegion,
    const BufferRegion& originalRequest,
    LruFileDataCacheSystemAccessResult* pAccessResult) NN_NOEXCEPT
{
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
    auto printCacheMiss = [](const BufferRegion& region) -> void
    {
        NN_FS_DETAIL_CACHE_SYSTEM_LOG("<<< cache miss: offset = %11lld, size = %10zd\n", region.offset, region.size);
    };
#endif

    // 要求は 2 つ
    //   * cacheRegion で示された領域をキャッシュする
    //   * cacheRegion と originalRequest の重なっている領域 (userRequestedRegion) を pBuffer にコピーする

    int64_t storageSize;
    NN_RESULT_DO(pStorage->GetSize(&storageSize));

    const size_t maxCacheSize = GetCacheSize();
    const BufferRegion userRequestedRegion = BufferRegion::GetIntersection(originalRequest, cacheRegion);
    if (userRequestedRegion.size == 0)
    {
        // 重なっていないなら、キャッシュだけすれば良い
        // maxCacheSize を超えないようにできるだけキャッシュする
        const BufferRegion actuallyCachedRegion = cacheRegion.GetEndRegionWithSizeLimit(maxCacheSize);
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
        printCacheMiss(actuallyCachedRegion);
#endif

        void* cacheBuffer;
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            cacheBuffer = ReserveFreePages(pStorage, actuallyCachedRegion.offset, actuallyCachedRegion.size);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddReservePageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        RegionAttributedBuffer actuallyCachedRegionBuffer(cacheBuffer, actuallyCachedRegion);
        {
            Result result;
            {
                ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);

                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                result = actuallyCachedRegionBuffer.Read(
                    pStorage, BufferRegion(actuallyCachedRegion.offset, static_cast<size_t>(std::min(actuallyCachedRegion.GetEndOffset(), storageSize) - actuallyCachedRegion.offset)));
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddFileReadTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
            pAccessResult->AddCacheFetchedRegion(actuallyCachedRegion);

            if (result.IsFailure())
            {
                ReturnBackReservedPages(cacheBuffer, actuallyCachedRegion.size);
                return result;
            }
        }
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            CommitCachePages(pStorage, cacheBuffer, actuallyCachedRegion.offset, actuallyCachedRegion.size);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCommitPageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }

        NN_RESULT_SUCCESS;
    }

    // 重なっている場合は、IPC が 2 回にならないようにしつつキャッシュするサイズが最大になるように努める

    RegionAttributedBuffer originalRequestBuffer(pBuffer, originalRequest);

    if (userRequestedRegion.Includes(cacheRegion))
    {
        // この場合は簡単
        // ユーザバッファにファイルを読み込んでから、キャッシュにコピーすれば良い
        // ただし maxCacheSize を超えないこと
        {
            ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);

            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            NN_RESULT_DO(originalRequestBuffer.Read(
                pStorage, BufferRegion(userRequestedRegion.offset, static_cast<size_t>(std::min(userRequestedRegion.GetEndOffset(), storageSize) - userRequestedRegion.offset))));
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddFileReadTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        pAccessResult->AddCacheFetchedRegion(userRequestedRegion);

        const BufferRegion actuallyCachedRegion = cacheRegion.GetEndRegionWithSizeLimit(maxCacheSize);
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
        printCacheMiss(actuallyCachedRegion);
#endif

        void* cacheBuffer;
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            cacheBuffer = ReserveFreePages(pStorage, actuallyCachedRegion.offset, actuallyCachedRegion.size);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddReservePageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        RegionAttributedBuffer actuallyCachedRegionBuffer(cacheBuffer, actuallyCachedRegion);
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            const size_t copiedSize = actuallyCachedRegionBuffer.CopyFrom(originalRequestBuffer);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCacheStoreTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));

            NN_UNUSED(copiedSize);
            NN_SDK_ASSERT(copiedSize == actuallyCachedRegion.size);
        }
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            CommitCachePages(pStorage, cacheBuffer, actuallyCachedRegion.offset, actuallyCachedRegion.size);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCommitPageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
    }
    else if (cacheRegion.Includes(userRequestedRegion))
    {
        // 基本的にはキャッシュにファイルを読み込んでからユーザバッファにコピーすれば良いが
        // すべてをキャッシュし切れない場合は、きちんとユーザバッファを含むようにしながらキャッシュしないといけない

        BufferRegion actuallyCachedRegion = cacheRegion;
        if (actuallyCachedRegion.size > maxCacheSize)
        {
            // とりあえず末尾を再選択して、userRequestedRegion を含むか確認する
            actuallyCachedRegion = cacheRegion.GetEndRegionWithSizeLimit(maxCacheSize);
            if (!actuallyCachedRegion.Includes(userRequestedRegion))
            {
                // userRequestedRegion を含む maxCacheSize 分の領域を再選択して actuallyCachedRegion に入れる
                actuallyCachedRegion = BufferRegion(util::align_down(userRequestedRegion.offset, PageSize), maxCacheSize);
            }
        }

        if (actuallyCachedRegion.Includes(userRequestedRegion))
        {
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
            printCacheMiss(actuallyCachedRegion);
#endif

            void* cacheBuffer;
            {
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                cacheBuffer = ReserveFreePages(pStorage, actuallyCachedRegion.offset, actuallyCachedRegion.size);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddReservePageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
            RegionAttributedBuffer actuallyCachedRegionBuffer(cacheBuffer, actuallyCachedRegion);
            {
                Result result;
                {
                    ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);

                    NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                    result = actuallyCachedRegionBuffer.Read(
                        pStorage, BufferRegion(actuallyCachedRegion.offset, static_cast<size_t>(std::min(actuallyCachedRegion.GetEndOffset(), storageSize) - actuallyCachedRegion.offset)));
                    NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                    NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddFileReadTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
                }
                pAccessResult->AddCacheFetchedRegion(actuallyCachedRegion);

                if (result.IsFailure())
                {
                    ReturnBackReservedPages(cacheBuffer, actuallyCachedRegion.size);
                    return result;
                }
            }
            {
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                CommitCachePages(pStorage, cacheBuffer, actuallyCachedRegion.offset, actuallyCachedRegion.size);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCommitPageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
            {
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                originalRequestBuffer.CopyFrom(actuallyCachedRegionBuffer);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCacheStoreTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
        }
        else
        {
            // 調整してもダメなら、多分 userRequestedRegion が大きすぎてうまくキャッシュし切れない
            NN_SDK_ASSERT(userRequestedRegion.ExpandAndAlign(PageSize).size > maxCacheSize);

            // ユーザバッファに読み込んでからキャッシュにコピーする方針に切り替える
            {
                ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);

                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                NN_RESULT_DO(originalRequestBuffer.Read(
                    pStorage, BufferRegion(userRequestedRegion.offset, static_cast<size_t>(std::min(userRequestedRegion.GetEndOffset(), storageSize) - userRequestedRegion.offset))));
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddFileReadTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
            pAccessResult->AddCacheFetchedRegion(userRequestedRegion);

            actuallyCachedRegion = userRequestedRegion.ShrinkAndAlign(PageSize).GetEndRegionWithSizeLimit(maxCacheSize);
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
            printCacheMiss(actuallyCachedRegion);
#endif

            void* cacheBuffer;
            {
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                cacheBuffer = ReserveFreePages(pStorage, actuallyCachedRegion.offset, actuallyCachedRegion.size);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddReservePageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
            RegionAttributedBuffer actuallyCachedRegionBuffer(cacheBuffer, actuallyCachedRegion);
            {
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                const size_t copiedSize = actuallyCachedRegionBuffer.CopyFrom(originalRequestBuffer);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCacheStoreTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));

                NN_UNUSED(copiedSize);
                NN_SDK_ASSERT(copiedSize == actuallyCachedRegion.size);
            }
            {
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
                CommitCachePages(pStorage, cacheBuffer, actuallyCachedRegion.offset, actuallyCachedRegion.size);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
                NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCommitPageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
            }
        }
    }
    else
    {
        // 中途半端に重なっていて最も起きてほしくないパターン
        // 仕方ないのでユーザバッファにファイルを読み込んだ後、cacheRegion と重なっているところだけキャッシュする
        {
            ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);

            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            NN_RESULT_DO(originalRequestBuffer.Read(
                pStorage, BufferRegion(userRequestedRegion.offset, static_cast<size_t>(std::min(userRequestedRegion.GetEndOffset(), storageSize) - userRequestedRegion.offset))));
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddFileReadTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        pAccessResult->AddCacheFetchedRegion(userRequestedRegion);

        const BufferRegion actuallyCachedRegion = userRequestedRegion.ShrinkAndAlign(PageSize).GetEndRegionWithSizeLimit(maxCacheSize);
#if NN_FS_DETAIL_CACHE_SYSTEM_LOG_ENABLED
        printCacheMiss(actuallyCachedRegion);
#endif

        void* cacheBuffer;
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            cacheBuffer = ReserveFreePages(pStorage, actuallyCachedRegion.offset, actuallyCachedRegion.size);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddReservePageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
        RegionAttributedBuffer actuallyCachedRegionBuffer(cacheBuffer, actuallyCachedRegion);
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            const size_t copiedSize = actuallyCachedRegionBuffer.CopyFrom(originalRequestBuffer);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCacheStoreTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));

            NN_UNUSED(copiedSize);
            NN_SDK_ASSERT(copiedSize == actuallyCachedRegion.size);
        }
        {
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_START(0);
            CommitCachePages(pStorage, cacheBuffer, actuallyCachedRegion.offset, actuallyCachedRegion.size);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_END(0);
            NN_FS_DETAIL_CACHE_SYSTEM_TIME_MEASUREMENT_EXPRESSION(pAccessResult->AddCommitPageTime(NN_FS_DETAIL_CACHE_SYSTEM_GET_LAST_TIME_MEASUREMENT(0)));
        }
    }
    NN_RESULT_SUCCESS;

} // NOLINT(impl/function_size)

void* LruFileDataCacheSystem::ReserveFreePages(IStorage* pStorage, int64_t offset, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size <= GetCacheSize());

    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    NN_SDK_ASSERT(util::is_aligned(offset, PageSize));
    NN_SDK_ASSERT(util::is_aligned(size, PageSize));

    if (size == 0)
    {
        return nullptr;
    }

    if (size == PageSize * 1)
    {
        return ReserveFreePagesBySinglePageReservation(pStorage, offset, size);
    }
    else
    {
        return ReserveFreePagesByGatherCompaction(pStorage, offset, size);
    }
}

void* LruFileDataCacheSystem::ReserveFreePagesBySinglePageReservation(IStorage* pStorage, int64_t offset, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    const int requiredPageCount = static_cast<int>(size / PageSize);
    NN_SDK_ASSERT(requiredPageCount == 1 && requiredPageCount <= m_LruList.CountTotal());

    while (NN_STATIC_CONDITION(true))
    {
        LruList<Key, Page>::EntryNode* pNode = m_LruList.Get(Key(pStorage, offset));
        if (pNode)
        {
            AddFreePage(m_LruList.Remove(*pNode));
        }

        if (m_FreePageCount < requiredPageCount)
        {
            if (m_LruList.CountUsed() == 0)
            {
                NN_SDK_ASSERT(m_ReservedPageCount > 0);
            }
            else
            {
                AddFreePage(m_LruList.Remove());
            }
        }

        if (m_FreePageCount < requiredPageCount)
        {
            ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);
            m_CommitEvent.Wait();
        }
        else
        {
            break;
        }
    }

    PageRegion& freeRegion = m_FreePageRegionList.front();

    int cacheIndex = freeRegion.index;

    freeRegion.index++;
    freeRegion.count--;
    if (freeRegion.count == 0)
    {
        m_FreePageRegionList.erase(m_FreePageRegionList.iterator_to(freeRegion));
        m_UnusedPageRegionList.push_front(freeRegion);
    }
    m_FreePageCount--;

    PageRegion& reservedRegion = m_UnusedPageRegionList.front();
    m_UnusedPageRegionList.pop_front();

    reservedRegion.index = cacheIndex;
    reservedRegion.count = requiredPageCount;

    AddPageRegionToSortedList(reservedRegion, m_ReservedPageRegionList);
    m_ReservedPageCount += requiredPageCount;

    return GetCacheOfIndex(cacheIndex);
}

void* LruFileDataCacheSystem::ReserveFreePagesByGatherCompaction(IStorage* pStorage, int64_t offset, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    const int requiredPageCount = static_cast<int>(size / PageSize);
    NN_SDK_ASSERT(requiredPageCount > 0 && requiredPageCount <= m_LruList.CountTotal());

    while (NN_STATIC_CONDITION(true))
    {
        // 既にキャッシュ済みのページがある場合は優先して空ける
        for (int64_t position = offset; position < (offset + static_cast<int64_t>(size)); position += PageSize)
        {
            LruList<Key, Page>::EntryNode* pNode = m_LruList.Get(Key(pStorage, position));
            if (pNode)
            {
                AddFreePage(m_LruList.Remove(*pNode));
            }
        }
        // 必要なページ数分だけ空ける
        while (m_FreePageCount < requiredPageCount)
        {
            // もし空けられるページがないのであれば抜ける (その場合、他のスレッドが reserve している可能性がある)
            if (m_LruList.CountUsed() == 0)
            {
                NN_SDK_ASSERT(m_ReservedPageCount > 0);
                break;
            }
            AddFreePage(m_LruList.Remove());
        }

        if (m_FreePageCount >= requiredPageCount && GetMaxAllocatableConsecutivePageCount() >= requiredPageCount)
        {
            // 空き領域が十分あって、かつ連続領域にできるなら OK
            break;
        }
        else
        {
            // そうでないなら、reserve された領域が動くのを待つ
            ScopedExclusiveLockRelease releasedLock(&m_CacheOperationLock);
            m_CommitEvent.Wait();
        }
    }

    // コンパクションして一番大きな連続した空き領域を得る
    PageRegion* pMaxSizeRegion = DoCompaction();
    NN_SDK_ASSERT(pMaxSizeRegion->count >= requiredPageCount);

    // 空き領域を減らす
    int cacheIndex = pMaxSizeRegion->index;

    pMaxSizeRegion->index += requiredPageCount;
    pMaxSizeRegion->count -= requiredPageCount;
    if (pMaxSizeRegion->count == 0)
    {
        m_FreePageRegionList.erase(m_FreePageRegionList.iterator_to(*pMaxSizeRegion));
        m_UnusedPageRegionList.push_front(*pMaxSizeRegion);
    }
    m_FreePageCount -= requiredPageCount;

    // ページを予約する
    PageRegion& reservedRegion = m_UnusedPageRegionList.front();
    m_UnusedPageRegionList.pop_front();

    reservedRegion.index = cacheIndex;
    reservedRegion.count = requiredPageCount;

    AddPageRegionToSortedList(reservedRegion, m_ReservedPageRegionList);
    m_ReservedPageCount += requiredPageCount;

    return GetCacheOfIndex(cacheIndex);
}

void LruFileDataCacheSystem::CommitCachePages(IStorage* pStorage, void* pBuffer, int64_t offset, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    NN_SDK_ASSERT(util::is_aligned(offset, PageSize));
    NN_SDK_ASSERT(util::is_aligned(size, PageSize));

    for (int64_t position = offset; position < static_cast<int64_t>(offset + size); position += PageSize)
    {
        Bit8* p = reinterpret_cast<Bit8*>(pBuffer) + (position - offset);
        Page* pPage = GetPageOfIndex(GetIndexOfCache(p))->reversePointer;

        LruListType::EntryNode* pNode = m_LruList.Get(Key(pStorage, position));
        if (pNode)
        {
            m_LruList.Promote(pNode);

            // promote したということはキャッシュの重複なので free として計上しないといけない
            AddFreePage(pPage);
        }
        else
        {
            m_LruList.Add(Key(pStorage, position), pPage);
        }
    }

    if (size > 0)  // 0 の時もありうる (その場合 pBuffer == nullptr)
    {
        // 予約したページを解除する
        for (auto it = m_ReservedPageRegionList.begin(); it != m_ReservedPageRegionList.end(); it++)
        {
            if (it->index == GetIndexOfCache(pBuffer))
            {
                NN_SDK_ASSERT(it->count == static_cast<int>(size / PageSize));

                PageRegion& region = *it;
                m_ReservedPageRegionList.erase(it);
                m_ReservedPageCount -= region.count;

                m_UnusedPageRegionList.push_front(region);

                m_CommitEvent.Signal();
                return;
            }
        }
        NN_ABORT("reserved page not found");
    }
}

void LruFileDataCacheSystem::ReturnBackReservedPages(void* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    for (auto it = m_ReservedPageRegionList.begin(); it != m_ReservedPageRegionList.end(); it++)
    {
        if (it->index == GetIndexOfCache(pBuffer))
        {
            NN_SDK_ASSERT(it->count == static_cast<int>(size / PageSize));

            PageRegion& region = *it;
            m_ReservedPageRegionList.erase(it);
            m_ReservedPageCount -= region.count;

            AddPageRegionToSortedList(region, m_FreePageRegionList);
            m_FreePageCount += region.count;
            return;
        }
    }
    NN_ABORT("reserved region not found");
}

void LruFileDataCacheSystem::GetMaxAllocatableConsecutivePageRegion(int* pOutIndex, int* pOutCount) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    if (m_ReservedPageRegionList.empty())
    {
        *pOutIndex = 0;
        *pOutCount = m_TotalPageCount;
        return;
    }

    int index = 0;
    int count = 0;

    int lastEndIndex = 0;
    for (auto it = m_ReservedPageRegionList.begin(); it != m_ReservedPageRegionList.end(); it++)
    {
        int freeCount = it->index - lastEndIndex;
        if (count < freeCount)
        {
            index = lastEndIndex;
            count = freeCount;
        }
        lastEndIndex = it->index + it->count;
    }
    if (count < m_TotalPageCount - lastEndIndex)
    {
        index = lastEndIndex;
        count = m_TotalPageCount - lastEndIndex;
    }

    NN_SDK_ASSERT(count <= m_TotalPageCount - m_ReservedPageCount);

    *pOutIndex = index;
    *pOutCount = count;
}

int LruFileDataCacheSystem::GetMaxAllocatableConsecutivePageCount() const NN_NOEXCEPT
{
    int index;
    int count;
    GetMaxAllocatableConsecutivePageRegion(&index, &count);
    return count;
}

void LruFileDataCacheSystem::AddPageRegionToSortedList(PageRegion& pageRegion, PageRegionList& pageRegionList) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());
    NN_SDK_ASSERT(!pageRegion.IsLinked());

    for (auto it = pageRegionList.cbegin(); it != pageRegionList.cend(); it++)
    {
        if (pageRegion.index < it->index)
        {
            m_FreePageRegionList.insert(it, pageRegion);
            return;
        }
    }
    pageRegionList.push_back(pageRegion);
}

void LruFileDataCacheSystem::AddFreePage(Page* pPage) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    const int freeIndex = GetIndexOfCache(pPage->address);

    // 既にある PageRegion にくっつけられないか調べる
    for (auto it = m_FreePageRegionList.begin(); it != m_FreePageRegionList.end(); it++)
    {
        bool added = false;
        if (it->index == freeIndex + 1)
        {
            // 左側にくっつけられる
            it->index = freeIndex;
            it->count++;
            added = true;
        }
        else if (it->index + it->count == freeIndex)
        {
            // 右側にくっつけられる
            it->count++;
            added = true;
        }

        if (added)
        {
            m_FreePageCount++;

            // くっつけた場合は、周囲の PageRegion 同士で更にくっつけられないか調べる
            // まず左側を見る
            PageRegionList::iterator leftIt(it);
            for (; leftIt != m_FreePageRegionList.begin(); )
            {
                leftIt--;

                if (leftIt->GetEndIndex() == it->index)
                {
                    it->index = leftIt->index;
                    it->count += leftIt->count;

                    PageRegion& unusedRegion = *leftIt;
                    leftIt = m_FreePageRegionList.erase(leftIt);
                    m_UnusedPageRegionList.push_front(unusedRegion);
                }
                else
                {
                    break;
                }
            }
            // 続けて右側を見る
            PageRegionList::iterator rightIt(it);
            for (rightIt++; rightIt != m_FreePageRegionList.end(); )
            {
                if (it->GetEndIndex() == rightIt->index)
                {
                    it->count += rightIt->count;

                    PageRegion& unusedRegion = *rightIt;
                    rightIt = m_FreePageRegionList.erase(rightIt);
                    m_UnusedPageRegionList.push_front(unusedRegion);
                }
                else
                {
                    break;
                }
            }
            return;
        }
    }

    // くっつけられなかったので、1 ページだけの PageRegion として新規作成する
    PageRegion& newFreeRegion = m_UnusedPageRegionList.front();
    m_UnusedPageRegionList.pop_front();

    newFreeRegion.index = freeIndex;
    newFreeRegion.count = 1;

    AddPageRegionToSortedList(newFreeRegion, m_FreePageRegionList);
    m_FreePageCount++;
}

LruFileDataCacheSystem::PageRegion* LruFileDataCacheSystem::DoCompaction() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CacheOperationLock.IsWriteLockHeldByCurrentThread());

    // 空きがないならコンパクション不要なはず
    NN_SDK_ASSERT(!m_FreePageRegionList.empty());

    // 一番でかく空き領域を作れる領域を見つける
    int maxAllocatableConsecutiveRegionIndex;
    int maxAllocatableConsecutiveRegionCount;
    GetMaxAllocatableConsecutivePageRegion(&maxAllocatableConsecutiveRegionIndex, &maxAllocatableConsecutiveRegionCount);

    // 既存の一番でかい空き領域を見つける
    PageRegion* pMaxSizeRegion = nullptr;
    PageRegion* pMaxSizeRegionInMaxAllocatableConsecutiveRegion = nullptr;
    for (auto it = m_FreePageRegionList.begin(); it != m_FreePageRegionList.end(); it++)
    {
        if (!pMaxSizeRegion || pMaxSizeRegion->count < it->count)
        {
            pMaxSizeRegion = &(*it);
        }
        if (maxAllocatableConsecutiveRegionIndex <= it->index && it->GetEndIndex() <= maxAllocatableConsecutiveRegionIndex + maxAllocatableConsecutiveRegionCount)
        {
            if (!pMaxSizeRegionInMaxAllocatableConsecutiveRegion || pMaxSizeRegionInMaxAllocatableConsecutiveRegion->count < it->count)
            {
                pMaxSizeRegionInMaxAllocatableConsecutiveRegion = &(*it);
            }
        }
    }
    NN_SDK_ASSERT_NOT_NULL(pMaxSizeRegion);

    // 既存の一番でかい空き領域は、一番でかく空き領域を作れる領域の中に入っているか
    if (pMaxSizeRegion == pMaxSizeRegionInMaxAllocatableConsecutiveRegion)
    {
        // 入っているので、そこへ空き領域を集めれば良い
        return DoCompactionCore(
            pMaxSizeRegion,
            maxAllocatableConsecutiveRegionIndex,
            maxAllocatableConsecutiveRegionIndex + maxAllocatableConsecutiveRegionCount);
    }
    else
    {
        // 入っていないので、既存の一番でかい空き領域のまわりに残りの空き領域を集めるか、
        // 一番でかく空き領域を作れる領域の中に大きな空き領域を作り直すか判断する
        // コンパクションをした結果、より大きい空き領域を作れる方を選ぶ

        // まず既存の一番でかい空き領域のまわりに、残りの空き領域をいくつ集められるかを調べる
        int maxConsectiveRegionLeftEnd = 0;
        {
            int index = pMaxSizeRegion->index;
            for (auto it = m_ReservedPageRegionList.cbegin(); it != m_ReservedPageRegionList.cend(); it++)
            {
                if (it->index > index)
                {
                    break;
                }
                NN_SDK_ASSERT(it->GetEndIndex() <= index);
                maxConsectiveRegionLeftEnd = it->GetEndIndex();
            }
        }
        int maxConsectiveRegionRightEnd = m_TotalPageCount;
        {
            int index = pMaxSizeRegion->index + pMaxSizeRegion->count;
            for (auto it = m_ReservedPageRegionList.cbegin(); it != m_ReservedPageRegionList.cend(); it++)
            {
                if (it->GetEndIndex() < index)
                {
                    continue;
                }
                NN_SDK_ASSERT(it->index >= index);
                maxConsectiveRegionRightEnd = it->index;
                break;
            }
        }

        PageRegion* ret;
        // 既存の一番でかい空き領域のまわりに残りの空き領域がすべて収まるか確認
        if (m_FreePageCount > (maxConsectiveRegionRightEnd - maxConsectiveRegionLeftEnd))
        {
            // 収まらないので、一番でかく空き領域を作れる領域に大きな空き領域を作り直す
            ret = DoCompactionCore(
                pMaxSizeRegionInMaxAllocatableConsecutiveRegion,  // nullptr の可能性もある。nullptr でなければこのまわりに集めさせる
                maxAllocatableConsecutiveRegionIndex,
                maxAllocatableConsecutiveRegionIndex + maxAllocatableConsecutiveRegionCount);
        }
        else
        {
            // 収まるので、このまま まわりに集める
            ret = DoCompactionCore(
                pMaxSizeRegion,
                maxConsectiveRegionLeftEnd,
                maxConsectiveRegionRightEnd);
        }

        return ret;
    }
}

LruFileDataCacheSystem::PageRegion* LruFileDataCacheSystem::DoCompactionCore(PageRegion* pTargetRegion, int leftLimit, int rightLimit) NN_NOEXCEPT
{
    // 要求:
    //    * pTargetRegion のまわりに既存の空き領域をできるだけ集める。ただし pTargetRegion == nullptr ならば、limit 内のどこに空き領域を集めても構わない
    //    * 空き領域は leftLimit, rightLimit で指定されるインデックスを超えてはならない (このすぐ外側のページは予約されているか、またはキャッシュメモリの外)

    NN_SDK_ASSERT(leftLimit < rightLimit);

#if NN_FS_DETAIL_CACHE_SYSTEM_ASSERT_CHECK
    for (auto it = m_ReservedPageRegionList.cbegin(); it != m_ReservedPageRegionList.cend(); it++)
    {
        // leftLimit ~ rightLimit 内に reservedRegion がない
        NN_SDK_ASSERT(it->index + it->count <= leftLimit || it->index >= rightLimit);
    }
    {
        // leftLimit, rightLimit のすぐ外側は無効な領域 or reservedRegion
        bool leftOk = leftLimit == 0;
        bool rightOk = rightLimit == m_TotalPageCount;
        for (auto it = m_ReservedPageRegionList.cbegin(); it != m_ReservedPageRegionList.cend(); it++)
        {
            leftOk = leftOk || it->index + it->count == leftLimit;
            rightOk = rightOk || it->index == rightLimit;
        }
        NN_SDK_ASSERT(leftOk);
        NN_SDK_ASSERT(rightOk);
    }
#endif

    if (pTargetRegion == nullptr)
    {
        // とりあえず適当に 1 ページだけ移動させる
        PageRegion& freeRegion = m_FreePageRegionList.front();

        Page* pEmpty = GetPageOfIndex(freeRegion.index)->reversePointer;
        Page* pTarget = GetPageOfIndex(leftLimit)->reversePointer;
        MovePage(pEmpty, pTarget);

        freeRegion.index++;
        freeRegion.count--;
        if (freeRegion.count == 0)
        {
            m_FreePageRegionList.pop_front();
            m_UnusedPageRegionList.push_front(freeRegion);
        }

        // 既存の FreeRegion にはくっつけられないはず
#if NN_FS_DETAIL_CACHE_SYSTEM_ASSERT_CHECK
        for (auto it = m_FreePageRegionList.cbegin(); it != m_FreePageRegionList.cend(); it++)
        {
            NN_SDK_ASSERT(leftLimit != it->index - 1);
            NN_SDK_ASSERT(leftLimit != it->index + it->count);
        }
#endif
        PageRegion& newFreeRegion = m_UnusedPageRegionList.front();
        m_UnusedPageRegionList.pop_front();

        newFreeRegion.index = leftLimit;
        newFreeRegion.count = 1;

        AddPageRegionToSortedList(newFreeRegion, m_FreePageRegionList);

        pTargetRegion = &newFreeRegion;
    }

    // この時点で limit を超えてたらおかしい
    NN_SDK_ASSERT(pTargetRegion->index >= leftLimit);
    NN_SDK_ASSERT(pTargetRegion->GetEndIndex() <= rightLimit);

#if NN_FS_DETAIL_CACHE_SYSTEM_ASSERT_CHECK
    {
        // FreePageRegionList がちゃんとソートされていて、各 Region の重なりがないことを確認
        int lastEndIndex = 0;
        for (auto it = m_FreePageRegionList.cbegin(); it != m_FreePageRegionList.cend(); it++)
        {
            NN_SDK_ASSERT(lastEndIndex <= it->index);
            lastEndIndex = it->index + it->count;
        }
        NN_SDK_ASSERT(lastEndIndex <= m_TotalPageCount);
    }
#endif

    // pTargetRegion のまわりに、残りの空き領域を集めていく
    auto itFreeRegion = m_FreePageRegionList.begin();
    if (itFreeRegion == m_FreePageRegionList.iterator_to(*pTargetRegion))
    {
        itFreeRegion++;
    }
    // まず左側をいっぱいまで使って空き領域を集める
    while (pTargetRegion->index > leftLimit && itFreeRegion != m_FreePageRegionList.end())
    {
        // pTargetRegion の 1 つ左の空き領域を確認し、その右端点 (localLeftLimit) を得る
        // 得られた右端点か、または leftLimit のいずれかに当たるまで pTargetRegion の左に向かって空き領域を増やしたい
        int localLeftLimit;
        {
            auto itLeftRegion = m_FreePageRegionList.iterator_to(*pTargetRegion);
            if (itLeftRegion == m_FreePageRegionList.begin())
            {
                // pTargetRegion の左に空き領域はない
                localLeftLimit = leftLimit;
            }
            else
            {
                itLeftRegion--;
                localLeftLimit = std::max(itLeftRegion->GetEndIndex(), leftLimit);
            }
            NN_SDK_ASSERT(localLeftLimit <= pTargetRegion->index);

            // まだ leftLimit に到達しておらず、かつ pTargetRegion が右端点に当たっている場合は、空き領域を結合する
            while (leftLimit < localLeftLimit && localLeftLimit == pTargetRegion->index)
            {
                pTargetRegion->index = itLeftRegion->index;
                pTargetRegion->count += itLeftRegion->count;
                NN_SDK_ASSERT(pTargetRegion->index >= leftLimit);

                if (itLeftRegion == itFreeRegion)
                {
                    itFreeRegion++;
                    NN_SDK_ASSERT(itFreeRegion == m_FreePageRegionList.iterator_to(*pTargetRegion));
                    itFreeRegion++;
                }

                PageRegion& unusedRegion = *itLeftRegion;
                itLeftRegion = m_FreePageRegionList.erase(itLeftRegion);
                m_UnusedPageRegionList.push_front(unusedRegion);

                if (itLeftRegion == m_FreePageRegionList.begin())
                {
                    localLeftLimit = leftLimit;
                    NN_SDK_ASSERT(localLeftLimit <= pTargetRegion->index);
                    break;
                }
                else
                {
                    itLeftRegion--;
                    localLeftLimit = std::max(itLeftRegion->GetEndIndex(), leftLimit);
                    NN_SDK_ASSERT(localLeftLimit <= pTargetRegion->index);
                }
            }
        }

        // pTargetRegion の左に向かって空き領域を増やす
        // localLeftLimit に当たったら一旦終了して、次の localLeftLimit を更新する
        while (pTargetRegion->index > localLeftLimit && itFreeRegion != m_FreePageRegionList.end())
        {
            Page* pEmpty = GetPageOfIndex(itFreeRegion->index)->reversePointer;
            Page* pTarget = GetPageOfIndex(pTargetRegion->index - 1)->reversePointer;
            MovePage(pEmpty, pTarget);

            itFreeRegion->index++;
            itFreeRegion->count--;
            pTargetRegion->index--;
            pTargetRegion->count++;

            if (itFreeRegion->count == 0)
            {
                PageRegion& unusedRegion = *itFreeRegion;
                itFreeRegion = m_FreePageRegionList.erase(itFreeRegion);
                if (itFreeRegion == m_FreePageRegionList.iterator_to(*pTargetRegion))
                {
                    itFreeRegion++;
                }
                m_UnusedPageRegionList.push_front(unusedRegion);
            }
        }
    }
    // 次は右側をいっぱいまで使って空き領域を集める
    while (pTargetRegion->index + pTargetRegion->count < rightLimit && itFreeRegion != m_FreePageRegionList.end())
    {
        // pTargetRegion の 1 つ右の空き領域を確認し、その左端点 (localRightLimit) を得る
        // 得られた左端点か、または rightLimit のいずれかに当たるまで pTargetRegion の右に向かって空き領域を増やしたい
        int localRightLimit;
        {
            auto itRightRegion = m_FreePageRegionList.iterator_to(*pTargetRegion);
            itRightRegion++;

            if (itRightRegion == m_FreePageRegionList.end())
            {
                // pTargetRegion の右に空き領域はない
                localRightLimit = rightLimit;
            }
            else
            {
                localRightLimit = std::min(itRightRegion->index, rightLimit);
            }
            NN_SDK_ASSERT(pTargetRegion->GetEndIndex() <= localRightLimit);

            // まだ rightLimit に到達しておらず、かつ pTargetRegion が左端点に当たっている場合は、空き領域を結合する
            while (localRightLimit < rightLimit && localRightLimit == pTargetRegion->GetEndIndex())
            {
                pTargetRegion->count += itRightRegion->count;
                NN_SDK_ASSERT(pTargetRegion->GetEndIndex() <= rightLimit);

                if (itRightRegion == itFreeRegion)
                {
                    itFreeRegion++;
                }

                PageRegion& unusedRegion = *itRightRegion;
                itRightRegion = m_FreePageRegionList.erase(itRightRegion);
                m_UnusedPageRegionList.push_front(unusedRegion);

                if (itRightRegion == m_FreePageRegionList.end())
                {
                    localRightLimit = rightLimit;
                    NN_SDK_ASSERT(pTargetRegion->GetEndIndex() <= localRightLimit);
                    break;
                }
                else
                {
                    localRightLimit = std::min(itRightRegion->index, rightLimit);
                    NN_SDK_ASSERT(pTargetRegion->GetEndIndex() <= localRightLimit);
                }
            }
        }

        // pTargetRegion の右に向かって空き領域を増やす
        // localRightLimit に当たったら一旦終了して、次の localRightLimit を更新する
        while (pTargetRegion->GetEndIndex() < localRightLimit && itFreeRegion != m_FreePageRegionList.end())
        {
            Page* pEmpty = GetPageOfIndex(itFreeRegion->GetEndIndex() - 1)->reversePointer;
            Page* pTarget = GetPageOfIndex(pTargetRegion->GetEndIndex())->reversePointer;
            MovePage(pEmpty, pTarget);

            itFreeRegion->count--;
            pTargetRegion->count++;

            if (itFreeRegion->count == 0)
            {
                PageRegion& unusedRegion = *itFreeRegion;
                itFreeRegion = m_FreePageRegionList.erase(itFreeRegion);
                if (itFreeRegion == m_FreePageRegionList.iterator_to(*pTargetRegion))
                {
                    itFreeRegion++;
                }
                m_UnusedPageRegionList.push_front(unusedRegion);
            }
        }
    }

    // 最終的に limit を超えていないことを確認して、集めた空き領域を返す
    NN_SDK_ASSERT(pTargetRegion->index >= leftLimit);
    NN_SDK_ASSERT(pTargetRegion->GetEndIndex() <= rightLimit);
    return pTargetRegion;

} // NOLINT(impl/function_size)

void LruFileDataCacheSystem::MovePage(Page* pDestination, Page* pSource) NN_NOEXCEPT
{
    if (pDestination != pSource)
    {
        std::memcpy(pDestination->address, pSource->address, PageSize);
        std::swap(pDestination->address, pSource->address);

        Page* pPagePointingToDestination = GetForwardPointer(pDestination);
        Page* pPagePointingToSource = GetForwardPointer(pSource);
        std::swap(pPagePointingToDestination->reversePointer, pPagePointingToSource->reversePointer);
    }
}

bool LruFileDataCacheSystem::ShouldPerformCoalescedRead(const BufferRegion& cacheRegion1, const BufferRegion& cacheRegion2) NN_NOEXCEPT
{
    const size_t cacheMissRegionSizeSum = cacheRegion1.size + cacheRegion2.size;
    const size_t coalescedRegionSize = BufferRegion::GetInclusion(cacheRegion1, cacheRegion2).size;
    NN_SDK_ASSERT(coalescedRegionSize >= cacheMissRegionSizeSum);
    const size_t cacheHitRegionSize = coalescedRegionSize - cacheMissRegionSizeSum;

    // TODO: 調整する
    if (cacheMissRegionSizeSum < 16 * 1024)
    {
        return cacheHitRegionSize < (cacheMissRegionSizeSum * 2 / 3);
    }
    else if (cacheMissRegionSizeSum < 64 * 1024)
    {
        return cacheHitRegionSize < (cacheMissRegionSizeSum * 2 / 4);
    }
    else if (cacheMissRegionSizeSum < 128 * 1024)
    {
        return cacheHitRegionSize < (cacheMissRegionSizeSum * 2 / 5);
    }
    else
    {
        return cacheHitRegionSize < (cacheMissRegionSizeSum / 10);
    }
}

void* LruFileDataCacheSystem::GetCacheOfIndex(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(index >= 0 && index < m_LruList.CountTotal());
    return reinterpret_cast<Bit8*>(m_pCache) + index * PageSize;
}

LruFileDataCacheSystem::Page* LruFileDataCacheSystem::GetPageOfIndex(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(index >= 0 && index < m_LruList.CountTotal());
    return &m_pPages[index];
}

int LruFileDataCacheSystem::GetIndexOfCache(void* pCache) const NN_NOEXCEPT
{
    uintptr_t start = reinterpret_cast<uintptr_t>(m_pCache);
    uintptr_t input = reinterpret_cast<uintptr_t>(pCache);
    NN_UNUSED(start);
    NN_UNUSED(input);
    NN_SDK_ASSERT(input >= start && input < (start + GetCacheSize()));

    return static_cast<int>((reinterpret_cast<Bit8*>(pCache) - reinterpret_cast<Bit8*>(m_pCache)) / PageSize);
}

int LruFileDataCacheSystem::GetIndexOfPage(Page* pPage) const NN_NOEXCEPT
{
    uintptr_t start = reinterpret_cast<uintptr_t>(m_pPages);
    uintptr_t input = reinterpret_cast<uintptr_t>(pPage);
    NN_UNUSED(start);
    NN_UNUSED(input);
    NN_SDK_ASSERT(input >= start && input < (start + sizeof(Page) * m_LruList.CountTotal()));

    return static_cast<int>((reinterpret_cast<Bit8*>(pPage) - reinterpret_cast<Bit8*>(m_pPages)) / sizeof(Page));
}

LruFileDataCacheSystem::Page* LruFileDataCacheSystem::GetForwardPointer(Page* pPage) const NN_NOEXCEPT
{
    return &m_pPages[GetIndexOfCache(pPage->address)];
}

}}}  // namespace nn::fs::detail
