﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os/os_ReaderWriterLock.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fs_FileDataCache.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include "fs_FileDataCacheAccessor.h"
#include "fs_FileDataCacheDetail.h"
#include "fs_RomFsFileDataCache.h"


namespace nn { namespace fs {

namespace {

detail::FileDataCacheAccessor* g_pGlobalFileDataCacheAccessor = nullptr;
os::ReaderWriterLock g_GlobalFileDataCacheAccessorLock;

}  // namespace unnamed


namespace detail {

bool IsGlobalFileDataCacheEnabled() NN_NOEXCEPT
{
    return g_pGlobalFileDataCacheAccessor != nullptr;
}

bool TryGetGlobalFileDataCacheAccessor(GlobalFileDataCacheAccessorReadableScopedPointer* pOutPointer) NN_NOEXCEPT
{
    g_GlobalFileDataCacheAccessorLock.AcquireReadLock();

    bool accessorWillBeUsedOutsideOfThisFunction = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!accessorWillBeUsedOutsideOfThisFunction)
        {
            g_GlobalFileDataCacheAccessorLock.ReleaseReadLock();
        }
    };

    if (g_pGlobalFileDataCacheAccessor)
    {
        pOutPointer->Set(g_pGlobalFileDataCacheAccessor, &g_GlobalFileDataCacheAccessorLock);
        accessorWillBeUsedOutsideOfThisFunction = true;
    }

    return g_pGlobalFileDataCacheAccessor != nullptr;
}

}  // namespace nn::fs::detail


void EnableGlobalFileDataCache(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    auto enableImpl = [&]() NN_NOEXCEPT -> Result
    {
        g_GlobalFileDataCacheAccessorLock.AcquireWriteLock();
        NN_UTIL_SCOPE_EXIT
        {
            g_GlobalFileDataCacheAccessorLock.ReleaseWriteLock();
        };

        // 事前条件チェック
        NN_RESULT_THROW_UNLESS(pBuffer != nullptr, ResultNullptrArgument());
        NN_RESULT_THROW_UNLESS(bufferSize >= 1 * 1024 * 1024, ResultInvalidArgument());
        NN_RESULT_THROW_UNLESS(g_pGlobalFileDataCacheAccessor == nullptr, ResultGlobalFileDataCacheAlreadyEnabled());

        // デフォルトの FileDataCache 作成
        const uintptr_t pStart = reinterpret_cast<uintptr_t>(pBuffer);
        const uintptr_t pEnd = pStart + bufferSize;
        NN_ABORT_UNLESS(pStart <= pEnd);

        uintptr_t p = util::align_up(pStart, NN_ALIGNOF(RomFsFileDataCache));
        NN_RESULT_THROW_UNLESS(p + sizeof(RomFsFileDataCache) <= pEnd, ResultFileDataCacheMemorySizeTooSmall());
        RomFsFileDataCache* pRomFsFileDataCache = new (reinterpret_cast<RomFsFileDataCache*>(p)) RomFsFileDataCache();
        p += sizeof(RomFsFileDataCache);

        p = util::align_up(p, NN_ALIGNOF(detail::FileDataCacheAccessor));
        NN_RESULT_THROW_UNLESS(p + sizeof(detail::FileDataCacheAccessor) <= pEnd, ResultFileDataCacheMemorySizeTooSmall());
        g_pGlobalFileDataCacheAccessor = new (reinterpret_cast<detail::FileDataCacheAccessor*>(p)) detail::FileDataCacheAccessor(pRomFsFileDataCache);
        p += sizeof(detail::FileDataCacheAccessor);

        NN_RESULT_DO(pRomFsFileDataCache->Initialize(reinterpret_cast<void*>(p), bufferSize - (p - reinterpret_cast<uintptr_t>(pBuffer))));

        NN_RESULT_SUCCESS;
    };
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_FILE_DATA_CACHE(enableImpl(), NN_DETAIL_FS_ACCESS_LOG_FORMAT_SIZE, bufferSize));
}

void DisableGlobalFileDataCache() NN_NOEXCEPT
{
    auto disableImpl = [&]() NN_NOEXCEPT -> Result
    {
        g_GlobalFileDataCacheAccessorLock.AcquireWriteLock();
        NN_UTIL_SCOPE_EXIT
        {
            g_GlobalFileDataCacheAccessorLock.ReleaseWriteLock();
        };

        if (g_pGlobalFileDataCacheAccessor)
        {
            // グローバルな FileDataCacheAccessor を破棄
            g_pGlobalFileDataCacheAccessor->~FileDataCacheAccessor();
            g_pGlobalFileDataCacheAccessor = nullptr;
        }

        NN_RESULT_SUCCESS;
    };
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_FILE_DATA_CACHE(disableImpl(), ""));
}

}} // namespace nn::fs
