﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

namespace {
    Result MountDeviceSaveDataImpl(const char* name, const SaveDataAttribute& attribute) NN_NOEXCEPT
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        NN_RESULT_DO(fileSystemProxy->OpenSaveDataFileSystem(&fileSystem, static_cast<uint32_t>(SaveDataSpaceId::User), attribute));

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInDeviceSaveDataA());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAbstract), nullptr, false, true, true));
        NN_RESULT_SUCCESS;
    }
}

Result MountDeviceSaveData(const char* name) NN_NOEXCEPT
{
    auto attribute = SaveDataAttribute::Make(InvalidProgramId, SaveDataType::Device, InvalidUserId, InvalidSystemSaveDataId);
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(MountDeviceSaveDataImpl(name, attribute), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountDeviceSaveData(const char* name, const nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Device, InvalidUserId, InvalidSystemSaveDataId);
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(MountDeviceSaveDataImpl(name, attribute), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTDEVICESAVEDATA_APPLICATIONID(name, applicationId)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

}}

