﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_RomFsFileSystem.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_Data.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/ncm/ncm_StorageId.h>

#include "fsa/fs_FileSystemAccessor.h"
#include "fsa/fs_UserMountTable.h"
#include "fsa/fs_MountUtility.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fs_StorageServiceObjectAdapter.h"


namespace nn { namespace fs { namespace detail {

namespace {

    Result OpenDataStorageByDataId(std::unique_ptr<nn::fs::IStorage>* outValue, nn::ncm::DataId dataId, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IStorage> storage;

        NN_FS_RESULT_DO_WITH_HANDLER(
            fileSystemProxy->OpenDataStorageByDataId(
                nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IStorage>>(&storage),
                dataId,
                static_cast<nn::Bit8>(storageId)
            ),
            [](Result result) -> Result
            {
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(nn::ncm::ResultContentMetaNotFound)
                    {
                        NN_RESULT_THROW(ResultTargetNotFound()); // TORIAEZU: 存在しない場合は ResultTargetNotFound
                    }
                NN_RESULT_END_TRY
                NN_RESULT_SUCCESS;
            }
        );

        std::unique_ptr<nn::fs::IStorage> storageAbstract;
        storageAbstract.reset(new StorageServiceObjectAdapter(std::move(storage)));
        NN_RESULT_THROW_UNLESS(storageAbstract, ResultAllocationMemoryFailedInDataA());

        *outValue = std::move(storageAbstract);
        NN_RESULT_SUCCESS;
    }

    Result MountDataImpl(
        const char* name,
        nn::ncm::DataId dataId,
        nn::ncm::StorageId storageId,
        void* pFileSystemCacheBuffer,
        size_t fileSystemCacheBufferSize,
        bool isFileSystemCacheUsed,
        bool makeFileDataCacheAvailable,
        bool makePathBasedFileDataCacheAvailable
    ) NN_NOEXCEPT
    {
        std::unique_ptr<IStorage> storage;
        NN_RESULT_DO(OpenDataStorageByDataId(&storage, dataId, storageId));

        std::unique_ptr<RomFsFileSystem> fileSystem(new RomFsFileSystem());
        NN_RESULT_THROW_UNLESS(fileSystem != nullptr, ResultAllocationMemoryFailedInDataB());
        NN_RESULT_DO(
            fileSystem->Initialize(
                std::move(storage),
                pFileSystemCacheBuffer,
                fileSystemCacheBufferSize,
                isFileSystemCacheUsed
            )
        );

        return fsa::Register(name, std::move(fileSystem), nullptr, makeFileDataCacheAvailable, makePathBasedFileDataCacheAvailable, false);
    }

}

Result QueryMountDataCacheSize(
    size_t* pOutValue,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId
) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

    std::unique_ptr<IStorage> storage;
    NN_FS_RESULT_DO(OpenDataStorageByDataId(&storage, dataId, storageId));

    size_t size = 0;
    NN_FS_RESULT_DO(RomFsFileSystem::GetRequiredWorkingMemorySize(&size, storage.get()));
    *pOutValue = size;

    // サイズ 0 でメモリ確保すると問題のある環境があるため 0 ではない値を返します。
    static const size_t MinCacheSize = 32;
    if( *pOutValue < MinCacheSize )
    {
        *pOutValue = MinCacheSize;
    }
    NN_RESULT_SUCCESS;
}

Result MountData(
    const char* name,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId
) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(CheckMountName(name)); // マウント名チェック

    NN_FS_RESULT_DO(MountDataImpl(name, dataId, storageId, nullptr, 0, false, false, false));
    NN_RESULT_SUCCESS;
}

Result MountData(
    const char* name,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId,
    void* pFileSystemCacheBuffer,
    size_t fileSystemCacheBufferSize
) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(CheckMountName(name)); // マウント名チェック
    NN_FS_RESULT_THROW_UNLESS(pFileSystemCacheBuffer != nullptr, ResultNullptrArgument());

    NN_FS_RESULT_DO(MountDataImpl(name, dataId, storageId, pFileSystemCacheBuffer, fileSystemCacheBufferSize, true, false, false));
    NN_RESULT_SUCCESS;
}

Result MountData(
    const char* name,
    nn::ncm::DataId dataId,
    nn::ncm::StorageId storageId,
    void* pFileSystemCacheBuffer,
    size_t fileSystemCacheBufferSize,
    bool makeFileDataCacheAvailable,
    bool makePathBasedFileDataCacheAvailable
) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(CheckMountName(name)); // マウント名チェック
    NN_FS_RESULT_THROW_UNLESS(pFileSystemCacheBuffer != nullptr, ResultNullptrArgument());

    NN_FS_RESULT_DO(MountDataImpl(name, dataId, storageId, pFileSystemCacheBuffer, fileSystemCacheBufferSize, true, makeFileDataCacheAvailable, makePathBasedFileDataCacheAvailable));
    NN_RESULT_SUCCESS;
}

}}}
