﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_OperateRange.h>
#include <nn/fs/detail/fs_Newable.h>
#include "fs_IntrusiveList.h"

namespace nn { namespace fs {

    namespace fsa {
        class IFile;
    }

namespace detail{

    class FileSystemAccessor;
    struct FilePathHash;

    enum WriteState
    {
        WriteStateNon,
        WriteStateNeedsFlush,
        WriteStateFailed
    };

    class FileDataCacheAccessResult;

    class FileAccessor : public Newable
    {
        NN_DISALLOW_COPY(FileAccessor);

    public:

        FileAccessor(std::unique_ptr<fsa::IFile>&& file, FileSystemAccessor* pParent, int mode) NN_NOEXCEPT;
        ~FileAccessor() NN_NOEXCEPT;

        Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT;
        Result Write(int64_t offset, const void* buffer, size_t size, const WriteOption& option) NN_NOEXCEPT;
        Result Flush() NN_NOEXCEPT;
        Result SetSize(int64_t size) NN_NOEXCEPT;
        Result GetSize(int64_t *outValue) NN_NOEXCEPT;
        int GetOpenMode() const NN_NOEXCEPT;
        WriteState GetWriteState() NN_NOEXCEPT;
        FileSystemAccessor* GetParent() NN_NOEXCEPT
        {
            return m_pParent;
        }
        Result OperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT;

        IntrusiveListNode m_ListNode;

    public:
        void SetFilePathHash(std::unique_ptr<FilePathHash>&& filePathHash, int index) NN_NOEXCEPT;
        Result ReadWithoutCacheAccessLog(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option) NN_NOEXCEPT;

    private:
        Result UpdateLastResult(Result result) NN_NOEXCEPT;
        Result ReadWithCacheAccessLog(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option, bool usePathBasedFileDataCache, bool useFileDataCache) NN_NOEXCEPT;

    private:
        std::unique_ptr<fsa::IFile> m_Impl;
        FileSystemAccessor* const m_pParent;
        WriteState m_WriteState;
        Result m_LastWriteResult;
        const int m_OpenMode;
        std::unique_ptr<FilePathHash> m_FilePathHash;
        int m_FilePathHashIndex;
    };
}}}
