﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/detail/fs_Newable.h>

#include <nn/fs/fs_DbmRomTypes.h>
#include <nn/fs/fs_RomFsFileSystem.h>

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>

#include <nn/fs/detail/fs_LruFileDataCacheSystem.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

#include <nn/util/util_BitUtil.h>

#include "shim/fs_FileDataCacheAccessor.h"
#include "shim/fs_RomFsFileDataCache.h"

namespace nn { namespace fs {


namespace {

    nn::Result ConvertNcaCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultNcaCorrupted::Includes(result));

        if( nn::fs::ResultInvalidNcaFileSystemType::Includes(result) )
        {
            return nn::fs::ResultInvalidRomNcaFileSystemType();
        }
        else if( nn::fs::ResultInvalidAcidFileSize::Includes(result) )
        {
            return nn::fs::ResultInvalidRomAcidFileSize();
        }
        else if( nn::fs::ResultInvalidAcidSize::Includes(result) )
        {
            return nn::fs::ResultInvalidRomAcidSize();
        }
        else if( nn::fs::ResultInvalidAcid::Includes(result) )
        {
            return nn::fs::ResultInvalidRomAcid();
        }
        else if( nn::fs::ResultAcidVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomAcidVerificationFailed();
        }
        else if( nn::fs::ResultInvalidNcaSignature::Includes(result) )
        {
            return nn::fs::ResultInvalidRomNcaSignature();
        }
        else if( nn::fs::ResultNcaHeaderSignature1VerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomNcaHeaderSignature1VerificationFailed();
        }
        else if( nn::fs::ResultNcaHeaderSignature2VerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomNcaHeaderSignature2VerificationFailed();
        }
        else if( nn::fs::ResultNcaFsHeaderHashVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomNcaFsHeaderHashVerificationFailed();
        }
        else if( nn::fs::ResultInvalidNcaKeyIndex::Includes(result) )
        {
            return nn::fs::ResultInvalidRomNcaKeyIndex();
        }
        else if( nn::fs::ResultInvalidNcaFsHeaderHashType::Includes(result) )
        {
            return nn::fs::ResultInvalidRomNcaFsHeaderHashType();
        }
        else if( nn::fs::ResultInvalidNcaFsHeaderEncryptionType::Includes(result) )
        {
            return nn::fs::ResultInvalidRomNcaFsHeaderEncryptionType();
        }
        else if( nn::fs::ResultInvalidHierarchicalSha256BlockSize::Includes(result) )
        {
            return nn::fs::ResultInvalidRomHierarchicalSha256BlockSize();
        }
        else if( nn::fs::ResultInvalidHierarchicalSha256LayerCount::Includes(result) )
        {
            return nn::fs::ResultInvalidRomHierarchicalSha256LayerCount();
        }
        else if( nn::fs::ResultHierarchicalSha256BaseStorageTooLarge::Includes(result) )
        {
            return nn::fs::ResultRomHierarchicalSha256BaseStorageTooLarge();
        }
        else if( nn::fs::ResultHierarchicalSha256HashVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomHierarchicalSha256HashVerificationFailed();
        }
        else if( nn::fs::ResultNcaCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultRomNcaCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertIntegrityVerificationStorageCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result));

        if( nn::fs::ResultIncorrectIntegrityVerificationMagicCode::Includes(result) )
        {
            return nn::fs::ResultIncorrectRomIntegrityVerificationMagicCode();
        }
        else if( nn::fs::ResultInvalidZeroHash::Includes(result) )
        {
            return nn::fs::ResultInvalidRomZeroSignature();
        }
        else if( nn::fs::ResultNonRealDataVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomNonRealDataVerificationFailed();
        }
        else if( nn::fs::ResultInvalidHierarchicalIntegrityVerificationLayerCount::Includes(result) )
        {
            return nn::fs::ResultInvalidRomHierarchicalIntegrityVerificationLayerCount();
        }
        else if( nn::fs::ResultClearedRealDataVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultClearedRomRealDataVerificationFailed();
        }
        else if( nn::fs::ResultUnclearedRealDataVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultUnclearedRomRealDataVerificationFailed();
        }
        else if( nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultRomIntegrityVerificationStorageCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertPartitionFileSystemCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultPartitionFileSystemCorrupted::Includes(result));

        if( nn::fs::ResultInvalidSha256PartitionHashTarget::Includes(result) )
        {
            return nn::fs::ResultInvalidRomSha256PartitionHashTarget();
        }
        else if( nn::fs::ResultSha256PartitionHashVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomSha256PartitionHashVerificationFailed();
        }
        else if( nn::fs::ResultPartitionSignatureVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomPartitionSignatureVerificationFailed();
        }
        else if( nn::fs::ResultSha256PartitionSignatureVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomSha256PartitionSignatureVerificationFailed();
        }
        else if( nn::fs::ResultInvalidPartitionEntryOffset::Includes(result) )
        {
            return nn::fs::ResultInvalidRomPartitionEntryOffset();
        }
        else if( nn::fs::ResultInvalidSha256PartitionMetaDataSize::Includes(result) )
        {
            return nn::fs::ResultInvalidRomSha256PartitionMetaDataSize();
        }
        else if( nn::fs::ResultPartitionFileSystemCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultRomPartitionFileSystemCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertBuiltInStorageCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultBuiltInStorageCorrupted::Includes(result));

        if( nn::fs::ResultGptHeaderSignatureVerificationFailed::Includes(result) )
        {
            return nn::fs::ResultRomGptHeaderSignatureVerificationFailed();
        }
        else if( nn::fs::ResultBuiltInStorageCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultRomBuiltInStorageCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertHostFileSystemCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultHostFileSystemCorrupted::Includes(result));

        if( nn::fs::ResultHostEntryCorrupted::Includes(result) )
        {
            return nn::fs::ResultRomHostEntryCorrupted();
        }
        else if( nn::fs::ResultHostFileDataCorrupted::Includes(result) )
        {
            return nn::fs::ResultRomHostFileDataCorrupted();
        }
        else if( nn::fs::ResultHostFileCorrupted::Includes(result) )
        {
            return nn::fs::ResultRomHostFileCorrupted();
        }
        else if( nn::fs::ResultInvalidHostHandle::Includes(result) )
        {
            return nn::fs::ResultInvalidRomHostHandle();
        }
        else if( nn::fs::ResultHostFileSystemCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultRomHostFileSystemCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertDatabaseCorruptedResult(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(nn::fs::ResultDatabaseCorrupted::Includes(result));

        if( nn::fs::ResultInvalidAllocationTableBlock::Includes(result) )
        {
            return nn::fs::ResultInvalidRomAllocationTableBlock();
        }
        else if( nn::fs::ResultInvalidKeyValueListElementIndex::Includes(result) )
        {
            return nn::fs::ResultInvalidRomKeyValueListElementIndex();
        }
        else if( nn::fs::ResultDatabaseCorrupted::Includes(result) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultRomDatabaseCorrupted();
        }
        else
        {
            NN_SDK_ASSERT(false);
            return result;
        }
    }

    nn::Result ConvertRomFsDriverPrivateResult(nn::Result result) NN_NOEXCEPT
    {
        // through success
        if( result.IsSuccess() )
        {
            NN_RESULT_SUCCESS;
        }

        if( nn::fs::ResultUnsupportedVersion::Includes(result) )
        {
            return nn::fs::ResultUnsupportedRomVersion();
        }
        else if( nn::fs::ResultNcaCorrupted::Includes(result) )
        {
            return ConvertNcaCorruptedResult(result);
        }
        else if( nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result) )
        {
            return ConvertIntegrityVerificationStorageCorruptedResult(result);
        }
        else if( nn::fs::ResultBuiltInStorageCorrupted::Includes(result) )
        {
            return ConvertBuiltInStorageCorruptedResult(result);
        }
        else if( nn::fs::ResultPartitionFileSystemCorrupted::Includes(result) )
        {
            return ConvertPartitionFileSystemCorruptedResult(result);
        }
        else if( nn::fs::ResultFatFileSystemCorrupted::Includes(result) )
        {
            // FAT のエラーはどのパーティションで問題があったかが重要なのでリザルトを振り分けずにそのまま返す
            return result;
        }
        else if( nn::fs::ResultHostFileSystemCorrupted::Includes(result) )
        {
            return ConvertHostFileSystemCorruptedResult(result);
        }
        else if( nn::fs::ResultDatabaseCorrupted::Includes(result) )
        {
            return ConvertDatabaseCorruptedResult(result);
        }
        else if( nn::fs::ResultNotFound::Includes(result) )
        {
            return nn::fs::ResultPathNotFound();
        }
        else if( nn::fs::ResultInvalidOffset::Includes(result) )
        {
            return nn::fs::ResultOutOfRange();
        }
        else if( nn::fs::ResultPermissionDenied::Includes(result) )
        {
            return nn::fs::ResultTargetLocked();
        }
        else if( nn::fs::ResultFileNotFound::Includes(result)
                 || nn::fs::ResultIncompatiblePath::Includes(result) )
        {
            return nn::fs::ResultPathNotFound();
        }

        return result;
    }

    size_t CalculateRequiredWorkingMemorySize(const RomFileSystemInformation* pHeader) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHeader);

        return static_cast<size_t>(nn::util::align_up(
            pHeader->sizeBucketDirectory
                + pHeader->sizeDirectoryEntry
                + pHeader->sizeBucketFile
                + pHeader->sizeFileEntry,
            8
        ));
    }

    /**
    * @brief    ファイルからデータを読み込みます。
    *
    * @param[in]    pStorage    読み込むファイル
    * @param[in]    offset      読み込むオフセット
    * @param[in]    pBuffer     データの読み込み先バッファ
    * @param[in]    size        読み込むサイズ
    *
    * @pre          pStorage != nullptr
    * @pre          offset >= 0
    * @pre          pBuffer != nullptr
    */
    Result ReadFile(IStorage* pStorage, int64_t offset, void* pBuffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
        NN_SDK_REQUIRES(pBuffer != nullptr || size == 0);

        return ConvertRomFsDriverPrivateResult(pStorage->Read(offset, pBuffer, size));
    }

    /**
    * @brief    ファイルからヘッダを読み込みます
    *
    * @param[in]    pStorage    読み込むファイル
    * @param[in]    pBuffer     ヘッダの読み込み先バッファ
    *
    * @pre          pStorage != nullptr
    * @pre          pBuffer != nullptr
    */
    Result ReadFileHeader(IStorage* pStorage, RomFileSystemInformation* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        return ReadFile(pStorage, 0, pBuffer, sizeof(RomFileSystemInformation));
    }


    class RomFsFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
    {
    public:

        RomFsFile(RomFsFileSystem* pParent, int64_t head, int64_t tail) NN_NOEXCEPT
        : m_pParent(pParent),
          m_Head(head),
          m_Tail(tail)
        {
        }

        virtual ~RomFsFile() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        Result VerifyArguments(
            size_t* pOutValue,
            int64_t offset,
            void* pBuffer,
            size_t size,
            const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_RESULT_DO(DryRead(pOutValue, offset, size, option, nn::fs::OpenMode_Read));

            NN_UNUSED(pBuffer);
            NN_SDK_REQUIRES_NOT_NULL(m_pParent->GetBaseStorage());
            NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
            NN_SDK_REQUIRES(pBuffer != nullptr || size == 0);

            NN_RESULT_SUCCESS;
        }

        Result ConvertToApplicationResult(Result result) NN_NOEXCEPT
        {
            return ConvertRomFsDriverPrivateResult(result);
        }

        virtual Result DoRead(
            size_t* pOutValue,
            int64_t offset,
            void* pBuffer,
            size_t size,
            const nn::fs::ReadOption& option
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            size_t readSize = 0;
            NN_RESULT_DO(VerifyArguments(&readSize, offset, pBuffer, size, option));

            NN_RESULT_DO(ConvertToApplicationResult(
                m_pParent->GetBaseStorage()->Read(offset + m_Head, pBuffer, readSize)));
            *pOutValue = readSize;

            NN_RESULT_SUCCESS;
        }

        virtual Result DoGetSize(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);

            *pOutValue = GetSize();
            NN_RESULT_SUCCESS;
        }

        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
            case OperationId::QueryRange:
                {
                    NN_RESULT_THROW_UNLESS(offset >= 0, ResultOutOfRange());
                    NN_RESULT_THROW_UNLESS(GetSize() >= offset, ResultOutOfRange());

                    auto operateSize = size;
                    if( offset + operateSize > GetSize() || offset + operateSize < offset )
                    {
                        operateSize = GetSize() - offset;
                    }

                    return m_pParent->GetBaseStorage()->OperateRange(
                        outBuffer,
                        outBufferSize,
                        operationId,
                        m_Head + offset,
                        operateSize,
                        inBuffer,
                        inBufferSize);
                }
            default:
                return nn::fs::ResultUnsupportedOperation();
            }
        }

        virtual Result DoWrite(
            int64_t offset, const void* pBuffer, size_t size, const nn::fs::WriteOption& option
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(pBuffer);
            NN_UNUSED(size);
            NN_UNUSED(option);
            return ResultUnsupportedOperation();
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            return ResultUnsupportedOperation();
        }

        int64_t GetOffset() const NN_NOEXCEPT
        {
            return m_Head;
        }

        int64_t GetSize() const NN_NOEXCEPT
        {
            return m_Tail - m_Head;
        }

        IStorage* GetStorage() NN_NOEXCEPT
        {
            return m_pParent->GetBaseStorage();
        }

    private:
        RomFsFileSystem* m_pParent;
        int64_t m_Head;
        int64_t m_Tail;

    };

    class RomFsDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
    {
    public:

        RomFsDirectory(
            RomFsFileSystem* pParent,
            const RomFsFileSystem::RomFileTable::FindPosition& findPosition,
            nn::fs::OpenDirectoryMode mode
        ) NN_NOEXCEPT
        : m_pParent(pParent),
          m_FindPosition(findPosition),
          m_FirstFindPosition(findPosition),
          m_Mode(mode)
        {
        }

        virtual ~RomFsDirectory() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual Result DoRead(
            int64_t* pOutValue, nn::fs::DirectoryEntry* pEntryBuffer, int64_t entryBufferCount
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            return ReadInternal(pOutValue, &m_FindPosition, pEntryBuffer, entryBufferCount);
        }

        virtual Result DoGetEntryCount(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            // FIXME: 毎回カウントする必要はない
            FindPosition findPosition = m_FirstFindPosition;
            return ReadInternal(pOutValue, &findPosition, nullptr, 0);
        }

    private:
        typedef RomFsFileSystem::RomFileTable::FindPosition FindPosition;

    private:
        Result ReadInternal(
            int64_t* pOutValue,
            FindPosition* pFindPosition,
            nn::fs::DirectoryEntry* entryBuffer,
            int64_t entryBufferCount
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);
            NN_SDK_REQUIRES_NOT_NULL(pFindPosition);
            static const size_t BufNameLength = nn::fs::EntryNameLengthMax + 1;
            const auto bufNameDeleter
                = [](RomPathChar* ptr) NN_NOEXCEPT
                {
                    nn::fs::detail::Deallocate(ptr, sizeof(RomPathChar) * BufNameLength);
                };
            std::unique_ptr<RomPathChar, decltype(bufNameDeleter)> bufName(
                reinterpret_cast<RomPathChar*>(
                    nn::fs::detail::Allocate(sizeof(RomPathChar) * BufNameLength)
                ),
                bufNameDeleter
            );
            NN_RESULT_THROW_UNLESS(bufName != nullptr, nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemE());

            int32_t i = 0;

            if( (m_Mode & nn::fs::OpenDirectoryMode_Directory) != 0 )
            {
                for( ; i < entryBufferCount || entryBuffer == nullptr; ++i )
                {
                    NN_RESULT_TRY(
                        m_pParent->GetRomFileTable()->FindNextDirectory(
                            bufName.get(), pFindPosition, BufNameLength
                        )
                    )
                        NN_RESULT_CATCH(ResultDbmFindFinished)
                        {
                            break;
                        }
                    NN_RESULT_END_TRY

                    if( entryBuffer )
                    {
                        NN_SDK_ASSERT(i < entryBufferCount);
                        NN_RESULT_THROW_UNLESS(
                            strnlen(bufName.get(), BufNameLength) < BufNameLength,
                            nn::fs::ResultTooLongPath());
                        strncpy(entryBuffer[i].name, bufName.get(), nn::fs::EntryNameLengthMax);
                        entryBuffer[i].name[nn::fs::EntryNameLengthMax] = '\0';
                        entryBuffer[i].directoryEntryType = nn::fs::DirectoryEntryType_Directory;
                        entryBuffer[i].fileSize = 0;
                    }
                }
            }

            if( (m_Mode & nn::fs::OpenDirectoryMode_File) != 0 )
            {
                for( ; i < entryBufferCount || entryBuffer == nullptr; ++i )
                {
                    RomFsFileSystem::RomFileTable::StoragePosition filePosition
                        = pFindPosition->nextPositionFile;

                    NN_RESULT_TRY(
                        m_pParent->GetRomFileTable()->FindNextFile(
                            bufName.get(), pFindPosition, BufNameLength
                        )
                    )
                        NN_RESULT_CATCH(ResultDbmFindFinished)
                        {
                            break;
                        }
                    NN_RESULT_END_TRY

                    if( entryBuffer )
                    {
                        NN_SDK_ASSERT(i < entryBufferCount);
                        NN_RESULT_THROW_UNLESS(
                            strnlen(bufName.get(), BufNameLength) < BufNameLength,
                            nn::fs::ResultTooLongPath());
                        strncpy(entryBuffer[i].name, bufName.get(), nn::fs::EntryNameLengthMax);
                        entryBuffer[i].name[nn::fs::EntryNameLengthMax] = '\0';
                        RomFsFileSystem::RomFileTable::FileInfo fileInfo = {};
                        NN_RESULT_DO(
                            m_pParent->GetRomFileTable()->OpenFile(
                                &fileInfo,
                                m_pParent->GetRomFileTable()->PositionToFileId(filePosition)
                            )
                        );
                        entryBuffer[i].directoryEntryType = nn::fs::DirectoryEntryType_File;
                        entryBuffer[i].fileSize = fileInfo.size.Get();
                    }
                }
            }

            *pOutValue = i;
            NN_RESULT_SUCCESS;
        }

    private:
        RomFsFileSystem* m_pParent;
        FindPosition m_FindPosition;
        FindPosition m_FirstFindPosition;
        nn::fs::OpenDirectoryMode m_Mode;

    };

} // namespace

    RomFsFileSystem::RomFsFileSystem() NN_NOEXCEPT
    : m_pBaseStorage(NULL)
    {
    }

    RomFsFileSystem::~RomFsFileSystem() NN_NOEXCEPT
    {
    }

    IStorage* RomFsFileSystem::GetBaseStorage() NN_NOEXCEPT
    {
        return m_pBaseStorage;
    }
    RomFsFileSystem::RomFileTable* RomFsFileSystem::GetRomFileTable() NN_NOEXCEPT
    {
        return &m_RomFileTable;
    }

    Result RomFsFileSystem::GetRequiredWorkingMemorySize(
        size_t* pOutValue, IStorage* pStorage
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pStorage);

        RomFileSystemInformation header = {};
        NN_RESULT_DO(ReadFileHeader(pStorage, &header));

        *pOutValue = CalculateRequiredWorkingMemorySize(&header);

        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::Initialize(
        std::unique_ptr<IStorage>&& pBaseStorage,
        void* pWorkingMemory,
        size_t workingMemorySize,
        bool isFileSystemCacheUsed
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!isFileSystemCacheUsed || pWorkingMemory != nullptr);

        m_pUniqueBaseStorage = std::move(pBaseStorage);
        return Initialize(m_pUniqueBaseStorage.get(), pWorkingMemory, workingMemorySize, isFileSystemCacheUsed);
    }

    Result RomFsFileSystem::Initialize(
        IStorage* pBaseStorage, void* pWorkingMemory, size_t workingMemorySize, bool isFileSystemCacheUsed
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
        NN_SDK_REQUIRES(!isFileSystemCacheUsed || pWorkingMemory != nullptr);

        RomFileSystemInformation header = {};
        NN_RESULT_DO(ReadFileHeader(pBaseStorage, &header));

        // ファイルシステムのヘッダ情報をキャッシュします
        if( isFileSystemCacheUsed )
        {
            // 必要なワーキングメモリサイズのチェック
            size_t requiredWorkingMemorySize = CalculateRequiredWorkingMemorySize(&header);
            NN_SDK_REQUIRES_LESS_EQUAL(requiredWorkingMemorySize, workingMemorySize);
            if( requiredWorkingMemorySize > workingMemorySize )
            {
                return nn::fs::ResultPreconditionViolation();
            }

            Bit8* pBuf = static_cast<Bit8*>(pWorkingMemory);
            auto pBufDirectoryBucket = pBuf;
            NN_RESULT_DO(
                ReadFile(
                    pBaseStorage,
                    header.offsetBucketDirectory,
                    pBufDirectoryBucket,
                    static_cast<size_t>(header.sizeBucketDirectory)
                )
            );

            pBuf += header.sizeBucketDirectory;
            auto pBufDirectoryEntry = pBuf;
            NN_RESULT_DO(
                ReadFile(
                    pBaseStorage,
                    header.offsetDirectoryEntry,
                    pBufDirectoryEntry,
                    static_cast<size_t>(header.sizeDirectoryEntry)
                )
            );

            pBuf += header.sizeDirectoryEntry;
            auto pBufFileBucket = pBuf;
            NN_RESULT_DO(
                ReadFile(
                    pBaseStorage,
                    header.offsetBucketFile,
                    pBufFileBucket,
                    static_cast<size_t>(header.sizeBucketFile)
                )
            );

            pBuf += header.sizeBucketFile;
            auto pBufFileEntry = pBuf;
            NN_RESULT_DO(
                ReadFile(
                    pBaseStorage,
                    header.offsetFileEntry,
                    pBufFileEntry,
                    static_cast<size_t>(header.sizeFileEntry)
                )
            );

            m_pStorageDirectoryBucket.reset(
                new MemoryStorage(pBufDirectoryBucket, header.sizeBucketDirectory)
            );
            m_pStorageDirectoryEntry.reset(
                new MemoryStorage(pBufDirectoryEntry, header.sizeDirectoryEntry)
            );
            m_pStorageFileBucket.reset(
                new MemoryStorage(pBufFileBucket, header.sizeBucketFile)
            );
            m_pStorageFileEntry.reset(
                new MemoryStorage(pBufFileEntry, header.sizeFileEntry)
            );
        }
        else
        {
            m_pStorageDirectoryBucket.reset(
                new SubStorage(pBaseStorage, header.offsetBucketDirectory, header.sizeBucketDirectory)
            );
            m_pStorageDirectoryEntry.reset(
                new SubStorage(pBaseStorage, header.offsetDirectoryEntry, header.sizeDirectoryEntry)
            );
            m_pStorageFileBucket.reset(
                new SubStorage(pBaseStorage, header.offsetBucketFile, header.sizeBucketFile)
            );
            m_pStorageFileEntry.reset(
                new SubStorage(pBaseStorage, header.offsetFileEntry, header.sizeFileEntry)
            );
        }

        if( m_pStorageDirectoryBucket == nullptr ||
            m_pStorageDirectoryEntry  == nullptr ||
            m_pStorageFileBucket      == nullptr ||
            m_pStorageFileEntry       == nullptr )
        {
            return nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemA();
        }

        // ファイルテーブルをマウントします。
        {
            SubStorage subStorageDirectoryBucket(
                           m_pStorageDirectoryBucket.get(),
                           0,
                           header.sizeBucketDirectory
                       );
            SubStorage subStorageDirectoryEntry(
                           m_pStorageDirectoryEntry.get(),
                           0,
                           header.sizeDirectoryEntry
                       );
            SubStorage subStorageFileBucket(
                           m_pStorageFileBucket.get(),
                           0,
                           header.sizeBucketFile
                       );
            SubStorage subStorageFileEntry(
                           m_pStorageFileEntry.get(),
                           0,
                           header.sizeFileEntry
                       );
            NN_RESULT_DO(
                m_RomFileTable.Initialize(
                    subStorageDirectoryBucket,
                    subStorageDirectoryEntry,
                    subStorageFileBucket,
                    subStorageFileEntry
                )
            );
        }

        m_EntrySize = header.offsetFileBody;
        m_pBaseStorage = pBaseStorage;

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result RomFsFileSystem::DoOpenFile(
        std::unique_ptr<nn::fs::fsa::IFile>* pOutValue, const char* path, nn::fs::OpenMode mode
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(path);

        // 書き込みモードで開こうとしたら失敗
        NN_RESULT_THROW_UNLESS(
            (mode & (OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)) == OpenMode_Read,
            ResultInvalidArgument()
        );

        RomFsFileSystem::RomFileTable::FileInfo fileInfo = {};
        NN_RESULT_DO(GetFileInfo(&fileInfo, path));

        std::unique_ptr<RomFsFile> file(
            new RomFsFile(
                this,
                m_EntrySize + fileInfo.offset.Get(),
                m_EntrySize + fileInfo.offset.Get() + fileInfo.size.Get()
            )
        );
        if( file == nullptr )
        {
            return nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemB();
        }

        *pOutValue = std::move(file);
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoOpenDirectory(
        std::unique_ptr<nn::fs::fsa::IDirectory>* pOutValue,
        const char* path,
        nn::fs::OpenDirectoryMode mode
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(path);

        RomFsFileSystem::RomFileTable::FindPosition findPosition = {};
        NN_RESULT_TRY(m_RomFileTable.FindOpen(&findPosition, path))
            NN_RESULT_CATCH(ResultDbmNotFound)
            {
                return nn::fs::ResultPathNotFound();
            }
            NN_RESULT_CATCH(ResultDbmInvalidOperation)
            {
                return nn::fs::ResultPathNotFound();
            }
        NN_RESULT_END_TRY

        std::unique_ptr<RomFsDirectory> directory(new RomFsDirectory(this, findPosition, mode));
        if( directory == nullptr )
        {
            return nn::fs::ResultAllocationMemoryFailedInRomFsFileSystemC();
        }

        *pOutValue = std::move(directory);
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoGetEntryType(
        nn::fs::DirectoryEntryType* pOutValue, const char* path
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(path);

        RomDirectoryInfo directoryInfo = {};

        NN_RESULT_TRY(m_RomFileTable.GetDirectoryInformation(&directoryInfo, path))
            NN_RESULT_CATCH(ResultDbmNotFound) // 親ディレクトリが見つからない場合
            {
                return nn::fs::ResultPathNotFound();
            }
            NN_RESULT_CATCH(ResultDbmInvalidOperation) // 対象がファイルである場合（を含む）
            {
                RomFsFileSystem::RomFileTable::FileInfo fileInfo = {};
                NN_RESULT_DO(GetFileInfo(&fileInfo, path));

                *pOutValue = nn::fs::DirectoryEntryType_File;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *pOutValue = nn::fs::DirectoryEntryType_Directory;
        NN_RESULT_SUCCESS;
    }


    Result RomFsFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_UNUSED(size);
        NN_UNUSED(option);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FS_RESULT_DO(ResultUnsupportedOperation());
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoRenameDirectory(
        const char* currentPath, const char* newPath
    ) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoCommit() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
    {
        NN_UNUSED(counter);
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoRollback() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::DoGetFreeSpaceSize(int64_t* pOutValue, const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::DoGetTotalSpaceSize(int64_t* pOutValue, const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(path);
        return ResultUnsupportedOperation();
    }

    Result RomFsFileSystem::GetFileBaseOffset(int64_t* pOutValue, const char* path) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(path);

        RomFsFileSystem::RomFileTable::FileInfo fi;
        NN_RESULT_DO(GetFileInfo(&fi, path));
        *pOutValue = m_EntrySize + fi.offset.Get();
        NN_RESULT_SUCCESS;
    }

    Result RomFsFileSystem::GetFileInfo(
        RomFileTable::FileInfo* pOutValue, const char* path
    ) NN_NOEXCEPT
    {
        NN_RESULT_TRY(m_RomFileTable.OpenFile(pOutValue, path))
            NN_RESULT_CATCH(ResultDbmNotFound)
            {
                return nn::fs::ResultPathNotFound();
            }
            NN_RESULT_CATCH(ResultDbmInvalidOperation)
            {
                return nn::fs::ResultPathNotFound();
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    //---------------------------------------------------------------------------------------------

    RomFsFileDataCache::RomFsFileDataCache() NN_NOEXCEPT
        : m_pLruFileDataCacheSystem(nullptr)
    {
    }

    RomFsFileDataCache::~RomFsFileDataCache() NN_NOEXCEPT
    {
        Finalize();
    }

    Result RomFsFileDataCache::Initialize(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pLruFileDataCacheSystem == nullptr);

        const uintptr_t pStart = reinterpret_cast<uintptr_t>(pBuffer);
        const uintptr_t pEnd = pStart + bufferSize;
        NN_ABORT_UNLESS(pStart <= pEnd);

        uintptr_t p = util::align_up(pStart, NN_ALIGNOF(detail::LruFileDataCacheSystem));
        NN_RESULT_THROW_UNLESS(p + sizeof(detail::LruFileDataCacheSystem) <= pEnd, ResultFileDataCacheMemorySizeTooSmall());
        m_pLruFileDataCacheSystem = new (reinterpret_cast<detail::LruFileDataCacheSystem*>(p)) detail::LruFileDataCacheSystem();
        p += sizeof(detail::LruFileDataCacheSystem);

        NN_RESULT_TRY(m_pLruFileDataCacheSystem->Initialize(reinterpret_cast<void*>(p), bufferSize - (p - reinterpret_cast<uintptr_t>(pBuffer))))
            NN_RESULT_CATCH_ALL
            {
                m_pLruFileDataCacheSystem->~LruFileDataCacheSystem();
                m_pLruFileDataCacheSystem = nullptr;

                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    void RomFsFileDataCache::Finalize() NN_NOEXCEPT
    {
        if (m_pLruFileDataCacheSystem)
        {
            m_pLruFileDataCacheSystem->~LruFileDataCacheSystem();
            m_pLruFileDataCacheSystem = nullptr;
        }
    }

    Result RomFsFileDataCache::DoRead(
        fsa::IFile* pFile,
        size_t* outValue,
        int64_t offset,
        void* pBuffer,
        size_t size,
        const ReadOption& option,
        detail::FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pLruFileDataCacheSystem);

        // TORIAEZU: FileDataCache はまだ一種類しかないので RomFsFile と仮定してしまう
        // TODO: もう少し厳密にチェックするようにしたい
        RomFsFile* pRomFsFile = static_cast<RomFsFile*>(pFile);

        NN_RESULT_DO(pRomFsFile->VerifyArguments(outValue, offset, pBuffer, size, option));

        int64_t offsetBase = pRomFsFile->GetOffset();
        int64_t offsetLimit = offsetBase + pRomFsFile->GetSize();

        NN_RESULT_DO(pRomFsFile->ConvertToApplicationResult(
            m_pLruFileDataCacheSystem->Read(pRomFsFile->GetStorage(), offsetBase + offset, pBuffer, *outValue, offsetBase, offsetLimit, pAccessResult)));

        NN_RESULT_SUCCESS;
    }

    void RomFsFileDataCache::Purge(fsa::IFileSystem* pFileSystem) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pLruFileDataCacheSystem);

        // TORIAEZU: FileDataCache はまだ一種類しかないので RomFsFileSystem と仮定してしまう
        // TODO: もう少し厳密にチェックするようにしたい
        RomFsFileSystem* pRomFsFileSystem = static_cast<RomFsFileSystem*>(pFileSystem);

        m_pLruFileDataCacheSystem->Purge(pRomFsFileSystem->GetBaseStorage());
    }

}}  // namespace nn::fs
