﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_CharacterEncoding.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_PathUtility.h>
#include "fsa/fs_MountUtility.h"

namespace nn { namespace fs {


namespace {

    class PathVerifier
    {
    public:
        PathVerifier() NN_NOEXCEPT
        {
            auto pConvertedInvalidChar = m_InvalidChars;
            for( auto pInvalidChar = ":*?<>|"; *pInvalidChar != '\0'; ++pInvalidChar )
            {
                NN_SDK_ASSERT_LESS(pConvertedInvalidChar, std::end(m_InvalidChars));
                const auto result = nn::util::ConvertCharacterUtf8ToUtf32(
                    pConvertedInvalidChar,
                    pInvalidChar);
                NN_SDK_ASSERT_EQUAL(result, nn::util::CharacterEncodingResult_Success);
                NN_UNUSED(result);
                ++pConvertedInvalidChar;
            }
            NN_SDK_ASSERT_EQUAL(pConvertedInvalidChar, std::end(m_InvalidChars));

            auto pConvertedSeparator = m_Separators;
            for( auto pSeparator = "/\\"; *pSeparator != '\0'; ++pSeparator )
            {
                NN_SDK_ASSERT_LESS(pConvertedSeparator, std::end(m_Separators));
                const auto result = nn::util::ConvertCharacterUtf8ToUtf32(
                    pConvertedSeparator,
                    pSeparator);
                NN_SDK_ASSERT_EQUAL(result, nn::util::CharacterEncodingResult_Success);
                NN_UNUSED(result);
                ++pConvertedSeparator;
            }
            NN_SDK_ASSERT_EQUAL(pConvertedSeparator, std::end(m_Separators));
        }

        Result Verify(const char* path, int pathLengthMax, int nameLengthMax) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(path);

            auto currentPath = path;
            auto nameLength = 0;

            for( auto pathLength = 0;
                pathLength <= pathLengthMax && nameLength <= nameLengthMax;
                ++pathLength )
            {
                if( *currentPath == '\0' )
                {
                    NN_RESULT_SUCCESS;
                }

                nn::util::CharacterEncodingResult result;

                char pathBuffer[4] = {};
                result = nn::util::PickOutCharacterFromUtf8String(pathBuffer, &currentPath);
                if( result != nn::util::CharacterEncodingResult_Success )
                {
                    return nn::fs::ResultInvalidCharacter();
                }

                uint32_t pathChar = 0;
                result = nn::util::ConvertCharacterUtf8ToUtf32(&pathChar, pathBuffer);
                if( result != nn::util::CharacterEncodingResult_Success )
                {
                    return nn::fs::ResultInvalidCharacter();
                }

                for( const auto invalidChar : m_InvalidChars )
                {
                    if( pathChar == invalidChar )
                    {
                        return nn::fs::ResultInvalidCharacter();
                    }
                }

                ++nameLength;
                for( const auto separator : m_Separators )
                {
                    if( pathChar == separator )
                    {
                        nameLength = 0;
                        break;
                    }
                }
            }

            return nn::fs::ResultTooLongPath();
        }

    private:
        uint32_t m_InvalidChars[6];
        uint32_t m_Separators[2];
    } g_PathVerifier;

}

    Result VerifyPath(const char* path, int pathLengthMax, int nameLengthMax) NN_NOEXCEPT
    {
        return g_PathVerifier.Verify(path, pathLengthMax, nameLengthMax);
    }

}}
