﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief    侵入型リストです。
* @details  侵入型リストを定義します。
*/

#pragma once

#include <iterator>

namespace nn { namespace fs {

/**
* @brief    リストノードです。
* @details  リストノードを定義します。
*/
class IntrusiveListNode
{
    NN_DISALLOW_COPY(IntrusiveListNode);
public:
    /**
    * @brief    コンストラクタです。
    * @details  コンストラクタを定義します。
    */
    IntrusiveListNode() NN_NOEXCEPT
    {
        m_Prev = this;
        m_Next = this;
    }

    /**
    * @brief    前の要素を取得します。
    * @return   前の要素です。
    * @details  前の要素を取得するメソッドです。
    */
    IntrusiveListNode* GetPrev() NN_NOEXCEPT
    {
        return m_Prev;
    }

    /**
    * @brief    前の要素を取得します。
    * @return   前の要素です。
    * @details  前の要素を取得するメソッドです。
    */
    const IntrusiveListNode* GetPrev() const NN_NOEXCEPT
    {
        return m_Prev;
    }

    /**
    * @brief    次の要素を取得します。
    * @return   次の要素です。
    * @details  次の要素を取得するメソッドです。
    */
    IntrusiveListNode* GetNext() NN_NOEXCEPT
    {
        return m_Next;
    }

    /**
    * @brief    次の要素を取得します。
    * @return   次の要素です。
    * @details  次の要素を取得するメソッドです。
    */
    const IntrusiveListNode* GetNext() const NN_NOEXCEPT
    {
        return m_Next;
    }

    /**
    * @brief    前に要素を連結します。
    * @param[in]    node    連結する要素を指定します。
    * @details  前に要素を連結するメソッドです。
    */
    void LinkPrev(IntrusiveListNode* node) NN_NOEXCEPT
    {
        LinkPrev(node, node);
    }

    /**
    * @brief    前に要素列を連結します。
    * @param[in]    first   連結する要素列の先頭要素を指定します。
    * @param[in]    last    連結する要素列の終端要素を指定します。
    * @details  前に要素列を連結するメソッドです。
    */
    void LinkPrev(IntrusiveListNode* first, IntrusiveListNode* last) NN_NOEXCEPT
    {
        IntrusiveListNode* node = last->m_Prev;
        first->m_Prev = m_Prev;
        node->m_Next = this;
        m_Prev->m_Next = first;
        m_Prev = node;
    }

    /**
    * @brief    次に要素を連結します。
    * @param[in]    node    連結する要素を指定します。
    * @details  次に要素を連結するメソッドです。
    */
    void LinkNext(IntrusiveListNode* node) NN_NOEXCEPT
    {
        LinkNext(node, node);
    }

    /**
    * @brief    次に要素列を連結します。
    * @param[in]    first   連結する要素列の先頭要素を指定します。
    * @param[in]    last    連結する要素列の終端要素を指定します。
    * @details  次に要素列を連結するメソッドです。
    */
    void LinkNext(IntrusiveListNode* first, IntrusiveListNode* last) NN_NOEXCEPT
    {
        IntrusiveListNode* node = last->m_Prev;
        first->m_Prev = this;
        node->m_Next = m_Next;
        m_Next->m_Prev = node;
        m_Next = first;
    }

    /**
    * @brief    要素の連結を切断します。
    * @details  要素の連結を切断するメソッドです。
    *           自身を切断します。
    */
    void Unlink() NN_NOEXCEPT
    {
        Unlink(m_Next);
    }

    /**
    * @brief    要素列の連結を切断します。
    * @param[in]    last    要素列の終端要素を指定します。
    * @details  要素列の連結を切断するメソッドです。
    *           自身を要素列の開始要素として切断します。
    */
    void Unlink(IntrusiveListNode* last) NN_NOEXCEPT
    {
        IntrusiveListNode* node = last->m_Prev;
        m_Prev->m_Next = last;
        last->m_Prev = m_Prev;
        node->m_Next = this;
        m_Prev = node;
    }

    /**
    * @brief    他の要素に連結しているかを取得します。
    * @return   他の要素に連結していればtrueを、それ以外ならばfalseを返します。
    * @details  他の要素に連結しているかを取得するメソッドです。
    */
    bool IsLinked() const NN_NOEXCEPT
    {
        return m_Next != this;
    }

private:
    IntrusiveListNode* m_Prev;
    IntrusiveListNode* m_Next;
};

/**
* @brief    nn::util::detail名前空間です。
* @details  外部に公開しない要素が定義されます。
*/
namespace detail
{
    class IntrusiveListImplementation
    {
        NN_DISALLOW_COPY(IntrusiveListImplementation);
    public:
        class const_iterator
        {
        public:
            typedef const IntrusiveListNode value_type;
            typedef int difference_type;
            typedef value_type* pointer;
            typedef value_type& reference;
            typedef std::bidirectional_iterator_tag iterator_category;

        public:
            explicit const_iterator(pointer node) NN_NOEXCEPT
                : m_Node(node)
            {
            }

            reference operator*() const NN_NOEXCEPT
            {
                return *m_Node;
            }

            pointer operator->() const NN_NOEXCEPT
            {
                return m_Node;
            }

            const_iterator& operator++() NN_NOEXCEPT
            {
                m_Node = m_Node->GetNext();
                return *this;
            }

            const_iterator operator++(int) NN_NOEXCEPT
            {
                const_iterator temporary(*this);
                ++(*this);
                return temporary;
            }

            const_iterator& operator--() NN_NOEXCEPT
            {
                m_Node = m_Node->GetPrev();
                return *this;
            }

            const_iterator operator--(int) NN_NOEXCEPT
            {
                const_iterator temporary(*this);
                --(*this);
                return temporary;
            }

            bool operator==(const const_iterator& target) const NN_NOEXCEPT
            {
                return m_Node == target.m_Node;
            }

            bool operator!=(const const_iterator& target) const NN_NOEXCEPT
            {
                return m_Node != target.m_Node;
            }

        private:
            pointer m_Node;
        };

        class iterator
        {
        public:
            typedef IntrusiveListNode value_type;
            typedef int difference_type;
            typedef value_type* pointer;
            typedef value_type& reference;
            typedef std::bidirectional_iterator_tag iterator_category;

        public:
            explicit iterator(pointer node) NN_NOEXCEPT
                : m_Node(node)
            {
            }

            NN_IMPLICIT operator const_iterator() const NN_NOEXCEPT
            {
                return const_iterator(m_Node);
            }

            reference operator*() const NN_NOEXCEPT
            {
                return *m_Node;
            }

            pointer operator->() const NN_NOEXCEPT
            {
                return m_Node;
            }

            iterator& operator++() NN_NOEXCEPT
            {
                m_Node = m_Node->GetNext();
                return *this;
            }

            iterator operator++(int) NN_NOEXCEPT
            {
                iterator temporary(*this);
                ++(*this);
                return temporary;
            }

            iterator& operator--() NN_NOEXCEPT
            {
                m_Node = m_Node->GetPrev();
                return *this;
            }

            iterator operator--(int) NN_NOEXCEPT
            {
                iterator temporary(*this);
                --(*this);
                return temporary;
            }

            bool operator==(const iterator& target) const NN_NOEXCEPT
            {
                return m_Node == target.m_Node;
            }

            bool operator!=(const iterator& target) const NN_NOEXCEPT
            {
                return m_Node != target.m_Node;
            }

        private:
            pointer m_Node;
        };

        typedef IntrusiveListNode value_type;
        typedef int difference_type;
        typedef value_type* pointer;
        typedef value_type& reference;
        typedef const value_type& const_reference;

    public:
        IntrusiveListImplementation() NN_NOEXCEPT
            : m_Root()
        {
        }

        void push_back(reference node) NN_NOEXCEPT
        {
            m_Root.LinkPrev(&node);
        }

        void push_front(reference node) NN_NOEXCEPT
        {
            m_Root.LinkNext(&node);
        }

        void pop_back() NN_NOEXCEPT
        {
            m_Root.GetPrev()->Unlink();
        }

        void pop_front() NN_NOEXCEPT
        {
            m_Root.GetNext()->Unlink();
        }

        reference back() NN_NOEXCEPT
        {
            return *m_Root.GetPrev();
        }

        const_reference back() const NN_NOEXCEPT
        {
            return *m_Root.GetPrev();
        }

        reference front() NN_NOEXCEPT
        {
            return *m_Root.GetNext();
        }

        const_reference front() const NN_NOEXCEPT
        {
            return *m_Root.GetNext();
        }

        iterator begin() NN_NOEXCEPT
        {
            return iterator(m_Root.GetNext());
        }

        const_iterator begin() const NN_NOEXCEPT
        {
            return const_iterator(m_Root.GetNext());
        }

        iterator end() NN_NOEXCEPT
        {
            return iterator(&m_Root);
        }

        const_iterator end() const NN_NOEXCEPT
        {
            return const_iterator(&m_Root);
        }

        int size() const NN_NOEXCEPT
        {
            return static_cast<int>(std::distance(begin(), end()));
        }

        bool empty() const NN_NOEXCEPT
        {
            return !m_Root.IsLinked();
        }

        iterator erase(const_iterator position) NN_NOEXCEPT
        {
            if (position == end())
            {
                return end();
            }
            iterator temporary(iterator(const_cast<IntrusiveListNode*>(&*position)));
            (temporary++)->Unlink();
            return temporary;
        }

        void clear() NN_NOEXCEPT
        {
            while (!empty())
            {
                pop_front();
            }
        }

        iterator insert(const_iterator position, reference node) NN_NOEXCEPT
        {
            iterator(const_cast<IntrusiveListNode*>(&*position))->LinkPrev(&node);
            return iterator(&node);
        }

        void splice(const_iterator position, IntrusiveListImplementation& other) NN_NOEXCEPT
        {
            SpliceImplementation(position, other.begin(), other.end());
        }

        void splice(const_iterator position, IntrusiveListImplementation& other, iterator element) NN_NOEXCEPT
        {
            NN_UNUSED(other);
            iterator nextIterator(element);
            std::advance(nextIterator, 1);
            SpliceImplementation(position, element, nextIterator);
        }

        void splice(const_iterator position, IntrusiveListImplementation& other, const_iterator first, const_iterator last) NN_NOEXCEPT
        {
            NN_UNUSED(other);
            SpliceImplementation(position, first, last);
        }

    private:
        void SpliceImplementation(const_iterator position, const_iterator first, const_iterator last) NN_NOEXCEPT
        {
            if (first == last)
            {
                return;
            }
            iterator iteratorPosition(iterator(const_cast<IntrusiveListNode*>(&*position)));
            iterator iteratorFirst(iterator(const_cast<IntrusiveListNode*>(&*first)));
            iterator iteratorLast(iterator(const_cast<IntrusiveListNode*>(&*last)));
            iteratorFirst->Unlink(&*iteratorLast);
            iteratorPosition->LinkPrev(&*iteratorFirst, &*iteratorFirst);
        }

    private:
        IntrusiveListNode m_Root;
    };
}

/**
* @brief    侵入型リストです。
* @details  侵入型リストを定義します。
*/
template<class T, class NodeTraits>
class IntrusiveList
{
    NN_DISALLOW_COPY(IntrusiveList);
public:
    /**
    * @brief    constイテレータです。
    * @details  constイテレータを定義します。
    */
    class const_iterator
    {
    public:
        typedef const T value_type; //!< value_type型を定義します。
        typedef int difference_type; //!< difference_type型を定義します。
        typedef value_type* pointer; //!< pointer型を定義します。
        typedef value_type& reference; //!< reference型を定義します。
        typedef std::bidirectional_iterator_tag iterator_category; //!< iterator_category型を定義します。

    public:
        /**
        * @brief    コンストラクタです。
        * @param[in]    sourceIterator  イテレータを指定します。
        * @details  コンストラクタを定義します。
        *           ユーザによって呼ばれることは想定していません。
        */
        explicit const_iterator(detail::IntrusiveListImplementation::const_iterator sourceIterator) NN_NOEXCEPT
            : m_Iterator(sourceIterator)
        {
        }

        /**
        * @brief    *演算子のオーバーロードです。
        * @details  *演算子のオーバーロードを定義します。
        */
        reference operator*() const NN_NOEXCEPT
        {
            return NodeTraits::GetItem(*m_Iterator);
        }

        /**
        * @brief    ->演算子のオーバーロードです。
        * @details  ->演算子のオーバーロードを定義します。
        */
        pointer operator->() const NN_NOEXCEPT
        {
            return &NodeTraits::GetItem(*m_Iterator);
        }

        /**
        * @brief    ++演算子のオーバーロードです。
        * @details  ++演算子のオーバーロードを定義します。
        */
        const_iterator& operator++() NN_NOEXCEPT
        {
            ++m_Iterator;
            return *this;
        }

        /**
        * @brief    ++演算子のオーバーロードです。
        * @details  ++演算子のオーバーロードを定義します。
        */
        const_iterator operator++(int) NN_NOEXCEPT
        {
            const_iterator temporary(*this);
            ++m_Iterator;
            return temporary;
        }

        /**
        * @brief    --演算子のオーバーロードです。
        * @details  --演算子のオーバーロードを定義します。
        */
        const_iterator& operator--() NN_NOEXCEPT
        {
            --m_Iterator;
            return *this;
        }

        /**
        * @brief    --演算子のオーバーロードです。
        * @details  --演算子のオーバーロードを定義します。
        */
        const_iterator operator--(int) NN_NOEXCEPT
        {
            const_iterator temporary(*this);
            --m_Iterator;
            return temporary;
        }

        /**
        * @brief    ==演算子のオーバーロードです。
        * @details  ==演算子のオーバーロードを定義します。
        */
        bool operator==(const const_iterator& target) const NN_NOEXCEPT
        {
            return m_Iterator == target.m_Iterator;
        }

        /**
        * @brief    !=演算子のオーバーロードです。
        * @details  !=演算子のオーバーロードを定義します。
        */
        bool operator!=(const const_iterator& target) const NN_NOEXCEPT
        {
            return m_Iterator != target.m_Iterator;
        }

        /**
        * @brief    実装クラスを返します。
        * @details  実装クラスを返すメソッドを定義します。
        */
        detail::IntrusiveListImplementation::const_iterator GetImplementationIterator() NN_NOEXCEPT
        {
            return m_Iterator;
        }

    private:
        detail::IntrusiveListImplementation::const_iterator m_Iterator;
    };

    /**
    * @brief    イテレータです。
    * @details  イテレータを定義します。
    */
    class iterator
    {
    public:
        typedef T value_type; //!< value_type型を定義します。
        typedef int difference_type; //!< difference_type型を定義します。
        typedef value_type* pointer; //!< pointer型を定義します。
        typedef value_type& reference; //!< reference型を定義します。
        typedef std::bidirectional_iterator_tag iterator_category; //!< iterator_category型を定義します。

    public:
        /**
        * @brief    コンストラクタです。
        * @param[in]    sourceIterator  イテレータを指定します。
        * @details  コンストラクタを定義します。
        *           ユーザによって呼ばれることは想定していません。
        */
        explicit iterator(detail::IntrusiveListImplementation::iterator sourceIterator) NN_NOEXCEPT
            : m_Iterator(sourceIterator)
        {
        }

        /**
        * @brief    const_iteratorキャスト演算子のオーバーロードです。
        * @details  const_iteratorキャスト演算子のオーバーロードを定義します。
        */
        NN_IMPLICIT operator const_iterator() const NN_NOEXCEPT
        {
            return const_iterator(static_cast<detail::IntrusiveListImplementation::const_iterator>(m_Iterator));
        }

        /**
        * @brief    *演算子のオーバーロードです。
        * @details  *演算子のオーバーロードを定義します。
        */
        reference operator*() const NN_NOEXCEPT
        {
            return NodeTraits::GetItem(*m_Iterator);
        }

        /**
        * @brief    ->演算子のオーバーロードです。
        * @details  ->演算子のオーバーロードを定義します。
        */
        pointer operator->() const NN_NOEXCEPT
        {
            return &NodeTraits::GetItem(*m_Iterator);
        }

        /**
        * @brief    ++演算子のオーバーロードです。
        * @details  ++演算子のオーバーロードを定義します。
        */
        iterator& operator++() NN_NOEXCEPT
        {
            ++m_Iterator;
            return *this;
        }

        /**
        * @brief    ++演算子のオーバーロードです。
        * @details  ++演算子のオーバーロードを定義します。
        */
        iterator operator++(int) NN_NOEXCEPT
        {
            iterator temporary(*this);
            ++m_Iterator;
            return temporary;
        }

        /**
        * @brief    --演算子のオーバーロードです。
        * @details  --演算子のオーバーロードを定義します。
        */
        iterator& operator--() NN_NOEXCEPT
        {
            --m_Iterator;
            return *this;
        }

        /**
        * @brief    --演算子のオーバーロードです。
        * @details  --演算子のオーバーロードを定義します。
        */
        iterator operator--(int) NN_NOEXCEPT
        {
            iterator temporary(*this);
            --m_Iterator;
            return temporary;
        }

        /**
        * @brief    ==演算子のオーバーロードです。
        * @details  ==演算子のオーバーロードを定義します。
        */
        bool operator==(const iterator& target) const NN_NOEXCEPT
        {
            return m_Iterator == target.m_Iterator;
        }

        /**
        * @brief    !=演算子のオーバーロードです。
        * @details  !=演算子のオーバーロードを定義します。
        */
        bool operator!=(const iterator& target) const NN_NOEXCEPT
        {
            return m_Iterator != target.m_Iterator;
        }

        /**
        * @brief    実装クラスを返します。
        * @details  実装クラスを返すメソッドを定義します。
        */
        detail::IntrusiveListImplementation::iterator GetImplementationIterator() NN_NOEXCEPT
        {
            return m_Iterator;
        }

    private:
        detail::IntrusiveListImplementation::iterator m_Iterator;
    };

    typedef T& reference; //!< reference型を定義します。
    typedef const T& const_reference; //!< const_reference型を定義します。要求があるまでは暫定的にTにconstが含まれないこと前提の仕様としています。
    typedef std::reverse_iterator<iterator> reverse_iterator; //!< reverse_iterator型を定義します。
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator; //!< const_reverse_iterator型を定義します。

public:
    /**
    * @brief    コンストラクタです。
    * @details  コンストラクタを定義します。
    */
    IntrusiveList() NN_NOEXCEPT
        : m_Implementation()
    {
    }

    /**
    * @brief    値をリスト終端に挿入します。
    * @param[in]    value   挿入する値を指定します。
    * @details  値をリスト終端に挿入するメソッドです。
    */
    void push_back(reference value) NN_NOEXCEPT
    {
        m_Implementation.push_back(ToNode(value));
    }

    /**
    * @brief    値をリスト先頭に挿入します。
    * @param[in]    value   挿入する値を指定します。
    * @details  値をリスト先頭に挿入するメソッドです。
    */
    void push_front(reference value) NN_NOEXCEPT
    {
        m_Implementation.push_front(ToNode(value));
    }

    /**
    * @brief    リスト終端の要素を削除します。
    * @details  リスト終端の要素を削除するメソッドです。
    *           デストラクタは一切呼ばれません。
    */
    void pop_back() NN_NOEXCEPT
    {
        m_Implementation.pop_back();
    }

    /**
    * @brief    リスト先頭の要素を削除します。
    * @details  リスト先頭の要素を削除するメソッドです。
    */
    void pop_front() NN_NOEXCEPT
    {
        m_Implementation.pop_front();
    }

    /**
    * @brief    リスト先頭の要素への参照を返します。
    * @return   リスト先頭の要素への参照です。
    * @details  リスト先頭の要素への参照を返すメソッドです。
    */
    reference front() NN_NOEXCEPT
    {
        return ToReference(m_Implementation.front());
    }

    /**
    * @brief    リスト先頭の要素へのconst参照を返します。
    * @return   リスト先頭の要素へのconst参照です。
    * @details  リスト先頭の要素へのconst参照を返すメソッドです。
    */
    const_reference front() const NN_NOEXCEPT
    {
        return ToReference(m_Implementation.front());
    }

    /**
    * @brief    リスト終端の要素への参照を返します。
    * @return   リスト終端の要素への参照です。
    * @details  リスト終端の要素への参照を返すメソッドです。
    */
    reference back() NN_NOEXCEPT
    {
        return ToReference(m_Implementation.back());
    }

    /**
    * @brief    リスト終端の要素へのconst参照を返します。
    * @return   リスト終端の要素へのconst参照です。
    * @details  リスト終端の要素へのconst参照を返すメソッドです。
    */
    const_reference back() const NN_NOEXCEPT
    {
        return ToReference(m_Implementation.back());
    }

    /**
    * @brief    リスト先頭の要素のイテレータを返します。
    * @return   リスト先頭の要素のイテレータです。
    * @details  リスト先頭の要素のイテレータを返すメソッドです。
    */
    iterator begin() NN_NOEXCEPT
    {
        return iterator(m_Implementation.begin());
    }

    /**
    * @brief    リスト先頭の要素のconstイテレータを返します。
    * @return   リスト先頭の要素のconstイテレータです。
    * @details  リスト先頭の要素のconstイテレータを返すメソッドです。
    */
    const_iterator begin() const NN_NOEXCEPT
    {
        return const_iterator(m_Implementation.begin());
    }

    /**
    * @brief    リスト先頭の要素のconstイテレータを返します。
    * @return   リスト先頭の要素のconstイテレータです。
    * @details  リスト先頭の要素のconstイテレータを返すメソッドです。
    */
    const_iterator cbegin() const NN_NOEXCEPT
    {
        return begin();
    }

    /**
    * @brief    リスト終端の要素のイテレータを返します。
    * @return   リスト終端の要素のイテレータです。
    * @details  リスト終端の要素のイテレータを返すメソッドです。
    */
    iterator end() NN_NOEXCEPT
    {
        return iterator(m_Implementation.end());
    }

    /**
    * @brief    リスト終端の要素のconstイテレータを返します。
    * @return   リスト終端の要素のconstイテレータです。
    * @details  リスト終端の要素のconstイテレータを返すメソッドです。
    */
    const_iterator end() const NN_NOEXCEPT
    {
        return const_iterator(m_Implementation.end());
    }

    /**
    * @brief    リスト終端の要素のconstイテレータを返します。
    * @return   リスト終端の要素のconstイテレータです。
    * @details  リスト終端の要素のconstイテレータを返すメソッドです。
    */
    const_iterator cend() const NN_NOEXCEPT
    {
        return end();
    }

    /**
    * @brief    リスト先頭へのreverseイテレータを返します。
    * @return   リスト先頭へのreverseイテレータです。
    * @details  リスト先頭へのreverseイテレータを返すメソッドです。
    */
    reverse_iterator rbegin() NN_NOEXCEPT
    {
        return reverse_iterator(end());
    }

    /**
    * @brief    リスト先頭へのconst reverseイテレータを返します。
    * @return   リスト先頭へのconst reverseイテレータです。
    * @details  リスト先頭へのconst reverseイテレータを返すメソッドです。
    */
    const_reverse_iterator rbegin() const NN_NOEXCEPT
    {
        return const_reverse_iterator(end());
    }

    /**
    * @brief    リスト先頭へのconst reverseイテレータを返します。
    * @return   リスト先頭へのconst reverseイテレータです。
    * @details  リスト先頭へのconst reverseイテレータを返すメソッドです。
    */
    const_reverse_iterator crbegin() const NN_NOEXCEPT
    {
        return rbegin();
    }

    /**
    * @brief    リスト終端へのreverseイテレータを返します。
    * @return   リスト終端へのreverseイテレータです。
    * @details  リスト終端へのreverseイテレータを返すメソッドです。
    */
    reverse_iterator rend() NN_NOEXCEPT
    {
        return reverse_iterator(begin());
    }

    /**
    * @brief    リスト終端へのconst reverseイテレータを返します。
    * @return   リスト終端へのconst reverseイテレータです。
    * @details  リスト終端へのconst reverseイテレータを返すメソッドです。
    */
    const_reverse_iterator rend() const NN_NOEXCEPT
    {
        return const_reverse_iterator(begin());
    }

    /**
    * @brief    リスト終端へのconst reverseイテレータを返します。
    * @return   リスト終端へのconst reverseイテレータです。
    * @details  リスト終端へのconst reverseイテレータを返すメソッドです。
    */
    const_reverse_iterator crend() const NN_NOEXCEPT
    {
        return rend();
    }

    /**
    * @brief    リストに含まれる要素数を返します。
    * @return   リストに含まれる要素数です。
    * @details  リストに含まれる要素数を返すメソッドです。
    */
    int size() const NN_NOEXCEPT
    {
        return m_Implementation.size();
    }

    /**
    * @brief    リストが空かどうかを判定します。
    * @return   リストが空であればtrueを返し、それ以外であればfalseを返します。
    * @details  リストが空かどうかを判定するメソッドです。
    */
    bool empty() const NN_NOEXCEPT
    {
        return m_Implementation.empty();
    }

    /**
    * @brief    イテレータが指す要素をリストから削除します。
    * @param[in]    i   削除する要素を指すイテレータを指定します。
    * @return   削除した要素のイテレータです。
    * @details  イテレータが指す要素をリストから削除するメソッドです。
    *           デストラクタは一切呼ばれません。
    */
    iterator erase(const_iterator i) NN_NOEXCEPT
    {
        detail::IntrusiveListImplementation::iterator result = m_Implementation.erase(i.GetImplementationIterator());
        return iterator(result);
    }

    /**
    * @brief    リスト内の全要素を削除します。
    * @details  リスト内の全要素を削除するメソッドです。
    *           デストラクタは一切呼ばれません。
    */
    void clear() NN_NOEXCEPT
    {
        m_Implementation.clear();
    }

    /**
    * @brief    イテレータが指す要素の直前に値を挿入します。
    * @param[in]    p   値を挿入する位置をイテレータで指定します。
    * @param[in]    value   挿入する値を指定します。
    * @return   挿入した要素のイテレータです。
    * @details  イテレータが指す要素の直前に値を挿入するメソッドです。
    */
    iterator insert(const_iterator p, reference value) NN_NOEXCEPT
    {
        detail::IntrusiveListImplementation::iterator result = m_Implementation.insert(p.GetImplementationIterator(), ToNode(value));
        return iterator(result);
    }

    /**
    * @brief    別のリストから全要素を転送します。
    * @param[in]    p   転送先の位置をイテレータで指定します。
    * @param[in]    x   転送元のリストを指定します。
    * @details  別のリストから全要素を転送するメソッドです。
    *           指定したイテレータの指す要素の直前に転送されます。
    *           デストラクタおよびコピーコンストラクタは一切呼ばれません。
    */
    void splice(const_iterator p, IntrusiveList& x) NN_NOEXCEPT
    {
        m_Implementation.splice(p.GetImplementationIterator(), x.m_Implementation);
    }

    /**
    * @brief    別のリストから指定した値を転送します。
    * @param[in]    p   転送先の位置をイテレータで指定します。
    * @param[in]    x   転送元のリストを指定します。
    * @param[in]    new_ele 転送する値を指したイテレータを指定します。
    * @details  別のリストから指定した値を転送するメソッドです。
    *           指定したイテレータの指す要素の直前に転送されます。
    *           デストラクタおよびコピーコンストラクタは一切呼ばれません。
    */
    void splice(const_iterator p, IntrusiveList& x, iterator new_ele) NN_NOEXCEPT
    {
        m_Implementation.splice(p.GetImplementationIterator(), x.m_Implementation, new_ele.GetImplementationIterator());
    }

    /**
    * @brief    別のリストから指定した領域を転送します。
    * @param[in]    p   転送先の位置をイテレータで指定します。
    * @param[in]    x   転送元のリストを指定します。
    * @param[in]    f   転送する領域の先頭位置をイテレータで指定します。
    * @param[in]    e   転送する領域の終端位置をイテレータで指定します。
    * @details  別のリストから指定した領域を転送するメソッドです。
    *           指定したイテレータの指す要素の直前に転送されます。
    *           デストラクタおよびコピーコンストラクタは一切呼ばれません。
    */
    void splice(const_iterator p, IntrusiveList& x, const_iterator f, const_iterator e) NN_NOEXCEPT
    {
        m_Implementation.splice(p.GetImplementationIterator(), x.m_Implementation, f.GetImplementationIterator(), e.GetImplementationIterator());
    }

private:
    IntrusiveListNode& ToNode(reference ref) const NN_NOEXCEPT
    {
        return NodeTraits::GetNode(ref);
    }

    const IntrusiveListNode& ToNode(const_reference ref) const NN_NOEXCEPT
    {
        return NodeTraits::GetNode(ref);
    }

    reference ToReference(IntrusiveListNode& node) const NN_NOEXCEPT
    {
        return NodeTraits::GetItem(node);
    }

    const_reference ToReference(const IntrusiveListNode& node) const NN_NOEXCEPT
    {
        return NodeTraits::GetItem(node);
    }

private:
    detail::IntrusiveListImplementation m_Implementation;
};

/**
* @brief    ノードのオフセットを管理するクラスです。
* @details  ノードのオフセットを管理するクラスを定義します。
*/
template<class T, IntrusiveListNode T::*Member>
class IntrusiveListMemberNodeTraits
{
    NN_DISALLOW_COPY(IntrusiveListMemberNodeTraits);
public:
    /**
    * @brief    リストからノードを取得します。
    * @param[in]    リストを指定します。
    * @return   ノードです。
    * @details  リストからノードを取得する関数です。
    */
    static IntrusiveListNode& GetNode(T& ref) NN_NOEXCEPT
    {
        return ref.*Member;
    }

    /**
    * @brief    リストからノードを取得します。
    * @param[in]    リストを指定します。
    * @return   ノードです。
    * @details  リストからノードを取得する関数です。
    */
    static const IntrusiveListNode& GetNode(const T& ref) NN_NOEXCEPT
    {
        return ref.*Member;
    }

    /**
    * @brief    ノードからリストを取得します。
    * @param[in]    ノードを指定します。
    * @return   リストです。
    * @details  ノードからリストを取得する関数です。
    */
    static T& GetItem(IntrusiveListNode& node) NN_NOEXCEPT
    {
        return *reinterpret_cast<T*>(reinterpret_cast<char*>(&node) - GetOffset());
    }

    /**
    * @brief    ノードからリストを取得します。
    * @param[in]    ノードを指定します。
    * @return   リストです。
    * @details  ノードからリストを取得する関数です。
    */
    static const T& GetItem(const IntrusiveListNode& node) NN_NOEXCEPT
    {
        return *reinterpret_cast<const T*>(reinterpret_cast<const char*>(&node) - GetOffset());
    }

private:
    static uintptr_t GetOffset() NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(&((reinterpret_cast<T*>(0))->*Member));
    }
};

}}
