﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstring>
#include <nn/fs_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_DbmRomTypes.h>
#include <nn/fs/fs_DbmRomPathTool.h>

namespace nn { namespace fs {

/*!
    @brief コンストラクタです。
*/
RomPathTool::PathParser::PathParser() NN_NOEXCEPT
    : m_pPreviousBeginPath(nullptr)
    , m_pPreviousEndPath(nullptr)
    , m_pNextPath(nullptr)
    , m_IsParseFinished(false)
{
}

/*!
    @brief パーサを初期化します。

    @param[in] pFullPath フルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDbmInvalidPathFormat  不正なパスです。

    @pre    pFullPath != nullptr
*/
Result RomPathTool::PathParser::Initialize(const RomPathChar* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // フルパスがルートディレクトリから始まっているかどうかチェックします。
    if( !IsSeparator(pFullPath[0]) )
    {
        // 不正なパスです。
        return ResultDbmInvalidPathFormat();
    }
    // セパレーターが連続していたら読み進めておきます
    while( IsSeparator(pFullPath[1]) )
    {
        pFullPath++;
    }

    // 初回のみ先頭のセパレーターを指し示すようにします。
    m_pPreviousBeginPath = pFullPath;

    // 初回のみ先頭のセパレーターを指し示すようにします。
    m_pPreviousEndPath = m_pPreviousBeginPath;

    // m_pNextPath は常に区切り文字の次の文字を指すようにします。
    m_pNextPath = &pFullPath[1];
    while( IsSeparator(m_pNextPath[0]) )
    {
        m_pNextPath++;
    }

    NN_RESULT_SUCCESS;
}

/*!
    @brief パーサを破棄します。
*/
void RomPathTool::PathParser::Finalize() NN_NOEXCEPT
{
    m_pPreviousBeginPath = nullptr;
    m_pPreviousEndPath = nullptr;
    m_pNextPath = nullptr;
    m_IsParseFinished = false;
}

/*!
    @brief パスのパースが完了したかどうかを取得します。
*/
bool RomPathTool::PathParser::IsParseFinished() const NN_NOEXCEPT
{
    return m_IsParseFinished;
}

/*!
    @brief  パスがディレクトリだったかどうかを取得します。

            IsParseFinished が true の時のみ使用できます。
*/
bool RomPathTool::PathParser::IsDirectoryPath() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pNextPath);
    // パスの最後が区切り文字かどうかで判定します。
    if( (m_pNextPath[0] == StringTraitsRom::Nul)
        && (*(m_pNextPath - 1) == StringTraitsRom::DirectorySeparator) )
    {
        return true;
    }

    // カレントディレクトリを示す文字であれば true を返します。
    if( IsCurrentDirectory(m_pNextPath) )
    {
        return true;
    }

    // 親ディレクトリを示す文字であれば true を返します。
    if( IsParentDirectory(m_pNextPath) )
    {
        return true;
    }

    return false;
}

/*!
    @brief 最上位ディレクトリ部分を取得します。

    @param[out] pOutValue ディレクトリ名

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。

    @pre    初期化済み
    @pre    pOutValue != nullptr
*/
Result RomPathTool::PathParser::GetNextDirectoryName(RomEntryName* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPreviousBeginPath);
    NN_SDK_REQUIRES_NOT_NULL(m_pPreviousEndPath);
    NN_SDK_REQUIRES_NOT_NULL(m_pNextPath);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // 現在のディレクトリ名をコピーします。
    pOutValue->length = m_pPreviousEndPath - m_pPreviousBeginPath;
    pOutValue->path = m_pPreviousBeginPath;

    // 次のパスの始点へ進めます
    m_pPreviousBeginPath = m_pNextPath;

    // 次の区切り文字を探索します。
    const RomPathChar* pChar = m_pNextPath;
    for( size_t directoryNameLength = 0; ; directoryNameLength++ )
    {
        // 区切り文字が見つかれば探索終了です。
        if( IsSeparator(pChar[directoryNameLength]) )
        {
            if( directoryNameLength >= MaxPathLength )
            {
                // ディレクトリ名が長すぎます。
                return ResultDbmDirectoryNameTooLong();
            }

            // 区切り文字の一文字手前を指すようにします。
            m_pPreviousEndPath = &pChar[directoryNameLength];
            m_pNextPath = m_pPreviousEndPath + 1;

            // 連続区切り文字の次の文字を指すようにします。
            while( IsSeparator(*m_pNextPath) )
            {
                m_pNextPath++;
            }
            if( *m_pNextPath == StringTraitsRom::Nul )
            {
                m_IsParseFinished = true;
            }
            break;
        }

        // 終端文字が見つかれば探索終了です。
        if( pChar[directoryNameLength] == StringTraitsRom::Nul )
        {
            m_IsParseFinished = true;
            m_pPreviousEndPath = m_pNextPath = &pChar[directoryNameLength];
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

/*!
    @brief 文字列の先頭から次の１節をディレクトリ名として取得します。

    @param[out] pOutValue ディレクトリ名

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。

    @pre    初期化済み
    @pre    pOutValue != nullptr
*/
Result RomPathTool::PathParser::GetAsDirectoryName(RomEntryName* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pNextPath);
    NN_SDK_REQUIRES_NOT_NULL(m_pPreviousBeginPath);
    NN_SDK_REQUIRES_NOT_NULL(m_pPreviousEndPath);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const size_t nameLength = m_pPreviousEndPath - m_pPreviousBeginPath;
    if( nameLength > MaxPathLength )
    {
        // ディレクトリ名が長すぎます。
        return ResultDbmDirectoryNameTooLong();
    }

    // ディレクトリ名データへの参照を設定します。
    pOutValue->length = nameLength;
    pOutValue->path = m_pPreviousBeginPath;

    NN_RESULT_SUCCESS;
}

/*!
    @brief 文字列の先頭から次の１節をファイル名として取得します。

    @param[out] pOutValue ファイル名

    @return 関数の処理結果を返します。

    @retval ResultSuccess                   成功しました。
    @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。

    @pre    初期化済み
    @pre    pOutValue != nullptr
*/
Result RomPathTool::PathParser::GetAsFileName(RomEntryName* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pNextPath);
    NN_SDK_REQUIRES_NOT_NULL(m_pPreviousBeginPath);
    NN_SDK_REQUIRES_NOT_NULL(m_pPreviousEndPath);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const size_t nameLength = m_pPreviousEndPath - m_pPreviousBeginPath;
    if( nameLength > MaxPathLength )
    {
        // ファイル名が長すぎます。
        return ResultDbmFileNameTooLong();
    }

    // ファイル名データへの参照を設定します。
    pOutValue->length = nameLength;
    pOutValue->path = m_pPreviousBeginPath;

    NN_RESULT_SUCCESS;
}

/*!
    @brief 親ディレクトリ名を抽出します。

           フルパス:"/AAA/BBB/CCC" 開始:"AAA" => 親ディレクトリ:"" (ルートディレクトリ)
           フルパス:"/AAA/BBB/CCC" 開始:"BBB" => 親ディレクトリ:"AAA"
           フルパス:"/AAA/BBB/CCC/." 開始:"." => 親ディレクトリ:"BBB"
           フルパス:"/AAA/BBB/CCC/.." 開始:".." => 親ディレクトリ:"AAA"
           フルパス:"/AAA/.." 開始:".." => ResultDirectoryUnobtainable()

    @param[out] pOutValue   親ディレクトリ名
    @param[in]  base        カレントディレクトリ名。
                            フルパス内の文字列を指している必要があります。
    @param[in]  pFullPath   フルパス

    @return 関数の処理結果を返します。

    @retval ResultSuccess               成功しました。
    @retval ResultDirectoryUnobtainable ルートディレクトリの親を取得しようとしました。

    @pre    pOutValue != nullptr
    @pre    pHead != nullptr
*/
Result RomPathTool::GetParentDirectoryName(
    RomEntryName* pOutValue, const RomEntryName& base, const RomPathChar* pHead
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pHead);

    const RomPathChar* pFirst = base.path;
    const RomPathChar* pLast = base.path + base.length - 1;

    // スキップするセパレータの個数を決定します。
    int32_t depth = 1;

    // カレントディレクトリに当たる部分が親ディレクトリを
    // 指していればセパレータを 1 個余分にスキップします。
    if( IsParentDirectory(base) )
    {
        depth++;
    }

    if( base.path > pHead )
    {
        // パスの先頭に向かって親ディレクトリの探索を行います。
        size_t length = 0;
        const RomPathChar* pChar = base.path - 1;
        while( pChar >= pHead )
        {
            if( IsSeparator(*pChar) )
            {
                // カレントディレクトリであればもう 1 つ親のディレクトリまで
                // 探索を行います。
                if( IsCurrentDirectory(pChar + 1, length) )
                {
                    depth++;
                }

                // 親ディレクトリであればもう 2 つ親のディレクトリまで
                // 探索を行います。
                if( IsParentDirectory(pChar + 1, length) )
                {
                    depth += 2;
                }

                // 親ディレクトリに到達しました。
                if( depth == 0 )
                {
                    pFirst = pChar + 1;
                    break;
                }

                // 連続するセパレータをスキップします。
                while( IsSeparator(*pChar) )
                {
                    pChar--;
                }

                // 抽出する範囲の末尾を更新します。
                pLast = pChar;
                length = 0;
                depth--;
            }

            length++;
            pChar--;
        }

        // ルートディレクトリの親ディレクトリは取得できません。
        if( depth != 0 )
        {
            return ResultDirectoryUnobtainable();
        }

        // 先頭のセパレータまで到達すれば、セパレータの次の文字を
        // 開始地点とします。
        if( pChar == pHead )
        {
            pFirst = pHead + 1;
        }
    }

    if( pLast <= pHead )
    {
        // ルートディレクトリです。
        pOutValue->path = pHead;
        pOutValue->length = 0;
    }
    else
    {
        // 開始地点から範囲の末尾までを抽出します。
        pOutValue->path = pFirst;
        pOutValue->length = pLast - pFirst + 1;
    }

    NN_RESULT_SUCCESS;
}

}}
