﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <functional>
#include <mutex>

#include <nn/nn_Version.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_CloudBackupWorkStorage.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_Priority.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/detail/fs_MemoryManagementPrivate.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_FileSystemProxyTypes.h>
#include <nn/fs/detail/fs_Log.h>

#include <nn/fs/detail/fs_ResultHandlingUtility.h>

#include "fsa/fs_UserMountTable.h"
#include "fsa/fs_DirectoryAccessor.h"
#include "fsa/fs_FileAccessor.h"
#include "fsa/fs_FileSystemAccessor.h"
#include "shim/fs_FileSystemProxyServiceObject.h"

#define NN_DETAIL_FS_STARTLOG_SDK_VERSION "sdk_version: " \
    NN_MACRO_STRINGIZE(NN_SDK_VERSION_MAJOR) "." NN_MACRO_STRINGIZE(NN_SDK_VERSION_MINOR) "." NN_MACRO_STRINGIZE(NN_SDK_VERSION_MICRO)
#if defined(NN_BUILD_CONFIG_SPEC_NX)
#define NN_DETAIL_FS_STARTLOG_SPEC "spec: NX"
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
#define NN_DETAIL_FS_STARTLOG_SPEC "spec: Generic"
#else
#define NN_DETAIL_FS_STARTLOG_SPEC "spec: Unknown"
#endif

namespace nn { namespace fs {
    PriorityRaw GetPriorityRawOnCurrentThreadImpl() NN_NOEXCEPT;
}}

namespace nn { namespace fs {

namespace {
    uint32_t g_GlobalAccessLogMode = AccessLogMode_Off;
    uint32_t g_LocalAccessLogTarget = nn::fs::detail::AccessLogTarget_None;
    bool g_IsAccessLogInitialized = false;

    void SetLocalAccessLogImpl(bool isEnabled) NN_NOEXCEPT
    {
        if (isEnabled)
        {
            g_LocalAccessLogTarget |= nn::fs::detail::AccessLogTarget_Application;
        }
        else
        {
            g_LocalAccessLogTarget &= ~nn::fs::detail::AccessLogTarget_Application;
        }
    }
}

nn::Result GetGlobalAccessLogMode(uint32_t* pOutAccessLogMode) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->GetGlobalAccessLogMode(nn::sf::Out<uint32_t>(pOutAccessLogMode)));
    NN_RESULT_SUCCESS;
}

nn::Result SetGlobalAccessLogMode(uint32_t accessLogMode) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->SetGlobalAccessLogMode(accessLogMode));
    NN_RESULT_SUCCESS;
}

void SetLocalAccessLog(bool isEnabled) NN_NOEXCEPT
{
    SetLocalAccessLogImpl(isEnabled);
}

void SetLocalApplicationAccessLog(bool isEnabled) NN_NOEXCEPT
{
    SetLocalAccessLogImpl(isEnabled);
}

#if !defined(NN_SDK_BUILD_RELEASE)

void SetLocalSystemAccessLogForDebug(bool isEnabled) NN_NOEXCEPT
{
    if (isEnabled)
    {
        g_LocalAccessLogTarget |= nn::fs::detail::AccessLogTarget_System | nn::fs::detail::AccessLogTarget_Application;
    }
    else
    {
        g_LocalAccessLogTarget &= ~(nn::fs::detail::AccessLogTarget_System | nn::fs::detail::AccessLogTarget_Application);
    }
}

#endif

}}

namespace nn { namespace fs { namespace detail {

const char* IdString::ToValueString(int id) NN_NOEXCEPT
{
    const int length = nn::util::SNPrintf(m_StringBuffer, sizeof(m_StringBuffer), "%d", id);
    NN_SDK_ASSERT_LESS(static_cast<size_t>(length), sizeof(m_StringBuffer));
    NN_UNUSED(length);
    return m_StringBuffer;
}

template<>
const char* IdString::ToString<nn::fs::Priority>(nn::fs::Priority id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::Priority_Realtime:
        return "Realtime";
    case nn::fs::Priority_Normal:
        return "Normal";
    case nn::fs::Priority_Low:
        return "Low";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<nn::fs::PriorityRaw>(nn::fs::PriorityRaw id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::PriorityRaw_Realtime:
        return "Realtime";
    case nn::fs::PriorityRaw_Normal:
        return "Normal";
    case nn::fs::PriorityRaw_Low:
        return "Low";
    case nn::fs::PriorityRaw_Background:
        return "Background";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<ImageDirectoryId>(ImageDirectoryId id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::ImageDirectoryId::Nand:
        return "Nand";
    case nn::fs::ImageDirectoryId::SdCard:
        return "SdCard";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<ContentStorageId>(ContentStorageId id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::ContentStorageId::User:
        return "User";
    case nn::fs::ContentStorageId::System:
        return "System";
    case nn::fs::ContentStorageId::SdCard:
        return "SdCard";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<CloudBackupWorkStorageId>(CloudBackupWorkStorageId id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::CloudBackupWorkStorageId::Nand:
        return "Nand";
    case nn::fs::CloudBackupWorkStorageId::SdCard:
        return "SdCard";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<GameCardPartition>(GameCardPartition id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::GameCardPartition::Update:
        return "Update";
    case nn::fs::GameCardPartition::Normal:
        return "Normal";
    case nn::fs::GameCardPartition::Secure:
        return "Secure";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<SaveDataSpaceId>(SaveDataSpaceId id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::SaveDataSpaceId::System:
        return "System";
    case nn::fs::SaveDataSpaceId::User:
        return "User";
    case nn::fs::SaveDataSpaceId::SdSystem:
        return "SdSystem";
    case nn::fs::SaveDataSpaceId::ProperSystem:
        return "ProperSystem";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<ContentType>(ContentType id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::ContentType_Meta:
        return "Meta";
    case nn::fs::ContentType_Control:
        return "Control";
    case nn::fs::ContentType_Manual:
        return "Manual";
    case nn::fs::ContentType_Logo:
        return "Logo";
    case nn::fs::ContentType_Data:
        return "Data";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

template<>
const char* IdString::ToString<BisPartitionId>(BisPartitionId id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::BisPartitionId::BootPartition1Root:
        return "BootPartition1Root";
    case nn::fs::BisPartitionId::BootPartition2Root:
        return "BootPartition2Root";
    case nn::fs::BisPartitionId::UserDataRoot:
        return "UserDataRoot";
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part1:
        return "BootConfigAndPackage2Part1";
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part2:
        return "BootConfigAndPackage2Part2";
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part3:
        return "BootConfigAndPackage2Part3";
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part4:
        return "BootConfigAndPackage2Part4";
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part5:
        return "BootConfigAndPackage2Part5";
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part6:
        return "BootConfigAndPackage2Part6";
    case nn::fs::BisPartitionId::CalibrationBinary:
        return "CalibrationBinary";
    case nn::fs::BisPartitionId::CalibrationFile:
        return "CalibrationFile";
    case nn::fs::BisPartitionId::SafeMode:
        return "SafeMode";
    case nn::fs::BisPartitionId::User:
        return "User";
    case nn::fs::BisPartitionId::System:
        return "System";
    case nn::fs::BisPartitionId::SystemProperEncryption:
        return "SystemProperEncryption";
    case nn::fs::BisPartitionId::SystemProperPartition:
        return "SystemProperPartition";
    case nn::fs::BisPartitionId::Invalid:
        return "Invalid";
    default:
        break;
    }
    return ToValueString(static_cast<int>(id));
}

namespace {
    const char* GetPriorityRawName() NN_NOEXCEPT
    {
        return nn::fs::detail::IdString().ToString(nn::fs::GetPriorityRawOnCurrentThreadImpl());
    }

    nn::Result OutputAccessLogToSdCard(const char* buffer, size_t length) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_FS_RESULT_DO(fileSystemProxy->OutputAccessLogToSdCard(nn::sf::InBuffer(buffer, length)));
        NN_RESULT_SUCCESS;
    }

    void OutputAccessLogImpl(const char* buffer, size_t length) NN_NOEXCEPT
    {
        if( (g_GlobalAccessLogMode & AccessLogMode_Log) != 0 )
        {
            // デフォルトのログオブザーバーが長い文字列に対応していないため、
            // NN_DETAIL_FS_PUT_INFO を使わずに、フォーマットを通して少しずつオブザーバーに送信する
            NN_DETAIL_FS_INFO("%s", buffer);
        }
        if( (g_GlobalAccessLogMode & AccessLogMode_SdCard) != 0 )
        {
            OutputAccessLogToSdCard(buffer, length - 1);  // '\0' 終端は除く
        }
    }

    void OutputAccessLog(
        nn::Result result,
        const char* priorityName,
        nn::os::Tick tickStart,
        nn::os::Tick tickEnd,
        const char* apiName,
        const void* pHandle,
        const char* fmt,
        std::va_list va
    ) NN_NOEXCEPT
    {
        // 入れ子用のバッファ構築
        int nestBufferSize = 1024;
        auto nestBuffer = MakeUnique<char[]>(nestBufferSize);
        for(;;)
        {
            if( nestBuffer == nullptr )
            {
                return;
            }
            const auto size = nn::util::VSNPrintf(nestBuffer.get(), nestBufferSize, fmt, va);
            if( size < nestBufferSize )
            {
                break;
            }
            nestBufferSize = size + 1;  // '\0' 終端分のサイズを加える
            nestBuffer = MakeUnique<char[]>(nestBufferSize);
        }

        // 出力用バッファ構築
        decltype(nestBuffer) buffer;
        int bufferLength = 0;
        {
            const int64_t startMilliSecond = nn::os::ConvertToTimeSpan(tickStart).GetMilliSeconds();
            const int64_t endMilliSecond = nn::os::ConvertToTimeSpan(tickEnd).GetMilliSeconds();
            const char formatString[] = "FS_ACCESS: { "
                "start: %9lld, "
                "end: %9lld, "
                "result: 0x%08X, "
                "handle: 0x%p, "
                "priority: %s, "
                "function: \"%s\""
                "%s"
                " }\n";

            int bufferSize = std::max<int>(nestBufferSize + sizeof(formatString) + 256, 1024);
            for(;;)
            {
                buffer = MakeUnique<char[]>(bufferSize);
                if( buffer == nullptr )
                {
                    return;
                }
                bufferLength = nn::util::SNPrintf(
                    buffer.get(),
                    bufferSize,
                    formatString,
                    startMilliSecond,
                    endMilliSecond,
                    result.GetInnerValueForDebug(),
                    pHandle,
                    priorityName,
                    apiName,
                    nestBuffer.get()
                    );
                // '\0' 終端分のサイズを加える
                bufferLength += 1;
                if( bufferLength <= bufferSize )
                {
                    break;
                }
                bufferSize = bufferLength;
            }
        }

        // 出力
        OutputAccessLogImpl(buffer.get(), bufferLength);
    }

    void OutputAccessLogStart()
    {
        const char startString[] = "FS_ACCESS: { "
            NN_DETAIL_FS_STARTLOG_SDK_VERSION ", "
            NN_DETAIL_FS_STARTLOG_SPEC
            " }\n";
        OutputAccessLogImpl(startString, sizeof(startString));
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    void OutputAccessLogStartForSystem()
    {
        const char startString[] = "FS_ACCESS: { "
            NN_DETAIL_FS_STARTLOG_SDK_VERSION ", "
            NN_DETAIL_FS_STARTLOG_SPEC
            ", for_system: true }\n";
        OutputAccessLogImpl(startString, sizeof(startString));
    }
#endif
}

bool IsEnabledAccessLog(uint32_t target) NN_NOEXCEPT
{
    if( (g_LocalAccessLogTarget & target) == 0 )
    {
        return false;
    }
    if( !g_IsAccessLogInitialized )
    {
        g_IsAccessLogInitialized = true;
#if !defined(NN_SDK_BUILD_RELEASE)
        if( (g_LocalAccessLogTarget & ::nn::fs::detail::AccessLogTarget_System) != 0 )
        {
            // システム用の場合は常に Log モードとする
            g_GlobalAccessLogMode = AccessLogMode_Log;
            OutputAccessLogStartForSystem();
        }
        else
#endif
        {
            // システム用でなければ設定を問い合わせ
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(GetGlobalAccessLogMode(&g_GlobalAccessLogMode));
            OutputAccessLogStart();
        }
    }
    return g_GlobalAccessLogMode != AccessLogMode_Off;
}

bool IsEnabledAccessLog() NN_NOEXCEPT
{
    return IsEnabledAccessLog(nn::fs::detail::AccessLogTarget_Application | nn::fs::detail::AccessLogTarget_System);
}

void OutputAccessLog(
    nn::Result result,
    nn::fs::Priority priority,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, nn::fs::detail::IdString().ToString(priority), tickStart, tickEnd, apiName, pHandle, fmt, vaList );
    va_end(vaList);
}

void OutputAccessLog(
    nn::Result result,
    nn::fs::PriorityRaw priorityRaw,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, nn::fs::detail::IdString().ToString(priorityRaw), tickStart, tickEnd, apiName, pHandle, fmt, vaList );
    va_end(vaList);
}

void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::FileHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::DirectoryHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::detail::IdentifyAccessLogHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLog(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, pHandle, fmt, vaList);
    va_end(vaList);
}

void OutputAccessLogForPrecondition(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::FileHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    if( result.IsSuccess() )
    {
        return;
    }
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLogForPrecondition(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::DirectoryHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    if( result.IsSuccess() )
    {
        return;
    }
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLogForPrecondition(
    nn::Result result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    if( result.IsSuccess() )
    {
        return;
    }
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(result, GetPriorityRawName(), tickStart, tickEnd, apiName, pHandle, fmt, vaList);
    va_end(vaList);
}

void OutputAccessLogForPrecondition(
    bool result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::FileHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    if( result )
    {
        return;
    }
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(nn::fs::ResultPreconditionViolation(), GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLogForPrecondition(
    bool result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    nn::fs::DirectoryHandle handle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    if( result )
    {
        return;
    }
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(nn::fs::ResultPreconditionViolation(), GetPriorityRawName(), tickStart, tickEnd, apiName, handle.handle, fmt, vaList);
    va_end(vaList);
}
void OutputAccessLogForPrecondition(
    bool result,
    nn::os::Tick tickStart,
    nn::os::Tick tickEnd,
    const char* apiName,
    const void* pHandle,
    const char* fmt,
    ...
) NN_NOEXCEPT
{
    if( result )
    {
        return;
    }
    std::va_list vaList;
    va_start(vaList, fmt);
    OutputAccessLog(nn::fs::ResultPreconditionViolation(), GetPriorityRawName(), tickStart, tickEnd, apiName, pHandle, fmt, vaList);
    va_end(vaList);
}

bool IsEnabledHandleAccessLog(FileHandle handle) NN_NOEXCEPT
{
    detail::FileAccessor* pAccessor = reinterpret_cast<detail::FileAccessor*>(handle.handle);
    if( pAccessor == nullptr )
    {
        return true;
    }

    auto pParent = pAccessor->GetParent();
    if (pParent == nullptr)
    {
        return false;
    }

    return pParent->IsEnabledAccessLog();
}
bool IsEnabledHandleAccessLog(DirectoryHandle handle) NN_NOEXCEPT
{
    detail::DirectoryAccessor* pAccessor = reinterpret_cast<detail::DirectoryAccessor*>(handle.handle);
    if( pAccessor == nullptr )
    {
        return true;
    }
    return pAccessor->GetParent().IsEnabledAccessLog();
}
bool IsEnabledHandleAccessLog(detail::IdentifyAccessLogHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    return true;
}
bool IsEnabledHandleAccessLog(const void* pHandle) NN_NOEXCEPT
{
    if( pHandle == nullptr )
    {
        return true;
    }
    NN_SDK_ASSERT(pHandle == nullptr, "handle type must be FileHandle or DirectoryHandle? please cast to handle type.");
    return false;
}

bool IsEnabledFileSystemAccessorAccessLog(const char* mountName) NN_NOEXCEPT
{
    detail::FileSystemAccessor* pSystem;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(detail::Find(&pSystem, mountName));
    return pSystem->IsEnabledAccessLog();
}

void EnableFileSystemAccessorAccessLog(const char* mountName) NN_NOEXCEPT
{
    detail::FileSystemAccessor* pSystem;
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(detail::Find(&pSystem, mountName));
    pSystem->SetAccessLog(true);
}

}}}
