﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Utilities for ethernet driver.
 */

#pragma once

#include <nn/dd.h>
#include <nn/nn_SdkAssert.h>

namespace nn {
namespace eth {
namespace util {

// address に指定したレジスタの読み出しを実行します。
NN_FORCEINLINE void DummyRead(volatile uint32_t* address)
{
    (void)(*address);
}

NN_FORCEINLINE void DummyRead(uintptr_t base, uint32_t offset) NN_NOEXCEPT
{
    DummyRead(reinterpret_cast<volatile uint32_t*>(base + offset));
}

NN_FORCEINLINE void WriteMasked32(volatile uint32_t* address, uint32_t value, uint32_t mask)
{
    *address = (*address & ~mask) | (value & mask);
}

NN_FORCEINLINE void WriteMasked32(uintptr_t base, uint32_t offset, uint32_t val, uint32_t mask) NN_NOEXCEPT
{
    WriteMasked32(reinterpret_cast<volatile uint32_t*>(base + offset), val, mask);
}

NN_FORCEINLINE void Write32(volatile uint32_t* reg, uint32_t val) NN_NOEXCEPT
{
    *reg = val;
}

NN_FORCEINLINE void Write32(uintptr_t base, uint32_t offset, uint32_t val) NN_NOEXCEPT
{
    Write32(reinterpret_cast<volatile uint32_t*>(base + offset), val);
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOn32(volatile uint32_t* reg, uint32_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp |= flags;
    *reg = temp;
}

NN_FORCEINLINE void SetBitOn32(uintptr_t base, uint32_t offset, uint32_t flags) NN_NOEXCEPT
{
    SetBitOn32(reinterpret_cast<volatile uint32_t*>(base + offset), flags);
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOff32(volatile uint32_t* reg, uint32_t flags) NN_NOEXCEPT
{
    auto temp = *reg;
    temp &= ~flags;
    *reg = temp;
}

NN_FORCEINLINE void SetBitOff32(uintptr_t base, uint32_t offset, uint32_t flags) NN_NOEXCEPT
{
    SetBitOff32(reinterpret_cast<volatile uint32_t*>(base + offset), flags);
}

// Overwrite partly on specified register
// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void Modify32(volatile uint32_t* reg, uint32_t value, uint32_t mask) NN_NOEXCEPT
{
    auto temp = *reg;
    temp &= ~mask;
    temp |= value;
    *reg = temp;
}

NN_FORCEINLINE void Modify32(uintptr_t base, uint32_t offset, uint32_t value, uint32_t mask) NN_NOEXCEPT
{
    Modify32(reinterpret_cast<volatile uint32_t*>(base + offset), value, mask);
}

NN_FORCEINLINE uint32_t Read32(volatile uint32_t* reg) NN_NOEXCEPT
{
    return *reg;
}

NN_FORCEINLINE uint32_t Read32(uintptr_t base, uint32_t offset) NN_NOEXCEPT
{
    return Read32(reinterpret_cast<volatile uint32_t*>(base + offset));
}

NN_FORCEINLINE void CopyToIo(volatile uint32_t* reg, void* pSrc, size_t byteCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(byteCount % sizeof(uint32_t), 0);

    const size_t count = byteCount / sizeof(uint32_t);
    uint32_t* const values = reinterpret_cast<uint32_t*>(pSrc);

    for (size_t i = 0; i < count; i++)
    {
        Write32(&reg[i], values[i]);
    }
}

NN_FORCEINLINE void CopyToIo(uintptr_t base, uint32_t offset, void* pSrc, size_t byteCount) NN_NOEXCEPT
{
    CopyToIo(reinterpret_cast<volatile uint32_t*>(base + offset), pSrc, byteCount);
}

NN_FORCEINLINE void CopyFromIo(void* pOutDst, volatile uint32_t* reg, size_t byteCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(byteCount % sizeof(uint32_t), 0);

    const size_t count = byteCount / sizeof(uint32_t);
    uint32_t* const values = reinterpret_cast<uint32_t*>(pOutDst);

    for (size_t i = 0; i < count; i++)
    {
        values[i] = Read32(&reg[i]);
    }
}

NN_FORCEINLINE void CopyFromIo(void* pOutDst, uintptr_t base, uint32_t offset, size_t byteCount) NN_NOEXCEPT
{
    CopyFromIo(pOutDst, reinterpret_cast<volatile uint32_t*>(base + offset), byteCount);
}

NN_FORCEINLINE uintptr_t GetVirtualAddress(nn::dd::PhysicalAddress physicalAddress, size_t addressSize) NN_NOEXCEPT
{
    uintptr_t virtualAddress = nn::dd::QueryIoMappingAddress(physicalAddress, addressSize);
    NN_ABORT_UNLESS(virtualAddress != 0, "I/O registers for 0x%llx are not mapped. "
        "Make sure the capability setting is properly set for this process.\n", physicalAddress);
    return virtualAddress;
}

} // util
} // eth
} // nn
