﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Synopsys DesignWare Ethernet QoS memory map
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

#include <nn/eth/eth_Result.h>

#include "../eth_Gmii.h"
#include "../eth_SocketIf.h"
#include "../eth_Worker.h"

#include "eth_SynopsysDwcEtherQos.descriptor.h"
#include "eth_SynopsysDwcEtherQos.reg.h"
#include "pcv/eth_Pcv.h"

//#define NN_DETAIL_ETH_ENABLE_DEBUG_FUNCTIONS
//#define NN_DETAIL_ETH_ENABLE_DEBUG_LOGS

#if defined(NN_DETAIL_ETH_ENABLE_DEBUG_LOGS)
#define NN_DETAIL_ETH_DEBUG_TRACE(...)  NN_DETAIL_ETH_TRACE(__VA_ARGS__)
#define NN_DETAIL_ETH_DEBUG_INFO(...)   NN_DETAIL_ETH_INFO(__VA_ARGS__)
#define NN_DETAIL_ETH_DEBUG_WARN(...)   NN_DETAIL_ETH_WARN(__VA_ARGS__)
#define NN_DETAIL_ETH_DEBUG_ERROR(...)  NN_DETAIL_ETH_ERROR(__VA_ARGS__)
#define NN_DETAIL_ETH_DEBUG_FATAL(...)  NN_DETAIL_ETH_FATAL(__VA_ARGS__)
#else
#define NN_DETAIL_ETH_DEBUG_TRACE(...)
#define NN_DETAIL_ETH_DEBUG_INFO(...)
#define NN_DETAIL_ETH_DEBUG_WARN(...)
#define NN_DETAIL_ETH_DEBUG_ERROR(...)
#define NN_DETAIL_ETH_DEBUG_FATAL(...)
#endif

#define ADDR_HIGH(addr) (static_cast<uint32_t>(addr >> 32))
#define ADDR_LOW(addr)  (static_cast<uint32_t>(addr))

namespace nn {
namespace eth {
namespace device {
namespace tx2 {

const uint8_t MacAddressVendorCode[] = {0x00, 0x04, 0x4b}; // NVidia

const auto PhyThreadPollIntervalMs  = 128;
const auto PhyThreadPollInterval    = nn::TimeSpan::FromMilliSeconds(PhyThreadPollIntervalMs);

const auto DmaResetPollInterval = nn::TimeSpan::FromMilliSeconds(10);
const auto DmaResetPollTimeout  = nn::TimeSpan::FromSeconds(1);

const auto PhyResetPollInterval = nn::TimeSpan::FromMilliSeconds(10);
const auto PhyResetPollTimeout  = nn::TimeSpan::FromSeconds(1);

const auto PhyReadPollInterval  = nn::TimeSpan::FromMilliSeconds(1);
const auto PhyReadPollTimeout   = nn::TimeSpan::FromSeconds(1);

const auto PhyWritePollInterval = nn::TimeSpan::FromMilliSeconds(1);
const auto PhyWritePollTimeout  = nn::TimeSpan::FromSeconds(1);

const size_t DmaTxBurstLength               = 32;
const size_t DmaRxBurstLength               = 32;
const size_t MtlTxQueueSize                 = 16 * 1024;    // 16KB
const size_t MtlRxQueueSize                 = 16 * 1024;    // 16KB
const size_t MtlRxFlowControlOnThreshold    = 4 * 1024;     // FULL - 4KB
const size_t MtlRxFlowControlOffThreshold   = 8 * 1024;     // FULL - 8KB

const pcv::ClockHz TxClockHzForSpeed_10 = 2500 * 1000;
const pcv::ClockHz TxClockHzForSpeed_100 = 25 * 1000 * 1000;
const pcv::ClockHz TxClockHzForSpeed_1000 = 125 * 1000 * 1000;

const size_t PrintableEqosRegSize = 0x1200;
const size_t PrintableMiiRegSize = 32;

class SynopsysDwcEtherQos :
    private Gmii,
    private SocketIf
{
public:
    SynopsysDwcEtherQos();
    virtual ~SynopsysDwcEtherQos();
    nn::Result Initialize();
    void Finalize();

private:
    enum Speed
    {
        Speed_10,   // 10Mbps
        Speed_100,  // 100Mbps
        Speed_1000, // 1000Mbps
    };

    // Utility functions
    NN_FORCEINLINE uint32_t GetTxDescPhysAddrHigh(int index) NN_NOEXCEPT
    {
        return ADDR_HIGH(m_TxDescRingPhysAddr) + index * sizeof(TxDesc);
    }

    NN_FORCEINLINE uint32_t GetTxDescPhysAddrLow(int index) NN_NOEXCEPT
    {
        return ADDR_LOW(m_TxDescRingPhysAddr) + index * sizeof(TxDesc);
    }

    NN_FORCEINLINE uint32_t GetRxDescPhysAddrHigh(int index) NN_NOEXCEPT
    {
        return ADDR_HIGH(m_RxDescRingPhysAddr) + index * sizeof(RxDesc);
    }

    NN_FORCEINLINE uint32_t GetRxDescPhysAddrLow(int index) NN_NOEXCEPT
    {
        return ADDR_LOW(m_RxDescRingPhysAddr) + index * sizeof(RxDesc);
    }

    NN_FORCEINLINE nn::Result GetSpeed(Speed* pOutSpeed, MediaType mediaType) NN_NOEXCEPT
    {
        switch(EthSpeed(mediaType))
        {
        case 10:
            *pOutSpeed = Speed_10;
            break;
        case 100:
            *pOutSpeed = Speed_100;
            break;
        case 1000:
            *pOutSpeed = Speed_1000;
            break;
        default:
            return nn::eth::ResultMediaNotSupported();
        }

        return nn::ResultSuccess();
    }

    // Debug functions
#if defined(NN_DETAIL_ETH_ENABLE_DEBUG_FUNCTIONS)
    nn::dd::PhysicalAddress GetPhysicalAddress(uintptr_t base, uint32_t offset) NN_NOEXCEPT;
    void Print(uintptr_t addr, size_t size) NN_NOEXCEPT;
    void PrintEqosReg() NN_NOEXCEPT;
    void PrintTxDesc(int index) NN_NOEXCEPT;
    void PrintTxDescAll() NN_NOEXCEPT;
    void PrintRxDesc(int index) NN_NOEXCEPT;
    void PrintRxDescAll() NN_NOEXCEPT;
    void PrintTxBuffer(int index) NN_NOEXCEPT;
    void PrintRxBuffer(int index) NN_NOEXCEPT;
    void PrintMiiRegister() NN_NOEXCEPT;
#endif

    // Descriptor control
    NN_FORCEINLINE void ReadTdes3(Tdes3* pOutTdes3, int index) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(Tdes3) == sizeof(uint32_t));

        volatile uint32_t* const pSrc = reinterpret_cast<volatile uint32_t*>(&m_pTxDescRing[index].tdes3);
        uint32_t* const pDst = reinterpret_cast<uint32_t*>(pOutTdes3);

        *pDst = *pSrc;
    }

    NN_FORCEINLINE void ReadRdes3(Rdes3* pOutRdes3, int index) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(Rdes3) == sizeof(uint32_t));

        volatile uint32_t* const pSrc = reinterpret_cast<volatile uint32_t*>(&m_pRxDescRing[index].rdes3);
        uint32_t* const pDst = reinterpret_cast<uint32_t*>(pOutRdes3);

        *pDst = *pSrc;
    }

    NN_FORCEINLINE void WriteTdes3(Tdes3* pTdes3, int index) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(Tdes3) == sizeof(uint32_t));

        uint32_t* const pSrc = reinterpret_cast<uint32_t*>(pTdes3);
        volatile uint32_t* const pDst = reinterpret_cast<volatile uint32_t*>(&m_pTxDescRing[index].tdes3);

        *pDst = *pSrc;
    }

    NN_FORCEINLINE void WriteRdes3(Rdes3* pRdes3, int index) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(Rdes3) == sizeof(uint32_t));

        uint32_t* const pSrc = reinterpret_cast<uint32_t*>(pRdes3);
        volatile uint32_t* const pDst = reinterpret_cast<volatile uint32_t*>(&m_pRxDescRing[index].rdes3);

        *pDst = *pSrc;
    }

    NN_FORCEINLINE void ReadTxDesc(TxDesc* pOutDesc, int index) NN_NOEXCEPT
    {
        volatile uint32_t* const pSrc = reinterpret_cast<volatile uint32_t*>(&m_pTxDescRing[index]);
        uint32_t* const pDst = reinterpret_cast<uint32_t*>(pOutDesc);

        // 16 byte のデータを 4 byte ずつコピー
        for (int i = 0; i < 4; i++)
        {
            pDst[i] = pSrc[i];
        }
    }

    NN_FORCEINLINE void ReadRxDesc(RxDesc* pOutDesc, int index) NN_NOEXCEPT
    {
        volatile uint32_t* const pSrc = reinterpret_cast<volatile uint32_t*>(&m_pRxDescRing[index]);
        uint32_t* const pDst = reinterpret_cast<uint32_t*>(pOutDesc);

        // 16 byte のデータを 4 byte ずつコピー
        for (int i = 0; i < 4; i++)
        {
            pDst[i] = pSrc[i];
        }
    }

    NN_FORCEINLINE void WriteTxDesc(TxDesc* pDesc, int index) NN_NOEXCEPT
    {
        uint32_t* const pSrc = reinterpret_cast<uint32_t*>(pDesc);
        volatile uint32_t* const pDst = reinterpret_cast<volatile uint32_t*>(&m_pTxDescRing[index]);

        // 16 byte のデータを 4 byte ずつコピー
        for (int i = 0; i < 4; i++)
        {
            pDst[i] = pSrc[i];
        }
    }

    NN_FORCEINLINE void WriteRxDesc(RxDesc* pDesc, int index) NN_NOEXCEPT
    {
        uint32_t* const pSrc = reinterpret_cast<uint32_t*>(pDesc);
        volatile uint32_t* const pDst = reinterpret_cast<volatile uint32_t*>(&m_pRxDescRing[index]);

        // 16 byte のデータを 4 byte ずつコピー
        for (int i = 0; i < 4; i++)
        {
            pDst[i] = pSrc[i];
        }
    }

    NN_FORCEINLINE void InvalidateTxDescCache(int index) NN_NOEXCEPT
    {
        nn::dd::InvalidateDataCache(&m_pTxDescRing[index], sizeof(TxDesc));
    }

    NN_FORCEINLINE void InvalidateRxDescCache(int index) NN_NOEXCEPT
    {
        nn::dd::InvalidateDataCache(&m_pRxDescRing[index], sizeof(RxDesc));
    }

    NN_FORCEINLINE void FlushTxDescCache(int index) NN_NOEXCEPT
    {
        nn::dd::FlushDataCache(&m_pTxDescRing[index], sizeof(TxDesc));
    }

    NN_FORCEINLINE void FlushRxDescCache(int index) NN_NOEXCEPT
    {
        nn::dd::FlushDataCache(&m_pRxDescRing[index], sizeof(RxDesc));
    }

    void InitializeTxDesc(int index) NN_NOEXCEPT;
    void InitializeRxDesc(int index) NN_NOEXCEPT;
    void PrepareTxDescToSend(int index, size_t dataSize) NN_NOEXCEPT;
    void PrepareRxDescToReceive(int index) NN_NOEXCEPT;
    bool CheckTxDesc(const TxDesc& desc) NN_NOEXCEPT;
    bool CheckRxDesc(const RxDesc& desc) NN_NOEXCEPT;
    void PrintTxDescError(const TxDesc& desc) NN_NOEXCEPT;
    void PrintRxDescError(const RxDesc& desc) NN_NOEXCEPT;

    // Initialize functions
    void InitializeMacAddress() NN_NOEXCEPT;
    void InitializeDesc() NN_NOEXCEPT;
    void InitializeBuffer() NN_NOEXCEPT;
    void InitializeDma() NN_NOEXCEPT;
    void InitializeMtl() NN_NOEXCEPT;
    void InitializeMac() NN_NOEXCEPT;

    // Phy access functions (overrides Gmii)
    nn::Result PhyReset() NN_NOEXCEPT NN_OVERRIDE;
    nn::Result PhyReadReg(uint16_t miiRegister, uint16_t* pMiiValue) NN_NOEXCEPT NN_OVERRIDE;
    nn::Result PhyWriteReg(uint16_t miiRegister, uint16_t miiValue) NN_NOEXCEPT NN_OVERRIDE;
    nn::Result PhyGetMedia(MediaType* pMediaType) NN_NOEXCEPT NN_OVERRIDE;
    nn::Result PhySetMedia(MediaType type) NN_NOEXCEPT NN_OVERRIDE;
    nn::Result PhyAdjustMedia(MediaType type) NN_NOEXCEPT NN_OVERRIDE;

    // Phy related functions
    void ProcessLinkStateChange(bool linkup);

    // Mac related functions
    void UpdateMac(Speed speed, bool fullDuplex, bool flowControlEnable) NN_NOEXCEPT;

    // Mtl related functions
    void UpdateMtl(bool fullDuplex) NN_NOEXCEPT;

    // Clock related functions
    void UpdateClock(Speed speed) NN_NOEXCEPT;

    // Socket related (overrides for SocketIf)
    void Start() NN_NOEXCEPT NN_OVERRIDE;

    // Thread functions
    void TxThread() NN_NOEXCEPT;
    void ProcessTxCompleteInterrupt() NN_NOEXCEPT;
    void ProcessRxInterrupt() NN_NOEXCEPT;
    void InterruptThread() NN_NOEXCEPT;
    void PhyThread() NN_NOEXCEPT;

    // Thread objects
    Worker m_TxThread;
    Worker m_InterruptThread;
    Worker m_PhyThread;

    // virtual address of registers
    uintptr_t m_pEqos;

    // Event
    nn::os::Event m_TxRequestEvent;
    nn::os::TimerEvent m_PhyTimerEvent;
    nn::os::InterruptEventType m_InterruptEvent[InterruptCount];
    nn::os::MultiWaitHolderType m_InterruptEventHolder[InterruptCount];
    nn::os::MultiWaitType m_InterruptMultiWait;

    // Mutex
    nn::os::Mutex m_PhyMutex; // PHY レジスタを読み書きするスレッドがロックする
    nn::os::Mutex m_TxDescMutex; // m_TxDescConditionalVariable と共に使う mutex

    // Conditional Variable
    nn::os::ConditionVariable m_TxDescConditionalVariable; // m_AvailableTxDescCount を操作する際に使う条件変数

    // Descriptor
    nn::dd::PhysicalAddress m_TxDescRingPhysAddr;
    nn::dd::PhysicalAddress m_RxDescRingPhysAddr;
    nn::dd::PhysicalAddress m_TxDescRingTailPhysAddr;
    nn::dd::PhysicalAddress m_RxDescRingTailPhysAddr;
    TxDesc* m_pTxDescRing;
    RxDesc* m_pRxDescRing;
    int m_TxDescHead; // 次に CPU が送信データを書き込む descriptor のインデックス
    int m_TxDescTail; // 次に DMA が送信完了する descriptor のインデックス
    int m_RxDescHead; // 次に DMA が受信データを書き込み descriptor のインデックス
    int m_AvailableTxDescCount; // CPU が使用可能な空き送信ディスクリプタの数

    // Buffer
    TxBuffer* m_pTxBuffer;
    RxBuffer* m_pRxBuffer;
    nn::dd::PhysicalAddress m_TxBufferPhysAddr[TxDescRingLength];
    nn::dd::PhysicalAddress m_RxBufferPhysAddr[RxDescRingLength];
};

} // tx2
} // device
} // eth
} // nn
