﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Synopsys DesignWare Ethernet QoS memory map
 */

#pragma once

#include <nn/nn_StaticAssert.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_BitPack.h>

namespace nn {
namespace eth {
namespace device {
namespace tx2 {

const size_t CacheLineSize = 64;
const size_t DescPaddingSize = 48;

enum Owner : bool
{
    Owner_Cpu = false,
    Owner_Dma = true,
};

struct Tdes0 : public nn::util::BitPack32
{
    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > BufferAddressLow;

    // Normal Descriptor (Write-Back format) and Context Descriptor
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > TimestampLow;
};

struct Tdes1 : public nn::util::BitPack32
{
    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > BufferAddressHigh;

    // Normal Descriptor (Write-Back format) and Context Descriptor
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > TimestampHigh;
};

struct Tdes2 : public nn::util::BitPack32
{
    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field< 31,  1, bool     > Ioc;             // Interrupt on Completion
    typedef nn::util::BitPack32::Field< 30,  1, bool     > Ttse;            // Transmit Timestamp Enable
    typedef nn::util::BitPack32::Field< 16, 14, uint32_t > Buffer2Length;   // unused
    typedef nn::util::BitPack32::Field< 14,  2, uint32_t > Vtir;            // VLAN Tag Insertion or Replacement
    typedef nn::util::BitPack32::Field<  0, 14, uint32_t > BufferLength;

    // Normal Descriptor (Write-Back format)
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > Reserved0;

    // Context Descriptor
    typedef nn::util::BitPack32::Field< 16, 32, uint32_t > Ivt;     // Inner VLAN Tag
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Reserved1;
    typedef nn::util::BitPack32::Field<  0, 15, uint32_t > Mss;     // Maximum Segment Size
};

struct Tdes3 : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 31,  1, bool     > Own;     // 1:owner=DMA  0:owner=CPU
    typedef nn::util::BitPack32::Field< 30,  1, bool     > Ctxt;    // Context Type

    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field< 29,  1, bool     > Fd;      // First Descriptor
    typedef nn::util::BitPack32::Field< 28,  1, bool     > Ld;      // Last descriptor
    typedef nn::util::BitPack32::Field< 26,  2, uint32_t > Cpc;     // CRC Pad Control
    typedef nn::util::BitPack32::Field< 23,  3, uint32_t > Saic;    // Source Address Insertion Control
    typedef nn::util::BitPack32::Field< 19,  4, uint32_t > Thl;     // TCP Header Length (valid only if TSE=1)
    typedef nn::util::BitPack32::Field< 18,  1, bool     > Tse;     // TCP Segmentation Enable
    typedef nn::util::BitPack32::Field< 16,  2, uint32_t > Cic;     // Checksum Insertion Control
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Reserved0;
    typedef nn::util::BitPack32::Field< 0,  15, uint32_t > PacketLength;

    // Normal Descriptor (Write-Back format)
    typedef nn::util::BitPack32::Field< 16, 12, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Es;      // Error Summary
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Jt;      // Jabber Timeout
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Ff;      // Packet Flushed
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Pce;     // Payload Checksum Error
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Loc;     // Loss of Carrier
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Nc;      // No Carrier
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Lc;      // Late Colision
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Ec;      // Excessive Collision
    typedef nn::util::BitPack32::Field<  4,  4, uint32_t > Cc;      // Collision Count
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Ed;      // Excessive Deferral
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Uf;      // Underflow Error
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Db;      // Deferred Bit
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Ihe;     // IP Header Error

    // Context Descriptor
    typedef nn::util::BitPack32::Field< 16, 14, uint32_t > Control; // See TRM 40.6.6
    typedef nn::util::BitPack32::Field<  0, 16, uint32_t > Ivt;     // Inner VLAN Tag
};

struct NN_ALIGNAS(CacheLineSize) TxDesc
{
    Tdes0 tdes0;
    Tdes1 tdes1;
    Tdes2 tdes2;
    Tdes3 tdes3;
    uint8_t padding[DescPaddingSize];
};
NN_STATIC_ASSERT(sizeof(TxDesc) == CacheLineSize);

struct Rdes0 : public nn::util::BitPack32
{
    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > BufferAddressLow;

    // Normal Descriptor (Write-Back format)
    typedef nn::util::BitPack32::Field< 16, 32, uint32_t > Ivt; // Inner VLAN Tag
    typedef nn::util::BitPack32::Field<  0, 16, uint32_t > Ovt; // Outer VLAN Tag

    // Context Descriptor
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > TimestampLow;
};

struct Rdes1 : public nn::util::BitPack32
{
    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > BufferAddressHigh;

    // Normal Descriptor (Write-Back format)
    typedef nn::util::BitPack32::Field< 16, 32, uint32_t > Opc;     // OAM code or MAC Control Opcode
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Td;      // Timestamp Dropped
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Tsa;     // Timestamp Available
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Pv;      // PTP Version
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Pft;     // PTP Packet Type
    typedef nn::util::BitPack32::Field<  8,  4, uint32_t > Pmt;     // PTP Message Type
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Ipce;    // IP Payload Error
    typedef nn::util::BitPack32::Field<  6,  1, bool     > Ipcb;    // IP Checksum Bypassed
    typedef nn::util::BitPack32::Field<  5,  1, bool     > Ipv6;    // IPv6 Header Present
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Ipv4;    // IPv4 Header Present
    typedef nn::util::BitPack32::Field<  3,  1, bool     > Iphe;    // IP Header Error
    typedef nn::util::BitPack32::Field<  0,  3, uint32_t > Pt;      // Payload Type

    // Context Descriptor
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > TimestampHigh;
};

struct Rdes2 : public nn::util::BitPack32
{
    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > Payload;

    // Normal Descriptor (Write-Back format)
    typedef nn::util::BitPack32::Field< 29, 32, uint32_t > L3l4fm;      // Layer 3 and Layer 4 Filter Number Matched
    typedef nn::util::BitPack32::Field< 28,  1, bool     > L4fm;        // Layer 4 Filter Match
    typedef nn::util::BitPack32::Field< 27,  1, bool     > L3fm;        // Layer 3 Filter Match
    typedef nn::util::BitPack32::Field< 19,  8, uint32_t > Madrm;       // MAC Address Match or Hash Value
    typedef nn::util::BitPack32::Field< 18,  1, bool     > Hf;          // Hash Filter Status
    typedef nn::util::BitPack32::Field< 17,  1, bool     > Daf;         // Destination Address Filter Fail
    typedef nn::util::BitPack32::Field< 16,  1, bool     > Saf;         // Source Address Filter Fail
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Vf;          // VLAN Filter Status
    typedef nn::util::BitPack32::Field< 11,  4, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 10,  1, bool     > Arpnr;       // ARP Reply Not Generated
    typedef nn::util::BitPack32::Field<  0, 10, uint32_t > Hl;          // L3/L4 Header Length

    // Context Descriptor
    typedef nn::util::BitPack32::Field<  0, 32, uint32_t > Reserved1;
};

struct Rdes3 : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 31,  1, bool     > Own;     // Own Bit
    typedef nn::util::BitPack32::Field< 30,  1, bool     > Ctxt;    // Receive Context Descriptor

    // Normal Descriptor (Read format)
    typedef nn::util::BitPack32::Field< 30,  1, bool     > Ioc;     // Interrupt on Completion
    typedef nn::util::BitPack32::Field< 26,  4, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 25,  1, bool     > Buf2v;   // Buffer 2 Address Valid
    typedef nn::util::BitPack32::Field< 24,  1, bool     > Buf1v;   // Buffer 1 Address Valid
    typedef nn::util::BitPack32::Field<  0, 24, uint32_t > Reserved1;

    // Normal Descriptor (Write-Back format)
    typedef nn::util::BitPack32::Field< 29,  1, bool     > Fd;      // First Descriptor
    typedef nn::util::BitPack32::Field< 28,  1, bool     > Ld;      // Last Descriptor
    typedef nn::util::BitPack32::Field< 27,  1, bool     > Rs2v;    // Receive Status RDES2 Valid
    typedef nn::util::BitPack32::Field< 26,  1, bool     > Rs1v;    // Receive Status RDES1 Valid
    typedef nn::util::BitPack32::Field< 25,  1, bool     > Rs0v;    // Receive Status RDES0 Valid
    typedef nn::util::BitPack32::Field< 24,  1, bool     > Ce;      // CRC Error
    typedef nn::util::BitPack32::Field< 23,  1, bool     > Gp;      // Giant Packet
    typedef nn::util::BitPack32::Field< 22,  1, bool     > Rwt;     // Receive Watchdog Timeout
    typedef nn::util::BitPack32::Field< 21,  1, bool     > Oe;      // Overflow Error
    typedef nn::util::BitPack32::Field< 20,  1, bool     > Re;      // Receive Error
    typedef nn::util::BitPack32::Field< 19,  1, bool     > De;      // Dribble Bit Error
    typedef nn::util::BitPack32::Field< 16,  3, uint32_t > Lt;      // Length/Type Field
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Es;      // Error Summary
    typedef nn::util::BitPack32::Field<  0, 15, uint32_t > PacketLength;

    // Context Descriptor
    typedef nn::util::BitPack32::Field<  0, 30, uint32_t > Reserved2;
};

struct NN_ALIGNAS(CacheLineSize) RxDesc
{
    Rdes0 rdes0;
    Rdes1 rdes1;
    Rdes2 rdes2;
    Rdes3 rdes3;
    uint8_t padding[DescPaddingSize];
};
NN_STATIC_ASSERT(sizeof(RxDesc) == CacheLineSize);

// ディスクリプタの数
const size_t TxDescRingLength = 64;
const size_t RxDescRingLength = 64;

// ディスクリプタの合計サイズ
const size_t TxDescTotalSize = sizeof(TxDesc) * TxDescRingLength;
const size_t RxDescTotalSize = sizeof(RxDesc) * RxDescRingLength;

// TORIAEZU: 4KB を超える場合は物理アドレス連続性 / 4GB 境界などに注意する必要がある
// TODO: SMMU 対応で物理アドレス連続性や 4GB 境界を超えないことを担保
NN_STATIC_ASSERT(TxDescTotalSize <= 4096);
NN_STATIC_ASSERT(RxDescTotalSize <= 4096);

// TORIAEZU: 4KB ページ1つに収める
const size_t TxBufferSize = 4096;
struct TxBuffer
{
    uint8_t bytes[TxBufferSize];
};

const size_t RxBufferSize = 4096;
struct RxBuffer
{
    uint8_t bytes[RxBufferSize];
};

// 送受信バッファの合計サイズ
const size_t TxBufferTotalSize = sizeof(TxBuffer) * TxDescRingLength;
const size_t RxBufferTotalSize = sizeof(RxBuffer) * RxDescRingLength;

} // tx2
} // device
} // eth
} // nn
