﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/dd.h>
#include <cstring>
#include <mutex>
#include "eth_EthernetDriver.h"
#include "eth_EthernetRegister.h"
#include <nn/svc/svc_Dd.h>

namespace nn { namespace os {
size_t GetMemoryHeapSize();
}}
namespace nn { namespace drivers { namespace eth {

namespace {
constexpr size_t RoundUp(size_t value, size_t align)
{
    return ((value + (align - 1)) & ~(align - 1));
}

const size_t STACK_SIZE  = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char g_stack[STACK_SIZE];

void InterruptThread(void* p)
{
    EthernetDriver* enet = reinterpret_cast<EthernetDriver*>(p);
    enet->HandleInterrupt();
}

}

uintptr_t g_EnetBegin;

nn::Result EthernetDriver::Initialize()
{
    m_TxMutex.Initialize();
    m_ConditionVariable.Initialize();

    m_RequestedFinalize = false;
    m_TxProducer = 0;
    m_TxConsumer = 0;
    m_TxFree = TXBD_NUM;
    m_RxIndex = 0;
    m_SendCompletedCallBack = NULL;
    m_FrameReceivedCallBack = NULL;

    g_EnetBegin = nn::dd::QueryIoMappingAddress(EnetRegionAddr, EnetRegionSize);
    NN_ABORT_UNLESS(g_EnetBegin != 0);

    m_DeviceBufferSize = RoundUp(
            (1 + (RXBD_NUM + TXBD_NUM) / (nn::os::MemoryPageSize / FRAME_BUFFER_SIZE)), nn::os::MemoryHeapUnitSize);

    static_assert((sizeof(RxBd) * RXBD_NUM + sizeof(TxBd) * TXBD_NUM) <= nn::os::MemoryPageSize, "");
    static_assert(FRAME_BUFFER_SIZE <= nn::os::MemoryPageSize, "");
    NN_ABORT_UNLESS(nn::os::SetMemoryHeapSize(m_DeviceBufferSize + nn::os::GetMemoryHeapSize()).IsSuccess());
    NN_ABORT_UNLESS(nn::os::AllocateMemoryBlock(&m_DeviceBuffer, m_DeviceBufferSize).IsSuccess());
    std::memset(reinterpret_cast<void*>(m_DeviceBuffer), 0, m_DeviceBufferSize);
    nn::dd::FlushDataCache(reinterpret_cast<void*>(m_DeviceBuffer), m_DeviceBufferSize);

    nn::dd::PhysicalAddress paddr;
    NN_ABORT_UNLESS(nn::dd::QuerySinglePhysicalAddress(&paddr, reinterpret_cast<const void*>(m_DeviceBuffer), 1).IsSuccess());

    m_RxBd = reinterpret_cast<RxBd*>(m_DeviceBuffer);
    m_TxBd = reinterpret_cast<TxBd*>(m_RxBd + RXBD_NUM);
    uintptr_t physRxBd = static_cast<uintptr_t>(paddr);
    uintptr_t physTxBd = static_cast<uintptr_t>(paddr) + (reinterpret_cast<uintptr_t>(m_TxBd) - reinterpret_cast<uintptr_t>(m_RxBd));

    uintptr_t vaddr = m_DeviceBuffer + nn::os::MemoryPageSize;
    for (int i = 0; i < RXBD_NUM; i++)
    {
        NN_ABORT_UNLESS(nn::dd::QuerySinglePhysicalAddress(&paddr, reinterpret_cast<const void*>(vaddr), 1).IsSuccess());
        m_RxBdBuffer[i] = vaddr;
        m_RxBd[i].status0 = (i == RXBD_NUM - 1)? ENET_RXBD_STS0_W | ENET_RXBD_STS0_E : ENET_RXBD_STS0_E;
        m_RxBd[i].length = 0;
        m_RxBd[i].pointer = paddr;
        m_RxBd[i].status1 = ENET_RXBD_STS1_INT;
        m_RxBd[i].status2 = 0;
        m_RxBd[i].header = 0;
        m_RxBd[i].timestamp = 0;
        m_RxBd[i].dummy1 = 0;
        m_RxBd[i].dummy2 = 0;
        vaddr += nn::os::MemoryPageSize / (nn::os::MemoryPageSize / FRAME_BUFFER_SIZE);
    }
    nn::dd::StoreDataCache(m_RxBd, sizeof(RxBd) * RXBD_NUM);

    for (int i = 0; i < TXBD_NUM; i++)
    {
        NN_ABORT_UNLESS(nn::dd::QuerySinglePhysicalAddress(&paddr, reinterpret_cast<const void*>(vaddr), 1).IsSuccess());
        m_TxBdBuffer[i] = vaddr;
        m_TxBd[i].status0 = (i == TXBD_NUM - 1)?ENET_TXBD_STS0_W: 0;
        m_TxBd[i].length = 0;
        m_TxBd[i].pointer = paddr;
        m_TxBd[i].status1 = ENET_TXBD_STS1_INT;
        m_TxBd[i].status2 = 0;
        m_TxBd[i].dummy0 = 0;
        m_TxBd[i].timestamp = 0;
        m_TxBd[i].dummy1 = 0;
        m_TxBd[i].dummy2 = 0;
        vaddr += nn::os::MemoryPageSize / (nn::os::MemoryPageSize / FRAME_BUFFER_SIZE);
    }
    nn::dd::StoreDataCache(m_TxBd, sizeof(TxBd) * TXBD_NUM);

    // MACアドレスは u-boot で初期化される前提
    // リセット後に再設定する
    uint32_t palr = ENET_PALR;
    uint32_t paur = ENET_PAUR;

    // レジスタの設定
    ENET_ECR = ENET_ECR_RESET;
    while (ENET_ECR & ENET_ECR_RESET)
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1) );
    }

    ENET_EIMR = 0;
    ENET_EIR = ENET_EIR;
    ENET_RCR = ENET_RCR_NLC | (MAX_FRAME_SIZE << ENET_RCR_MAX_FL_SHIFT) |
        ENET_RCR_RGMII_EN | ENET_RCR_FCE | ENET_RCR_MII_MODE;
    ENET_RSFL = RX_SECTION_FULL;
    ENET_RSEM = RX_SECTION_EMPTY;
    ENET_RAEM = RX_ALMOST_EMPTY;
    ENET_RAFL = RX_ALMOST_FULL;
    ENET_OPD  = PAUSE_DUR;

    ENET_PALR = palr;
    ENET_PAUR = paur;

    ENET_MSCR = (((MII_CLOCK / 1000000) + 2) / 5) << 1;
    ENET_OPD  = 0x00010020;
    ENET_TFWR = 0x00000002;
    ENET_IAUR = 0;
    ENET_IALR = 0;
    ENET_GAUR = 0;
    ENET_GALR = 0;
    ENET_MRBR = FRAME_BUFFER_SIZE;
    ENET_RDSR = physRxBd;
    ENET_TDSR = physTxBd;

    // Phy Reset & Auto-negotiation
    StartAutoNegotiation();

    WaitLinkUp();

    nn::os::CreateThread(&m_InterruptTthread, &InterruptThread, this, g_stack, STACK_SIZE, nn::os::HighestSystemThreadPriority);
    nn::os::StartThread(&m_InterruptTthread);

    ENET_TCR = ENET_TCR_FDEN;

    ENET_TFWR |= ENET_TFWR_STRFWD;
    ENET_EIMR = ENET_EIMR_TXF | ENET_EIMR_RXF;

    ENET_ECR |= ENET_ECR_DBSWP | ENET_ECR_EN1588 | ENET_ECR_ETHEREN;

    ENET_RDAR = 0x01000000;

    return ResultSuccess();
}

nn::Result EthernetDriver::Finalize()
{
    m_RequestedFinalize = true;
    nn::os::WaitThread(&m_InterruptTthread);
    nn::os::DestroyThread(&m_InterruptTthread);
    ENET_ECR = 0;
    ENET_EIMR = 0;

    m_TxMutex.Finalize();
    m_ConditionVariable.Finalize();

    nn::os::FreeMemoryBlock(m_DeviceBuffer, m_DeviceBufferSize);

    return ResultSuccess();
}

nn::Result EthernetDriver::Send(const void* frameData, int frameBytes)
{
    if (frameBytes > MAX_FRAME_SIZE)
    {
        return nn::os::ResultInvalidParameter();
    }

    {
        std::lock_guard<InternalMutex> lock(m_TxMutex);

        while (m_TxFree == 0)
        {
            m_ConditionVariable.Wait(&m_TxMutex);
        }

        TxBd* txBd = &m_TxBd[m_TxProducer];
        nn::dd::FlushDataCache(txBd, sizeof(*txBd));

        void* buffer = reinterpret_cast<void*>(m_TxBdBuffer[m_TxProducer]);
        std::memcpy(buffer, frameData, frameBytes);
        nn::dd::StoreDataCache(buffer, frameBytes);

        txBd->status2 = 0;
        txBd->timestamp = 0;
        txBd->length = frameBytes;
        uint16_t status = (m_TxProducer == TXBD_NUM - 1)?ENET_TXBD_STS0_W: 0;
        nn::dd::EnsureMemoryAccess();
        txBd->status0 = status | ENET_TXBD_STS0_R | ENET_TXBD_STS0_L | ENET_TXBD_STS0_TC;
        nn::dd::StoreDataCache(txBd, sizeof(*txBd));

        m_TxDataBuffer[m_TxProducer] = frameData;

        m_TxProducer = (m_TxProducer == TXBD_NUM - 1)? 0: m_TxProducer + 1;
        m_TxFree--;
        ENET_TDAR = 0x01000000;
    }

    return ResultSuccess();
}

nn::Result EthernetDriver::Send(nnnetOslMbuf* pMbufHead)
{
    std::lock_guard<InternalMutex> lock(m_TxMutex);

    while (m_TxFree == 0)
    {
        m_ConditionVariable.Wait(&m_TxMutex);
    }

    TxBd* txBd = &m_TxBd[m_TxProducer];
    nn::dd::FlushDataCache(txBd, sizeof(*txBd));

    uint8_t* buffer = reinterpret_cast<uint8_t*>(m_TxBdBuffer[m_TxProducer]);

    nnnetOslMbuf* pMbuf = pMbufHead;
    size_t offset = 0;
    for (;;)
    {
        void* segmentData = nnnetOslMbuf_tod(pMbuf);
        int  segmentBytes = pMbufHead->m_len;
        bool isFirstSegment = (pMbuf == pMbufHead);
        bool isLastSegment = (pMbuf->m_next == NULL);
        std::memcpy(buffer + offset, segmentData, segmentBytes);
        offset += segmentBytes;
        if (offset > MAX_FRAME_SIZE)
        {
            return nn::os::ResultInvalidParameter();
        }

        if (isLastSegment)
        {
            break;
        }
        pMbuf = pMbuf->m_next;
    }
    nn::dd::StoreDataCache(buffer, offset);

    txBd->status2 = 0;
    txBd->timestamp = 0;
    txBd->length = offset;
    uint16_t status = (m_TxProducer == TXBD_NUM - 1)?ENET_TXBD_STS0_W: 0;
    nn::dd::EnsureMemoryAccess();
    txBd->status0 = status | ENET_TXBD_STS0_R | ENET_TXBD_STS0_L | ENET_TXBD_STS0_TC;
    nn::dd::StoreDataCache(txBd, sizeof(*txBd));

    m_TxDataBuffer[m_TxProducer] = pMbufHead;

    m_TxProducer = (m_TxProducer == TXBD_NUM - 1)? 0: m_TxProducer + 1;
    m_TxFree--;
    ENET_TDAR = 0x01000000;

    return ResultSuccess();
}

void EthernetDriver::HandleInterrupt()
{
    const nn::os::InterruptName InterruptNameEnet = 150;
    nn::os::InterruptEventType  interruptEvent;
    nn::os::InitializeInterruptEvent(&interruptEvent,
                                     InterruptNameEnet,
                                     nn::os::EventClearMode_ManualClear);

    while (!m_RequestedFinalize)
    {
        nn::os::WaitInterruptEvent(&interruptEvent);

        // 割り込みクリア
        ENET_EIR = ENET_EIR;

        Receive();
        SendComplete();

        nn::os::ClearInterruptEvent(&interruptEvent);
    }
    nn::os::FinalizeInterruptEvent(&interruptEvent);
}

void EthernetDriver::SendComplete()
{
    bool wakeupSender = false;

    std::lock_guard<InternalMutex> lock(m_TxMutex);

    while (m_TxFree < TXBD_NUM)
    {
        TxBd* txBd = &m_TxBd[m_TxConsumer];
        nn::dd::FlushDataCache(txBd, sizeof(*txBd));
        if (txBd->status0 & ENET_TXBD_STS0_R)
        {
            break;
        }
        const void* frameData = m_TxDataBuffer[m_TxConsumer];

        m_TxConsumer = (m_TxConsumer == TXBD_NUM - 1)? 0: m_TxConsumer + 1;
        if (m_TxFree++ == 0)
        {
            wakeupSender = true;
        }

        if (m_SendCompletedCallBack)
        {
            m_SendCompletedCallBack(const_cast<void*>(frameData));
        }
    }

    if (wakeupSender)
    {
        m_ConditionVariable.Broadcast();
    }
}

void EthernetDriver::Receive()
{
    // 1周したら明け渡す。
    for (int i = 0; i < RXBD_NUM; i++)
    {
        RxBd* rxBd = &m_RxBd[m_RxIndex];
        nn::dd::FlushDataCache(rxBd, sizeof(*rxBd));
        uint16_t status0 = rxBd->status0;
        nn::dd::EnsureMemoryAccess();
        uint16_t status1 = rxBd->status1;
        if (status0 & ENET_RXBD_STS0_E)
        {
            break;
        }

        void* buffer = reinterpret_cast<void*>((m_RxBdBuffer[m_RxIndex]));

        if (((status0 & (ENET_RXBD_STS0_L | ENET_RXBD_STS0_ERROR)) == ENET_RXBD_STS0_L) &&
                ((status1 & ENET_RXBD_STS1_ERROR) == 0))
        {
            if (m_FrameReceivedCallBack)
            {
                nn::dd::FlushDataCache(buffer, rxBd->length);
                m_FrameReceivedCallBack(buffer, rxBd->length);
            }
        }
        nn::dd::FlushDataCache(buffer, FRAME_BUFFER_SIZE);
        rxBd->length = 0;
        rxBd->status2 = 0;
        rxBd->header = 0;
        rxBd->timestamp = 0;
        nn::dd::EnsureMemoryAccess();
        rxBd->status0 = (m_RxIndex == RXBD_NUM - 1)? ENET_RXBD_STS0_W | ENET_RXBD_STS0_E : ENET_RXBD_STS0_E;
        nn::dd::StoreDataCache(rxBd, sizeof(*rxBd));
        m_RxIndex = (m_RxIndex == RXBD_NUM - 1)? 0: m_RxIndex + 1;
        ENET_RDAR = 0x01000000;
    }
    ENET_RDAR = 0x01000000;
}

nn::Result EthernetDriver::GetMacAddress(uint8_t pMacAddress[NN_NET_MAC_ADDRESS_SIZE]) const
{
    uint32_t palr = ENET_PALR;
    uint32_t paur = ENET_PAUR;

    pMacAddress[0] = (palr >> 24) & 0xFF;
    pMacAddress[1] = (palr >> 16) & 0xFF;
    pMacAddress[2] = (palr >>  8) & 0xFF;
    pMacAddress[3] = (palr >>  0) & 0xFF;
    pMacAddress[4] = (paur >> 24) & 0xFF;
    pMacAddress[5] = (paur >> 16) & 0xFF;

    return ResultSuccess();
}

uint16_t EthernetDriver::PhyRead(uint8_t addr)
{
    // MII割り込みクリア
    ENET_EIR = ENET_EIR_MII;

    ENET_MMFR = ENET_MMFR_ST | ENET_MMFR_OP_READ |
        (PHY_ADDRESS << ENET_MMFR_PA_SHIFT) |
        (addr << ENET_MMFR_RA_SHIFT) | ENET_MMFR_TA;

    // MII割り込みのポーリング
    while ((ENET_EIR & ENET_EIR_MII) == 0)
    {
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(100) );
    }

    // MII割り込みクリア
    ENET_EIR = ENET_EIR_MII;

    uint16_t reg = ENET_MMFR & 0xFFFF;
    return reg;
}

void EthernetDriver::PhyWrite(uint8_t addr, uint16_t val)
{
    // MII割り込みクリア
    ENET_EIR = ENET_EIR_MII;

    ENET_MMFR = ENET_MMFR_ST | ENET_MMFR_OP_WRITE |
        (PHY_ADDRESS << ENET_MMFR_PA_SHIFT) |
        (addr << ENET_MMFR_RA_SHIFT) | ENET_MMFR_TA | val;

    // MII割り込みのポーリング
    while ((ENET_EIR & ENET_EIR_MII) == 0)
    {
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(100) );
    }

    // MII割り込みクリア
    ENET_EIR = ENET_EIR_MII;
}

void EthernetDriver::StartAutoNegotiation()
{
    PhyWrite(0, 0x8000);
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1) );

    PhyWrite(4, 0x0100 | 0x0080 | 0x0040 | 0x0020 | 0x0001);
    PhyWrite(0, 0x1200);
}

void EthernetDriver::WaitLinkUp()
{
    // Auto-Negotiation 完了待ち (最大10秒)
    // Link Up も同時に待つ。
    // Auto-Negotiation 完了直後で Link Up前に送信を開始すると
    // Link Upに失敗することがある。
    uint16_t bmsr;
    for (int i = 0; i < 100; i++)
    {
        bmsr = PhyRead(1);
        if ((bmsr & 0x24) == 0x24)
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    }
    if ((bmsr & 0x24) == 0x24)
    {
        uint16_t gtsr = PhyRead(10);
        if (gtsr & 0x0C00)
        {
            ENET_ECR |= ENET_ECR_SPEED;
        }
        else
        {
            uint16_t anlpar = PhyRead(5);
            if (anlpar & 0x0180)
            {
            }
            else if (anlpar & 0x0060)
            {
                ENET_RCR |= ENET_RCR_RMII_10T;
            }
        }
    }
    else
    {
        // Link up 失敗時は 1000Base-Tにしておく
        ENET_ECR |= ENET_ECR_SPEED;
    }
}

void EthernetDriver::DumpRegister()
{
    NN_SDK_LOG("ENET_EIR=0x%08x\n", ENET_EIR);
    NN_SDK_LOG("ENET_EIMR=0x%08x\n", ENET_EIMR);
    NN_SDK_LOG("ENET_RDAR=0x%08x\n", ENET_RDAR);
    NN_SDK_LOG("ENET_TDAR=0x%08x\n", ENET_TDAR);
    NN_SDK_LOG("ENET_ECR=0x%08x\n", ENET_ECR);
    NN_SDK_LOG("ENET_MMFR=0x%08x\n", ENET_MMFR);
    NN_SDK_LOG("ENET_MSCR=0x%08x\n", ENET_MSCR);
    NN_SDK_LOG("ENET_MIBC=0x%08x\n", ENET_MIBC);
    NN_SDK_LOG("ENET_RCR=0x%08x\n", ENET_RCR);
    NN_SDK_LOG("ENET_TCR=0x%08x\n", ENET_TCR);
    NN_SDK_LOG("ENET_PALR=0x%08x\n", ENET_PALR);
    NN_SDK_LOG("ENET_PAUR=0x%08x\n", ENET_PAUR);
    NN_SDK_LOG("ENET_OPD=0x%08x\n", ENET_OPD);
    NN_SDK_LOG("ENET_IAUR=0x%08x\n", ENET_IAUR);
    NN_SDK_LOG("ENET_IALR=0x%08x\n", ENET_IALR);
    NN_SDK_LOG("ENET_GAUR=0x%08x\n", ENET_GAUR);
    NN_SDK_LOG("ENET_GALR=0x%08x\n", ENET_GALR);
    NN_SDK_LOG("ENET_TFWR=0x%08x\n", ENET_TFWR);
    NN_SDK_LOG("ENET_RDSR=0x%08x\n", ENET_RDSR);
    NN_SDK_LOG("ENET_TDSR=0x%08x\n", ENET_TDSR);
    NN_SDK_LOG("ENET_MRBR=0x%08x\n", ENET_MRBR);
    NN_SDK_LOG("ENET_RSFL=0x%08x\n", ENET_RSFL);
    NN_SDK_LOG("ENET_RSEM=0x%08x\n", ENET_RSEM);
    NN_SDK_LOG("ENET_RAEM=0x%08x\n", ENET_RAEM);
    NN_SDK_LOG("ENET_RAFL=0x%08x\n", ENET_RAFL);
    NN_SDK_LOG("ENET_TSEM=0x%08x\n", ENET_TSEM);
    NN_SDK_LOG("ENET_TAEM=0x%08x\n", ENET_TAEM);
    NN_SDK_LOG("ENET_TAFL=0x%08x\n", ENET_TAFL);
    NN_SDK_LOG("ENET_TIPG=0x%08x\n", ENET_TIPG);
    NN_SDK_LOG("ENET_FTRL=0x%08x\n", ENET_FTRL);
    NN_SDK_LOG("ENET_TACC=0x%08x\n", ENET_TACC);
    NN_SDK_LOG("ENET_RACC=0x%08x\n", ENET_RACC);
    NN_SDK_LOG("ENET_ATCR=0x%08x\n", ENET_ATCR);
    NN_SDK_LOG("ENET_ATVR=0x%08x\n", ENET_ATVR);
    NN_SDK_LOG("ENET_ATOFF=0x%08x\n", ENET_ATOFF);
    NN_SDK_LOG("ENET_ATPER=0x%08x\n", ENET_ATPER);
    NN_SDK_LOG("ENET_ATCOR=0x%08x\n", ENET_ATCOR);
    NN_SDK_LOG("ENET_ATINC=0x%08x\n", ENET_ATINC);
    NN_SDK_LOG("ENET_ATSTMP=0x%08x\n", ENET_ATSTMP);
    NN_SDK_LOG("ENET_TGSR=0x%08x\n", ENET_TGSR);
    NN_SDK_LOG("ENET_TCSR0=0x%08x\n", ENET_TCSR0);
    NN_SDK_LOG("ENET_TCCR0=0x%08x\n", ENET_TCCR0);
    NN_SDK_LOG("ENET_TCSR1=0x%08x\n", ENET_TCSR1);
    NN_SDK_LOG("ENET_TCCR1=0x%08x\n", ENET_TCCR1);
    NN_SDK_LOG("ENET_TCSR2=0x%08x\n", ENET_TCSR2);
    NN_SDK_LOG("ENET_TCCR2=0x%08x\n", ENET_TCCR2);
    NN_SDK_LOG("ENET_TCSR3=0x%08x\n", ENET_TCSR3);
    NN_SDK_LOG("ENET_TCCR3=0x%08x\n", ENET_TCCR3);
    NN_SDK_LOG("\n");
}

}}}
