﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <new>
#include <unordered_map>

#include <crtdbg.h>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>

#include <nn/nn_Common.h>
#include <nn/diag/detail/diag_DetailTranslation.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_ScopeExit.h>

#include "detail/diag_IgnoreBlockAllocator-compiler.vc.h"

#if !NN_BUILD_CONFIG_TOOLCHAIN_SUPPORTS_VC
    #error  "VC コンパイラが利用出来るビルド条件ではありません。"
#endif

namespace nn { namespace diag { namespace detail {
    struct UnorderedMapHash
    {
        size_t operator()(const wchar_t* wstr) const
        {
            if (wstr == nullptr)
            {
                return static_cast<size_t>(0);
            }

            // 32bit / 64bit FNV-1
            const size_t fnvPrime = (sizeof(size_t) == sizeof(int64_t)) ? 1099511628211 : 16777619;
            const size_t offsetBasis = (sizeof(size_t) == sizeof(int64_t)) ? 14695981039346656037 : 2166136261;
            size_t hash = offsetBasis;

            while(*wstr)
            {
                hash *= fnvPrime;
                hash ^= static_cast<size_t>(*wstr++);
            }

            return hash;
        }
    };

    struct UnorderedMapKeyEqual
    {
        bool operator()(const wchar_t* lhs, const wchar_t* rhs) const
        {
            return wcscmp(lhs, rhs) == 0;
        }
    };

    typedef std::unordered_map<
        const wchar_t*,
        const char*,
        UnorderedMapHash,
        UnorderedMapKeyEqual,
        IgnoreBlockAllocator<const wchar_t*>
    > unordered_map;
    nn::diag::detail::unordered_map* g_pMultiByteSdkText = nullptr;

    nn::os::MutexType g_MultiByteSdkTextLock = NN_OS_MUTEX_INITIALIZER(false);
    const char g_SubstituteCharacter = '?';

    const char* CreateMultiByteString(const wchar_t* pWideString) NN_NOEXCEPT
    {
        const int convertedElementCount = WideCharToMultiByte(CP_UTF8, 0, pWideString, -1, NULL, 0, NULL, NULL);
        char* pMultibyteString = reinterpret_cast<char*>(_malloc_dbg(convertedElementCount, _IGNORE_BLOCK, __FILE__, __LINE__));
        int result = WideCharToMultiByte(CP_UTF8, 0, pWideString, -1, pMultibyteString, convertedElementCount, NULL, NULL);
        NN_ABORT_UNLESS(result != 0, "WideCharToMultiByte() failed.\n");
        return pMultibyteString;
    }

    const char* GetSdkMultiByteStringFromWideString(const wchar_t* pWideString) NN_NOEXCEPT
    {
        nn::os::LockMutex(&g_MultiByteSdkTextLock);
        NN_UTIL_SCOPE_EXIT { nn::os::UnlockMutex(&g_MultiByteSdkTextLock); };

        // unordered_map をグローバル変数にすると、プロセス終了時に _free_dbg が assert 落ちするので、
        // 最初に使うときにオブジェクトを作成する。
        if (g_pMultiByteSdkText == nullptr)
        {
            void* p = _malloc_dbg(sizeof(nn::diag::detail::unordered_map), _IGNORE_BLOCK, __FILE__, __LINE__);
            g_pMultiByteSdkText = new (p) nn::diag::detail::unordered_map();
        }

        // unordered_map を検索
        auto e = g_pMultiByteSdkText->find(pWideString);
        if (e != g_pMultiByteSdkText->end())
        {
            return e->second;
        }

        // 検索にヒットしなかったら、unordered_map に挿入
        const char* pMultibyteString = CreateMultiByteString(pWideString);
        g_pMultiByteSdkText->insert(std::make_pair(pWideString, pMultibyteString));

        return pMultibyteString;
    }

}}}
