﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/diag/detail/diag_DetailLog.h>
#include <nn/diag/detail/diag_FormatStringOutputBufferedDebugPrint.h>
#include <nn/diag/diag_LogObserver.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include "diag_LogImpl.h"
#include <nn/cstd/cstd_CStdArg.h>
#include <cstdio>

namespace nn { namespace diag {

namespace detail {

void CallAllLogObserver(const LogMetaData& logMetaData, const LogBody& logBody) NN_NOEXCEPT;

namespace {

    struct CallPrintDebugString
    {
        void operator()(const LogMetaData& logMetaData, const char* text, size_t length, bool isHead, bool isTail) NN_NOEXCEPT
        {
            LogBody logBody;
            logBody.message = text;
            logBody.messageBytes = length;
            logBody.isHead = isHead;
            logBody.isTail = isTail;
            CallAllLogObserver(logMetaData, logBody);
        }
    };

} // anonymous namespace

void VLogImpl(const LogMetaInfo& logMetaInfo, const char* format, std::va_list formatArg) NN_NOEXCEPT
{
    LogMetaData logMetaData;
    logMetaData.sourceInfo.lineNumber = logMetaInfo.lineNumber;
    logMetaData.sourceInfo.fileName = logMetaInfo.fileName;
    logMetaData.sourceInfo.functionName = "";
    logMetaData.moduleName = "";
    logMetaData.severity = LogSeverity_Info;
    logMetaData.verbosity = 0;
    logMetaData.useDefaultLocaleCharset = false;
    logMetaData.pAdditionalData = NULL;
    logMetaData.additionalDataBytes = 0;

    VLogImpl(logMetaData, format, formatArg);
}

void LogImpl(const LogMetaInfo& logMetaInfo, const char* format, ...) NN_NOEXCEPT
{
    std::va_list list;
    va_start(list, format);
    VLogImpl(logMetaInfo, format, list);
    va_end(list);
}

void VLogImpl(const LogMetaData& logMetaData, const char* format, std::va_list formatArg) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Windows 版はログの連結処理がないため、マルチバイト文字が分割されて文字化けしないように、
    // 書式化に必要な長さのバッファを動的に確保して利用する。
    const int outputBufferLength = nn::util::VSNPrintf(nullptr, 0, format, formatArg) + 1;
    char* outputBuffer = reinterpret_cast<char*>(std::malloc(outputBufferLength));
    NN_ABORT_UNLESS(outputBuffer != nullptr);
    NN_UTIL_SCOPE_EXIT { std::free(outputBuffer); };
    FormatStringOutputBufferedDebugPrint<CallPrintDebugString> out(CallPrintDebugString(), outputBuffer, outputBufferLength, logMetaData);
#else
    char outputBuffer[DebugPrintfBufferLength];
    FormatStringOutputBufferedDebugPrint<CallPrintDebugString> out(CallPrintDebugString(), outputBuffer, DebugPrintfBufferLength, logMetaData);
#endif
    nn::util::VFormatString(out.Output, reinterpret_cast<uintptr_t>(&out), format, formatArg);
    out.Flush(true);
}

void LogImpl(const LogMetaData& logMetaData, const char* format, ...) NN_NOEXCEPT
{
    std::va_list list;
    va_start(list, format);
    VLogImpl(logMetaData, format, list);
    va_end(list);
}

void PutImpl(const LogMetaData& logMetaData, const char* text, size_t length) NN_NOEXCEPT
{
    CallPrintDebugString()(logMetaData, text, length, true, true);
}

} // namespace detail

}}
