﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Windows.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_ScopeExit.h>
#include "diag_DbgHelp.h"

#if !(defined(NN_BUILD_CONFIG_CPU_X86) || defined(NN_BUILD_CONFIG_CPU_X64))
    #error "未サポートのアーキテクチャです。"
#endif

// SIGLO-21432: DbgHelp.h を VS2015 でビルドすると警告が出るので、抑制
#pragma warning(push)
#pragma warning(disable: 4091)
#include <DbgHelp.h>
#pragma warning(pop)

#include "diag_DumpStackTrace.h"

#pragma comment(lib, "DbgHelp.lib")

namespace nn { namespace diag { namespace detail {

namespace {
    ADDRESS64 MakeFlatAddress64(DWORD64 flatAddress)
    {
        ADDRESS64 address64;
        address64.Mode = AddrModeFlat;
        address64.Segment = 0;
        address64.Offset = flatAddress;
        return address64;
    }

    // シンボル名のバッファ長
    const int FunctionNameBufferLength = 128;
    // 表示するスタックトレースの数の最大（無限ループに陥るのを防ぐため）
    const int MaxStackTraceDepth = 64;
}

void TentativeDumpStackTrace() NN_NOEXCEPT
{
    // サイズが大きいので、静的に確保しておきます
    static CONTEXT s_Context = {};
    static STACKFRAME64 s_StackFrame = {};
    struct
    {
        SYMBOL_INFOW symbolInfo;
        // SYMBOL_INFO の最後のメンバーが WCHAR name[1] となっており、それにバッファを追加している
        WCHAR nameBuffer[FunctionNameBufferLength - 1];
    } s_SymbolInfoBuffer = {};

    // DbgHelp が複数スレッドから同時に使用できないため、排他します
    diag::detail::LockDbgHelpMutex();
    NN_UTIL_SCOPE_EXIT {
        diag::detail::UnlockDbgHelpMutex();
    };

    NN_SDK_LOG("----------------Stack Trace----------------\n");

    const HANDLE hCurrentProcess = GetCurrentProcess();

    // 現在のスタックフレームとプログラムカウンタを取得する
    RtlCaptureContext(&s_Context);

#if defined(NN_BUILD_CONFIG_CPU_X86)
    s_StackFrame.AddrPC = MakeFlatAddress64(s_Context.Eip);
    s_StackFrame.AddrFrame = MakeFlatAddress64(s_Context.Ebp);
    s_StackFrame.AddrStack = MakeFlatAddress64(s_Context.Esp);
#else
    s_StackFrame.AddrPC = MakeFlatAddress64(s_Context.Rip);
    s_StackFrame.AddrFrame = MakeFlatAddress64(s_Context.Rbp);
    s_StackFrame.AddrStack = MakeFlatAddress64(s_Context.Rsp);
#endif

    // スタックをさかのぼって情報を表示する
    for (int i = 0; i < MaxStackTraceDepth; i ++)
    {
        if (!StackWalk64(
        #if defined(NN_BUILD_CONFIG_CPU_X86)
            IMAGE_FILE_MACHINE_I386,
        #else
            IMAGE_FILE_MACHINE_AMD64,
        #endif
            hCurrentProcess,
            GetCurrentThread(),
            &s_StackFrame,
            &s_Context,
            NULL,
            &SymFunctionTableAccess64,
            &SymGetModuleBase64,
            NULL))
        {
            break;
        }

        // シンボルを取得
        DWORD64 symbolDisplacement;
        s_SymbolInfoBuffer.symbolInfo.SizeOfStruct = sizeof(s_SymbolInfoBuffer.symbolInfo);
        s_SymbolInfoBuffer.symbolInfo.MaxNameLen = FunctionNameBufferLength;
        if (SymFromAddrW(hCurrentProcess, s_StackFrame.AddrPC.Offset, &symbolDisplacement, &s_SymbolInfoBuffer.symbolInfo))
        {
            NN_SDK_LOG("%ls+%zx",
                s_SymbolInfoBuffer.symbolInfo.Name,
                static_cast<size_t>(symbolDisplacement));
        }
        else
        {
            NN_SDK_LOG("(unknown)");
        }

#if defined(NN_BUILD_CONFIG_CPU_X86)
        NN_SDK_LOG(" (eip:%p esp:%p ebp:%p)\n",
            static_cast<uintptr_t>(s_StackFrame.AddrPC.Offset),
            static_cast<uintptr_t>(s_StackFrame.AddrStack.Offset),
            static_cast<uintptr_t>(s_StackFrame.AddrFrame.Offset));
#else
        NN_SDK_LOG(" (rip:%p rsp:%p rbp:%p)\n",
            static_cast<uintptr_t>(s_StackFrame.AddrPC.Offset),
            static_cast<uintptr_t>(s_StackFrame.AddrStack.Offset),
            static_cast<uintptr_t>(s_StackFrame.AddrFrame.Offset));
#endif
    }
    NN_SDK_LOG("-------------------------------------------\n");
}

}}}
