﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/ddsf/ddsf_IEventHandler.h>
#include <nn/ddsf/ddsf_EventHandlerLooper.h>
#include <nn/ddsf/detail/ddsf_Log.h>

namespace nn { namespace ddsf {

IEventHandler::IEventHandler() NN_NOEXCEPT :
    m_pUserArg(0),
    m_IsInitialized(false),
    m_IsLinked(false)
{
}

IEventHandler::~IEventHandler() NN_NOEXCEPT
{
    if ( m_IsLinked )
    {
        Unlink();
    }
    if ( m_IsInitialized )
    {
        Finalize();
    }
}

//---------------------------------------------------------------------------------------------------------------------

namespace
{
    enum class LooperControlCommand
    {
        None,
        Register,
        Unregister,
        Terminate
    };
}

struct EventHandlerLooper::LooperControlCommandParam
{
    LooperControlCommandParam() NN_NOEXCEPT :
    command(LooperControlCommand::None),
        pTarget(nullptr)
    {}
    LooperControlCommandParam(LooperControlCommand command_, IEventHandler* pTarget_) NN_NOEXCEPT :
        command(command_),
        pTarget(pTarget_)
    {}

    LooperControlCommand command{ LooperControlCommand::None };
    IEventHandler* pTarget{ nullptr };
};

EventHandlerLooper::EventHandlerLooper() NN_NOEXCEPT
{
}

EventHandlerLooper::~EventHandlerLooper() NN_NOEXCEPT
{
    if ( m_IsLooping )
    {
        // LoopAuto しているスレッド中からデストラクタが呼ばれる状況は有り得ない
        NN_SDK_ASSERT(!IsRunningOnEventHandlerContext());
        RequestStop();
    }
    if ( m_IsInitialized )
    {
        Finalize();
    }
}

void EventHandlerLooper::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsInitialized, "Looper is already initialized.\n");
    if( m_IsInitialized )
    {
        return;
    }

    nn::os::InitializeMultiWait(&m_Waiter);
    nn::os::InitializeMultiWaitHolder(&m_LooperControlEventHolder, m_LooperControlEvent.GetBase());
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_LooperControlEventHolder);

    m_IsInitialized = true;
}

void EventHandlerLooper::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsLooping, "Looper is still running.\n");
    NN_SDK_REQUIRES(m_IsInitialized, "Looper is not initialized.\n");
    if ( !m_IsInitialized )
    {
        return;
    }

    nn::os::UnlinkMultiWaitHolder(&m_LooperControlEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_LooperControlEventHolder);
    nn::os::FinalizeMultiWait(&m_Waiter);

    m_IsInitialized = false;
}

void EventHandlerLooper::RegisterHandler(IEventHandler* pHandler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandler);
    NN_SDK_REQUIRES(pHandler->IsInitialized());
    LooperControlCommandParam param(LooperControlCommand::Register, pHandler);
    ConfigureMultiWait(&param);
}

void EventHandlerLooper::UnregisterHandler(IEventHandler* pHandler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHandler);
    NN_SDK_REQUIRES(pHandler->IsInitialized());
    LooperControlCommandParam param(LooperControlCommand::Unregister, pHandler);
    ConfigureMultiWait(&param);
}

void EventHandlerLooper::ConfigureMultiWait(LooperControlCommandParam* pParam) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);
    NN_SDK_ASSERT_NOT_NULL(pParam);
    std::lock_guard<decltype(m_LooperControlLock)> lock(m_LooperControlLock);

    // ルーパスレッドが回っている状態で、ルーパスレッド以外のコンテキストから呼び出された場合は、ルーパスレッドに向かってシグナルを出してそちらで多重待ちオブジェクトの更新をさせる
    // それ以外はこの呼び出しのコンテキスト内で直接更新する
    if ( !m_IsLooping || IsRunningOnEventHandlerContext() )
    {
        ConfigureMultiWaitBody(pParam);
    }
    else
    {
        m_pLooperControlCommandParam = pParam;
        m_LooperControlEvent.Signal();
        m_LooperControlCommandDoneEvent.Wait();
    }
}
void EventHandlerLooper::ConfigureMultiWaitBody(LooperControlCommandParam* pParam) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_LooperControlLock.IsLockedByCurrentThread() || !m_LooperControlLock.TryLock()); // ロック状態で呼ばれている必要がある（ロックしているのは自分自身とは限らない）
    NN_SDK_ASSERT_NOT_NULL(pParam);
    NN_SDK_ASSERT_NOT_NULL(pParam->pTarget);

    switch ( pParam->command )
    {
        case LooperControlCommand::Register:
            pParam->pTarget->Link(&m_Waiter);
            break;
        case LooperControlCommand::Unregister:
            pParam->pTarget->Unlink();
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void EventHandlerLooper::LoopAuto() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsLooping, "Looper is already running.\n");
    m_pLooperThread = nn::os::GetCurrentThread();
    m_IsLooping = true;
    m_IsLoopingCondition.Broadcast();
    NN_UTIL_SCOPE_EXIT
    {
        m_IsLooping = false;
        m_pLooperThread = nullptr;
        m_IsLoopingCondition.Broadcast();
    };

    bool isTerminating = false;
    while (!isTerminating )
    {
        // 登録されたイベントの発生またはスレッド終了シグナル待ち
        nn::os::MultiWaitHolderType* pEventHolder = nn::os::WaitAny(&m_Waiter);

        NN_SDK_ASSERT_NOT_NULL(pEventHolder);
        if ( pEventHolder == &m_LooperControlEventHolder )
        {
            if ( m_LooperControlEvent.TryWait() ) // AutoClear
            {
                NN_SDK_ASSERT_NOT_NULL(m_pLooperControlCommandParam);
                switch ( m_pLooperControlCommandParam->command )
                {
                    case LooperControlCommand::Register:
                    case LooperControlCommand::Unregister:
                        ConfigureMultiWaitBody(m_pLooperControlCommandParam);
                        break;
                    case LooperControlCommand::Terminate:
                        isTerminating = true;
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;
                }
                m_pLooperControlCommandParam = nullptr;
                m_LooperControlCommandDoneEvent.Signal();
            }
        }
        else
        {
            auto& handler = IEventHandler::ToEventHandler(pEventHolder);
            handler.HandleEvent();
        }
    }
}

void EventHandlerLooper::WaitLoopEnter() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_LooperControlLock)> lock(m_LooperControlLock);
    while ( !m_IsLooping )
    {
        m_IsLoopingCondition.Wait(m_LooperControlLock);
    }
}

void EventHandlerLooper::WaitLoopExit() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_LooperControlLock)> lock(m_LooperControlLock);
    while ( m_IsLooping )
    {
        m_IsLoopingCondition.Wait(m_LooperControlLock);
    }
}

void EventHandlerLooper::RequestStop() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsLooping, "Looper is not running.\n");
    NN_SDK_REQUIRES(!IsRunningOnEventHandlerContext(), "RequestStop() cannot be called on the event handler callback context.\n");
    if ( m_IsLooping )
    {
        std::lock_guard<decltype(m_LooperControlLock)> lock(m_LooperControlLock);
        LooperControlCommandParam param(LooperControlCommand::Terminate, nullptr);
        m_pLooperControlCommandParam = &param;
        m_LooperControlEvent.Signal();
        m_LooperControlCommandDoneEvent.Wait();
    }
}

bool EventHandlerLooper::IsRunningOnEventHandlerContext() const NN_NOEXCEPT
{
    return nn::os::GetCurrentThread() == m_pLooperThread;
}

}} // nn::ddsf
