﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcpc_PrivateIncludes.h"
#include "nn/bsdsocket/cfg/cfg_Types.h"
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/bgtc.h>

namespace nn { namespace bsdsocket { namespace dhcpc {

namespace {
    bgtc::Interval CalculateWakeupSchedule(TimeSpan timeRenew, TimeSpan timeRebind)
    {
        return std::max<TimeSpan>(timeRenew, timeRebind - TimeSpan::FromSeconds(60)).GetSeconds();
    }
}

const char *Interface::EventNames[Event_Maximum] =
{
    "LinkStateChange", "LinkUp", "Try",
    "Proceed",  "Error", "ReTransmitTimer", "ReceivedBootP", "ReceivedOffer",
    "ReceivedAck", "ReceivedNack", "ReceivedForceRenew", "RenewTimer",
    "RebindTimer", "ExpireTimer", "Finalize", "UnrecoverableError"
};

const char *Interface::StateNames[State_Maximum] =
{
    "Null", "WaitBeforeInit", "Init", "Selecting", "Request", "Bound",
    "WaitBeforeInitReboot", "InitReboot", "Rebooting",
    "Renew", "Rebind","Inform", "Probe", "Failure"
};

const nn::socket::InAddr Interface::m_InAddrAny = { nn::socket::InAddr_Any };
const nn::socket::InAddr Interface::m_InAddrBroadcast = { nn::socket::InAddr_Broadcast };

Interface::Interface()
    : Fsm()
    , m_pMain(nullptr)
    , m_Xid(0)
    , m_SendPacketSize(0)
    , m_ShutdownOptions(nn::bsdsocket::cfg::ShutdownOption_None)
    , m_UpTimeSeconds(0)
    , m_RetryCount(0)
    , m_PendingClientIndication(Indication_InvalidLease)
    , m_pDeferredFinalize(nullptr)
    , m_Mt(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()))
{
    memset(&m_IfConfig, 0, sizeof(m_IfConfig));
    memset(&m_Packet, 0, sizeof(m_Packet));
    memset(&m_ToABuffer, 0, sizeof(m_ToABuffer));
    memset(&m_ClientId, 0, sizeof(m_ClientId));
    memset(&m_ReceivePacket, 0, sizeof(m_ReceivePacket));
    memset(&m_OptionBuffer, 0, sizeof(m_OptionBuffer));
    memset(&m_SendPacket, 0, sizeof(m_SendPacket));
    memset(&m_NextOffer, 0, sizeof(m_NextOffer));
    memset(&m_NextLease, 0, sizeof(m_NextLease));
    memset(&m_Offer, 0, sizeof(m_Offer));
    memset(&m_Lease, 0, sizeof(m_Lease));
}

Result Interface::Initialize(DhcpcMain *pMain,
                             const InterfaceConfigType *pIfConfig)
{

    Result result;

    char fsmName[Fsm::MAX_FSM_NAME_SIZE];
    m_pMain = pMain;
    m_IfConfig = *pIfConfig;
    snprintf(fsmName, sizeof(fsmName), "Interface-%s", m_IfConfig.ifName);
    m_PendingClientIndication = Indication_InvalidLease;
    m_FailureCause = ResultSuccess();
    InitStaticTimedFsmEvent(&m_TryTimerEvent, Event_Try);
    InitStaticTimedFsmEvent(&m_ReTransTimerEvent, Event_ReTransmitTimer);
    InitStaticTimedFsmEvent(&m_RenewTimerEvent, Event_T1RenewTimer);
    InitStaticTimedFsmEvent(&m_RebindTimerEvent, Event_T2RebindTimer);
    InitStaticTimedFsmEvent(&m_ExpireTimerEvent, Event_ExpireTimer);
    do
    {
        if(m_IfConfig.requestedAddr != 0)
        {
            DHCPC_BREAK_UPON_ERROR(Fsm::Initialize(pMain, fsmName,
                                               EventNames, Event_Maximum,
                                               StateNames, State_Maximum,
                                               State_WaitBeforeInitReboot));
        }
        else
        {
            DHCPC_BREAK_UPON_ERROR(Fsm::Initialize(pMain, fsmName,
                                               EventNames, Event_Maximum,
                                               StateNames, State_Maximum,
                                               State_WaitBeforeInit));
        }
        Fsm::SetFsmTraceEnable(true);
    }while (false);

    bgtc::Initialize();

    return result;
}

Result Interface::Stop(DeferredRequest *pDr)
{
    Result result = ResultSuccess();
    // Okay to finalize now?
    if (m_pDeferredFinalize == NULL)
    {
        // finalize now in progress
        m_pDeferredFinalize = pDr;
        // set shutdown options
        m_ShutdownOptions = pDr->m_Params.stopIf.options;
        // queue finalize event and trigger scheduling
        QueueFsmEvent(Event_Finalize);

        result = ResultDeferred();
    }
    else
    {
        result = ResultBusy();
    }

    return result;
}

Result Interface::Finalize()
{
    Result result;

    bgtc::Finalize();

    // FSM is done
    result = Fsm::Finalize();

    return result;
}

Result Interface::GetOptionData(DhcpProtOption option, void *pRetData,
                                size_t bufferSize, size_t *pRetSize)
{
    Result result = ResultSuccess();

    do
    {
        const uint8_t *pOpt;
        size_t optSize = 0;
        size_t limitedSize;

        if ( !DHCPC_HAS_PROT_OPTION_MASK(m_IfConfig.requestMask, option))
        {
            result = ResultProtOptionNotRequested();
            break;
        }
        if ( !m_Lease.isValid ||  !m_Offer.isValid)
        {
            result = ResultLeaseInvalid();
            break;
        }
        pOpt = Util::GetOption(m_OptionBuffer, &m_Offer.message, option, &optSize);
        if(!pOpt || !optSize)
        {
            result = ResultOptionMissing();
            break;
        }
        limitedSize = (optSize > bufferSize) ? bufferSize : optSize;
        memcpy(pRetData, pOpt, limitedSize);
        *pRetSize = limitedSize;
    }while (false);

    return result;
}

Result Interface::GetState(char *pReturnedStateName, size_t stateNameSizeLimit)
{
    nn::util::Strlcpy(pReturnedStateName, StateNames[Fsm::GetState()], stateNameSizeLimit);
    return ResultSuccess();
}

Result Interface::ForceRenew()
{
    Result result = ResultSuccess();
    do
    {
        if ( !m_Lease.isValid ||  !m_Offer.isValid)
        {
            result = ResultLeaseInvalid();
            break;
        }

        // queue finalize event and trigger scheduling
        QueueFsmEvent(Event_ReceivedForceRenew);

    }while (false);

    return result;
}

void Interface::GetHardwareAddress(uint8_t *pRetAddress)
{
    memcpy(pRetAddress, m_IfConfig.hardwareAddress, sizeof(m_IfConfig.hardwareAddress));
}

void Interface::GetInterfaceName(char *name, size_t maxSize)
{
    nn::util::Strlcpy(name, m_IfConfig.ifName, maxSize);
}

void Interface::HandleFsmError(Result result)
{
    if ( !result.IsSuccess())
    {
        QueueFsmEvent(Event_Error);
    }
}

Result Interface::FsmHandler(int32_t state, LocalEventDataType *pEvent)
{
    Result result = ResultInvalidInternalState();
    switch (state)
    {
    case State_Null:
        result = NullStateHandler(pEvent);
        break;
    case State_WaitBeforeInit:
        result = WaitBeforeInitStateHandler(pEvent);
        break;
    case State_Init:
        result = InitStateHandler(pEvent);
        break;
    case State_Selecting:
        result = SelectingStateHandler(pEvent);
        break;
    case State_Request:
        result = RequestStateHandler(pEvent);
        break;
    case State_Bound:
        result = BoundStateHandler(pEvent);
        break;
    case State_WaitBeforeInitReboot:
        result = WaitBeforeInitRebootStateHandler(pEvent);
        break;
    case State_InitReboot:
        result = InitRebootStateHandler(pEvent);
        break;
    case State_Rebooting:
        result = RebootingStateHandler(pEvent);
        break;
    case State_Renew:
        result = RenewStateHandler(pEvent);
        break;
    case State_Rebind:
        result = RebindStateHandler(pEvent);
        break;
    case State_Inform:
        result = InformStateHandler(pEvent);
        break;
    case State_Probe:
        result = ProbeStateHandler(pEvent);
        break;
    case State_Failure:
        result = FailureStateHandler(pEvent);
        break;
    default:
        break;
    }
    return result;
}

Result Interface::NullStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            // Cannot be valid at this point
            DeclareLeaseInvalid();

            HandleDeInit();

            // Release caller
            if (m_pDeferredFinalize != NULL)
            {
                m_pDeferredFinalize->CompleteRequest(ResultSuccess());
                m_pDeferredFinalize = NULL;
            }
            break;
        }
    case Event_Try:
    case Event_UnrecoverableError:
        break;
    case Fsm::Event_Exit:
        break;
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::WaitBeforeInitStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
    {
        // Cannot be valid at this point
        DeclareLeaseInvalid();
        // Clean up any prior operating state
        HandleDeInit();
        // Reset NAK backoff interval
        UpdateNakBackoffInterval(true);
        // Start wait timer
        m_InitWaitTime = TimeSpan::FromMicroSeconds(std::uniform_int_distribution<uint32_t>(StartupWaitTime_RandomMin * 1000,StartupWaitTime_RandomMax * 1000)(m_Mt));
        StartTimedFsmEvent(&m_TryTimerEvent, m_InitWaitTime);
        break;
    }
    case Event_Try:
    {
        SetState(State_Init);
        break;
    }
    case Event_Finalize:
    {
        HandleFinalizeEvent();
        break;
    }
    case Event_UnrecoverableError:
    {
        HandleUnrecoverableErrorEvent(pEvent);
        break;
    }
    case Fsm::Event_Exit:
    {
        break;
    }
    default:
    {
        result = ResultUnexpectedEvent();
        break;
    }
    }
    return result;
}

Result Interface::InitStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            // Cannot be valid at this point
            DeclareLeaseInvalid();
            // Clean up any prior operating state
            HandleDeInit();
            // Reset NAK backoff interval
            UpdateNakBackoffInterval(true);
            // Queue event to proceed
            QueueFsmEvent(Event_Try);
            break;
        }
    case Event_Try:
        {
            do
            {
                DHCPC_BREAK_UPON_ERROR(m_Packet.Initialize(this));
                DHCPC_BREAK_UPON_ERROR(m_pMain->m_FdMonitor.Register(m_Packet.GetRawFd(),
                                                                     FdMonRawPacketCallbackStatic, this));
                DHCPC_BREAK_UPON_ERROR(m_pMain->m_FdMonitor.SetPollPeriod(m_Packet.GetRawFd(),
                                                                          FdMonitor::FastPollPeriod));
            }while (false);
            if (result.IsSuccess())
            {
                SetState(State_Selecting);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Interface::SelectingStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            size_t dhcpMessageSize = 0;

            // Cannot be valid at this point
            DeclareLeaseInvalid();

            // Prepare request but don't send yet
            InitXid();
            memset(&m_SendPacket, 0, sizeof(m_SendPacket));
            DHCPC_BREAK_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Discover, &m_InAddrAny, &dhcpMessageSize));
            m_SendPacketSize = PacketManager::MakeUdpPacket(&m_SendPacket, dhcpMessageSize, &m_InAddrAny, &m_InAddrBroadcast);
            // Retransmit backoff is reset
            UpdateNakBackoffInterval(true);
            UpdateReTransBackoffInterval(true);
            // Respecting NAK backoff, schedule the first try
            StartTimedFsmEvent(&m_TryTimerEvent, m_NakBackoffInterval);
            DHCPC_BREAK_UPON_ERROR(m_pMain->m_FdMonitor.SetPollPeriod(m_Packet.GetRawFd(),
                                                                      FdMonitor::FastPollPeriod));
            break;
        }
    case Event_ReTransmitTimer:
        {

            if(IsEventStale(pEvent)) break;
            DHCPC_LOG_INFO("Re-sending discover after backoff of %d ms.\n", m_ReTransBackoffInterval.GetMilliSeconds());
            UpdateReTransBackoffInterval(false);
            // fall through
        }
    case Event_Try:
        {
            if(IsEventStale(pEvent)) break;

            if ((result = m_Packet.SendRawPacket(nn::socket::EtherType::EtherType_Ip, (const void *)&m_SendPacket, m_SendPacketSize)).IsSuccess())
            {
                StartTimedFsmEvent(&m_ReTransTimerEvent, m_ReTransBackoffInterval);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedOffer:
        {
            DHCPC_BREAK_UPON_ERROR(TakeReceiptOfNext(pEvent));
            SetState(State_Request);
            break;
        }
    case Event_ReceivedNack:
        {
            UpdateNakBackoffInterval(false);
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedBootP:
        {
            // Check if we will allow it

            // BOOTP does not need request, skip ahead to bound

            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_TryTimerEvent);
            StopTimedFsmEvent(&m_ReTransTimerEvent);
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::RequestStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            size_t dhcpMessageSize = 0;
            m_PendingClientIndication = Indication_NewLease;
            UpdateReTransBackoffInterval(true);
            // Clear retry count
            m_RetryCount = 0;

            memset(&m_SendPacket, 0, sizeof(m_SendPacket));
            DHCPC_BREAK_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Request, &m_InAddrAny, &dhcpMessageSize));
            m_SendPacketSize = PacketManager::MakeUdpPacket(&m_SendPacket, dhcpMessageSize, &m_InAddrAny, &m_InAddrBroadcast);
            QueueFsmEvent(Event_Try);
            break;
        }
    case Event_ReTransmitTimer:
        {
            static const uint8_t MaxRetryCount = 3;

            if(IsEventStale(pEvent)) break;

            if(++m_RetryCount > MaxRetryCount)
            {
                LogContextual(Util::LogLevel_Info,"Returning to selecting after %d failed requests.\n", m_RetryCount - 1);
                SetState(State_Selecting);
                break;
            }

            DHCPC_LOG_INFO("Re-sending request after backoff of %d ms.\n", m_ReTransBackoffInterval.GetMilliSeconds());
            UpdateReTransBackoffInterval(false);
            // fall through
        }
    case Event_Try:
        {
            if(IsEventStale(pEvent)) break;
            if ((result = m_Packet.SendRawPacket(nn::socket::EtherType::EtherType_Ip, (const void *)&m_SendPacket, m_SendPacketSize)).IsSuccess())
            {
                StartTimedFsmEvent(&m_ReTransTimerEvent, m_ReTransBackoffInterval);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedNack:
        {
            UpdateNakBackoffInterval(false);
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedAck:
        {
            if ((result = TakeReceiptOfNext(pEvent)).IsSuccess())
            {
                UpdateNakBackoffInterval(true);
                SetState(State_Bound);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedOffer:
        break;
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_ReTransTimerEvent);
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::BoundStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_Lease = m_NextLease;
            Util::ApplyLeaseLimits(&m_Lease);
            m_Lease.isValid = true;

            // No timers valid at this point yet
            StopTimedFsmEvent(&m_RenewTimerEvent);
            StopTimedFsmEvent(&m_RebindTimerEvent);
            StopTimedFsmEvent(&m_ExpireTimerEvent);
            // start lease timers from the point at which the request was sent
            TimeSpan currentUpTime;
            Util::GetUpTime(&currentUpTime);
            TimeSpan timeDiff = m_RequestSentTime - currentUpTime;

            // Start timers only if lease time is not infinite
            if (m_Lease.leaseTime != TimeSpan::FromSeconds(Config_InfiniteTime))
            {
                StartTimedFsmEvent(&m_RenewTimerEvent, m_Lease.renewalTime + timeDiff);
                StartTimedFsmEvent(&m_RebindTimerEvent, m_Lease.rebindTime + timeDiff);
                StartTimedFsmEvent(&m_ExpireTimerEvent, m_Lease.leaseTime + timeDiff);

                // schedule entering to "half-awake" for lease time extension
                bgtc::ScheduleTaskUnsafe(
                    CalculateWakeupSchedule(m_Lease.renewalTime + timeDiff, m_Lease.rebindTime + timeDiff)
                );
            }
            m_LeaseAquiredTime = m_RequestSentTime;
            DHCPC_BREAK_UPON_ERROR(m_pMain->m_FdMonitor.SetPollPeriod(m_Packet.GetRawFd(),
                                                                      FdMonitor::SlowPollPeriod));
            // Notify
            DoClientCallback(m_PendingClientIndication);
            break;
        }
    case Event_T1RenewTimer:
        {
            SetState(State_Renew);
            break;
        }
    case Event_T2RebindTimer:
        {
            SetState(State_Rebind);
            break;
        }
    case Event_ExpireTimer:
        {
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedAck:
        {
            // Expected if latent server responds multiple times after client retried
            if ((result = TakeReceiptOfNext(pEvent)).IsSuccess())
            {
                if (memcmp(&m_NextLease.addr, &m_Lease.addr, sizeof(m_Lease.addr)) != 0)
                {
                    result = ResultInvalidReissuedLease();
                    QueueFsmEvent(Event_UnrecoverableError, result);
                    break;
                }
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedForceRenew:
        {
            TimeSpan newRenew = TimeSpan::FromSeconds(0);
            TimeSpan newRebind = m_Lease.rebindTime - m_Lease.renewalTime;
            TimeSpan newlease = m_Lease.leaseTime - m_Lease.renewalTime;
            ReStartTimedFsmEvent(&m_RenewTimerEvent, newRenew);
            ReStartTimedFsmEvent(&m_RebindTimerEvent, newRebind);
            ReStartTimedFsmEvent(&m_ExpireTimerEvent, newlease);

            // re-schedule
            bgtc::ScheduleTaskUnsafe(
                CalculateWakeupSchedule(newRenew, newRebind)
            );
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            // cancel schedule to save battery
            bgtc::UnscheduleTask();
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::WaitBeforeInitRebootStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
    {
        // Cannot be valid at this point
        DeclareLeaseInvalid();
        // Clean up any prior operating state
        HandleDeInit();
        // Reset NAK backoff interval
        UpdateNakBackoffInterval(true);
        // Start wait timer
        m_InitWaitTime = TimeSpan::FromMicroSeconds(std::uniform_int_distribution<uint32_t>(StartupWaitTime_RandomMin * 1000000, StartupWaitTime_RandomMax * 1000000)(m_Mt));
        StartTimedFsmEvent(&m_TryTimerEvent, m_InitWaitTime);
        break;
    }
    case Event_Try:
    {
        SetState(State_InitReboot);
        break;
    }
    case Event_Finalize:
    {
        HandleFinalizeEvent();
        break;
    }
    case Event_UnrecoverableError:
    {
        HandleUnrecoverableErrorEvent(pEvent);
        break;
    }
    case Fsm::Event_Exit:
    {
        break;
    }
    default:
    {
        result = ResultUnexpectedEvent();
        break;
    }
    }
    return result;
}

Result Interface::InitRebootStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            // Cannot be valid at this point
            DeclareLeaseInvalid();

            // Clean up any prior operating state
            HandleDeInit();

            // Reset NAK backoff interval
            UpdateNakBackoffInterval(true);

            // Queue event to proceed
            QueueFsmEvent(Event_Try);
            break;
        }
    case Event_Try:
        {
            do
            {
                DHCPC_BREAK_UPON_ERROR(m_Packet.Initialize(this));
                DHCPC_BREAK_UPON_ERROR(m_pMain->m_FdMonitor.Register(m_Packet.GetRawFd(),
                                                                     FdMonRawPacketCallbackStatic, this));
            }while (false);

            if (result.IsSuccess())
            {
                SetState(State_Rebooting);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        {
            result = ResultUnexpectedEvent();
            break;
        }
    }
    return result;
}

Result Interface::RebootingStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {

            size_t dhcpMessageSize = 0;

            // Cannot be valid at this point
            DeclareLeaseInvalid();

            InitXid();

            GenerateRebootLease();

            // Clear retry count
            m_RetryCount = 0;

            memset(&m_SendPacket, 0, sizeof(m_SendPacket));
            m_PendingClientIndication = Indication_NewLease;
            UpdateReTransBackoffInterval(true);

            DHCPC_BREAK_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Request, &m_InAddrAny, &dhcpMessageSize));
            m_SendPacketSize = PacketManager::MakeUdpPacket(&m_SendPacket, dhcpMessageSize, &m_InAddrAny, &m_NextLease.serverAddr);
            QueueFsmEvent(Event_Try);
            break;
        }
    case Event_ReTransmitTimer:
        {
            static const uint8_t MaxRetryCount = 3;

            if(IsEventStale(pEvent)) break;

            if(++m_RetryCount > MaxRetryCount)
            {
                LogContextual(Util::LogLevel_Info,"Switching to selecting after %d failed requests.\n", m_RetryCount - 1);
                SetState(State_Selecting);
                break;
            }

            DHCPC_LOG_INFO("Re-sending request after backoff of %d ms.\n", m_ReTransBackoffInterval.GetMilliSeconds());
            UpdateReTransBackoffInterval(false);
            // fall through
        }
    case Event_Try:
        {
            if(IsEventStale(pEvent)) break;
            if ((result = m_Packet.SendRawPacket(nn::socket::EtherType::EtherType_Ip, (const void *)&m_SendPacket, m_SendPacketSize)).IsSuccess())
            {
                StartTimedFsmEvent(&m_ReTransTimerEvent, m_ReTransBackoffInterval);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedNack:
        {
            LogContextual(Util::LogLevel_Info,"Server refused reboot request for IP %s, switching state to selecting.\n",nn::socket::InetNtoa(m_NextLease.addr));
            UpdateNakBackoffInterval(false);
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedAck:
        {

            if ((result = TakeReceiptOfNext(pEvent)).IsSuccess())
            {
                UpdateNakBackoffInterval(true);
                SetState(State_Bound);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedOffer:
        break;
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_ReTransTimerEvent);
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::RenewStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_PendingClientIndication = Indication_RenewedLease;
            UpdateRenewReTransInterval();
            memset(&m_SendPacket, 0, sizeof(m_SendPacket));
            DHCPC_BREAK_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Request, &m_Lease.addr, &m_SendPacketSize));
            QueueFsmEvent(Event_Try);

            // FIXME: workaround for not discarding packets of DHCP Request by WLAN when half-awake is triggerd by lease time extension task.
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

            // Prevent re-entering to sleep during the transaction
            bgtc::NotifyTaskStarting();
            break;
        }
    case Event_ReTransmitTimer:
        {
            if(IsEventStale(pEvent)) break;

            // giving up immediately when system state is not full-awake
            if (!bgtc::IsInFullAwake())
            {
                DHCPC_LOG_INFO("Giving up renewal since system state is not full-awake.\n");
                SetState(State_Selecting);
                break;
            }

            DHCPC_LOG_INFO("Re-sending request after backoff of %d ms.\n", m_ReTransBackoffInterval.GetMilliSeconds());
            UpdateRenewReTransInterval();
            // fall through
        }
    case Event_Try:
        {
            if(IsEventStale(pEvent)) break;
            if ((result = m_Packet.SendUdpPacket(&m_Lease.serverAddr,&m_Lease.addr , &m_SendPacket.dhcp, m_SendPacketSize)).IsSuccess())
            {
                StartTimedFsmEvent(&m_ReTransTimerEvent, m_ReTransBackoffInterval);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedNack:
        {
            UpdateNakBackoffInterval(false);
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedAck:
        {
            if ((result = TakeReceiptOfNext(pEvent)).IsSuccess())
            {
                if (memcmp(&m_NextLease.addr, &m_Lease.addr, sizeof(m_Lease.addr)) != 0)
                {
                    result = ResultInvalidReissuedLease();
                    QueueFsmEvent(Event_UnrecoverableError, result);
                    break;
                }
                UpdateNakBackoffInterval(true);
                SetState(State_Bound);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_T2RebindTimer:
        {
            SetState(State_Rebind);
            break;
        }
    case Event_ExpireTimer:
        {
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedForceRenew:
        {
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_ReTransTimerEvent);

            // Allow entering to sleep. It must be called when the transaction is finished.
            bgtc::NotifyTaskFinished();
            // There is 3 seconds margin to prevent entering sleep immediately.
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::RebindStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            m_PendingClientIndication = Indication_ReboundLease;
            UpdateRebindReTransInterval();
            memset(&m_SendPacket, 0, sizeof(m_SendPacket));

            size_t dhcpMessageSize = 0;
            DHCPC_BREAK_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Request, &m_Lease.addr, &dhcpMessageSize));
            m_SendPacketSize = PacketManager::MakeUdpPacket(&m_SendPacket, dhcpMessageSize, &m_Lease.addr, &m_InAddrBroadcast);
            QueueFsmEvent(Event_Try);

            // Prevent re-entering to sleep during the transaction
            bgtc::NotifyTaskStarting();
            break;
        }
    case Event_ReTransmitTimer:
        {
            if(IsEventStale(pEvent)) break;

            // giving up immediately when system state is not full-awake
            if (!bgtc::IsInFullAwake())
            {
                DHCPC_LOG_INFO("Giving up renewal since system state is not full-awake.\n");
                SetState(State_Selecting);
                break;
            }

            DHCPC_LOG_INFO("Re-sending request after backoff of %d ms.\n", m_ReTransBackoffInterval.GetMilliSeconds());
            UpdateRebindReTransInterval();
            // fall through
        }
    case Event_Try:
        {
            if(IsEventStale(pEvent)) break;
            if ((result = m_Packet.SendRawPacket(nn::socket::EtherType::EtherType_Ip, (const void *)&m_SendPacket, m_SendPacketSize)).IsSuccess())
            {
                StartTimedFsmEvent(&m_ReTransTimerEvent, m_ReTransBackoffInterval);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ReceivedNack:
        {
            UpdateNakBackoffInterval(false);
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedAck:
        {
            if ((result = TakeReceiptOfNext(pEvent)).IsSuccess())
            {
                if (memcmp(&m_NextLease.addr, &m_Lease.addr, sizeof(m_Lease.addr)) != 0)
                {
                    result = ResultInvalidReissuedLease();
                    QueueFsmEvent(Event_UnrecoverableError, result);
                    break;
                }
                UpdateNakBackoffInterval(true);
                SetState(State_Bound);
            }
            else
            {
                QueueFsmEvent(Event_UnrecoverableError, result);
            }
            break;
        }
    case Event_ExpireTimer:
        {
            SetState(State_Selecting);
            break;
        }
    case Event_ReceivedForceRenew:
        {
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            StopTimedFsmEvent(&m_ReTransTimerEvent);

            // Allow entering to sleep. It must be called when the transaction is finished.
            bgtc::NotifyTaskFinished();
            // There is 3 seconds margin to prevent entering sleep immediately.
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::InformStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::ProbeStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            break;
        }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            HandleUnrecoverableErrorEvent(pEvent);
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

Result Interface::FailureStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        {
            DoClientCallback(Indication_Failure);
            break;
        }
    case Event_ReTransmitTimer:
    case Event_T1RenewTimer:
    case Event_T2RebindTimer:
    case Event_ExpireTimer:
    {
        //don't want to throw more errors if these timers expire after we've already failed.
        break;
    }
    case Event_Finalize:
        {
            HandleFinalizeEvent();
            break;
        }
    case Event_UnrecoverableError:
        {
            break;
        }
    case Fsm::Event_Exit:
        {
            break;
        }
    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}

void Interface::DispatchReceivedDhcpMessage(DhcpMessage *dhcp, nn::socket::InAddr *pFrom)
{
    int32_t eventId = -1;
    uint8_t msgType = 0xff;

    // We may have found a BOOTP server
    if (Util::GetOptionUint8(m_OptionBuffer, &msgType, dhcp,
                             (uint8_t)DhcpProtOption_MessageType) == -1)
    {
        msgType = 0;
    }

    switch (msgType)
    {
    case 0:
        eventId = Event_ReceivedBootP;
        break;
    case DhcpProtMsgType_ForceRenew:
        eventId = Event_ReceivedForceRenew;
        break;
    case DhcpProtMsgType_Offer:
        eventId = Event_ReceivedOffer;
        break;
    case DhcpProtMsgType_Ack:
        eventId = Event_ReceivedAck;
        break;
    case DhcpProtMsgType_Nak:
        eventId = Event_ReceivedNack;
        break;
    default:
        break;
    }

    if (eventId >= 0)
    {
        DispatchFsmEvent(eventId, dhcp, pFrom);
    }
    else
    {
        DHCPC_LOG_INFO("Discarding received message.\n");
    }
}

void Interface::HandleDeInit()
{
    // Stop all timers
    StopTimedFsmEvent(&m_RenewTimerEvent);
    StopTimedFsmEvent(&m_RebindTimerEvent);
    StopTimedFsmEvent(&m_ExpireTimerEvent);

    m_pMain->m_FdMonitor.SetPollPeriod(m_Packet.GetRawFd(),
                                       FdMonitor::SlowPollPeriod);

    // Stop monitoring for received packets
    m_pMain->m_FdMonitor.UnRegister(m_Packet.GetRawFd());

    // Finalize this instance's packet manager
    m_Packet.Finalize();
}

void Interface::InitXid()
{
    if (m_IfConfig.daemonOptions & ClientDaemonOption_XidHwaddr)
    {
        memcpy(&m_Xid, m_IfConfig.hardwareAddress + sizeof(m_IfConfig.hardwareAddress) -
               sizeof(m_Xid), sizeof(m_Xid));
    }
    else
    {
        m_Xid = std::uniform_int_distribution<uint32_t>(0, UINT32_MAX)(m_Mt);
    }
}

void Interface::GenerateRebootLease()
{
    memset(&m_NextLease,0,sizeof(Lease));
    m_NextLease.addr.S_addr = m_IfConfig.requestedAddr;
    m_NextLease.serverAddr.S_addr = m_IfConfig.serverAddr;
    m_NextLease.cookie = nn::socket::InetHtonl(DhcpMessageFlags_MagicCookie);
}

Result Interface::MakeMessage(DhcpMessage *dhcp, DhcpProtMsgType msgType,
                              const nn::socket::InAddr *ciaddr, size_t *pReturnedSize)
{
    Result result = ResultSuccess();
    uint8_t *lp;
    size_t len;
    size_t size = 0;
    uint8_t *m = (uint8_t *)dhcp;
    uint8_t *p = dhcp->options;

    *pReturnedSize = size;
    memset(dhcp, 0, sizeof(*dhcp));
    dhcp->ciaddr = ciaddr->S_addr;
    dhcp->op     = DhcpProtOpCode_BootRequest;
    dhcp->hwtype = static_cast<uint8_t>(nn::socket::ArpHardware::ArpHrd_Ether);
    dhcp->hwlen  = Config_MaxHardwareAddressSize;
    memcpy(&dhcp->chaddr, m_IfConfig.hardwareAddress, sizeof(m_IfConfig.hardwareAddress));

    if ((m_IfConfig.daemonOptions & ClientDaemonOption_Broadcast)
        && (dhcp->ciaddr == 0) && (msgType != DhcpProtMsgType_Decline) && (msgType != DhcpProtMsgType_Release))
    {
        dhcp->flags = nn::socket::InetHtons(DhcpProtMessageFlags_Broadcast);
    }

    if (msgType != DhcpProtMsgType_Decline && msgType != DhcpProtMsgType_Release)
    {
        if(msgType == DhcpProtMsgType_Request)
        {
            dhcp->secs = nn::socket::InetHtons(m_UpTimeSeconds);
            Util::GetUpTime(&m_RequestSentTime);
        }
        else
        {
            Util::GetSaturatedUpTimeSeconds(&m_UpTimeSeconds);
            dhcp->secs = nn::socket::InetHtons(m_UpTimeSeconds);
        }
    }

    dhcp->xid = nn::socket::InetHtonl(m_Xid);
    dhcp->cookie = nn::socket::InetHtonl(DhcpMessageFlags_MagicCookie);

    *p++ = DhcpProtOption_MessageType;
    *p++ = 1;
    *p++ = msgType;

    if (strnlen(m_ClientId, sizeof(m_ClientId)) != 0)
    {
        size_t saneSize =  DHCPC_MAX(static_cast<size_t>(m_ClientId[0]) + 1, sizeof(m_ClientId));
        *p++ = DhcpProtOption_ClientId;
        memcpy(p, m_ClientId, saneSize);
        p += saneSize;
    }

    if (m_NextLease.addr.S_addr && (m_NextLease.cookie == nn::socket::InetHtonl(DhcpMessageFlags_MagicCookie)))
    {
        if (msgType == DhcpProtMsgType_Decline ||
            (msgType == DhcpProtMsgType_Request &&
             m_NextLease.addr.S_addr != m_Lease.addr.S_addr))
        {
            DHCPC_PUTADDR(DhcpProtOption_IpAddress, m_NextLease.addr.S_addr);
            if (m_NextLease.serverAddr.S_addr) DHCPC_PUTADDR(DhcpProtOption_ServerId,
                                                             m_NextLease.serverAddr.S_addr);
        }

        if (msgType == DhcpProtMsgType_Release)
        {
            if (m_NextLease.serverAddr.S_addr) DHCPC_PUTADDR(DhcpProtOption_ServerId,
                                                             m_NextLease.serverAddr.S_addr);
        }
    }

    if (msgType == DhcpProtMsgType_Decline)
    {
        const char *pDad = "Duplicate address detected";
        *p++ = DhcpProtOption_Message;
        len = strlen(pDad);
        *p++ = (uint8_t)len;
        memcpy(p, pDad, len);
        p += len;
    }

    if (msgType == DhcpProtMsgType_Discover &&
        !(m_IfConfig.daemonOptions & ClientDaemonOption_Test) &&
        DHCPC_HAS_PROT_OPTION_MASK(m_IfConfig.requestMask, DhcpProtOption_RapidCommit))
    {
        /* RFC 4039 Section 3 */
        *p++ = DhcpProtOption_RapidCommit;
        *p++ = 0;
    }

    if (msgType == DhcpProtMsgType_Discover && m_IfConfig.daemonOptions & ClientDaemonOption_Request)
    {
        DHCPC_PUTADDR(DhcpProtOption_IpAddress, m_IfConfig.requestedAddr);
    }

    /* RFC 2563 Auto Configure */
    if (msgType == DhcpProtMsgType_Discover && m_IfConfig.daemonOptions & ClientDaemonOption_Ipv4ll)
    {
        *p++ = DhcpProtOption_AutoConfigure;
        *p++ = 1;
        *p++ = 1;
    }

    if (msgType == DhcpProtMsgType_Discover || msgType == DhcpProtMsgType_Inform || msgType == DhcpProtMsgType_Request)
    {
        uint16_t mtu, sz;
        bool isHostNameValid = (strnlen(m_IfConfig.hostName, sizeof(m_IfConfig.hostName)) != 0) ? true : false;
        *p++ = DhcpProtOption_MaxMessageSize;
        *p++ = 2;
        mtu = static_cast<uint16_t>(DoMtu(0));
        if (mtu < Config_MinMtu)
        {
            if (DoMtu(Config_MinMtu) == 0) sz = Config_MinMtu;
        }
        else if (mtu > Config_MaxMtu)
        {
            mtu = Config_MaxMtu;
        }
        sz = nn::socket::InetHtons(mtu);
        memcpy(p, &sz, 2);
        p += 2;
        if (m_IfConfig.userClass[0])
        {
            *p++ = DhcpProtOption_UserClass;
            memcpy(p, m_IfConfig.userClass, m_IfConfig.userClass[0] + 1);
            p += m_IfConfig.userClass[0] + 1;
        }

        if (m_IfConfig.vendorClassId[0])
        {
            *p++ = DhcpProtOption_VendorClassId;
            memcpy(p, m_IfConfig.vendorClassId, m_IfConfig.vendorClassId[0] + 1);
            p += m_IfConfig.vendorClassId[0] + 1;
        }

        if (msgType != DhcpProtMsgType_Inform)
        {
            if (m_IfConfig.leaseTime != 0)
            {
                uint32_t ul;
                *p++ = DhcpProtOption_LeaseTime;
                *p++ = 4;
                ul = nn::socket::InetHtonl(m_IfConfig.leaseTime);
                memcpy(p, &ul, 4);
                p += 4;
            }
        }

        /*
         * RFC4702 3.1 States that if we send the Client FQDN option
         * then we MUST NOT also send the Host Name option.
         * Technically we could, but that is not RFC conformant and
         * also seems to break some DHCP server implemetations such as
         * Windows. On the other hand, ISC dhcpd is just as non RFC
         * conformant by not accepting a partially qualified FQDN.
         */
        if (m_IfConfig.fqdn != DhcpProtFqdn_Disable)
        {
            /* IETF DHC-FQDN option (81), RFC4702 */
            *p++ = DhcpProtOption_Fqdn;
            lp = p;
            *p++ = 3;
            /*
             * Flags: 0000NEOS
             * S: 1 => Client requests Server to update
             *         a RR in DNS as well as PTR
             * O: 1 => Server indicates to client that
             *         DNS has been updated
             * E: 1 => Name data is DNS format
             * N: 1 => Client requests Server to not
             *         update DNS
             */
            *p++ = (isHostNameValid) ? ((m_IfConfig.fqdn & 0x09) | 0x04) : ((DhcpProtFqdn_None & 0x09) | 0x04);
            *p++ = 0; /* from server for PTR RR */
            *p++ = 0; /* from server for A RR if S=1 */
            if (isHostNameValid)
            {
                int i = Util::EncodeRfc1035(m_IfConfig.hostName, p);
                *lp += i;
                p += i;
            }
        }
        else if ((m_IfConfig.daemonOptions & ClientDaemonOption_Hostname) && isHostNameValid)
        {
            *p++ = DhcpProtOption_Hostname;
            len = strlen(m_IfConfig.hostName);
            *p++ = (uint8_t)len;
            memcpy(p, m_IfConfig.hostName, len);
            p += len;
        }

        /* vendor is already encoded correctly, so just add it */
        if (m_IfConfig.vendor[0])
        {
            *p++ = DhcpProtOption_Vendor;
            memcpy(p, m_IfConfig.vendor, m_IfConfig.vendor[0] + 1);
            p += m_IfConfig.vendor[0] + 1;
        }

        if (m_IfConfig.daemonOptions & ClientDaemonOption_SendRequire)
        {
            /* We support HMAC-MD5 */
            *p++ = DhcpProtOption_ForceRenewNonce;
            *p++ = 1;
            *p++ = DhcpProtAuth_AlgHmacMd5;
        }

        DHCPC_RETURN_UPON_ERROR(MakeVivoMessage(msgType, m, &p));
        DHCPC_RETURN_UPON_ERROR(MakeParamReqMessage(msgType, m, &p));
    }

    if (result.IsSuccess())
    {
        // Pad to minimum size
        do
        {
            *p++ = DhcpProtOption_Pad;
            size = (size_t)(p - m);
        }while (size < Config_MinBootpMessageSize);

        *pReturnedSize = size;
    }

    return result;
} // NOLINT(readability/fn_size)

Result Interface::MakeVivoMessage(DhcpProtMsgType msgType, uint8_t *m, uint8_t **pP)
{
    uint32_t vivoEnterprise;
    size_t requiredSize, remaningSize;
    uint8_t *p = *pP;
    NN_UNUSED(msgType);

    if (m_IfConfig.vivoDataSize == 0 || m_IfConfig.pVivoData == NULL)
    {
        return ResultSuccess();
    }

    // Is there enough room?
    requiredSize = 7 + m_IfConfig.vivoDataSize;
    remaningSize = (size_t)(p - m);
    if (requiredSize > remaningSize)
    {
        DHCPC_LOG_ERROR("%s: Overall DHCP message size exceeded during Vivo.\n",
                        m_IfConfig.ifName);
        return ResultMaximumExceeded();
    }

    // Build VIVO header
    *p++ = DhcpProtOption_Vivco;
    *p++ = sizeof(vivoEnterprise) + m_IfConfig.vivoDataSize + 1;
    vivoEnterprise = nn::socket::InetHtonl(m_IfConfig.vivoEnterprise);
    memcpy(p, &vivoEnterprise, sizeof(vivoEnterprise));
    p += sizeof(vivoEnterprise);
    *p++ = m_IfConfig.vivoDataSize;
    memcpy(p, m_IfConfig.pVivoData, m_IfConfig.vivoDataSize);
    p += m_IfConfig.vivoDataSize;

    // Pass back updated message pointer
    *pP = p;

    return ResultSuccess();
}


Result Interface::MakeParamReqMessage(DhcpProtMsgType msgType, uint8_t *m, uint8_t **pP)
{
    size_t requiredSize, remaningSize;
    uint8_t *pNumParams;
    int32_t numOptionsRequested = 0;
    uint8_t *p = *pP;

    // Determine number of requested options
    for (uint8_t option = DhcpProtOption_SubnetMask; option < DhcpProtOption_End; option++)
    {
        if (DHCPC_HAS_PROT_OPTION_MASK(m_IfConfig.requestMask, option))
        {
            numOptionsRequested++;
        }
    }

    // Nothing to do if options are not requested
    if (numOptionsRequested == 0)
    {
        return ResultSuccess();
    }

    // Is there enough room?
    requiredSize = 2 + numOptionsRequested;
    remaningSize = (size_t)(p - m);
    if (requiredSize > remaningSize)
    {
        DHCPC_LOG_ERROR("%s: Overall DHCP message size exceeded during ParamRequest.\n",
                        m_IfConfig.ifName);
        return ResultMaximumExceeded();
    }

    // Build header
    *p++ = DhcpProtOption_ParameterRequestList;
    pNumParams = p;
    *p++ = 0;

    // Build out options
    for (uint8_t option = DhcpProtOption_SubnetMask; option < DhcpProtOption_End; option++)
    {
        if (DHCPC_HAS_PROT_OPTION_MASK(m_IfConfig.requestMask, option))
        {
            if ( !(msgType == DhcpProtMsgType_Inform &&
                   (option == DhcpProtOption_RenewalTime || option == DhcpProtOption_RebindTime)))
            {
                *pNumParams = *pNumParams + 1;
                *p++ = option;
            }
        }
    }

    // Set last option to end option
    *p++ = DhcpProtOption_End;

    // Pass back updated message pointer
    *pP = p;

    return ResultSuccess();
}

int Interface::DoMtu(int mtu)
{
    int s, r;
    struct ifreq ifr;
    if ((s = nn::socket::SocketExempt(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) == -1)
    {
        return -1;
    }
    memset(&ifr, 0, sizeof(ifr));
    nn::util::Strlcpy(ifr.ifr_name, m_IfConfig.ifName, sizeof(ifr.ifr_name));
    ifr.ifr_mtu = mtu;
    nn::socket::IoctlCommand command = static_cast<nn::socket::IoctlCommand>
                                        (mtu ? nn::socket::IoctlCommandPrivate::SiocSIfMtu : nn::socket::IoctlCommandPrivate::SiocGIfMtu);
    r = nn::socket::Ioctl(s, command, &ifr, sizeof(ifr));
    nn::socket::Close(s);
    if (r == -1)
    {
        return -1;
    }
    return ifr.ifr_mtu;
}

void Interface::UpdateReTransBackoffInterval(bool initialize)
{
    if (initialize)
    {
        m_ReTransBackoffInterval =  TimeSpan::FromSeconds(RetransmitBackoff_BaseMin);
    }
    else
    {
        m_ReTransBackoffInterval = TimeSpan::FromMicroSeconds(m_ReTransBackoffInterval.GetMicroSeconds() * 2);
        if (m_ReTransBackoffInterval.GetSeconds() > RetransmitBackoff_BaseMax)
        {
            m_ReTransBackoffInterval = TimeSpan::FromSeconds(RetransmitBackoff_BaseMax);
        }
    }

    m_ReTransBackoffInterval += TimeSpan::FromMicroSeconds(std::uniform_int_distribution<int32_t>(RetransmitBackoff_RandomMin * 1000000, RetransmitBackoff_RandomMax * 1000000)(m_Mt));
}

void Interface::UpdateRenewReTransInterval()
{
    if (!bgtc::IsInFullAwake())
    {
        m_ReTransBackoffInterval = TimeSpan::FromSeconds(RenewRebindRetransmit_BaseMin);
    }
    else
    {
        TimeSpan currentTime;
        Util::GetUpTime(&currentTime);

        TimeSpan deltaTime = currentTime - m_LeaseAquiredTime;

        m_ReTransBackoffInterval = TimeSpan::FromSeconds((m_Lease.rebindTime.GetSeconds() - deltaTime.GetSeconds()) / 2);
        if (m_ReTransBackoffInterval.GetSeconds() < RenewRebindRetransmit_BaseMin)
        {
            m_ReTransBackoffInterval = TimeSpan::FromSeconds(RenewRebindRetransmit_BaseMin);
        }

        m_ReTransBackoffInterval += TimeSpan::FromMicroSeconds(std::uniform_int_distribution<int32_t>(RetransmitBackoff_RandomMin * 1000000, RetransmitBackoff_RandomMax * 1000000)(m_Mt));
    }
}

void Interface::UpdateRebindReTransInterval()
{
    if (!bgtc::IsInFullAwake())
    {
        m_ReTransBackoffInterval = TimeSpan::FromSeconds(RenewRebindRetransmit_BaseMin);
    }
    else
    {
        TimeSpan currentTime;
        Util::GetUpTime(&currentTime);

        TimeSpan deltaTime = currentTime - m_LeaseAquiredTime;

        m_ReTransBackoffInterval = TimeSpan::FromSeconds((m_Lease.leaseTime.GetSeconds() - deltaTime.GetSeconds()) / 2);
        if (m_ReTransBackoffInterval.GetSeconds() < RenewRebindRetransmit_BaseMin)
        {
            m_ReTransBackoffInterval = TimeSpan::FromSeconds(RenewRebindRetransmit_BaseMin);
        }

        m_ReTransBackoffInterval += TimeSpan::FromMicroSeconds(std::uniform_int_distribution<int32_t>(RetransmitBackoff_RandomMin * 1000000, RetransmitBackoff_RandomMax * 1000000)(m_Mt));
    }
}

void Interface::UpdateNakBackoffInterval(bool initialize)
{
    if (initialize)
    {
        m_NakBackoffInterval =  0;
    }
    else
    {
        m_NakBackoffInterval = TimeSpan::FromMicroSeconds(m_ReTransBackoffInterval.GetMicroSeconds() * 2);
        if (m_NakBackoffInterval.GetSeconds() > NakBackoff_Max)
        {
            m_NakBackoffInterval = TimeSpan::FromSeconds(NakBackoff_Max);
        }
    }
}

void Interface::DoClientCallback(Indication indication)
{
    IndicationDataType indData;
    memset(&indData, 0, sizeof(indData));
    indData.indication   = indication;
    indData.failureCause = m_FailureCause;
    nn::util::Strlcpy(indData.ifName, m_IfConfig.ifName, sizeof(indData.ifName));
    indData.lease = m_Lease;
    if ((m_IfConfig.callback != NULL) && (m_pDeferredFinalize == NULL))
    {
        (*m_IfConfig.callback)(m_IfConfig.callbackContext, &indData);
    }
}

Result Interface::HandleFinalizeEvent()
{
    Result result = ResultSuccess();
    if(m_ShutdownOptions != 0 && (Fsm::GetState() == State_Bound || Fsm::GetState() == State_Renew || Fsm::GetState() == State_Rebind))
    {
        if((m_ShutdownOptions & cfg::ShutdownOption_Release) > 0)
        {
            size_t dhcpMessageSize = 0;
            memset(&m_SendPacket, 0, sizeof(m_SendPacket));

            DHCPC_RETURN_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Release, &m_InAddrAny, &dhcpMessageSize));

            m_SendPacketSize = PacketManager::MakeUdpPacket(&m_SendPacket, dhcpMessageSize, &m_InAddrAny, &m_InAddrBroadcast);
            if(!(result = m_Packet.SendRawPacket(nn::socket::EtherType::EtherType_Ip, (const void *)&m_SendPacket, m_SendPacketSize)).IsSuccess())
            {
                DHCPC_LOG_ERROR("Failed To send DhcpProtMsgType_Release!\n");
            }
        }
        else if((m_ShutdownOptions & cfg::ShutdownOption_Decline) > 0)
        {
            size_t dhcpMessageSize = 0;
            memset(&m_SendPacket, 0, sizeof(m_SendPacket));

            DHCPC_RETURN_UPON_ERROR(MakeMessage(&m_SendPacket.dhcp, DhcpProtMsgType_Decline, &m_InAddrAny, &dhcpMessageSize));

            m_SendPacketSize = PacketManager::MakeUdpPacket(&m_SendPacket, dhcpMessageSize, &m_InAddrAny, &m_InAddrBroadcast);
            if(!(result = m_Packet.SendRawPacket(nn::socket::EtherType::EtherType_Ip, (const void *)&m_SendPacket, m_SendPacketSize)).IsSuccess())
            {
                DHCPC_LOG_ERROR("Failed To send DhcpProtMsgType_Decline!\n");
            }
        }
    }

    SetState(State_Null);
    return result;
}

void Interface::HandleUnrecoverableErrorEvent(LocalEventDataType *pEvent)
{
    m_FailureCause = pEvent->status;
    SetState(State_Failure);
}

Result Interface::TakeReceiptOfNext(LocalEventDataType *pEvent)
{
    Result result = ResultInvalidPacket();

    DhcpMessage *dhcp = reinterpret_cast<DhcpMessage *>(pEvent->args[0].pData);
    struct nn::socket::InAddr *pFrom = reinterpret_cast<struct nn::socket::InAddr *>(pEvent->args[1].pData);

    memset(&m_NextLease, 0, sizeof(m_NextLease));
    memset(&m_NextOffer, 0, sizeof(m_NextOffer));
    do
    {
        bool reject = false;

        // Check transaction ID
        if (m_Xid != nn::socket::InetNtohl(dhcp->xid))
        {
            DHCPC_LOG_WARN("Wrong XID - Expected 0x%x, receieved 0x%x.\n",
                           m_Xid, dhcp->xid);
            reject = true;
        }

        // Check for invalid address
        if ((dhcp->ciaddr == nn::socket::InetHtonl(nn::socket::InAddr_Any) || dhcp->ciaddr ==  nn::socket::InetHtonl(nn::socket::InAddr_Broadcast)) &&
            (dhcp->yiaddr == nn::socket::InetHtonl(nn::socket::InAddr_Any) || dhcp->yiaddr == nn::socket::InetHtonl(nn::socket::InAddr_Broadcast)))
        {
            DHCPC_LOG_WARN("Rejecting invalid address.\n");
            reject = true;
        }

        // Check for rejected options
        for (uint8_t option = DhcpProtOption_SubnetMask; option < DhcpProtOption_End; option++)
        {
            if (DHCPC_HAS_PROT_OPTION_MASK(m_IfConfig.rejectMask, option) &&
                (Util::GetOption(m_OptionBuffer, dhcp, option, NULL) != NULL))
            {
                LogContextual(Util::LogLevel_Warn,
                              "Rejecting response from %s because it contained rejected option %d.\n",
                              nn::socket::InetNtoa(*pFrom),
                              option);
                reject = true;
                break;
            }
        }

        // Check for required options
        for (uint8_t option = DhcpProtOption_SubnetMask; option < DhcpProtOption_End; option++)
        {
            if(DHCPC_HAS_PROT_OPTION_MASK(m_IfConfig.requireMask, option) &&
                Util::GetOption(m_OptionBuffer, dhcp, option, NULL) == NULL)
            {
                 LogContextual(Util::LogLevel_Warn,
                            "Rejecting response from %s because it is missing required option %d.\n",
                            nn::socket::InetNtoa(*pFrom),
                            option);
                reject = true;
                // Want to list all missing options so no break here.
            }
        }

        if (reject)
        {
            break;
        }

        // Passed sanity, record the offer
        memcpy(&m_Offer.message, dhcp, sizeof(m_Offer.message));
        m_Offer.isValid = true;

        // Extract lease parameters
        Util::GetLeaseFromDhcpMessage(m_OptionBuffer, &m_NextLease, &m_Offer.message);

        result = ResultSuccess();
    }while (false);

    return result;
}

Result Interface::DeclareLeaseInvalid()
{
    // Clear lease data
    memset(&m_Lease, 0, sizeof(m_Lease));
    memset(&m_NextLease, 0, sizeof(m_NextLease));
    // T1, T2 and lease timers never valid at this point
    StopTimedFsmEvent(&m_RenewTimerEvent);
    StopTimedFsmEvent(&m_RebindTimerEvent);
    StopTimedFsmEvent(&m_ExpireTimerEvent);
    // Tell upper layers
    DoClientCallback(Indication_InvalidLease);
    return ResultSuccess();
}

void Interface::FdMonRawPacketCallback()
{
    int flags = 0;
    uint32_t iteration = 0;

    while ( !(flags & PacketManager::RawFlags_EOF))
    {
        size_t bytes;
        nn::socket::InAddr from;
        DhcpMessage localDhcpCopy;
        bool IsWhitelisted = true;
        bool IsBlacklisted = false;
        memset(&localDhcpCopy, 0, sizeof(localDhcpCopy));
        iteration++;
        bytes = (size_t)m_Packet.ReadRawPacket(nn::socket::EtherType::EtherType_Ip, &m_ReceivePacket,
                                               sizeof(m_ReceivePacket), &flags);
        if ((ssize_t)bytes == -1)
        {
            DHCPC_LOG_VERBOSE("Interface-%s: ReadRawPacket() failed.\n", m_IfConfig.ifName);
            break;
        }
        if (bytes == 0)
        {
            break;
        }
        if (m_Packet.ValidateUdpPacket((const uint8_t *)&m_ReceivePacket, bytes,
                                       &from, flags & PacketManager::RawFlags_PartialCsum) == -1)
        {
            DHCPC_LOG_WARN("%s: invalid UDP packet from %s\n",
                           m_IfConfig.ifName, nn::socket::InetNtoa(from));
            continue;
        }

        //IsWhitelisted = m_IfSettings.whitelistedIps.IsAddressInList(from.S_addr);
        // IsBlacklisted = m_IfSettings.blacklistedIps.IsAddressInList(from.S_addr);
        if ( !IsWhitelisted /*&& (m_IfSettings.whitelistedIps.GetNumSubnets() > 0)*/)
        {
            DHCPC_LOG_WARN("%s: non whitelisted DHCP packet from %s\n",
                           m_IfConfig.ifName, nn::socket::InetNtoa(from));
            continue;
        }
        else if ( !IsWhitelisted && IsBlacklisted)
        {
            DHCPC_LOG_WARN("%s: blacklisted DHCP packet from %s\n",
                           m_IfConfig.ifName, nn::socket::InetNtoa(from));

        }

        bytes = nn::socket::InetNtohs(m_ReceivePacket.ip.ip_len) - sizeof(m_ReceivePacket.ip) - sizeof(m_ReceivePacket.udp);
        if (bytes > sizeof(localDhcpCopy))
        {
            DHCPC_LOG_WARN("%s: packet greater than DHCP size from %s\n",
                           m_IfConfig.ifName, nn::socket::InetNtoa(from));
            continue;
        }

        /* Make a local copy */
        memcpy(&localDhcpCopy, &m_ReceivePacket.dhcp, bytes);
        if (localDhcpCopy.cookie != nn::socket::InetHtonl(DhcpMessageFlags_MagicCookie))
        {
            DHCPC_LOG_WARN("%s: bogus cookie from %s\n",
                           m_IfConfig.ifName, nn::socket::InetNtoa(from));
            continue;
        }

        /* Ensure packet is for us */
        if (memcmp(localDhcpCopy.chaddr, m_IfConfig.hardwareAddress,
                   DHCPC_MIN(sizeof(localDhcpCopy.chaddr), sizeof(m_IfConfig.hardwareAddress))))
        {
            /*DHCPC_LOG_VERBOSE("%s: xid 0x%x is for hwaddr %s\n",
                              m_IfConfig.ifName, nn::socket::InetNtohl(localDhcpCopy.xid),
                              Util::HwAddrToA(localDhcpCopy.chaddr, localDhcpCopy.hwlen,
                                              &m_ToABuffer));*/
            continue;
        }

        DispatchReceivedDhcpMessage(&localDhcpCopy, &from);

        if (m_Packet.GetRawFd() == -1) break;
    }
}

void Interface::LogContextual(Util::LogLevel severity, const char *format, ...)
{
    size_t offset = 0;
    char buffer[256];
    va_list ap;
    va_start(ap, format);
    offset = snprintf(buffer, sizeof(buffer), "%s/%s - ",
                      StateNames[Fsm::GetState()], EventNames[GetCurrentEvent()]);
    offset = nn::util::VSNPrintf(offset + buffer, sizeof(buffer) - offset, format, ap);
    va_end(ap);
    Util::Log(severity, buffer);
}

} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn
