﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_KMemoryManager.h"
#include "../kern_KSharedMemory.h"
#include "../kern_KProcess.h"
#include "../kern_KHandleTable.h"
#include "kern_MemoryMap.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"

namespace nn { namespace kern { namespace svc {
namespace {
Result SvcCreateSharedMemory(nn::svc::Handle* pOut, size_t size, nn::svc::MemoryPermission myProtect, nn::svc::MemoryPermission otherProtect)
{
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }

    if (size >= NN_KERN_P_ADDR_MAIN_MEMORY_SIZE)
    {
        return nn::svc::ResultInvalidSize();
    }

    switch (myProtect)
    {
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_ReadWrite:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    switch (otherProtect)
    {
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_ReadWrite:
    case nn::svc::MemoryPermission_DontCare:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    Result result;
    KProcess& process = GetCurrentProcess();

    KSharedMemory* psm = KSharedMemory::Create();
    if (!psm)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KSharedMemory> autoCloser(psm);

    result = psm->Initialize(&process, size, myProtect, otherProtect);
    if (result.IsFailure())
    {
        return result;
    }

    result = KSharedMemory::Register(psm);
    if (result.IsSuccess())
    {
        // KHandleTableに登録
        result = process.GetHandleTable().Add(pOut, psm);
    }

    return result;
}

Result SvcMapSharedMemory(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (myProtect)
    {
        case nn::svc::MemoryPermission_Read:
        case nn::svc::MemoryPermission_ReadWrite:
        {
            break;
        }
        default:
        {
            return nn::svc::ResultInvalidNewMemoryPermission();
        }
    }

    KProcess& process = GetCurrentProcess();
    KSharedMemory* pMemory = process.GetHandleTable().GetObject<KSharedMemory>(handle);

    if(!pMemory)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KSharedMemory> autoCloser(pMemory);

    KProcessPageTable& pageTable = process.GetPageTable();
    if (!pageTable.IsInRange(addr, size, KMemoryState_Shared))
    {
        return nn::svc::ResultInvalidRegion();
    }

    Result result = process.AddSharedMemory(pMemory, addr, size);
    if (result.IsFailure())
    {
        return result;
    }

    result = pMemory->Map(&pageTable, addr, size, &process, myProtect);
    if (result.IsFailure())
    {
        process.DeleteSharedMemory(pMemory, addr, size);
        return result;
    }

    return result;
}

Result SvcUnmapSharedMemory(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess& process = GetCurrentProcess();
    KSharedMemory* pMemory = process.GetHandleTable().GetObject<KSharedMemory>(handle);

    if(!pMemory)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KSharedMemory> autoCloser(pMemory);

    KProcessPageTable& pageTable = process.GetPageTable();
    if (!pageTable.IsInRange(addr, size, KMemoryState_Shared))
    {
        return nn::svc::ResultInvalidRegion();
    }

    Result result = pMemory->Unmap(&pageTable, addr, size, &process);
    if (result.IsFailure())
    {
        return result;
    }

    process.DeleteSharedMemory(pMemory, addr, size);

    return result;
}
}
#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateSharedMemory32(nn::svc::Handle* pOut, size_t size, nn::svc::MemoryPermission myProtect, nn::svc::MemoryPermission otherProtect)
{
    Result result = SvcCreateSharedMemory(pOut, size, myProtect, otherProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapSharedMemory32(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    Result result = SvcMapSharedMemory(handle, addr, size, myProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapSharedMemory32(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    Result result = SvcUnmapSharedMemory(handle, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateSharedMemory64(nn::svc::Handle* pOut, size_t size, nn::svc::MemoryPermission myProtect, nn::svc::MemoryPermission otherProtect)
{
    Result result = SvcCreateSharedMemory(pOut, size, myProtect, otherProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapSharedMemory64(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    Result result = SvcMapSharedMemory(handle, addr, size, myProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapSharedMemory64(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    Result result = SvcUnmapSharedMemory(handle, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreateSharedMemory64From32(nn::svc::Handle* pOut, size_t size, nn::svc::MemoryPermission myProtect, nn::svc::MemoryPermission otherProtect)
{
    Result result = SvcCreateSharedMemory(pOut, size, myProtect, otherProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapSharedMemory64From32(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    Result result = SvcMapSharedMemory(handle, addr, size, myProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapSharedMemory64From32(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    Result result = SvcUnmapSharedMemory(handle, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
