﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../kern_Platform.h"
#include "kern_KUserPointer.h"
#include "../kern_Kernel.h"
#include "../kern_Utility.h"
#include "../kern_KScheduler.h"
#include "../kern_KSynchronization.h"
#include "../kern_ExceptionHandlerSelect.h"
#include "../kern_DebugSelect.h"
#include "../kern_KScopedSchedulingLock.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include "kern_SystemControl.h"

namespace nn { namespace kern { namespace svc {
namespace {
bool CheckCpuNoValue(int32_t cpuNo)
{
    return (0 <= cpuNo && cpuNo < KCPU::NUM_CORE);
}

Result SvcCreateProcess( nn::svc::Handle* pOut, const nn::svc::CreateProcessParameter& params, KUserPointer<const Bit32*> flags, int32_t numFlags )
{
    Result result;
    if (numFlags < 0)
    {
        return nn::svc::ResultInvalidPointer();
    }
    if (numFlags > 0)
    {
        if (((numFlags * sizeof(Bit32)) / sizeof(Bit32)) != static_cast<size_t>(numFlags))
        {
            return nn::svc::ResultInvalidPointer();
        }
        if (!GetCurrentProcess().GetPageTable().IsInRange(KProcessAddress(GetUnsafePointer(flags)), numFlags * sizeof(Bit32)))
        {
            return nn::svc::ResultInvalidPointer();
        }
    }

    if (params.flags & ~nn::svc::CreateProcessParameterFlag_All)
    {
        return nn::svc::ResultInvalidEnum();
    }

    if ((sizeof(void *) < sizeof(uint64_t)) && (params.flags & nn::svc::CreateProcessParameterFlag_64Bit))
    {
        return nn::svc::ResultInvalidCombination();
    }

    uintptr_t mapBegin;
    uintptr_t mapEnd;
    size_t maxMapSize;
    switch (params.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask)
    {
#if defined (NN_BUILD_CONFIG_ABI_LP64)
    case nn::svc::CreateProcessParameterFlag_AddressSpace64BitOld:
        {
            if (!(params.flags & nn::svc::CreateProcessParameterFlag_64Bit))
            {
                return nn::svc::ResultInvalidCombination();
            }
            mapBegin = NN_SVC_ADDR_SMALL_MAP36_BEGIN;
            mapEnd = NN_SVC_ADDR_SMALL_MAP36_END;
            maxMapSize = NN_SVC_ADDR_SMALL_MAP36_SIZE;
        }
        break;
    case nn::svc::CreateProcessParameterFlag_AddressSpace64Bit:
        {
            if (!(params.flags & nn::svc::CreateProcessParameterFlag_64Bit))
            {
                return nn::svc::ResultInvalidCombination();
            }
            mapBegin = NN_SVC_ADDR_MAP39_BEGIN;
            mapEnd = NN_SVC_ADDR_MAP39_END;
            // 物理メモリサイズより小さいはずなので、heap の最大サイズで制限をかけておく。
            maxMapSize = NN_SVC_ADDR_MEMORY_REGION_HEAP39_SIZE;
        }
        break;
#endif
    case nn::svc::CreateProcessParameterFlag_AddressSpace32Bit:
    case nn::svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved:
        {
            mapBegin  = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            mapEnd    = NN_SVC_ADDR_SMALL_MAP32_END;
            maxMapSize = NN_SVC_ADDR_SMALL_MAP32_SIZE;
        }
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    switch (params.flags & nn::svc::CreateProcessParameterFlag_UseMemoryMask)
    {
    case nn::svc::CreateProcessParameterFlag_UseMemory0:
    case nn::svc::CreateProcessParameterFlag_UseMemory1:
    case nn::svc::CreateProcessParameterFlag_UseMemory2:
    case nn::svc::CreateProcessParameterFlag_UseMemory3:
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    if (params.memoryAddress & (KProcess::ProcessAlign - 1))
    {
        return nn::svc::ResultInvalidAddress();
    }
    if (params.memoryNumPages < 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (params.extraResourcePageCount < 0)
    {
        return nn::svc::ResultInvalidSize();
    }

    size_t memoryNumPages = params.memoryNumPages;
    size_t extraResourcePageCount = params.extraResourcePageCount;

    if ((memoryNumPages * NN_KERN_FINEST_PAGE_SIZE) / NN_KERN_FINEST_PAGE_SIZE != memoryNumPages)
    {
        return nn::svc::ResultInvalidSize();
    }
    if ((extraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE) / NN_KERN_FINEST_PAGE_SIZE != extraResourcePageCount)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (memoryNumPages + extraResourcePageCount < memoryNumPages)
    {
        return nn::svc::ResultOutOfMemory();
    }
    if (((memoryNumPages + extraResourcePageCount) * NN_KERN_FINEST_PAGE_SIZE) / NN_KERN_FINEST_PAGE_SIZE != (memoryNumPages + extraResourcePageCount))
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(memoryNumPages < (maxMapSize / NN_KERN_FINEST_PAGE_SIZE)))
    {
        return nn::svc::ResultInvalidRegion();
    }

    if (!(mapBegin <= params.memoryAddress &&
                params.memoryAddress < params.memoryAddress + memoryNumPages * NN_KERN_FINEST_PAGE_SIZE &&
                params.memoryAddress + memoryNumPages * NN_KERN_FINEST_PAGE_SIZE - 1 <= mapEnd - 1))
    {
        return nn::svc::ResultInvalidRegion();
    }

    if (memoryNumPages >= NN_KERN_P_ADDR_MAIN_MEMORY_SIZE / NN_KERN_FINEST_PAGE_SIZE)
    {
        return nn::svc::ResultOutOfMemory();
    }

    if (extraResourcePageCount >= NN_KERN_P_ADDR_MAIN_MEMORY_SIZE / NN_KERN_FINEST_PAGE_SIZE)
    {
        return nn::svc::ResultOutOfMemory();
    }

    if (memoryNumPages + extraResourcePageCount >= NN_KERN_P_ADDR_MAIN_MEMORY_SIZE / NN_KERN_FINEST_PAGE_SIZE)
    {
        return nn::svc::ResultOutOfMemory();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KProcess* pCreatedProcess = KProcess::Create();
    if (!pCreatedProcess)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KProcess> autoCloser(pCreatedProcess);

    KResourceLimit* pResourceLimit = nullptr;
    if (params.resourceLimit != nn::svc::INVALID_HANDLE_VALUE)
    {
        pResourceLimit = handleTable.GetObject<KResourceLimit>(params.resourceLimit);
        if (!pResourceLimit)
        {
            return nn::svc::ResultInvalidHandle();
        }
    }

    KMemoryManager::Region region;
    switch (params.flags & nn::svc::CreateProcessParameterFlag_UseMemoryMask)
    {
    case nn::svc::CreateProcessParameterFlag_UseMemory0:
        region = KMemoryManager::Region_Application;
        break;
    case nn::svc::CreateProcessParameterFlag_UseMemory1:
        region = KMemoryManager::Region_Applet;
        break;
    case nn::svc::CreateProcessParameterFlag_UseMemory2:
        region = KMemoryManager::Region_SecureSystem;
        break;
    case nn::svc::CreateProcessParameterFlag_UseMemory3:
    default:
        region = KMemoryManager::Region_NonSecureSystem;
        break;
    }

    result = pCreatedProcess->Initialize(params, flags, numFlags, pResourceLimit?pResourceLimit: &Kernel::GetSystemResourceLimit(), region);

    if (result.IsSuccess())
    {
        result = KProcess::Register(pCreatedProcess);

        if (result.IsSuccess())
        {
            result = handleTable.Add(pOut, pCreatedProcess);
        }
    }

    if (pResourceLimit)
    {
        pResourceLimit->Close();
    }

    return result;
}

Result SvcStartProcess(nn::svc::Handle hProcess, int32_t priority, int32_t coreNo, uint64_t stackSize)
{
    if (stackSize != static_cast<size_t>(stackSize))
    {
        return nn::svc::ResultOutOfMemory();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, hProcess);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    if (!CheckCpuNoValue(coreNo))
    {
        return nn::svc::ResultInvalidCoreNumber();
    }
    Bit64 coreMask = pProcess->GetCoreMask();
    if ((coreMask | (1ull << coreNo)) != coreMask)
    {
        return nn::svc::ResultInvalidCoreNumber();
    }

    if (!(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority))
    {
        return nn::svc::ResultInvalidPriority();
    }
    if (!pProcess->CheckThreadPriority(priority))
    {
        return nn::svc::ResultInvalidPriority();
    }

    pProcess->SetIdealProcessor(coreNo);

    Result result = pProcess->Run(priority, stackSize);
    if (result.IsSuccess())
    {
        // Exit との対応のため 成功時に Open する
        pProcess->Open();
    }
    return result;
}

nn::Result SvcTerminateProcess(nn::svc::Handle hProcess)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, hProcess);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    Result result = ResultSuccess();
    if (pProcess == GetCurrentProcessPointer())
    {
        // Exit からは返ってこないうえ、自プロセスであるから
        // pProcess は Exit まで有効であることが保証される。
        // よって先に Close
        pProcess->Close();

        pProcess->Exit();
        NN_KERNEL_PANIC("Never reach here.");
    }
    else
    {
        result = pProcess->Terminate();

        // pProcess が Terminate まで有効であることが保証されない。
        // よって後で Close
        pProcess->Close();
    }

    return result;
}

void SvcExitProcess()
{
    GetCurrentProcess().Exit();
    NN_KERNEL_PANIC("Never reach here.");
}

Result SvcGetProcessId( Bit64* pOut, nn::svc::Handle handle )
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KAutoObject* pObject = GetObjectFromRealOrPseudoHandle(handleTable, handle);

    if (!pObject)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KAutoObject> autoCloser(pObject);
    KProcess* pProcess = NULL;

    if( pObject->IsDerivedFrom(KProcess::GetTypeObjStatic()) )
    {
        pProcess = reinterpret_cast<KProcess*>(pObject);
    }
    else if( pObject->IsDerivedFrom(KThread::GetTypeObjStatic()) )
    {
        pProcess = reinterpret_cast<KThread*>(pObject)->GetParentPointer();
    }
    else if( pObject->IsDerivedFrom(KDebug::GetTypeObjStatic()) )
    {
        pProcess = reinterpret_cast<KDebug*>(pObject)->GetProcess();

        if( pProcess != NULL )
        {
            autoCloser.CloseAndReset(pProcess);
        }
    }

    if( pProcess != NULL )
    {
        *pOut = pProcess->GetId();
        return ResultSuccess();
    }
    else
    {
        return nn::svc::ResultInvalidHandle();
    }
}

Result SvcGetProcessInfo(int64_t* pOut, nn::svc::Handle handle, nn::svc::ProcessInfoType type)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, handle);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    Result result;
    switch (type)
    {
    case nn::svc::ProcessInfoType_State:
        {
            KProcess::State state;
            {
                KScopedLightLock procLocker(&pProcess->GetStateMutex());
                KScopedSchedulingLock lock;
                state = pProcess->GetState();
            }
            result = ResultSuccess();
            switch (state)
            {
            case KProcess::State_Initializing:
                {
                    *pOut = nn::svc::ProcessState_Initializing;
                }
                break;
            case KProcess::State_PreAttached:
                {
                    *pOut = nn::svc::ProcessState_PreAttached;
                }
                break;
            case KProcess::State_Running:
                {
                    *pOut = nn::svc::ProcessState_Running;
                }
                break;
            case KProcess::State_WaitAttach:
                {
                    *pOut = nn::svc::ProcessState_WaitAttach;
                }
                break;
            case KProcess::State_Attached:
                {
                    *pOut = nn::svc::ProcessState_Attached;
                }
                break;
            case KProcess::State_Terminating:
                {
                    *pOut = nn::svc::ProcessState_Terminating;
                }
                break;
            case KProcess::State_Terminated:
                {
                    *pOut = nn::svc::ProcessState_Terminated;
                }
                break;
            case KProcess::State_Breaked:
                {
                    *pOut = nn::svc::ProcessState_Breaked;
                }
                break;
            default:
                {
                    NN_KERN_ABORT();
                }
                break;
            }
        }
        break;
    default:
        {
            result = nn::svc::ResultInvalidEnum();
        }
        break;
    }

    return result;
}

Result SvcSetProcessActivity(nn::svc::Handle handle, nn::svc::ProcessActivity activity)
{
    switch (activity)
    {
    case nn::svc::ProcessActivity_Runnable:
    case nn::svc::ProcessActivity_Paused:
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    KProcess* pProcess = GetCurrentProcess().GetHandleTable().GetObject<KProcess>(handle);
    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    if (pProcess == &GetCurrentProcess())
    {
        return nn::svc::ResultBusy();
    }

    return pProcess->SetActivity(activity);
}

}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateProcess32( nn::svc::Handle* pOut, KUserPointer<const nn::svc::ilp32::CreateProcessParameter*> userParams, KUserPointer<const Bit32*> flags, int32_t numFlags )
{
    nn::svc::CreateProcessParameter params;
    Result result = userParams.CopyTo(&params);
    if (result.IsFailure())
    {
        ClearSvcOutRegistersReturnResult();
        return result;
    }

    result = SvcCreateProcess(pOut, params, flags, numFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcStartProcess32( nn::svc::Handle hProcess, int32_t priority, int32_t coreNo, uint64_t stackSize)
{
    Result result = SvcStartProcess(hProcess, priority, coreNo, stackSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
nn::Result SvcTerminateProcess32( nn::svc::Handle hProcess )
{
    Result result = SvcTerminateProcess(hProcess);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcExitProcess32()
{
    SvcExitProcess();
}
Result SvcGetProcessId32( Bit64* pOut, nn::svc::Handle hProcess )
{
    Result result = SvcGetProcessId(pOut, hProcess);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetProcessInfo32(int64_t* pOut, nn::svc::Handle handle, nn::svc::ProcessInfoType type)
{
    Result result = SvcGetProcessInfo(pOut, handle, type);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetProcessActivity32(nn::svc::Handle handle, nn::svc::ProcessActivity activity)
{
    Result result = SvcSetProcessActivity(handle, activity);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateProcess64( nn::svc::Handle* pOut, KUserPointer<const nn::svc::lp64::CreateProcessParameter*> userParams, KUserPointer<const Bit32*> flags, int32_t numFlags )
{
    nn::svc::CreateProcessParameter params;
    Result result = userParams.CopyTo(&params);
    if (result.IsFailure())
    {
        ClearSvcOutRegistersReturnResult();
        return result;
    }

    result = SvcCreateProcess(pOut, params, flags, numFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcStartProcess64( nn::svc::Handle hProcess, int32_t priority, int32_t coreNo, uint64_t stackSize)
{
    Result result = SvcStartProcess(hProcess, priority, coreNo, stackSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
nn::Result SvcTerminateProcess64( nn::svc::Handle hProcess )
{
    Result result = SvcTerminateProcess(hProcess);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcExitProcess64()
{
    SvcExitProcess();
}
Result SvcGetProcessId64( Bit64* pOut, nn::svc::Handle hProcess )
{
    Result result = SvcGetProcessId(pOut, hProcess);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetProcessInfo64(int64_t* pOut, nn::svc::Handle handle, nn::svc::ProcessInfoType type)
{
    Result result = SvcGetProcessInfo(pOut, handle, type);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetProcessActivity64(nn::svc::Handle handle, nn::svc::ProcessActivity activity)
{
    Result result = SvcSetProcessActivity(handle, activity);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreateProcess64From32( nn::svc::Handle* pOut, KUserPointer<const nn::svc::ilp32::CreateProcessParameter*> userParams, KUserPointer<const Bit32*> flags, int32_t numFlags )
{
    nn::svc::Handle handle;

    nn::svc::ilp32::CreateProcessParameter params32;
    Result result = userParams.CopyTo(&params32);
    if (result.IsFailure())
    {
        ClearSvcOutRegistersReturnResult();
        return result;
    }

    nn::svc::CreateProcessParameter params = {};

    std::memcpy(params.name, params32.name, sizeof(params.name));
    static_assert(sizeof(params.name) == sizeof(params32.name), "");
    params.version = params32.version;
    params.programId = params32.programId;
    params.memoryAddress = params32.memoryAddress;
    params.memoryNumPages = params32.memoryNumPages;
    params.flags = params32.flags;
    params.resourceLimit = params32.resourceLimit ;
    params.extraResourcePageCount = params32.extraResourcePageCount;

    result = SvcCreateProcess(&handle, params, flags, numFlags);
    *pOut = static_cast<nn::svc::Handle>(handle);

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcStartProcess64From32( nn::svc::Handle hProcess, int32_t priority, int32_t coreNo, uint64_t stackSize)
{
    Result result = SvcStartProcess(hProcess, priority, coreNo, stackSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
nn::Result SvcTerminateProcess64From32( nn::svc::Handle hProcess )
{
    Result result = SvcTerminateProcess(hProcess);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcExitProcess64From32()
{
    SvcExitProcess();
}
Result SvcGetProcessId64From32( Bit64* pOut, nn::svc::Handle hProcess )
{
    Result result = SvcGetProcessId(pOut, hProcess);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetProcessInfo64From32(int64_t* pOut, nn::svc::Handle handle, nn::svc::ProcessInfoType type)
{
    Result result = SvcGetProcessInfo(pOut, handle, type);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetProcessActivity64From32(nn::svc::Handle handle, nn::svc::ProcessActivity activity)
{
    Result result = SvcSetProcessActivity(handle, activity);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
