﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Utility.h"
#include "../kern_KProcess.h"
#include "../kern_DebugSelect.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"

#include "../kern_KDumpObject.h"
#include "../kern_KThread.h"
#include "../kern_Kernel.h"
#include "../kern_DebugString.h"
#include "../kern_KTrace.h"
#include "../kern_KPageBuffer.h"
#include "kern_SystemControl.h"
#include "../kern_InitialProcess.h"

namespace nn { namespace kern { namespace svc {
namespace {
Result SvcGetInfo(Bit64* pOut, nn::svc::InfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result;

    switch (type)
    {
    case nn::svc::InfoType_CoreMask:
    case nn::svc::InfoType_PriorityMask:
    case nn::svc::InfoType_ReservedRegionAddress:
    case nn::svc::InfoType_ReservedRegionSize:
    case nn::svc::InfoType_HeapRegionAddress:
    case nn::svc::InfoType_HeapRegionSize:
    case nn::svc::InfoType_AslrRegionAddress:
    case nn::svc::InfoType_AslrRegionSize:
    case nn::svc::InfoType_StackRegionAddress:
    case nn::svc::InfoType_StackRegionSize:
    case nn::svc::InfoType_UsingUserPhysicalMemorySize:
    case nn::svc::InfoType_AssingedUserPhysicalMemorySize:
    case nn::svc::InfoType_ExtraResourceSize:
    case nn::svc::InfoType_ExtraResourceUsage:
    case nn::svc::InfoType_ProgramId:
    case nn::svc::InfoType_ProcessRegionAddress:
        {
            if (param == 0)
            {
                KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
                KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, handle);

                if (pProcess)
                {
                    KScopedAutoObject<KProcess> autoCloser(pProcess);
                    switch (type)
                    {
                    case nn::svc::InfoType_CoreMask:
                        {
                            *pOut = pProcess->GetCoreMask();
                        }
                        break;
                    case nn::svc::InfoType_PriorityMask:
                        {
                            *pOut = pProcess->GetPriorityMask();
                        }
                        break;
                    case nn::svc::InfoType_ReservedRegionAddress:
                        {
                            *pOut = GetAsInteger(pProcess->GetPageTable().GetReservedRegionBegin());
                        }
                        break;
                    case nn::svc::InfoType_ReservedRegionSize:
                        {
                            *pOut = pProcess->GetPageTable().GetReservedRegionSize();
                        }
                        break;
                    case nn::svc::InfoType_HeapRegionAddress:
                        {
                            *pOut = GetAsInteger(pProcess->GetPageTable().GetHeapRegionBegin());
                        }
                        break;
                    case nn::svc::InfoType_HeapRegionSize:
                        {
                            *pOut = pProcess->GetPageTable().GetHeapRegionSize();
                        }
                        break;
                    case nn::svc::InfoType_AslrRegionAddress:
                        {
                            *pOut = GetAsInteger(pProcess->GetPageTable().GetAslrRegionBegin());
                        }
                        break;
                    case nn::svc::InfoType_AslrRegionSize:
                        {
                            *pOut = pProcess->GetPageTable().GetAslrRegionSize();
                        }
                        break;
                    case nn::svc::InfoType_StackRegionAddress:
                        {
                            *pOut = GetAsInteger(pProcess->GetPageTable().GetStackRegionBegin());
                        }
                        break;
                    case nn::svc::InfoType_StackRegionSize:
                        {
                            *pOut = pProcess->GetPageTable().GetStackRegionSize();
                        }
                        break;
                    case nn::svc::InfoType_UsingUserPhysicalMemorySize:
                        {
                            *pOut = pProcess->GetUsingUserPhysicalMemorySize();
                        }
                        break;
                    case nn::svc::InfoType_AssingedUserPhysicalMemorySize:
                        {
                            *pOut = pProcess->GetAssingedUserPhysicalMemorySize();
                        }
                        break;
                    case nn::svc::InfoType_ExtraResourceSize:
                        {
                            *pOut = pProcess->GetExtraResourceSize();
                        }
                        break;
                    case nn::svc::InfoType_ExtraResourceUsage:
                        {
                            *pOut = pProcess->GetExtraResourceUsage();
                        }
                        break;
                    case nn::svc::InfoType_ProgramId:
                        {
                            *pOut = pProcess->GetProgramId();
                        }
                        break;
                    case nn::svc::InfoType_ProcessRegionAddress:
                        {
                            *pOut = GetAsInteger(pProcess->GetProcessLocalRegionAddr());
                        }
                        break;
                    default:
                        {
                            NN_KERN_ABORT();
                        }
                        break;
                    }
                    result = ResultSuccess();
                }
                else
                {
                    result = nn::svc::ResultInvalidHandle();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidCombination();
            }
        }
        break;
    case nn::svc::InfoType_DebuggerPresence:
        {
            if (nn::svc::Handle(0) == handle)
            {
                if (param == 0)
                {
                    if (GetCurrentProcess().GetDebugObject())
                    {
                        *pOut = 1;
                    }
                    else
                    {
                        *pOut = 0;
                    }
                    result = ResultSuccess();
                }
                else
                {
                    result = nn::svc::ResultInvalidCombination();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
    case nn::svc::InfoType_ResourceLimit:
        {
            if (nn::svc::Handle(0) == handle)
            {
                if (param == 0)
                {
                    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
                    KResourceLimit* pResourceLimit = GetCurrentProcess().GetResourceLimit();
                    if (pResourceLimit)
                    {
                        nn::svc::Handle tmp;
                        result = handleTable.Add(&tmp, pResourceLimit);
                        if (result.IsSuccess())
                        {
                            *pOut = static_cast<nnHandle>(tmp).value;
                        }
                    }
                    else
                    {
                        *pOut = nn::svc::INVALID_HANDLE_VALUE.value;
                        result = ResultSuccess();
                    }
                }
                else
                {
                    result = nn::svc::ResultInvalidCombination();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
    case nn::svc::InfoType_IdleTickCount:
        {
            if (nn::svc::Handle(0) == handle)
            {
                if (param == static_cast<Bit64>(-1ll) || param == static_cast<Bit64>(GetCurrentCpuNo()))
                {
                    *pOut = GetCoreScheduler().GetIdleThread()->GetCpuTime();
                    result = ResultSuccess();
                }
                else
                {
                    result = nn::svc::ResultInvalidCombination();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
    case nn::svc::InfoType_Random:
        {
            if (nn::svc::Handle(0) == handle)
            {
                if (param < 4)
                {
                    *pOut = GetCurrentProcess().GetRandomNumber(param);
                    result = ResultSuccess();
                }
                else
                {
                    result = nn::svc::ResultInvalidCombination();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
    case nn::svc::InfoType_ThreadTickCount:
        {
            if (param == static_cast<Bit64>(-1ll) || param < KCPU::NUM_CORE)
            {
                KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), handle);
                if (pThread)
                {
                    KScopedAutoObject<KThread> autoCloser(pThread);

                    int64_t tickCount;
                    if (param == static_cast<Bit64>(-1ll))
                    {
                        tickCount = pThread->GetCpuTime();
                        if (&GetCurrentThread() == pThread)
                        {
                            const int64_t newTick = KHardwareTimer::GetTick();
                            const int64_t prevSwitch = GetCoreScheduler().GetPrevSwitch();
                            tickCount += (newTick - prevSwitch);
                        }
                    }
                    else
                    {
                        tickCount = pThread->GetCpuTime(param);
                        if (&GetCurrentThread() == pThread && param == static_cast<Bit64>(GetCurrentCpuNo()))
                        {
                            const int64_t newTick = KHardwareTimer::GetTick();
                            const int64_t prevSwitch = GetCoreScheduler().GetPrevSwitch();
                            tickCount += (newTick - prevSwitch);
                        }
                    }
                    *pOut = tickCount;
                    result = ResultSuccess();
                }
                else
                {
                    result = nn::svc::ResultInvalidHandle();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidCombination();
            }
        }
        break;
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    case nn::svc::InfoType_PerformanceCounter:
        {
            if (param == 0)
            {
                *pOut = KCPU::GetCycleCounter();
            }
            else if ((param  - 1) < KCPU::NUM_PERFORMANCE_COUNTER)
            {
                uint32_t n = (param  - 1);
                KThread* pThread;
                KThread* pCurrentThread = &GetCurrentThread();
                if (nn::svc::Handle(0) == handle)
                {
                    pThread = pCurrentThread;
                }
                else
                {
                    pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), handle);
                }

                {
                    KDisableInterrupt kd;
                    if (pThread == pCurrentThread)
                    {
                        *pOut = (KCPU::GetPerformanceCounter(n) - Kernel::GetPrevPerformanceCounter(n)) + pThread->GetPerformanceCounter(n);
                        result = ResultSuccess();
                    }
                    else if (pThread)
                    {
                        *pOut = pThread->GetPerformanceCounter(n);
                        result = ResultSuccess();
                    }
                    else
                    {
                        result = nn::svc::ResultInvalidHandle();
                    }
                }
            }
            else
            {
                result = nn::svc::ResultInvalidCombination();
            }
        }
        break;
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
    case nn::svc::InfoType_IpcProfile:
        {
            KThread* pThread;
            if (nn::svc::Handle(0) == handle)
            {
                pThread = &GetCurrentThread();
            }
            else
            {
                pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), handle);
            }

            if (pThread)
            {
                switch (param)
                {
                case 0:
                    {
                        *pOut = pThread->GetNumIpc();
                        result = ResultSuccess();
                    }
                    break;
                case 1:
                    {
                        *pOut = pThread->GetNumIpcRecv();
                        result = ResultSuccess();
                    }
                    break;
                case 2:
                    {
                        *pOut = pThread->GetNumIpcReply();
                        result = ResultSuccess();
                    }
                    break;
                default:
                    {
                        result = nn::svc::ResultInvalidCombination();
                    }
                    break;
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
#endif
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
    case nn::svc::InfoType_CoreSwitchCount:
        {
            if (param == 0)
            {
                KThread* pThread = GetObjectFromRealOrPseudoHandle<KThread>(GetCurrentProcess().GetHandleTable(), handle);
                if (pThread)
                {
                    KScopedAutoObject<KThread> autoCloser(pThread);

                    *pOut = pThread->GetNumCoreSwitch();
                    result = ResultSuccess();
                }
                else
                {
                    result = nn::svc::ResultInvalidHandle();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidCombination();
            }
        }
        break;
#endif
    case nn::svc::InfoType_InitialProcessIdRangeObsolete:
    default:
        {
            result = nn::svc::ResultInvalidEnum();
        }
        break;
    }

    return result;
}

Result SvcGetSystemInfo(Bit64* pOut, nn::svc::SystemInfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result;

    switch (type)
    {
    case nn::svc::SystemInfoType_PhysicalMemorySize:
    case nn::svc::SystemInfoType_UsingPhysicalMemorySize:
        {
            if (nn::svc::Handle(0) == handle)
            {
                switch (param)
                {
                case KMemoryManager::Region_Application:
                case KMemoryManager::Region_Applet:
                case KMemoryManager::Region_SecureSystem:
                case KMemoryManager::Region_NonSecureSystem:
                    {
                        if (type == nn::svc::SystemInfoType_PhysicalMemorySize)
                        {
                            *pOut = Kernel::GetKernelHeapManager().GetSize(static_cast<KMemoryManager::Region>(param));
                        }
                        else if (type == nn::svc::SystemInfoType_UsingPhysicalMemorySize)
                        {
                            *pOut =
                                Kernel::GetKernelHeapManager().GetSize(static_cast<KMemoryManager::Region>(param)) -
                                Kernel::GetKernelHeapManager().GetFreeSize(static_cast<KMemoryManager::Region>(param));
                        }
                        else
                        {
                            NN_KERN_ABORT();
                        }
                        result = ResultSuccess();
                    }
                    break;
                default:
                    {
                        result = nn::svc::ResultInvalidCombination();
                    }
                    break;
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
    case nn::svc::SystemInfoType_InitialProcessIdRange:
        {
            if (nn::svc::Handle(0) == handle)
            {
                switch (param) {
                case 0:
                    {
                        NN_KERN_ABORT_UNLESS(GetInitialProcessIdMin() <= GetInitialProcessIdMax());
                        *pOut = GetInitialProcessIdMin();
                        result = ResultSuccess();
                    }
                    break;
                case 1:
                    {
                        NN_KERN_ABORT_UNLESS(GetInitialProcessIdMin() <= GetInitialProcessIdMax());
                        *pOut = GetInitialProcessIdMax();
                        result = ResultSuccess();
                    }
                    break;
                default:
                    {
                        result = nn::svc::ResultInvalidCombination();
                    }
                    break;
                }
            }
            else
            {
                result = nn::svc::ResultInvalidHandle();
            }
        }
        break;
    default:
        {
            result = nn::svc::ResultInvalidEnum();
        }
        break;
    }
    return result;
};

void SvcKernelDebug(nn::svc::KernelDebugType type, Bit64 param0, Bit64 param1, Bit64 param2)
{
    NN_UNUSED(type);
    NN_UNUSED(param0);
    NN_UNUSED(param1);
    NN_UNUSED(param2);
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return;
    }
    if (!KTargetSystem::IsInternalDebugEnabled())
    {
        return;
    }
#if defined NN_KERN_FOR_DEVELOPMENT
    switch (type)
    {
    case nn::svc::KernelDebugType_Thread:
        {
            if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpThread();
            }
            else
            {
                KDumpObject::DumpThread(param0);
            }
        }
        break;
    case nn::svc::KernelDebugType_ThreadCallstack:
        {
            if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpThreadCallstack();
            }
            else
            {
                KDumpObject::DumpThreadCallstack(param0);
            }
        }
        break;
    case nn::svc::KernelDebugType_Process:
        {
            if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpProcess();
            }
            else
            {
                KDumpObject::DumpProcess(param0);
            }
        }
        break;
    case nn::svc::KernelDebugType_Port:
        {
            if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpPort();
            }
            else
            {
                KDumpObject::DumpPort(param0);
            }
        }
        break;
    case nn::svc::KernelDebugType_Handle:
        {
            if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpHandle();
            }
            else
            {
                KDumpObject::DumpHandle(param0);
            }
        }
        break;
    case nn::svc::KernelDebugType_KernelObject:
        {
            KDumpObject::DumpKernelObject();
        }
        break;
    case nn::svc::KernelDebugType_Memory:
        {
            if (param0 == static_cast<Bit64>(-2ll))
            {
                KDumpObject::DumpKernelMemory();
            }
            else if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpMemory();
            }
            else
            {
                KDumpObject::DumpMemory(param0);
            }
        }
        break;

#ifdef NN_KERN_ENABLE_DUMP_PAGETABLE
    case nn::svc::KernelDebugType_PageTable:
        {
            if (param0 == static_cast<Bit64>(-2ll))
            {
                KDumpObject::DumpKernelPageTable();
            }
            else if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpPageTable();
            }
            else
            {
                KDumpObject::DumpPageTable(param0);
            }
        }
        break;
#endif

    case nn::svc::KernelDebugType_CpuUtilization:
        {
            int oldPri = GetCurrentThread().GetBasePriority();
            GetCurrentThread().SetPriority(3);

            if (param0 == static_cast<Bit64>(-1ll))
            {
                KDumpObject::DumpCpuUtilization();
            }
            else if (param0 == static_cast<Bit64>(-2ll))
            {
                KDumpObject::DumpKernelCpuUtilization();
            }
            else
            {
                KDumpObject::DumpCpuUtilization(param0);
            }
            GetCurrentThread().SetPriority(oldPri);
        }
        break;

    case nn::svc::KernelDebugType_SuspendProcess:
        {
            KProcess* pProcess = KProcess::GetProcessFromId(param0);
            if (pProcess)
            {
                KScopedAutoObject<KProcess> autoCloser(pProcess);
                if (pProcess->SetActivity(nn::svc::ProcessActivity_Paused).IsSuccess())
                {
                    NN_KERN_RELEASE_LOG("Suspend Process ID=%3lld\n", pProcess->GetId());
                }
            }
        }
        break;
    case nn::svc::KernelDebugType_ResumeProcess:
        {
            KProcess* pProcess = KProcess::GetProcessFromId(param0);
            if (pProcess)
            {
                KScopedAutoObject<KProcess> autoCloser(pProcess);
                if (pProcess->SetActivity(nn::svc::ProcessActivity_Runnable).IsSuccess())
                {
                    NN_KERN_RELEASE_LOG("Resume Process ID=%3lld\n", pProcess->GetId());
                }
            }
        }
        break;

#if defined NN_KERN_ENABLE_KTRACE
    case nn::svc::KernelDebugType_KTraceSetTypeFilter:
        {
            KTrace::SetTypeFilter(param0);
        }
        break;

    case nn::svc::KernelDebugType_KTraceSetProcessFilter0:
        {
            KTrace::SetProcessFilter(0, param0);
        }
        break;

    case nn::svc::KernelDebugType_KTraceSetProcessFilter1:
        {
            KTrace::SetProcessFilter(1, param0);
        }
        break;

    case nn::svc::KernelDebugType_KTraceInfo:
        {
            KTrace::Info();
        }
        break;
#endif

    default:
        break;
    }
#endif
}

}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcGetInfo32(Bit64* pOut, nn::svc::InfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result = SvcGetInfo(pOut, type, handle, param);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetSystemInfo32(Bit64* pOut, nn::svc::SystemInfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result = SvcGetSystemInfo(pOut, type, handle, param);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcKernelDebug32(nn::svc::KernelDebugType type, Bit64 param0, Bit64 param1, Bit64 param2)
{
    SvcKernelDebug(type, param0, param1, param2);
    ClearSvcOutRegisters();
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcGetInfo64(Bit64* pOut, nn::svc::InfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result = SvcGetInfo(pOut, type, handle, param);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetSystemInfo64(Bit64* pOut, nn::svc::SystemInfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result = SvcGetSystemInfo(pOut, type, handle, param);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcKernelDebug64(nn::svc::KernelDebugType type, Bit64 param0, Bit64 param1, Bit64 param2)
{
    SvcKernelDebug(type, param0, param1, param2);
    ClearSvcOutRegisters();
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcGetInfo64From32(Bit64* pOut, nn::svc::InfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result = SvcGetInfo(pOut, type, handle, param);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetSystemInfo64From32(Bit64* pOut, nn::svc::SystemInfoType type, nn::svc::Handle handle, Bit64 param)
{
    Result result = SvcGetSystemInfo(pOut, type, handle, param);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcKernelDebug64From32(nn::svc::KernelDebugType type, Bit64 param0, Bit64 param1, Bit64 param2)
{
    SvcKernelDebug(type, param0, param1, param2);
    ClearSvcOutRegisters();
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
