﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_KEvent.h"
#include "../kern_KReadableEvent.h"
#include "../kern_KWritableEvent.h"
#include "../kern_KProcess.h"
#include "../kern_KHandleTable.h"
#include "kern_SvcHandlers.h"
#include "../kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern { namespace svc {
namespace {

Result SvcCreateEvent(nn::svc::Handle* pWriteHandle, nn::svc::Handle* pReadHandle)
{
    Result result;

    KProcess& process = GetCurrentProcess();
    KHandleTable& handleTable = process.GetHandleTable();

    KScopedResourceLimitTester eventTester(&process, nn::svc::LimitableResource_EventCountMax);
    if (eventTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    KEvent* pEvent = KEvent::Create();
    if (pEvent)
    {
        pEvent->Initialize();

        eventTester.Accepted();

        result = KEvent::Register(pEvent);

        if (result.IsSuccess())
        {
            result = handleTable.Add(pWriteHandle, &pEvent->GetWritableEvent());
            if (result.IsSuccess())
            {
                result = handleTable.Add(pReadHandle, &pEvent->GetReadableEvent());
                if (result.IsFailure())
                {
                    handleTable.Remove(*pWriteHandle);
                }
            }
        }

        // 失敗した場合でも、WritableEvent と ReadableEvent を Close()することで、KEvent を Closeさせる
        // 通常シーケンスでは、ハンドルテーブルでAddした分 Close() する
        pEvent->GetWritableEvent().Close();
        pEvent->GetReadableEvent().Close();
    }
    else
    {
        result = nn::svc::ResultOutOfResource();
    }

    return result;
}

Result SvcSignalEvent(nn::svc::Handle event)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KWritableEvent* pWritableEvent = handleTable.GetObject<KWritableEvent>(event);
    if (pWritableEvent)
    {
        KScopedAutoObject<KWritableEvent> autoCloser(pWritableEvent);
        return pWritableEvent->Signal();
    }

    return nn::svc::ResultInvalidHandle();
}

Result SvcClearEvent(nn::svc::Handle event)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KWritableEvent* pWritableEvent = handleTable.GetObject<KWritableEvent>(event);
    if (pWritableEvent)
    {
        KScopedAutoObject<KWritableEvent> autoCloser(pWritableEvent);
        return pWritableEvent->Clear();
    }

    KReadableEvent* pReadableEvent = handleTable.GetObject<KReadableEvent>(event);
    if (pReadableEvent)
    {
        KScopedAutoObject<KReadableEvent> autoCloser(pReadableEvent);
        return pReadableEvent->Clear();
    }

    return nn::svc::ResultInvalidHandle();
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateEvent32(nn::svc::Handle* pWriteHandle, nn::svc::Handle* pReadHandle)
{
    Result result = SvcCreateEvent(pWriteHandle, pReadHandle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSignalEvent32(nn::svc::Handle event)
{
    Result result = SvcSignalEvent(event);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcClearEvent32(nn::svc::Handle event)
{
    Result result = SvcClearEvent(event);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateEvent64(nn::svc::Handle* pWriteHandle, nn::svc::Handle* pReadHandle)
{
    Result result = SvcCreateEvent(pWriteHandle, pReadHandle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSignalEvent64(nn::svc::Handle event)
{
    Result result = SvcSignalEvent(event);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcClearEvent64(nn::svc::Handle event)
{
    Result result = SvcClearEvent(event);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreateEvent64From32(nn::svc::Handle* pWriteHandle, nn::svc::Handle* pReadHandle)
{
    Result result = SvcCreateEvent(pWriteHandle, pReadHandle);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSignalEvent64From32(nn::svc::Handle event)
{
    Result result = SvcSignalEvent(static_cast<nn::svc::Handle>(event));
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcClearEvent64From32(nn::svc::Handle event)
{
    Result result = SvcClearEvent(static_cast<nn::svc::Handle>(event));
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

}}}
